/* hp.c - support for HP Powertrust UPS

   Copyright (C) 2002  Richard Muratti <rick@ccoz.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/*
I have two HP Powertrust units
A2997A 1800VA Rack Mount
A2941A  600VA Floor Standing

Here's what i have discovered about these units.

+--------+-----+------+------+
|Cables  | UPS | DB25 | DB9* |
|--------+-----+------+------+
|Receive |  1  |  3   |  3   |
|Transmit|  2  |  2   |  2   |
|Ground  |  9  |  7   |  5   |
|--------+-----+------+------+

*) Cable used for A2994A; RX and TX appear to be exchanged

Comms
1200N81

Commands
CAPS
	Online  Offline Comment
	
A	$	!	Status % Low Batt
B	066.7	066.7	Battery Voltage
C	027.0	027.0	Temp
D	??	??
E	??	??
F	050.0	050.0	Frequency
G	??	??
H	??	??
I	??	??	
G	??	??
K	>	<	KnnnKnnn Kill UPS after nnn seconds (ie dont restart)
L	233.0	000.0	Line Voltage
M	??	??
N	NM	NM	[CR] gives CM response
O	231.0	231.0	Output Voltage
P	ddd.d	ddd.d	unsupported/future values
Q	<	<	
R	ddd.d	ddd.d	unsupported/future values
S	<	>	SnnnSnnn Shutdown UPS after nnn is seconds
T	??	??
U	>	>	Self test 10 Sec
V	1.00	1.00	Firmware Version
W	HP^...  HP^...	ID String + Capablilities
X	>	>
Y	??	??
Z	>	<	ZnnnZnnn does something but what???


Notes:
On power up UPS will send '('
On power good UPS will send '$' ie after going off batt
On power fail UPS will send '!'
On low batt UPS will send '%'

It seems that they are sent only once
so its best to poll for them using 'A'

Shutdown/kill only work if UPS is on battery.
To force set UPS in test mode 'U' then issue
SnnnSnnn or KnnnKnnn

UPS sends ')' on successful shutdown



TODO:
What do P & R values represent.
What does ZnnnZnnn do.
What does N do.


if you can add to the above please feel fee to do so
but email me with any changes.

Cheers
Rick
*/


#include "main.h"
#include <sys/ioctl.h>

#define ENDCHAR 13
#define IGNCHARS  "\n"

#define NUM_MODELS  sizeof(modeltab)/sizeof(modeltab[0])
#define SD_RESTART  0
#define SD_HALT     1


/* used external variables */
extern int debug_level;      /* debug level, set by "-D" in main.c */
extern int sddelay;          /* shutdown delay, set by "-d $delay" in main.c */
extern int do_forceshutdown; /* shutdown delay, set by "-k" in main.c */



struct {
	char	*model;
	char	*id_string;
}	modeltab[] =
{
	{ "A2997A","HP^V230^F50^R01800^O230^P1"},
	{ "A2941A","HP^V230^F50^R00600^O230^P1"},
	{ "A2994A","HP^V230^F50^R01300^O230^P1"}
	/* 
	 Add More Models Here
	 Some Models may have more than one id string
	 ie 110 volt countries
	*/
};

int	sdtype;


/* TODO: roll this into upscommon ala bestups */
void sendstring(char * string, int len, int delay)
{
	int	i;

	for (i=0; (i<len); i++) {
	        upssendchar(string[i]);
		usleep(delay);
	}
}


void instcmd (int auxcmd, int dlen, char *data)
{

	/* TODO: reply to upsd? */

	switch (auxcmd) {
		/* Special cases */
		case CMD_BTEST0:	/* Stop Battery Test*/
		        
			break;
		case CMD_BTEST1:	/* Start Battery Test*/
			upssendchar ('U'); 
			break;
		default:
			upslogx(LOG_INFO,"instcmd: unknown or unimplemented type 0x%04x\n",auxcmd);
			return;
	}
}



void setuphandlers()
{
	upsh.instcmd = instcmd;
}



void init_serial(void)
{
        int     clr_bit = TIOCM_DTR | TIOCM_RTS;
        ioctl(upsfd, TIOCMBIC, &clr_bit);
}

/* replacement for the old "installinfo" hack-job */
static void poll_data(const char *var, char reqchar)
{
        char    tmp[SMALLBUF];

        upssendchar(reqchar);
        upsrecv(tmp, sizeof(tmp), ENDCHAR, IGNCHARS);

        dstate_setinfo(var, "%s", tmp);
}

void ups_model (void)
{
	int i;

        /* Get Ident String	*/
	poll_data("ups.id", 'W');


        /* Lookup UPS Model */
        for (i=0; i < NUM_MODELS;i++) {
           if (strncmp (dstate_getinfo("ups.id"),modeltab[i].id_string,strlen(modeltab[i].id_string)) == 0)
	      dstate_setinfo("ups.model", "%s", modeltab[i].model);
	}
}

void upsdrv_initinfo(void)
{
	dstate_setinfo("ups.mfr", "HP");

	dstate_addcmd("battery.test.start");
	dstate_addcmd("battery.test.stop");
	
	ups_model ();

	printf("Detected %s [%s] on %s\n", dstate_getinfo("ups.mfr"),
		dstate_getinfo("ups.model"), device_path);

	setuphandlers();

}

void upsdrv_updateinfo(void)
{
	unsigned char recBuf[10];


	poll_data("input.voltage", 'L');
	poll_data("output.voltage", 'O');
	poll_data("input.frequency", 'F');
	poll_data("ups.temperature", 'C');
	poll_data("battery.voltage", 'B');

        upssendchar ('A');
	upsrecv((char *)recBuf,sizeof(recBuf)-1,ENDCHAR,IGNCHARS);

	status_init();		

        switch (recBuf[0]) {

		case '%':		/* on battery, low battery */
			status_set("LB");

			/* FALLTHROUGH */

		case '!':		/* on battery */
			status_set("OB");
			break;
		
		case '$':		/* on line */
			status_set("OL");
			break;

		default:
			upslogx(LOG_INFO,"unknown UPS status byte 0x%04x\n",
				recBuf[0]);

			return;		/* don't commit */
	}

	status_commit();
}

void upsdrv_shutdown(void)
{
	char  tmp [10];


        /* Put UPS in Self Test Mode */
        upssendchar ('U');
	upsrecvchars(tmp,1);
        usleep(100);


        /* Issue Shutdown/Kill Command */
	if (sdtype == SD_RESTART) {
	   printf("UPS shutdown in '%d' seconds.\nUPS will RESTART on AC Power Good\n", sddelay);
  	   snprintf (tmp, sizeof(tmp), "S%03dS%03d",sddelay,sddelay);
	}else{
	   printf("UPS shutdown in '%d' seconds.\nUPS will NOT RESTART on AC Power Good\n", sddelay);
	   snprintf (tmp, sizeof(tmp), "K%03dK%03d",sddelay,sddelay);   
	}
		   
        sendstring (tmp,9,10);
}

void upsdrv_help(void)
{
	printf("\nShutdown types:\n");
	printf("  restart: UPS Will shutdown and will RESTART on AC Power Good (default)\n");
	printf("  halt:    UPS Will shutdown and will NOT RESTART on AC Power Good\n");

}

/* list flags and values that you want to receive via -x */
void upsdrv_makevartable(void)
{
	/* allow '-x xyzzy' */
	/* addvar(VAR_FLAG, "xyzzy", "Enable xyzzy mode"); */

	/* allow '-x foo=<some value>' */
	
	addvar(VAR_VALUE, "shutdown", "halt/<restart>");
}

void upsdrv_banner(void)
{
	printf("Network UPS Tools - HP Powertrust UPS driver 0.01 (%s)\n", UPS_VERSION);
}

void upsdrv_initups(void)
{
	/* this driver doesn't do sanity checks in upsdrv_updateinfo */
	broken_driver = 1;
	return;

	open_serial (device_path,B1200);
	init_serial();
	
	
         if (NULL == getval("shutdown"))
	    sdtype = SD_RESTART;
	 else
	    if (strcmp (getval("shutdown"),"halt") == 0)
	       sdtype = SD_HALT;

}

void upsdrv_cleanup(void)
{
}
