/*!
 * @file libhid.c
 * @brief HID Library - User API (Generic HID Access using MGE HIDParser)
 *
 * @author Copyright (C) 2003
 *	Arnaud Quette <arnaud.quette@free.fr> && <arnaud.quette@mgeups.com>
 *	Philippe Marzouk <philm@users.sourceforge.net> (dump_hex())
 *
 * This program is sponsored by MGE UPS SYSTEMS - opensource.mgeups.com
 *
 *      The logic of this file is ripped from mge-shut driver (also from
 *      Arnaud Quette), which is a "HID over serial link" UPS driver for
 *      Network UPS Tools <http://www.networkupstools.org/>
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * -------------------------------------------------------------------------- */

#include <stdio.h>
#include <string.h>
/* #include <math.h> */
#include "hidparser.h"
#include "hidtypes.h"
#include "libhid.h"

#include "hid-usb.h"

HIDDevice curDevice;

static HIDData   	hData;
static HIDParser 	hParser;

unsigned char raw_buf[100];
int replen; /* size of the last report retrieved */
unsigned char ReportDesc[4096];

#define MAX_REPORT_SIZE         0x1800

/* TODO: rework all that */
extern void upsdebugx(int level, const char *fmt, ...);
#define TRACE upsdebugx

/* Units and exponents table (HID PDC, 3.2.3) */
#define NB_HID_UNITS 10
const long HIDUnits[NB_HID_UNITS][2]=
{
  {0x00000000,0}, /* None */
  {0x00F0D121,7}, /* Voltage */
  {0x00100001,0}, /* Ampere */
  {0x0000D121,7}, /* VA */
  {0x0000D121,7}, /* Watts */
  {0x00001001,0}, /* second */
  {0x00010001,0}, /* K */
  {0x00000000,0}, /* percent */
  {0x0000F001,0}, /* Hertz */
  {0x00101001,0}, /* As */
};

/* support functions */
void logical_to_physical(HIDData *Data);
void physical_to_logical(HIDData *Data);
int hid_lookup_usage(char *name);
ushort lookup_path(const char *HIDpath, HIDData *data);
void dump_hex (const char *msg, const unsigned char *buf, int len);
long get_unit_expo(long UnitType);
float expo(int a, int b);


HIDDevice *HIDOpenDevice(const char *port, MatchFlags *flg)
{
  int ReportSize;

  /* Init structure */
  curDevice.Name = NULL;
  curDevice.Vendor = NULL;
  curDevice.VendorID = -1;
  curDevice.Product = NULL;
  curDevice.ProductID = -1;
  curDevice.Serial = NULL;
  curDevice.Application = -1;
  curDevice.fd = -1;

  /* get and parse descriptors (dev, cfg and report) */
  if ((ReportSize = libusb_open(&curDevice, flg, ReportDesc)) == -1) {
    return NULL;
  }
  else {
    TRACE(2, "Report Descriptor size = %d", ReportSize);
    dump_hex ("Report Descriptor", ReportDesc, 200);

    /* HID Parser Init */
    ResetParser(&hParser);
    hParser.ReportDescSize = ReportSize;
    memcpy(hParser.ReportDesc, ReportDesc, ReportSize);
    HIDParse(&hParser, &hData);

    /* CHeck for matching UsageCode (1rst collection == application == type of device) */
    if ((hData.Path.Node[0].UPage == ((flg->UsageCode & 0xFFFF0000) / 0x10000))
	&& (hData.Path.Node[0].Usage == (flg->UsageCode & 0x0000FFFF)))
      TRACE(2, "Found a device matching UsageCode (0x%08x)", flg->UsageCode);
    else {
      TRACE(2, "Found a device, but not matching UsageCode (0x%08x)", flg->UsageCode);
      return NULL;
    }
  }
  return &curDevice;
}

HIDItem *HIDGetItem(const char *ItemPath)
{
  return NULL;
}

float HIDGetItemValue(const char *path)
{
  int i, retcode;
  float Value;

  /* Prepare path of HID object */
  hData.Type = ITEM_FEATURE;
  hData.ReportID = 0;

  if((retcode = lookup_path(path, &hData)) > 0) {
    TRACE(2, "Path depth = %i", retcode);
    
    for (i = 0; i<retcode; i++)
      TRACE(2, "%i: UPage(%x), Usage(%x)", i,
		hData.Path.Node[i].UPage,
		hData.Path.Node[i].Usage);
    
    hData.Path.Size = retcode;

    /* Get info on object (reportID, offset and size) */
    if (FindObject(&hParser,&hData) == 1) {
      /* Get report with data */
      /* if ((replen=libusb_get_report(hData.ReportID, 
	 raw_buf, MAX_REPORT_SIZE)) > 0) { => doesn't work! */
      if ((replen=libusb_get_report(hData.ReportID, raw_buf, 10)) > 0) {
	/* Extract the data value */
	GetValue((const unsigned char *) raw_buf, &hData);

	TRACE(2, "=>> Before exponent: %ld, %i/%i)", hData.Value,
	      (int)hData.UnitExp, (int)get_unit_expo(hData.Unit) );

	/* Convert Logical Min, Max and Value in Physical */
	/* logical_to_physical(&hData); */

	Value = hData.Value;

	/* Process exponents */
	/* Value*=(float) pow(10,(int)hData.UnitExp - get_unit_expo(hData.Unit)); */
	Value*=(float) expo(10,(int)hData.UnitExp - get_unit_expo(hData.Unit));
	hData.Value = (long) Value;

	/* Convert Logical Min, Max and Value into Physical */
	logical_to_physical(&hData);

	dump_hex ("Report ", raw_buf, replen);

	return Value;
      }
      else
	TRACE(2, "Can't retrieve Report %i", hData.ReportID);
//exit(-1);
    }
    else
      TRACE(2, "Can't find object %s", path);
  }
  return -2; /* TODO: should be checked */
}

char *HIDGetItemString(const char *path)
{
  int i, retcode;
  
  /* Prepare path of HID object */
  hData.Type = ITEM_FEATURE;
  hData.ReportID = 0;
  
  if((retcode = lookup_path(path, &hData)) > 0) {
    TRACE(2, "Path depth = %i", retcode);
    
    for (i = 0; i<retcode; i++)
      TRACE(2, "%i: UPage(%x), Usage(%x)", i,
		hData.Path.Node[i].UPage,
		hData.Path.Node[i].Usage);
    
    hData.Path.Size = retcode;
    
    /* Get info on object (reportID, offset and size) */
    if (FindObject(&hParser,&hData) == 1) {
      if (libusb_get_report(hData.ReportID, raw_buf, 8) > 0) { /* MAX_REPORT_SIZE) > 0) { */
	GetValue((const unsigned char *) raw_buf, &hData);

	/* now get string */
	libusb_get_string(hData.Value, raw_buf);
	return raw_buf;
      }
      else
	TRACE(2, "Can't retrieve Report %i", hData.ReportID);
    }
    else
      TRACE(2, "Can't find object %s", path);

    return NULL;
  }
  return NULL;
}
 
bool HIDSetItemValue(const char *path, float value)
{
  float Value;

  /* Begin by a standard Get to fill in com structures ... */
  Value = HIDGetItemValue(path);

  /* ... And play with global vars */
  if (Value != -2) { /* Get succeed */

    TRACE(2, "=>> SET: Before set: %.2f (%ld)", Value, (long)value);

    /* Test if Item is settable */
    if (hData.Attribute != ATTR_DATA_CST) {
      /* Set new value for this item */
      /* And Process exponents restoration */
      Value = value * expo(10, get_unit_expo(hData.Unit) - (int)hData.UnitExp);

      hData.Value=(long) Value;

      TRACE(2, "=>> SET: after exp: %ld/%.2f (exp = %.2f)", hData.Value, Value,
	     expo(10, (int)get_unit_expo(hData.Unit) - (int)hData.UnitExp));

      /* Convert Physical Min, Max and Value in Logical */
      physical_to_logical(&hData);
      TRACE(2, "=>> SET: after PL: %ld", hData.Value);

      SetValue(&hData, raw_buf);

      dump_hex ("==> Report after setvalue", raw_buf, replen);

      if (libusb_set_report(hData.ReportID, raw_buf, replen) > 0) {
	TRACE(2, "Set report succeeded");
	return TRUE;
      } else {
	TRACE(2, "Set report failed");
	return FALSE;
      }
      /* check if set succeed! => doesn't work on *Delay (decremented!) */
      /*      Value = HIDGetItemValue(path);
      
      TRACE(2, "=>> SET: new value = %.2f (was set to %.2f)\n", 
      Value, (float) value);
      return TRUE;*/ /* (Value == value); */
    }
  }
  return FALSE;
}
HIDItem *HIDGetNextEvent(HIDDevice *dev)
{
  /*  unsigned char buf[20];

  upsdebugx(1, "Waiting for notifications\n");*/

  /* TODO: To be written */
  /* needs libusb-0.1.8 to work => use ifdef and autoconf */
  /* libusb_get_interrupt(&buf[0], 20, 5000); */
  return NULL;
}
void HIDCloseDevice(HIDDevice *dev)
{
  libusb_close(&curDevice);
}


/*******************************************************
 * support functions
 *******************************************************/

#define MAX_STRING      		64

void logical_to_physical(HIDData *Data)
{
  if(Data->PhyMax - Data->PhyMin > 0)
    {
      float Factor = (float)(Data->PhyMax - Data->PhyMin) / (Data->LogMax - Data->LogMin);
      /* Convert Value */
      Data->Value=(long)((Data->Value - Data->LogMin) * Factor) + Data->PhyMin;

      if(Data->Value > Data->PhyMax)
	Data->Value |= ~Data->PhyMax;
    }
  else /* => nothing to do!? */
    {
      /* Value.m_Value=(long)(pConvPrm->HValue); */
      if(Data->Value > Data->LogMax)
	Data->Value |= ~Data->LogMax;
    }
  
  /* if(Data->Value > Data->Value.m_Max)
    Value.m_Value |= ~Value.m_Max;
  */
}

void physical_to_logical(HIDData *Data)
{
  TRACE(2, "PhyMax = %ld, PhyMin = %ld, LogMax = %ld, LogMin = %ld",
	Data->PhyMax, Data->PhyMin, Data->LogMax, Data->LogMin);

  if(Data->PhyMax - Data->PhyMin > 0)
    {
      float Factor=(float)(Data->LogMax - Data->LogMin) / (Data->PhyMax - Data->PhyMin);
      /* Convert Value */
      Data->Value=(long)((Data->Value - Data->PhyMin) * Factor) + Data->LogMin;
    }
    /* else => nothing to do!?
       {
       m_ConverterTab[iTab].HValue=m_ConverterTab[iTab].DValue;
       } */
}

long get_unit_expo(long UnitType)
{
  int i = 0, exp = -1;

  while (i < NB_HID_UNITS) {
    if (HIDUnits[i][0] == UnitType) {
      exp = HIDUnits[i][1];
      break;
    }
    i++;
  }
  return exp;
}

/* exponent function: return a^b */
/* TODO: check if needed to replace libmath->pow */
float expo(int a, int b)
{
  if (b==0)
    return (float) 1;
  if (b>0)
    return (float) a * expo(a,b-1);
  if (b<0)
    return (float)((float)(1/(float)a) * (float) expo(a,b+1));
  
  /* not reached */
  return -1;
}

/* translate HID string path to numeric path and return path depth */
/* TODO: use usbutils functions (need to be externalised!) */
ushort lookup_path(const char *HIDpath, HIDData *data)
{
  ushort i = 0, cond = 1;
  int cur_usage;
  char buf[MAX_STRING];
  char *start, *end; 
  
  strncpy(buf, HIDpath, strlen(HIDpath));
  buf[strlen(HIDpath)] = '\0';
  start = end = buf;
  
  TRACE(2, "entering lookup_path(%s)", buf);
  
  while (cond) {
    
    if ((end = strchr(start, '.')) == NULL) {
      cond = 0;			
    }
    else
      *end = '\0';
    
    TRACE(2, "parsing %s", start);
    
    /* lookup code */
    if ((cur_usage = hid_lookup_usage(start)) == -1) {
      TRACE(2, "%s wasn't found", start);
      return 0;
    }
    else {
      data->Path.Node[i].UPage = (cur_usage & 0xFFFF0000) / 0x10000;
      data->Path.Node[i].Usage = cur_usage & 0x0000FFFF; 
      i++; 
    }
    
    if(cond)
      start = end +1 ;
  }
  data->Path.Size = i;
  return i;
}

/* Lookup this usage name to find its code (page + index) */
/* temporary usage code lookup */
typedef struct {
	const char *usage_name;
	int usage_code;
} usage_lkp_t;

static usage_lkp_t usage_lkp[] = {
	/* Power Device Page */
	{  "PresentStatus", 0x00840002 },
	{  "UPS", 0x00840004 },
	{  "BatterySystem", 0x00840010 },
	{  "Battery", 0x00840012 },
	{  "BatteryID", 0x00840013 },	
	{  "PowerConverter", 0x00840016 },
	{  "OutletSystem", 0x00840018 },
	{  "Input", 0x0084001a },
	{  "Output", 0x0084001c },
	{  "Outlet", 0x00840020 },
	{  "OutletID", 0x00840021 },
	{  "PowerSummary", 0x00840024 },
	{  "Voltage", 0x00840030 },
	{  "Current", 0x00840031 },
	{  "Frequency", 0x00840032 },
	{  "PercentLoad", 0x00840035 },
	{  "Temperature", 0x00840036 },
	{  "ConfigVoltage", 0x00840040 },
	{  "ConfigCurrent", 0x00840041 },
	{  "ConfigFrequency", 0x00840042 },
	{  "ConfigApparentPower", 0x00840043 },
	{  "LowVoltageTransfer", 0x00840053 },
	{  "HighVoltageTransfer", 0x00840054 },	
	{  "DelayBeforeReboot", 0x00840055 },	
	{  "DelayBeforeStartup", 0x00840056 },
	{  "DelayBeforeShutdown", 0x00840057 },
	{  "Test", 0x00840058 },
	{  "OverLoad", 0x00840065 }, /* mispelled in usb.ids */
	{  "ShutdownImminent", 0x00840069 },
	{  "SwitchOn/Off", 0x0084006b },
	{  "Switchable", 0x0084006c },
	{  "Boost", 0x0084006e },
	{  "Buck", 0x0084006f },
	{  "Flow", 0x0084001e },
	/* Battery System Page */
	{  "RemainingCapacityLimit", 0x00850029 },
	{  "BelowRemainingCapacityLimit", 0x00850042 },
	{  "RemainingCapacity", 0x00850066 },
	{  "RunTimeToEmpty", 0x00850068 },
	{  "ACPresent", 0x008500d0 },
	{  "Charging", 0x00850044 },
	{  "Discharging", 0x00850045 },
	{  "NeedReplacement", 0x0085004b },
/* TODO: per MFR specific usages */
	/* MGE UPS SYSTEMS Page */
	{  "iModel", 0xffff00f0 },
	{  "RemainingCapacityLimitSetting", 0xffff004d },
	{  "TestPeriod", 0xffff0045 },
	{  "LowVoltageBoostTransfer", 0xffff0050 },
	{  "HighVoltageBoostTransfer", 0xffff0051 },
	{  "LowVoltageBuckTransfer", 0xffff0052 },
	{  "HighVoltageBuckTransfer", 0xffff0053 },
	/* end of structure. */
	{  "\0", 0x0 }
};

int hid_lookup_usage(char *name)
{	
  int i;
	
  TRACE(2, "Looking up %s", name);
	
  if (name[0] == '[') /* manage indexed collection */
    return (0x00FF0000 + atoi(&name[1]));
  else {
    for (i = 0; (usage_lkp[i].usage_code != 0x0); i++)
      {
	if (!strcmp(usage_lkp[i].usage_name, name))
	  {
	    TRACE(2, "hid_lookup_usage: found %04x",
		      usage_lkp[i].usage_code);
	    return usage_lkp[i].usage_code;
	  }
      }
  }
  return -1;
}

#define NIBBLE(_i)    (((_i) < 10) ? '0' + (_i) : 'A' + (_i) - 10)

void dump_hex (const char *msg, const unsigned char *buf, int len)
{
  int i;
  int nlocal;
  const unsigned char *pc;
  char *out;
  const unsigned char *start;
  char c;
  char line[100];
  
  start = buf;
  out = line;
  
  for (i = 0, pc = buf, nlocal = len; i < 16; i++, pc++) {
    if (nlocal > 0) {
      c = *pc;
      
      *out++ = NIBBLE ((c >> 4) & 0xF);
      *out++ = NIBBLE (c & 0xF);
      
      nlocal--;
    }
    else {
      *out++ = ' ';
      *out++ = ' ';
    }				/* end else */
    *out++ = ' ';
  }				/* end for */
  *out++ = 0;
  
  TRACE(2, "%s: (%d bytes) => %s", msg, len, line);
  
  buf += 16;
  len -= 16;
} /* end dump */
