/*
 * vim: ts=4 sw=4
 */

/*
 * powernet.h -- part of SNMP interface driver for APC SNMP devices.
 * Dmitry Frolov <frolov@riss-telecom.ru>
 */

#include <unistd.h>
#include <ucd-snmp/ucd-snmp-includes.h>
#include <shared-tables.h> 

#define PN_UPSDRV_VERSION	"1.0"

/* SNMP OIDs set */
#define OID_POWERNET_MIB	".1.3.6.1.4.1.318"
/* info elements */
#define OID_MODEL_NAME		".1.3.6.1.4.1.318.1.1.1.1.1.1.0"
#define OID_UPSIDEN			".1.3.6.1.4.1.318.1.1.1.1.1.2.0"
#define OID_FIRMREV			".1.3.6.1.4.1.318.1.1.1.1.2.1.0"
#define OID_MFRDATE			".1.3.6.1.4.1.318.1.1.1.1.2.2.0"
#define OID_SERIAL			".1.3.6.1.4.1.318.1.1.1.1.2.3.0"
#define OID_BATT_STATUS		".1.3.6.1.4.1.318.1.1.1.2.1.1.0"
#define BATT_UNKNOWN		1	/* unknown -> INFO_STATUS/? */
#define BATT_NORMAL			2	/* batteryNormal -> INFO_STATUS/? */
#define BATT_LOW			3	/* batteryLow -> INFO_STATUS/LB */
#define OID_BATTDATE		".1.3.6.1.4.1.318.1.1.1.2.1.3.0"
#define OID_BATT_CHARGE		".1.3.6.1.4.1.318.1.1.1.2.2.1.0"
#define OID_UPSTEMP			".1.3.6.1.4.1.318.1.1.1.2.2.2.0"
#define OID_BATT_RUNTIME	".1.3.6.1.4.1.318.1.1.1.2.2.3.0"
#define OID_INVOLT			".1.3.6.1.4.1.318.1.1.1.3.2.1.0"
#define OID_INFREQ			".1.3.6.1.4.1.318.1.1.1.3.2.4.0"
#define OID_POWER_STATUS	".1.3.6.1.4.1.318.1.1.1.4.1.1.0"
#define PWR_OTHER			1	/* other -> INFO_STATUS/? */
#define PWR_NORMAL			2	/* normal -> INFO_STATUS/OL */
#define PWR_BATTERY			3	/* battery -> INFO_STATUS/OB */
#define PWR_BOOSTER			4	/* booster -> INFO_STATUS/BOOST */
#define PWR_SLEEPING		5	/* timedSleeping -> INFO_STATUS/OFF */
#define PWR_SOFT_BYPASS		6	/* bypass -> INFO_STATUS/? */
#define PWR_NONE			7	/* none -> INFO_STATUS/OFF */
#define PWR_REBOOTING		8	/* rebooting -> INFO_STATUS/? */
#define PWR_HARD_BYPASS		9	/* bypass -> INFO_STATUS/? */
#define PWR_FAIL_BYPASS		10	/* bypass -> INFO_STATUS/? */
#define PWR_SLEEPING2		11	/* sleepingUntilPowerReturn -> INFO_STATUS/OFF */
#define PWR_REDUCER			12	/* reducer -> INFO_STATUS/TRIM */
#define OID_OUTVOLT			".1.3.6.1.4.1.318.1.1.1.4.2.1.0"
#define OID_LOADPCT			".1.3.6.1.4.1.318.1.1.1.4.2.3.0"
#define OID_HIGHXFER		".1.3.6.1.4.1.318.1.1.1.5.2.2.0"
#define OID_LOWXFER			".1.3.6.1.4.1.318.1.1.1.5.2.3.0"
#define OID_SLFTSTRES		".1.3.6.1.4.1.318.1.1.1.7.2.3.0"
/* XXX can't find appropriate OID for INFO_BATTVOLT. */
/*#define OID_BATT_VOLTAGE	".1.3.6.1.4.1.318.???"*/
/* commands */
#define OID_OFF				".1.3.6.1.4.1.318.1.1.1.6.2.1.0"
#define OFF_DO				2
#define OFF_GRACEFUL		3
#define OID_REBOOT			".1.3.6.1.4.1.318.1.1.1.6.2.2.0"
#define REBOOT_DO			2
#define REBOOT_GRACEFUL		3
#if 0	/* not used. */
#define OID_SLEEP			".1.3.6.1.4.1.318.1.1.1.6.2.3.0"
#define SLEEP_ON			"2"
#define SLEEP_GRACEFUL		"3"
#endif
#define OID_SIMPWF			".1.3.6.1.4.1.318.1.1.1.6.2.4.0"
#define SIMPWF_DO			2
#define OID_FPTEST			".1.3.6.1.4.1.318.1.1.1.6.2.5.0"
#define FPTEST_DO			2
#define OID_ON				".1.3.6.1.4.1.318.1.1.1.6.2.6.0"
#define ON_DO				2
#define OID_BYPASS			".1.3.6.1.4.1.318.1.1.1.6.2.7.0"
#define BYPASS_ON			2
#define BYPASS_OFF			3
#define OID_SELFTEST		".1.3.6.1.4.1.318.1.1.1.7.2.2.0"
#define SELFTEST_DO			2
#define OID_CAL				".1.3.6.1.4.1.318.1.1.1.7.2.5.0"
#define CAL_DO				2
#define CAL_CANCEL			3
#define OID_CAL_RESULTS		".1.3.6.1.4.1.318.1.1.1.7.2.6.0"
#define CAL_OK				1
#define CAL_INVALID			2
#define CAL_INPROGRESS		3
/*#define OID_OUTPUT_TAB	"XXX"*/
#define OID_OUTCURRENT		".1.3.6.1.4.1.318.1.1.1.4.2.4.0"
#define OID_REQOUTVOLT		".1.3.6.1.4.1.318.1.1.1.5.2.1.0"
#define OID_RETCAPACITY		".1.3.6.1.4.1.318.1.1.1.5.2.6.0"
#define OID_CONSERVE		".1.3.6.1.4.1.318.1.1.1.6.1.1.0"
#define CONSERVE_DO			2
#define OID_NEEDREPLBATT	".1.3.6.1.4.1.318.1.1.1.2.2.4.0"
#define RB_NONEED			1
#define RB_NEED				2
#define OID_SENS			".1.3.6.1.4.1.318.1.1.1.5.2.7.0"
#define OID_GRACEDELAY		".1.3.6.1.4.1.318.1.1.1.5.2.10.0"
#define OID_RETDELAY		".1.3.6.1.4.1.318.1.1.1.5.2.9.0"
#define OID_LOBATTIME		".1.3.6.1.4.1.318.1.1.1.5.2.8.0"

#define PN_VAR_COMMUNITY	"community"
#define PN_VAR_SDTYPE		"sdtype"

#define PN_INFOSIZE		128
#define PN_BUFSIZE		32
#define PN_LARGEBUF		256

#define PN_STALE_RETRY	10		/* retry to retrieve stale element */
								/* after this number of iterations. */

/* modes to pn_ups_walk. */
#define PN_WALKMODE_INIT	0
#define PN_WALKMODE_UPDATE	1

/* use explicit booleans */
#ifndef FALSE
typedef enum ebool { FALSE, TRUE } bool;
#else
typedef int bool;
#endif

typedef struct {
	int info_type;				/* INFO_ or CMD_ element */
	int info_flags;				/* flags to set in addinfo */
	int info_len;				/* length of strings if STR, */
								/* cmd value if CMD. */
	char *OID;					/* SNMP OID or NULL */
	char *dfl;					/* default value */
	unsigned long flags;		/* my flags */
} pn_info_t;

#define PN_FLAG_OK			(1 << 0)	/* show element to upsd. */
#define PN_FLAG_STATIC		(1 << 1)	/* retrieve info only once. */
#define PN_FLAG_ABSENT		(1 << 2)	/* data is absent in the device, */
										/* use default value. */
#define PN_FLAG_STALE		(1 << 3)	/* data stale, don't try too often. */

/* status string components */
#define PN_STATUS_PWR		(0 << 8)	/* indicates power status element */
#define PN_STATUS_BATT		(1 << 8)	/* indicates battery status element */
#define PN_STATUS_CAL		(2 << 8)	/* indicates calibration status element */
#define PN_STATUS_RB		(3 << 8)	/* indicates replace battery status element */
#define PN_STATUS_NUM_ELEM	4
#define PN_STATUS_INDEX(t)	(((t) >> 8) & 7)

/* hints for pn_ups_set, applicable only to rw vars */
#define PN_TYPE_INT			(0 << 16)	/* cast to int when setting value */
#define PN_TYPE_STRING		(1 << 16)	/* cast to string */
#define PN_TYPE_TIME		(2 << 16)	/* cast to int */
#define PN_TYPE(t)			((t)->flags & 7 << 16)

#define PN_CMD_MASK			0x2000

void pn_snmp_init(const char *type, const char *host, const char *community);
void pn_snmp_cleanup(void);
struct snmp_pdu *pn_snmp_get(const char *OID);
bool pn_snmp_get_str(const char *OID, char *buf, size_t buf_len);
bool pn_snmp_get_int(const char *OID, long *pval);
bool pn_snmp_set(const char *OID, char type, const char *value);
bool pn_snmp_set_str(const char *OID, const char *value);
bool pn_snmp_set_int(const char *OID, long value);
bool pn_snmp_set_time(const char *OID, long value);
void pn_snmp_perror(struct snmp_session *, int, struct snmp_pdu *,
	const char *fmt, ...);
void pn_startup(void);
void pn_cleanup(void);
void pn_init_instcmds(void);
void pn_setuphandlers(void);
void pn_setinfo(int type, const char *value, int flags, int auxdata);
void pn_status_init(void);
void pn_status_set(pn_info_t *, long value);
void pn_status_commit(void);
pn_info_t *pn_find_info(int type);
struct netvars_t *pn_find_netvar(int type);
struct instcmds_t *pn_find_instcmd(int cmd);
void pn_ups_walk(int mode);
bool pn_ups_get(pn_info_t *pn_info_p);
void pn_ups_set(int type, int data_len, char *data);
void pn_ups_instcmd(int auxcmd, int data_len, char *data);
void pn_shutdown_ups(void);

pn_info_t pn_info[] = {

	/* info elements. */
	{ INFO_MFR, FLAG_STRING, PN_INFOSIZE, NULL, "APC",
		PN_FLAG_STATIC | PN_FLAG_ABSENT | PN_FLAG_OK },
	{ INFO_MODEL, FLAG_STRING, PN_INFOSIZE, OID_MODEL_NAME,
		"Generic Powernet SNMP device", PN_FLAG_STATIC | PN_FLAG_OK },
	{ INFO_SERIAL, FLAG_STRING, PN_INFOSIZE, OID_SERIAL, "",
		PN_FLAG_STATIC | PN_FLAG_OK },
	{ INFO_MFRDATE, FLAG_STRING, PN_INFOSIZE, OID_BATTDATE, "",
		PN_FLAG_OK | PN_FLAG_STATIC },
	{ INFO_UTILITY, 0, 0, OID_INVOLT, "", PN_FLAG_OK },
	{ INFO_BATTPCT, 0, 0, OID_BATT_CHARGE, "", PN_FLAG_OK },
	{ INFO_STATUS, FLAG_STRING, PN_INFOSIZE, OID_POWER_STATUS, "OFF",
		PN_FLAG_OK | PN_STATUS_PWR },
	{ INFO_STATUS, FLAG_STRING, PN_INFOSIZE, OID_BATT_STATUS, "",
		PN_FLAG_OK | PN_STATUS_BATT },
	{ INFO_STATUS, FLAG_STRING, PN_INFOSIZE, OID_CAL_RESULTS, "",
		PN_FLAG_OK | PN_STATUS_CAL },
	{ INFO_STATUS, FLAG_STRING, PN_INFOSIZE, OID_NEEDREPLBATT, "",
		PN_FLAG_OK | PN_STATUS_RB },
	{ INFO_UPSTEMP, 0, 0, OID_UPSTEMP, "", PN_FLAG_OK },
	{ INFO_ACFREQ, 0, 0, OID_INFREQ, "", PN_FLAG_OK },
	{ INFO_LOADPCT, 0, 0, OID_LOADPCT, "", PN_FLAG_OK },
	{ INFO_FIRMREV, 0, 0, OID_FIRMREV, "",
		PN_FLAG_STATIC | PN_FLAG_OK },
	{ INFO_RUNTIME, 0, 0, OID_BATT_RUNTIME, "", PN_TYPE_TIME |  PN_FLAG_OK },
	/* can't find appropriate OID for INFO_BATTVOLT. */
	/*{ INFO_BATTVOLT, 0, 1, OID_BATT_VOLTAGE, "", PN_FLAG_OK },*/
	{ INFO_OUTVOLT, 0, 0, OID_OUTVOLT, "", PN_FLAG_OK },
	{ INFO_UPSIDENT, FLAG_STRING | FLAG_RW, 8, OID_UPSIDEN, "",
		PN_FLAG_OK | PN_FLAG_STATIC | PN_TYPE_STRING },
	{ INFO_BATTDATE, FLAG_STRING | FLAG_RW, 8, OID_MFRDATE, "",
		PN_FLAG_OK | PN_FLAG_STATIC | PN_TYPE_STRING },
	{ INFO_SLFTSTRES, FLAG_STRING, PN_INFOSIZE, OID_SLFTSTRES, "",
		PN_FLAG_OK },
	{ INFO_LOWXFER, FLAG_STRING | FLAG_RW, 3, OID_LOWXFER, "",
		PN_TYPE_INT | PN_FLAG_OK },
	{ INFO_HIGHXFER, FLAG_STRING | FLAG_RW, 3, OID_HIGHXFER, "",
		PN_TYPE_INT | PN_FLAG_OK },
	{ INFO_CURRENT, 0, 0, OID_OUTCURRENT, "", PN_FLAG_OK },
	{ INFO_REQVOLT, FLAG_STRING | FLAG_RW, 3, OID_REQOUTVOLT, "",
		PN_TYPE_INT | PN_FLAG_OK },
	{ INFO_WAKETHRSH, FLAG_STRING | FLAG_RW, 3, OID_RETCAPACITY, "",
		PN_TYPE_INT | PN_FLAG_OK },
	{ INFO_LINESENS, FLAG_STRING | FLAG_RW, 1, OID_SENS, "",
		PN_TYPE_INT | PN_FLAG_OK },
	{ INFO_GRACEDELAY, FLAG_STRING | FLAG_RW, 3, OID_GRACEDELAY, "",
		PN_TYPE_TIME | PN_FLAG_OK },
	{ INFO_WAKEDELAY, FLAG_STRING | FLAG_RW, 3, OID_RETDELAY, "",
		PN_TYPE_TIME | PN_FLAG_OK },
	{ INFO_LOBATTIME, FLAG_STRING | FLAG_RW, 3, OID_LOBATTIME, "",
		PN_TYPE_TIME | PN_FLAG_OK },

	/* instant commands. */
	{ CMD_OFF, 0, OFF_DO, OID_OFF, "", PN_FLAG_OK },
	{ CMD_ON, 0, ON_DO, OID_ON, "", PN_FLAG_OK },
	{ CMD_SHUTDOWN, 0, OFF_GRACEFUL, OID_OFF, "", PN_FLAG_OK },
	{ CMD_SDRET, 0, REBOOT_GRACEFUL, OID_REBOOT, "", PN_FLAG_OK },
	{ CMD_SOFTDOWN, 0, CONSERVE_DO, OID_CONSERVE, "", PN_FLAG_OK },
	{ CMD_SIMPWF, 0, SIMPWF_DO, OID_SIMPWF, "", PN_FLAG_OK },
	{ CMD_FPTEST, 0, FPTEST_DO, OID_FPTEST, "", PN_FLAG_OK },
	{ CMD_BYPASS, 0, BYPASS_ON, OID_BYPASS, "", PN_FLAG_OK },
	{ CMD_BTEST1, 0, SELFTEST_DO, OID_SELFTEST, "", PN_FLAG_OK },
	{ CMD_CAL0, 0, CAL_CANCEL, OID_CAL, "", PN_FLAG_OK },
	{ CMD_CAL1, 0, CAL_DO, OID_CAL, "", PN_FLAG_OK },

	/* end of structure. */
	{ 0, 0, 0, NULL, NULL, 0 }
};

#define NUM_INFO_ITEMS	(sizeof(pn_info)/sizeof(pn_info[0]))
