/* upscommon.c - functions used in more than one model support module

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "upscommon.h"
#include "common.h"

#include <pwd.h>
#include <ctype.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include "timehead.h"
#include <sys/socket.h>
#include <sys/un.h>

#ifdef HAVE_UU_LOCK
#include <libutil.h>
#endif

#include "dstate.h"
#include "extstate.h"
#include "var-map.h"
#include "var-map-init.h"
#include "cmd-map.h"
#include "cmd-map-init.h"

	int	upsfd, upsc_debug = 0, do_lock_port = 1;

	char	*pidfn = NULL;
	char	status_buf[ST_MAX_VALUE_LEN];

	struct	ups_handler	upsh;

#ifdef HAVE_UU_LOCK
	static	char	*upsport = NULL;
#endif

/* need to pick a sane default.  maybe this should be set by open_serial */
	unsigned int upssend_delay = 0;
	char	upssend_endchar = '\0';
	
	int	flag_timeoutfailure = 0;

	struct	sigaction sa;
	sigset_t	upscom_sigmask;

	/* set by handlers for SIGTERM/SIGQUIT and similar signals */
	extern	int	exit_flag;

/* signal handler for SIGALRM when opening a serial port */
void openfail(int sig)
{
	fatal("Fatal error: serial port open timed out");
}

/* try whatever method(s) are available to lock the port for our use */
void lockport(int upsfd, const char *port)
{
	if (do_lock_port == 0) {
		upslogx(LOG_INFO, "Serial port locking disabled.");
		return;
	}

#ifdef HAVE_UU_LOCK
	if (upsport)		/* already locked? */
		return;

	/* save for later in case we need it at shutdown */
	upsport = xstrdup(xbasename(port));

        {
                int res = uu_lock(upsport);
 
                if (res != 0)
                        fatalx("Can't uu_lock %s: %s", upsport,
                               uu_lockerr(res));
        }
#elif defined(HAVE_FLOCK)
        if (flock(upsfd, LOCK_EX | LOCK_NB) != 0)
                fatalx("%s is locked by another process", port);
#elif defined(HAVE_LOCKF)
        lseek(upsfd, 0L, SEEK_SET);
        if (lockf(upsfd, F_TLOCK, 0L))
                fatalx("%s is locked by another process", port);
#endif
}

/* explain in full detail what's wrong when the open fails */
static void open_error(const char *port)
{
	struct	passwd	*user;
	struct	stat	fs;

	/* see if port even exists first - typo check */
	if (stat(port, &fs)) {
		upslogx(LOG_NOTICE, "Can't stat() %s - did you specify a nonexistent /dev file?",
		       port);
		fatal("Unable to open %s", port);
	}

	user = getpwuid(getuid());
	if (user)
		upslogx(LOG_NOTICE, "This program is currently running as %s (UID %d)",
		       user->pw_name, (int)user->pw_uid);

	user = getpwuid(fs.st_uid);
	if (user)
		upslogx(LOG_NOTICE, "%s is owned by user %s (UID %d), mode %04o", port, 
		       user->pw_name, (int)user->pw_uid,
                       (int)(fs.st_mode & 07777));

	upslogx(LOG_NOTICE, "Change the port name, or fix the permissions or ownership"
	       " of %s and try again.", port);

	fatal("Unable to open %s", port);
}

/* open the port, but don't touch the tty details (except CLOCAL and speed) */
void open_serial_simple(const char *port, speed_t speed, int flags)
{
	struct	termios	tio;

	signal(SIGALRM, openfail);
	alarm(3);

	if ((upsfd = open(port, O_RDWR | O_NONBLOCK)) == -1)
		open_error(port);

	alarm(0);

	lockport(upsfd, port);

	tcgetattr(upsfd, &tio);
	tio.c_cflag |= CLOCAL;
	if (speed) {
#ifndef HAVE_CFSETISPEED
#error This system lacks cfsetispeed() and has no other means to set the speed
#endif
		cfsetispeed(&tio, speed);
		cfsetospeed(&tio, speed);
	}
	tcsetattr(upsfd, TCSANOW, &tio);

	/* set "normal" flags - cable power, or whatever */
	if (ioctl(upsfd, TIOCMSET, &flags))
		fatal("ioctl");
}

/* open a serial port and setup the flags properly at a given speed */
void open_serial(const char *port, speed_t speed)
{
	struct	termios	tio;
	int	fcarg;

	/* open port for normal use */

	signal(SIGALRM, openfail);
	alarm(3);

	upsfd = open(port, O_RDWR | O_NOCTTY | O_EXCL | O_NONBLOCK);
	alarm(0);

	if (upsfd < 0)
		open_error(port);

	fcarg = fcntl(upsfd, F_GETFL, 0);
	if (fcarg < 0 || fcntl(upsfd, F_SETFL, fcarg & ~O_NONBLOCK) < 0)
		fatal("Unable to clear O_NONBLOCK 2 %s", port);

	signal(SIGALRM, SIG_IGN);

	lockport(upsfd, port);

	tcgetattr(upsfd, &tio);
	tio.c_cflag = CS8 | CLOCAL | CREAD;
	tio.c_iflag = IGNPAR;
	tio.c_oflag = 0;
	tio.c_lflag = 0;
	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 0;

#ifdef HAVE_CFSETISPEED
	cfsetispeed(&tio, speed);
	cfsetospeed(&tio, speed);
#else
#error This system lacks cfsetispeed() and has no other means to set the speed
#endif

	tcflush(upsfd, TCIFLUSH);
	tcsetattr(upsfd, TCSANOW, &tio);
}

/* function for erasing "timeout"-conditions */
void nolongertimeout(void)
{
	/* if override enabled, then return without changing anything */
	if (flag_timeoutfailure == -1)
		return;

	if (flag_timeoutfailure == 1)
		upslogx(LOG_NOTICE, "Serial port read ok again");
	
	flag_timeoutfailure = 0;
}

/* signal handler for SIGALRM when trying to read */
void timeout(int sig)
{
	sa.sa_handler = SIG_DFL;
	sigemptyset(&upscom_sigmask);
	sa.sa_mask = upscom_sigmask;
	sa.sa_flags = 0;
	sigaction(SIGALRM, &sa, NULL);

	/* if override enabled, then return without changing anything */
	if (flag_timeoutfailure == -1)
		return;

	if (flag_timeoutfailure == 0)
		upslogx(LOG_NOTICE, "Serial port read timed out");
	
	flag_timeoutfailure = 1;
	return;
}

/* wait for an answer in the form <data><endchar> and store in buf */
int upsrecv(char *buf, int buflen, char endchar, const char *ignchars)
{
	char	recvbuf[512], *ptr, in;
	int	ret, cnt;
	struct 	sigaction sa;
	sigset_t upscom_sigmask;

	memset(recvbuf, '\0', sizeof(recvbuf));

	sa.sa_handler = timeout;
	sigemptyset(&upscom_sigmask);
	sa.sa_mask = upscom_sigmask;
	sa.sa_flags = 0;
	sigaction(SIGALRM, &sa, NULL);

	alarm(3);

	ptr = recvbuf;
	*ptr = 0;
	cnt = 0;
	for (;;) {
		ret = read(upsfd, &in, 1);
		if (ret > 0) {

			if (in == endchar) {
				alarm(0);
				signal(SIGALRM, SIG_IGN);
				strlcpy(buf, recvbuf, buflen);
				if (upsc_debug > 0) {
					int i;
					printf("upsrecv: read %d bytes [", cnt);
					for (i = 0; i < cnt; ++i)
						printf(isprint((unsigned char)buf[i]) ? "%c" : "\\%03o",
							   buf[i]);
					printf("]\n");
				}
				return buflen;
			}

			if (strchr(ignchars, in) == NULL) {
				*ptr++ = in;
				*ptr = 0; /* tie off end */
				cnt++;
			}
			
			nolongertimeout();
		}
		else {
			alarm(0);
			signal(SIGALRM, SIG_IGN);
			strlcpy(buf, recvbuf, buflen);
			return -1;
		}

		/* keep from overrunning the buffer - lame hack */
		if (cnt > (sizeof(recvbuf) - 4)) {
			upslogx(LOG_NOTICE, "UPS is spewing wildly");
			return -1;
		}
	}

	return -1;	/* not reached */
}

/* read buflen chars and store in buf */
int upsrecvchars(char *buf, int buflen)
{
	int ret,count;
	char* bufptr;
	struct 	sigaction sa;
	sigset_t upscom_sigmask;

	sa.sa_handler = timeout;
	sigemptyset(&upscom_sigmask);
	sa.sa_mask = upscom_sigmask;
	sa.sa_flags = 0;
	sigaction(SIGALRM, &sa, NULL);

	alarm(3);
	
	bufptr=buf;
	count=buflen;
	while (count>0) {
		ret=read(upsfd,bufptr,count);
		if (ret>0) {
			bufptr+=ret;
			count-=ret;
		} else {
			alarm (0);
			signal (SIGALRM, SIG_IGN);
			return (-1);
		}
	}
	alarm (0);
	signal (SIGALRM, SIG_IGN);
	return buflen;
}

/* send a single byte to the UPS */
int upssendchar(char data)
{
	if (upsc_debug > 0) {
		printf("upssendchar: sending [");
		printf(isprint((unsigned char)data) ? "%c" : "\\%03o", data);
		printf("]\n");
	}

	tcflush(upsfd, TCIFLUSH);
	return(write (upsfd, &data, 1));
}

int upssend(const char *fmt, ...)
{
	char buf[1024], *p;
	int bytes_sent = 0;
	va_list ap;

	va_start(ap, fmt);

	if (vsnprintf(buf, sizeof(buf), fmt, ap) >= sizeof(buf))
		; /* truncated */

	va_end(ap);

	tcflush(upsfd, TCIFLUSH);

	for (p = buf; *p; p++) {
		if (upsc_debug > 0) {
/*			printf ("upssendchar: sending [");
 *			printf (isprint(data) ? "%c" : "\\%03o", data);
 *			printf ("]\n");
 */		}

		if (write(upsfd, p, 1) != 1)
			return -1;
		bytes_sent++;
		usleep(upssend_delay);
	}

	if (upssend_endchar) {
		if (write(upsfd, &upssend_endchar, 1) != 1)
			return -1;
		bytes_sent++;
		usleep(upssend_delay);
	}

	return bytes_sent;
}


/* read in any pending characters.  If expected is set then
 * just drop them, otherwise log an error unles they are
 * in ignore chars
 */
void upsflushin(int expected, int debugit, const char *ignchars)
{
	fd_set readfds;
	struct timeval timeout;
	char logstr[256];
	char * ptr;
	int left, rc;

	ptr = logstr;
	*ptr = 0;
	left = 250;		/* somewhat smaller than buffer */

	FD_ZERO(&readfds);
	FD_SET(upsfd, &readfds);

	timeout.tv_usec = 0;
	timeout.tv_sec = 0;

	while ((rc = select(upsfd+1,
			    &readfds,
			    NULL,
			    NULL,
			    &timeout)) > 0) {
		char in;

		read(upsfd, &in, 1);
		if (debugit) {
			printf ("upsflushin: read char ");
			printf(isprint(in) ? "%c" : "\\%03o",
			       in);
		}
		if ((!expected) &&
		    (strchr(ignchars, in) == NULL)) {
			/* log the excess characters */
			if (isprint(in)) {
				*ptr++ = in;
				*ptr = 0; /* tie of string */
				left--;
			} else {
				sprintf(ptr, "\\%03o", in);
				ptr += 4;
				left--;
			}
			/* check if buffer full - if so moan */
			if (left <= 0) {
				upslogx(LOG_INFO,
					"Unexpected UPS jabber [%s]",
					logstr);
				ptr = logstr;
				left = 250;
			}
		}
	}

	/* check for error returns from select */
	if (rc != 0) {
		upslogx(LOG_NOTICE, "select() on input flush returned error");
	}

	if (ptr != logstr) {
		upslogx(LOG_INFO,
			"Unexpected UPS output [%s]",
			logstr);
	}

	return;
}

/* these functions are legacy entry points for the existing drivers
 * and will disappear once the drivers all use the dstate_() functions
 * directly.. */

static void warn_old(const char *func)
{
	static	int	warned = 0;

	if (warned)
		return;

	upslogx(LOG_NOTICE, "driver uses old function %s", func);

	warned = 1;
}

static const char *oldvar_to_new(int type)
{
	int	i;
	static	char	temphex[8];

	for (i = 0; var_map[i].type != 0; i++)
		if (var_map[i].type == type)
			return var_map[i].name;

	/* not there - convert to the hex hack for now */
	snprintf(temphex, sizeof(temphex), "%04X", type);
	return temphex;
}

static const char *oldcmd_to_new(int cmd)
{
	int	i;
	static	char	temphex[8];

	for (i = 0; cmd_map[i].cmd != 0; i++)
		if (cmd_map[i].cmd == cmd)
			return cmd_map[i].name;

	/* not there - convert to the hex hack for now */
	snprintf(temphex, sizeof(temphex), "%04X", cmd);
	return temphex;
}

/* store data into the array */
void setinfo(int infotype, const char *fmt, ...)
{
	va_list ap;
	const	char	*varname;
	char	value[SMALLBUF];

	warn_old("setinfo");

	va_start(ap, fmt);
	if (vsnprintf(value, sizeof(value), fmt, ap) >= sizeof(value))
		; /* truncated */
	va_end(ap);

	varname = oldvar_to_new(infotype);
	dstate_setinfo(varname, value);
}

/* set a flag on an existing member of the array */
void setflag(int infotype, int newflags)
{
	int	st_flags;
	const	char	*tmpname;

	warn_old("setflag");

	tmpname = oldvar_to_new(infotype);

	st_flags = 0;
	if (newflags & FLAG_RW)
		st_flags |= ST_FLAG_RW;
	if (newflags & FLAG_STRING)
		st_flags |= ST_FLAG_STRING;

	dstate_setflags(tmpname, st_flags);
}

/* find data of a given type in the info array */
const char *getdata(int infotype)
{
	warn_old("getdata");

	return dstate_getinfo(oldvar_to_new(infotype));
}

/* add a member to the info struct */
void addinfo(int type, const char *value, int flags, int auxdata)
{
	int	st_flags;
	const	char	*tmpname;

	warn_old("addinfo");

	if (type == INFO_INSTCMD) {
		tmpname = oldcmd_to_new(auxdata);
		dstate_addcmd(tmpname);
		return;
	}

	/* normal data */
	tmpname = oldvar_to_new(type);
	dstate_setinfo(tmpname, value);

	/* make sure the string maxlen gets set */
	if (flags & FLAG_STRING)
		dstate_setaux(tmpname, auxdata);

	/* get the flags too */
	st_flags = 0;
	if (flags & FLAG_RW)
		st_flags |= ST_FLAG_RW;
	if (flags & FLAG_STRING)
		st_flags |= ST_FLAG_STRING;

	dstate_setflags(tmpname, st_flags);
}

/* add a new ENUM info entry and do other related housekeeping */
void addenum(int basetype, const char *value)
{
	warn_old("addenum");

	dstate_addenum(oldvar_to_new(basetype), value);
}

/* ----- end of old state-managing functions ----- */

void sigdie(int sig)
{
	upslogx(LOG_INFO, "Signal %d: exiting", sig);
	exit_flag = 1;
}

void setup_signals(void)
{
	sigemptyset(&upscom_sigmask);
	sa.sa_mask = upscom_sigmask;
	sa.sa_flags = 0;

	sa.sa_handler = sigdie;
	sigaction(SIGTERM, &sa, NULL);
	sigaction(SIGINT, &sa, NULL);
	sigaction(SIGQUIT, &sa, NULL);

	sa.sa_handler = SIG_IGN;
	sigaction(SIGHUP, &sa, NULL);
	sigaction(SIGPIPE, &sa, NULL);
}	

/* modify in - strip all trailing instances of <sep> */
void rtrim(char *in, char sep)
{
	char	*p = NULL;

	p = &in[strlen(in) - 1];

	while (p >= in) {
		if (*p != sep)
			return;

		*p-- = '\0';
	}
}

/* try to unlock the port using any methods that are available */
void unlockport(int upsfd, const char *port)
{
#ifdef HAVE_UU_LOCK
        int res;
    
        if (!upsport)		/* not already locked? */
		return;

	res = uu_unlock(upsport);

	/* free up allocated memory */
	free(upsport);

	if (res != 0)
		fatalx("Can't uu_unlock %s: %s", upsport, uu_lockerr(res));

#elif defined(HAVE_FLOCK)
        if (flock(upsfd, LOCK_UN) != 0)
                fatalx("can't unlock %s!", port);
#elif defined(HAVE_LOCKF)
        lseek(upsfd, 0L, SEEK_SET);
        if (lockf(upsfd, F_ULOCK, 0L))
                fatalx("can't unlock %s!", port);
#endif
}

/* status management functions - reducing duplication in the drivers */

/* clean out the temp space for a new pass */
void status_init(void)
{
	memset(&status_buf, '\0', sizeof(status_buf));
}

/* add a status element */
void status_set(char *buf)
{
	/* separate with a space if multiple elements are present */
	if (strlen(status_buf) != 0)
		snprintfcat(status_buf, sizeof(status_buf), " %s", buf);
	else
		snprintfcat(status_buf, sizeof(status_buf), "%s", buf);
}

/* write the status_buf into the info array */
void status_commit(void)
{
	dstate_setinfo("ups.status", "%s", status_buf);
}

void exit_cleanup(void)
{
	if (pidfn)
		unlink(pidfn);

	dstate_free();

#ifdef HAVE_UU_LOCK
	uu_unlock(upsport);
#endif
}
