/* oldnet.c - old network command handlers for upsd

   Copyright (C) 2003  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "common.h"
#include "upsd.h"
#include "upstype.h"
#include "user.h"
#include "neterr.h"

#include "shared-tables.h"
#include "shared-tables-init.h"

#include "sstate.h"
#include "extstate.h"
#include "var-map.h"
#include "var-map-init.h"
#include "cmd-map.h"
#include "cmd-map-init.h"

/* everything in here is for compatibility with old clients */

	extern	upstype *firstups;

/* parse varname[@upsname] into separate variables */
upstype *parsearg(const char *arg, char **varname)
{
	char	*argtmp, *upsname;
	upstype	*ups;

	argtmp = xstrdup(arg);

	upsname = strchr(argtmp, '@');
	if (upsname) {
		*upsname++ = '\0';
		ups = findups(upsname);
	} else {
		ups = firstups;
	}

	/* varname may be a subset of argtmp, so copy it and free argtmp */
	*varname = xstrdup(argtmp);
	free(argtmp);

	return ups;
}

static const char *oldvar_to_new(const char *old)
{
	int     i, type = 0;
	static  char    temphex[8];

	/* first look up the type number in netvars */
	for (i = 0; netvars[i].name != NULL; i++)
		if (!strcasecmp(netvars[i].name, old))
			type = netvars[i].type;

	if (!type)
		return NULL;

	for (i = 0; var_map[i].type != 0; i++)
		if (var_map[i].type == type)
			return var_map[i].name;

	/* not there - convert to the hex hack for now */
	snprintf(temphex, sizeof(temphex), "%04X", type);
	return temphex;
}

static const char *oldcmd_to_new(const char *old)
{
	int     i, cmd = 0;
	static  char    temphex[8];

	/* first look up the type number in instcmds[] */
	for (i = 0; instcmds[i].name != NULL; i++)
		if (!strcasecmp(instcmds[i].name, old))
			cmd = instcmds[i].cmd;

	if (!cmd)
		return NULL;

	for (i = 0; cmd_map[i].cmd != 0; i++)
		if (cmd_map[i].cmd == cmd)
			return cmd_map[i].name;

	/* not there - convert to the hex hack for now */
	snprintf(temphex, sizeof(temphex), "%04X", cmd);
	return temphex;
}

/* handler for "REQ" - send a reply */
void do_sendans(ctype *client, char *varin, char *rest)
{
	char	*varname;
	upstype	*ups;

	const	char	*tempvar, *tempval;

	if (varin == NULL) {
		send_err(client, NUT_ERR_MISSING_ARGUMENT);
		return;
	}

	ups = parsearg(varin, &varname);

	if (ups == NULL) {
		send_err(client, NUT_ERR_UNKNOWN_UPS);
		free(varname);
		return;
	}

	/* special case: numlogins is an internal value */
	if (!strcasecmp(varname, "NUMLOGINS")) {
		sendback(client, "ANS %s %d\n", varname, ups->numlogins);
		free(varname);
		return;
	}

	/* make sure this thing is still with us */
	if (!ups_available(ups, client)) {
		free(varname);
		return;
	}

	tempvar = oldvar_to_new(varname);
	free(varname);
	
	/* type wasn't resolved in the netvars[] array */
	if (!tempvar) {
		send_err(client, NUT_ERR_VAR_UNKNOWN);
		return;
	}

	/* make sure this thing is still with us */
	if (!ups_available(ups, client))
		return;

	tempval = sstate_getinfo(ups, tempvar, 1);

	if (!tempval) {
		send_err(client, NUT_ERR_VAR_NOT_SUPPORTED);
		return;
	}

	/* special case - make FSD show up in STATUS */
	if (!strcasecmp(tempvar, "ups.status") && (ups->fsd == 1)) 
		sendback(client, "ANS %s FSD %s\n", varin, tempval);
	else
		sendback(client, "ANS %s %s\n", varin, tempval);
}

/* handler for "LISTVARS" */
void do_listvars(ctype *client, char *arg, char *rest)
{
	char	ans[SMALLBUF];
	upstype *ups;

	ups = findups(arg);

	if (ups == NULL) {
		send_err(client, NUT_ERR_UNKNOWN_UPS);
		return;
	}

	/* make sure this thing is still with us */
	if (!ups_available(ups, client))
		return;

	memset(ans, '\0', sizeof(ans));

	/* insert "@upsname" if explicitly specified in request */
	if ((arg) && (strcmp(arg, "") != 0))
		snprintf(ans, sizeof(ans), " @%s", arg);

	sstate_makeinfolist(ups, ans, sizeof(ans));

	sendback(client, "VARS%s\n", ans);
}

/* handler for "LISTRW" */
void do_listrw(ctype *client, char *arg, char *rest)
{
	char	ans[SMALLBUF];
	upstype *ups;

	ups = findups(arg);

	if (ups == NULL) {
		send_err(client, NUT_ERR_UNKNOWN_UPS);
		return;
	}

	/* make sure this thing is still with us */
	if (!ups_available(ups, client))
		return;

	ans[0] = '\0';

	/* insert "@upsname" if explicitly specified in request */
	if ((arg) && (strcmp(arg, "") != 0)) {
		snprintfcat(ans, sizeof(ans), " @%s", arg);
	}		

	sstate_makerwlist(ups, ans, sizeof(ans));

	sendback(client, "RW%s\n", ans);
}

/* handler for "VARTYPE" */
void do_vartype(ctype *client, char *arg, char *rest)
{
	char	*varname;
	upstype *ups;

	const	char	*tempvar;
	int	flags, ecount;
	const	struct	enum_t	*etmp;

	if (arg == NULL) {
		send_err(client, NUT_ERR_MISSING_ARGUMENT);
		return;
	}

	ups = parsearg(arg, &varname);

	if (ups == NULL) {
		send_err(client, NUT_ERR_UNKNOWN_UPS);
		free(varname);
		return;
	}

	tempvar = oldvar_to_new(varname);
	free(varname);

	if (!tempvar) {
		send_err(client, NUT_ERR_VAR_UNKNOWN);
		return;
	}

	flags = sstate_getflags(ups, tempvar);

	/* see if this variable even exists */
	if (flags == -1) {
		send_err(client, NUT_ERR_VAR_NOT_SUPPORTED);
		return;
	}

	/* maybe it's a string */
	if (flags & ST_FLAG_STRING) {
		sendback(client, "TYPE STRING %d\n", sstate_getaux(ups, tempvar));
		return;
	}

	/* maybe it's an enum */
	etmp = sstate_getenumlist(ups, tempvar);

	ecount = 0;
	while (etmp) {
		ecount++;
		etmp = etmp->next;
	}

	if (ecount) {
		sendback(client, "TYPE ENUM %d\n", ecount);
		return;
	}

	/* unknown */
	send_err(client, NUT_ERR_UNKNOWN_TYPE);
}

/* handler for "VARDESC" */
void do_vardesc(ctype *client, char *arg, char *rest)
{
	int	i;

	/* find the variable type for the name */
	for (i = 0; netvars[i].name != NULL; i++) {
		if (!strcasecmp(netvars[i].name, arg)) {
			sendback(client, "DESC \"%s\"\n", netvars[i].desc);
			return;
		}
	}

	send_err(client, NUT_ERR_VAR_NOT_SUPPORTED);
}

/* handler for "ENUM" */
void do_enum(ctype *client, char *arg, char *rest)
{
	char	*varname;
	int	i, type;
	upstype *ups;

	const	char	*tempvar, *tempval;
	const	struct	enum_t	*etmp;

	if (arg == NULL) {
		send_err(client, NUT_ERR_MISSING_ARGUMENT);
		return;
	}

	ups = parsearg(arg, &varname);

	if (ups == NULL) {
		send_err(client, NUT_ERR_UNKNOWN_UPS);
		free(varname);
		return;
	}

	type = 0;

	/* find the variable type for the name */
	for (i = 0; netvars[i].name != NULL; i++)
		if (!strcasecmp(netvars[i].name, varname))
			type = netvars[i].type;

	if (type == 0) {	/* didn't find it */
		send_err(client, NUT_ERR_UNKNOWN_TYPE);
		free(varname);
		return;
	}

	tempvar = oldvar_to_new(varname);
	free(varname);

	if (!tempvar) {
		send_err(client, NUT_ERR_VAR_NOT_SUPPORTED);
		return;
	}

	/* see if this variable even exists first */
	tempval = sstate_getinfo(ups, tempvar, 1);

	if (!tempval) {
		send_err(client, NUT_ERR_VAR_NOT_SUPPORTED);
		return;
	}

	/* bail out here if the write fails */
	if (!sendback(client, "ENUM %s\n", varname))
		return;

	etmp = sstate_getenumlist(ups, tempvar);

	while (etmp) {
		char	resp[SMALLBUF];

		if (!strcmp(etmp->val, tempval))
			snprintf(resp, sizeof(resp), "OPTION \"%s\" SELECTED",
				etmp->val);
		else
			snprintf(resp, sizeof(resp), "OPTION \"%s\"",
				etmp->val);

		if (!sendback(client, "%s\n", resp))
			return;

		etmp = etmp->next;
	}

	sendback(client, "END\n");
}

/* handler for "SET" */
void do_set(ctype *client, char *arg, char *rest)
{
	char	*newval, *varname, *ptr;
	upstype *ups;
	int	i, type, found;

	char	cmd[SMALLBUF], esc[SMALLBUF];
	const	char	*tempvar, *tempval;
	const	struct	enum_t	*etmp;

	if ((arg == NULL) || (rest == NULL)) {
		send_err(client, NUT_ERR_MISSING_ARGUMENT);
		return;
	}

	ups = parsearg(arg, &varname);

	if (ups == NULL) {
		send_err(client, NUT_ERR_UNKNOWN_UPS);
		free(varname);
		return;
	}

	type = 0;
	/* make sure 'varname' is part of the protocol as we know it */
	for (i = 0; netvars[i].name != NULL; i++)
		if (!strcasecmp(netvars[i].name, varname))
			type = netvars[i].type;
	
	/* type wasn't resolved in the netvars[] array */
	if (type == 0) {
		send_err(client, NUT_ERR_VAR_UNKNOWN);
		free(varname);
		return;
	}

	/* make sure this thing is still with us */
	if (!ups_available(ups, client)) {
		free(varname);
		return;
	}

	tempvar = oldvar_to_new(varname);
	free(varname);
	
	/* not resolved in netvars[] */
	if (!tempvar) {
		send_err(client, NUT_ERR_VAR_UNKNOWN);
		return;
	}

	/* see if this UPS supports that variable */
	tempval = sstate_getinfo(ups, tempvar, 1);

	if (!tempval) {
		send_err(client, NUT_ERR_VAR_NOT_SUPPORTED);
		return;
	}

	/* make sure this variable is writable (RW) */
	if ((sstate_getflags(ups, tempvar) & ST_FLAG_RW) == 0) {
		send_err(client, NUT_ERR_READONLY);
		return;
	}

	/* clean up rest into something usable */
	newval = xstrdup(rest);

	ptr = strchr(newval, 13);
	if (ptr)
		*ptr = '\0';	/* strip trailing CR */

	ptr = strchr(newval, 10);
	if (ptr)
		*ptr = '\0';	/* strip trailing LF */

	/* finally, see if the new value is allowed for this variable */

	/* check the length if this is for a string */
	if (sstate_getflags(ups, tempvar) & ST_FLAG_STRING) {
		if (strlen(newval) > sstate_getaux(ups, tempvar)) {
			send_err(client, NUT_ERR_TOO_LONG);
			free(newval);
			return;
		}
	}

	/* see if it's enumerated */
	etmp = sstate_getenumlist(ups, tempvar);

	if (etmp) {
		found = 0;

		/* make sure this value is supported */

		while (etmp) {
			if (!strcmp(etmp->val, newval)) {
				found = 1;
				break;
			}

			etmp = etmp->next;
		}

		if (!found) {
			send_err(client, NUT_ERR_INVALID_VALUE);
			free(newval);
			return;
		}
	}
	
	upslogx(LOG_INFO, "Set variable: %s@%s set %s on %s to %s",
	       client->username, client->addr, varname, ups->name, 
	       newval);

	snprintf(cmd, sizeof(cmd), "SET %s \"%s\"\n",
		tempvar, pconf_encode(newval, esc, sizeof(esc)));

	free(newval);

	if (!sstate_sendline(ups, cmd)) {
		upslogx(LOG_INFO, "Set command send failed");
		send_err(client, NUT_ERR_SET_FAILED);
		return;
	}

	sendback(client, "OK\n");
}

/* handler for "INSTCMD" */
void do_instcmd(ctype *client, char *arg, char *rest)
{
	char	*cmdname;
	upstype *ups;
	int	found;

	char	sockcmd[SMALLBUF];
	const	char	*tempcn;
	const	struct	cmdlist_t	*ctmp;

	if (!arg) {
		send_err(client, NUT_ERR_MISSING_ARGUMENT);
		return;
	}		

	ups = parsearg(arg, &cmdname);

	if (ups == NULL) {
		send_err(client, NUT_ERR_UNKNOWN_UPS);
		free(cmdname);
		return;
	}

	tempcn = oldcmd_to_new(cmdname);
	free(cmdname);

	if (!tempcn) {
		send_err(client, NUT_ERR_UNKNOWN_INSTCMD);
		return;
	}

	/* now make sure the UPS can actually DO this command */

	ctmp = sstate_getcmdlist(ups);

	found = 0;
	while (ctmp) {
		if (!strcasecmp(ctmp->name, tempcn)) {
			found = 1;
			break;
		}

		ctmp = ctmp->next;
	}

	if (!found) {
		send_err(client, NUT_ERR_CMD_NOT_SUPPORTED);
		return;
	}

	/* see if this user is allowed to do this command */
	if (!user_checkinstcmd(&client->sock, client->username, 
		client->password, tempcn)) {
		send_err(client, NUT_ERR_ACCESS_DENIED);
		return;
	}

	upslogx(LOG_INFO, "Instant command: %s@%s did %s on %s",
	       client->username, client->addr, cmdname, 
	       ups->name);

	snprintf(sockcmd, sizeof(sockcmd), "INSTCMD %s\n", tempcn);

	if (!sstate_sendline(ups, sockcmd)) {
		upslogx(LOG_INFO, "Set command send failed");
		send_err(client, NUT_ERR_INSTCMD_FAILED);
		return;
	}

	sendback(client, "OK\n");
}

/* handler for "LISTINSTCMD" */
void do_listinstcmd(ctype *client, char *arg, char *rest)
{
	char	ans[SMALLBUF];
	upstype *ups;

	ups = findups(arg);

	if (ups == NULL) {
		send_err(client, NUT_ERR_UNKNOWN_UPS);
		return;
	}

	/* make sure this thing is still with us */
	if (!ups_available(ups, client))
		return;

	ans[0] = '\0';

	/* insert "@upsname" if explicitly specified in request */
	if ((arg) && (strcmp(arg, "") != 0))
		snprintfcat(ans, sizeof(ans), " @%s", arg); 

	sstate_makeinstcmdlist(ups, ans, sizeof(ans));

	sendback(client, "INSTCMDS%s\n", ans);
}

/* handler for "INSTCMDDESC" */
void do_instcmddesc(ctype *client, char *arg, char *rest)
{
	int	i;

	if (arg == NULL) {
		send_err(client, NUT_ERR_MISSING_ARGUMENT);
		return;
	}

	/* find the variable type for the name */
	for (i = 0; instcmds[i].name != NULL; i++) {
		if (!strcasecmp(instcmds[i].name, arg)) {
			sendback(client, "DESC \"%s\"\n", instcmds[i].desc);
			return;
		}
	}

	send_err(client, NUT_ERR_CMD_NOT_SUPPORTED);
}
