/* upsd.c - watches ups state files and answers queries 

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "upsd.h"
#include "upstype.h"
#include "conf.h"

#include "netcmds.h"
#include "upsconf.h"

#include <sys/un.h>
#include <sys/socket.h>

#ifdef HAVE_SSL
#include <openssl/err.h>
#include <openssl/ssl.h>
#endif

#include "user.h"
#include "access.h"
#include "ctype.h"
#include "ssl.h"
#include "sstate.h"
#include "desc.h"
#include "neterr.h"

	/* pointers to linked lists */
	upstype	*firstups = NULL;
	ctype	*firstclient = NULL, *currclient = NULL, *udpclient = NULL;

static	int	listenfd, udpfd, net_port = PORT, upsnum = 0;

	/* default is to listen on all local interfaces */
static	struct	in_addr	listenaddr;

	/* default 15 seconds before data is marked stale */
	int	maxage = 15;

	/* signal handlers */
	struct sigaction sa;
	sigset_t upsd_sigmask;

	/* pid file */
	char	pidfn[SMALLBUF];

	/* set by signal handlers */
	int	reload_flag = 0, stop_flag = 0;

	/* preloaded to STATEPATH in main, can be overridden via upsd.conf */
	char	*statepath = NULL;

	/* preloaded to DATADIR in main, can be overridden via upsd.conf */
	char	*datapath = NULL;

#ifdef SHUT_RDWR
#define shutdown_how SHUT_RDWR
#else
#define shutdown_how 2
#endif

/* return a pointer to the named ups if possible */
upstype *get_ups_ptr(const char *name)
{
	upstype	*tmp;

	if (!name)
		return NULL;

	for (tmp = firstups; tmp != NULL; tmp = tmp->next)
		if (!strcasecmp(tmp->name, name))
			return tmp;

	return NULL;
}

/* search for a ups given the name - and this one allows defaults */
upstype *findups(const char *upsname)
{
	upstype	*temp;

	temp = firstups;

	/* NULL name -> return first one */
	if (!upsname)
		return firstups;

	/* return first one if a good name isn't given */
	if (!strcmp(upsname, ""))
		return firstups;
	
	while (temp != NULL) {
		if (!strcasecmp(temp->name, upsname))
			return temp;

		temp = temp->next;
	}

	return NULL;
}

/* mark the data stale if this is new, otherwise cleanup any remaining junk */
static void ups_data_stale(upstype *ups)
{
	/* don't complain again if it's already known to be stale */
	if (ups->stale == 1)
		return;

	ups->stale = 1;

	upslogx(LOG_NOTICE, "Data for UPS [%s] is stale - check driver",
		ups->name);
}

/* mark the data ok if this is new, otherwise do nothing */
static void ups_data_ok(upstype *ups)
{
	if (ups->stale == 0)
		return;

	upslogx(LOG_NOTICE, "UPS [%s] data is no longer stale", ups->name);
	ups->stale = 0;
}

/* make sure this UPS is connected and has fresh data */
static void check_ups(upstype *ups)
{
	/* sanity checks */
	if ((!ups) || (!ups->fn))
		return;

	/* throw some warnings if it's not feeding us data any more */
	if (sstate_dead(ups, maxage))
		ups_data_stale(ups);
	else
		ups_data_ok(ups);

	/* see if we need to (re)connect to the socket */
	if (ups->sock_fd == -1)
		ups->sock_fd = sstate_connect(ups);
}

/* change the configuration of an existing UPS (used during reloads) */
void redefine_ups(const char *fn, const char *name, const char *desc)
{
	upstype	*temp;

	temp = findups(name);

	if (!temp) {
		upslogx(LOG_ERR, "UPS %s disappeared during reload", name);
		return;
	}

	/* paranoia */
	if (!temp->fn) {
		upslogx(LOG_ERR, "UPS %s had a NULL filename!", name);

		/* let's give it something quick to use later */
		temp->fn = xstrdup("");
	}

	/* when the filename changes, force a reconnect */
	if (strcmp(temp->fn, fn) != 0) {

		upslogx(LOG_NOTICE, "Redefined UPS [%s]", name);

		/* release all data */
		sstate_infofree(temp);
		sstate_cmdfree(temp);
		pconf_finish(&temp->sock_ctx);

		close(temp->sock_fd);
		temp->sock_fd = -1;
		temp->dumpdone = 0;

		/* now redefine the filename and wrap up */
		free(temp->fn);
		temp->fn = xstrdup(fn);
	}

	/* update the description */

	if (temp->desc)
		free(temp->desc);

	if (desc)
		temp->desc = xstrdup(desc);
	else
		temp->desc = NULL;

	/* always set this on reload */
	temp->retain = 1;
}		

/* start monitoring a ups called <name> from a file called <fn> */
void addups(const char *fn, const char *name, const char *desc)
{
	upstype	*temp, *last;

	temp = last = firstups;

	/* find end of linked list */
	while (temp != NULL) {
		last = temp;

		if (!strcasecmp(temp->name, name)) {
			upslogx(LOG_ERR, "UPS name [%s] is already in use!", 
				name);
			return;
		}

		temp = temp->next;
	}

	/* grab some memory and add the info */
	temp = xcalloc(1, sizeof(upstype));

	temp->fn = xstrdup(fn);
	temp->name = xstrdup(name);

	if (desc)
		temp->desc = xstrdup(desc);
	else
		temp->desc = NULL;

	temp->stale = 1;

	temp->numlogins = 0;
	temp->fsd = 0;
	temp->retain = 1;
	temp->next = NULL;

	temp->dumpdone = 0;
	temp->data_ok = 0;

	/* preload this to the current time to avoid false staleness */
	time(&temp->last_heard);

	temp->last_ping = 0;
	temp->last_connfail = 0;
	temp->inforoot = NULL;

	if (last == NULL)	/* first one */
		firstups = temp;
	else			/* not the first */
		last->next = temp;

	upsnum++;

	temp->sock_fd = sstate_connect(temp);
}

/* setup a socket for incoming udp requests */
static void setupudp(void)
{
	struct	sockaddr_in	recv;

	if ((udpfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
		fatal("Can't create socket");

	memset(&recv, '\0', sizeof(recv));
	recv.sin_addr = listenaddr;
	recv.sin_family = AF_INET;
	recv.sin_port = htons(net_port);

	if (bind(udpfd, (struct sockaddr *) &recv, sizeof(recv)) < 0)
		fatal("Can't bind to udp socket (port %d)", net_port);

	return;
}

/* create a listening socket for tcp connections */
static void setuptcp(void)
{
	struct	sockaddr_in	server;
	int	res, one = 1;

	if ((listenfd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
		fatal("socket");

	res = setsockopt(listenfd, SOL_SOCKET, SO_REUSEADDR, (void *) &one, 
		sizeof(one));

	if (res != 0)
		fatal("setsockopt(SO_REUSEADDR)");

	memset(&server, '\0', sizeof(server));
	server.sin_addr = listenaddr;
	server.sin_family = AF_INET;
	server.sin_port = htons(net_port);

	if (bind(listenfd, (struct sockaddr *) &server, sizeof(server)) == -1)
		fatal("Can't bind TCP port number %d", net_port);

	if ((res = fcntl(listenfd, F_GETFL, 0)) == -1)
		fatal("fcntl(get)");

	if (fcntl(listenfd, F_SETFL, res | O_NDELAY) == -1)
		fatal("fcntl(set)");

	if (listen(listenfd, 16))
		fatal("listen");

	return;
}

/* decrement the login counter for this ups */
static void declogins(const char *upsname)
{
	upstype	*ups;

	ups = findups(upsname);

	if (ups == NULL) {
		upslogx(LOG_INFO, "Tried to decrement invalid ups name (%s)", upsname);
		return;
	}

	ups->numlogins--;
}

/* remove a client struct from the linked list */
static void delclient(ctype *dclient)
{
	ctype	*tmp, *last;

	if (dclient == NULL)
		return;

	last = NULL;
	tmp = firstclient;

	while (tmp != NULL) {
		if (tmp == dclient) {		/* found it */
			shutdown(dclient->fd, 2);
			close(dclient->fd);

			free(tmp->addr);

			if (tmp->loginups != NULL) {
				declogins(tmp->loginups);
				free(tmp->loginups);
			}

			if (tmp->password != NULL)
				free(tmp->password);

#ifdef HAVE_SSL
			if (tmp->ssl)
				SSL_free(tmp->ssl);
#endif

			if (last == NULL)	/* deleting first entry */
				firstclient = tmp->next;
			else
				last->next = tmp->next;

			free(tmp);
			currclient = NULL;
			return;
		}
		last = tmp;
		tmp = tmp->next;
	}

	/* not found?! */
	upslogx(LOG_WARNING, "Tried to delete client struct that doesn't exist!");

	return;
}

/* send the buffer <sendbuf> of length <sendlen> to host <dest> */
int sendback(ctype *client, const char *fmt, ...)
{
	int	res, len;
	char ans[NUT_NET_ANSWER_MAX+1];
	va_list ap;

	if (!client)
		return -1;

	va_start(ap, fmt);
	vsnprintf(ans, sizeof(ans), fmt, ap);
	va_end(ap);

	len = strlen(ans);

	upsdebugx(2, "sendto: [destfd=%d] [len=%d] [%s]", 
		client->fd, len, ans);

	if (client->ssl)
		res = ssl_write(client, ans, len);
	else
		res = sendto(client->fd, ans, len, 0, 
			(struct sockaddr *) &client->sock,
			sizeof(struct sockaddr_in));

	if (len != res) {
		upslog(LOG_NOTICE, "sendto() failed for %s", client->addr);
		delclient(client);
		return 0;	/* failed */
	}

	return 1;	/* OK */
}

/* just a simple wrapper for now */
int send_err(ctype *client, const char *errtype)
{
	if (!client)
		return -1;

	upsdebugx(4, "Sending error [%s] to client %s", 
		errtype, client->addr);

	return sendback(client, "ERR %s\n", errtype);
}

/* increment the login counter for this ups */
static void inclogins(const char *upsname)
{
	upstype	*ups;

	ups = findups(upsname);

	if (ups == NULL) {
		upslogx(LOG_INFO, "Tried to increment invalid ups name");
		return;
	}

	ups->numlogins++;
}

/* disconnect anyone logged into this UPS */
void kick_login_clients(char *upsname)      
{
	ctype   *tmp, *next;

	tmp = firstclient;

	while (tmp) {
		next = tmp->next;

		/* if it's not logged in, don't check it */
		if (!tmp->loginups) {
			tmp = next;
			continue;
		}

		if (!strcmp(tmp->loginups, upsname)) {
			upslogx(LOG_INFO, "Kicking client %s (was on UPS [%s])\n", 
				tmp->addr, upsname);
			delclient(tmp);
		}
                
		tmp = next;
	}
}

/* make sure a UPS is sane - connected, with fresh data */
int ups_available(const upstype *ups, ctype *client)
{
	if (ups->sock_fd == -1) {
		send_err(client, NUT_ERR_DRIVER_NOT_CONNECTED);
		return 0;
	}

	if (ups->stale) {
		send_err(client, NUT_ERR_DATA_STALE);
		return 0;
	}

	/* must be OK */
	return 1;
}

/* handler for "VER" */
void do_sendver(ctype *client, char *arg, char *rest)
{
	sendback(client, 
		"Network UPS Tools upsd %s - http://www.networkupstools.org/\n", 
		UPS_VERSION);
}

/* handler for "HELP" */
void do_sendhelp(ctype *client, char *arg, char *rest)
{
	char	ans[SMALLBUF];
	int	i;

	memset(ans, '\0', sizeof(ans));
	for (i = 0; netcmds[i].name != NULL; i++)
		snprintfcat(ans, sizeof(ans), " %s", netcmds[i].name);

	sendback(client, "Commands:%s\n", ans);
}

/* handler for "LOGOUT" */
void do_logout(ctype *client, char *arg, char *rest)
{
	if (client->username)
		upslogx(LOG_INFO, "Client %s@%s logged out", 
			client->username, client->addr);
	else
		upslogx(LOG_INFO, "Client on %s logged out", client->addr);

	/* check return since sendback can call delclient itself */
	if (sendback(client, "OK Goodbye\n"))
		delclient(client);
}		

/* handler for "LOGIN" */
static void do_login(ctype *client, char *arg, char *rest)
{
	int	found = 0;
	upstype *ups;

	if (client->loginups != NULL) {
		upslogx(LOG_INFO, "Client on %s tried to login twice", client->addr);
		send_err(client, NUT_ERR_ALREADY_LOGGED_IN);
		return;
	}

	/* "LOGIN" only - no UPS name given as argument */
	if ((arg == NULL) || (strlen(arg) == 0)) {

		/* no UPSes defined? */
		if (!firstups) {
			send_err(client, NUT_ERR_UNKNOWN_UPS);
			return;
		}

		/* give it the name of the first one in the .conf */
		client->loginups = xstrdup(firstups->name);

		/* but warn them that this is going away some day */
		upslogx(LOG_WARNING, "Warning: LOGIN relied on the default UPS");
	}
	else {
		/* see if <arg> is a valid UPS name */

		for (ups = firstups; ups != NULL; ups = ups->next) {
			if (!strcasecmp(ups->name, arg))
				found = 1;
		}

		if (!found) {
			send_err(client, NUT_ERR_UNKNOWN_UPS);
			return;
		}

		client->loginups = xstrdup(arg);
	}

	inclogins(client->loginups);

	upslogx(LOG_INFO, "Client %s@%s logged into UPS [%s]%s",
		client->username, client->addr, client->loginups, 
		client->ssl ? " (SSL)" : "");

	sendback(client, "OK\n");
}

/* handler for "PASSWORD" */
static void do_password(ctype *client, char *arg, char *rest)
{
	if ((arg == NULL) || (strlen(arg) == 0)) {
		upslogx(LOG_INFO, "Client on %s sent a NULL password!", client->addr);
		send_err(client, NUT_ERR_INVALID_PASSWORD);
		return;	
	}

	if (client->password != NULL) {
		upslogx(LOG_INFO, "Client on %s tried to set a password twice", 
			client->addr);
		send_err(client, NUT_ERR_ALREADY_SET_PASSWORD);
		return;
	}

	client->password = xstrdup(arg);

	sendback(client, "OK\n");
}

/* handler for "FSD" */
static void do_fsd(ctype *client, char *arg, char *rest)
{
	upstype *ups;

	ups = findups(arg);

	if (ups == NULL) {
		send_err(client, NUT_ERR_UNKNOWN_UPS);
		return;
	}

	upslogx(LOG_INFO, "Setting FSD on UPS [%s]", ups->name);

	ups->fsd = 1;
	sendback(client, "OK FSD-SET\n");
}

/* handler for "MASTER" */
static void do_master(ctype *client, char *arg, char *rest)
{
	upstype *ups;

	ups = findups(arg);

	if (ups == NULL) {
		send_err(client, NUT_ERR_UNKNOWN_UPS);
		return;
	}

	/* really quite simple - this is just an access level check */
	sendback(client, "OK MASTER-GRANTED\n");
}

/* handler for "USERNAME" */
static void do_username(ctype *client, char *arg, char *rest)
{
	if ((arg == NULL) || (strlen(arg) ==0)) {
		upslogx(LOG_INFO, "Client on %s sent a NULL username!",
		       client->addr);
		send_err(client, NUT_ERR_INVALID_USERNAME);
		return;
	}

	if (client->username != NULL) {
		upslogx(LOG_INFO, "Client on %s tried to set a username twice",
		       client->addr);
		send_err(client, NUT_ERR_ALREADY_SET_USERNAME);
		return;
	}

	client->username = xstrdup(arg);

	sendback(client, "OK\n");
}

static int check_flags(ctype *client, int flags, const char *action, int udp)
{
	if (flags & FLAG_TCP) { 
		if (udp) {
			send_err(client, NUT_ERR_UNKNOWN_COMMAND);
			return 0;
		}
	}

	if (flags & FLAG_USER) {
		if (!client->username) {
			send_err(client, NUT_ERR_USERNAME_REQUIRED);
			return 0;
		}

		if (!client->password) {
			send_err(client, NUT_ERR_PASSWORD_REQUIRED);
			return 0;
		}

		/* instcmd is a special case - checked in do_instcmd */
		if (flags & FLAG_CMD)
			return 1;

		/* now check this specific action */

		if (!user_checkaction(&client->sock, client->username,
			client->password, action)) {
			send_err(client, NUT_ERR_ACCESS_DENIED);
			return 0;
		}
	}

	/* seems good so far... */
	return 1;
}

/* this handles new commands separately during 1.4 and will take over in 2.0 */
static int new_net_parse(ctype *client, int numarg, char **arg, int udp)
{
	int	i;

	if (!access_check(&client->sock, LEVEL_BASE)) {
		send_err(client, NUT_ERR_ACCESS_DENIED);
		return 1;	
	}

	if (numarg < 1)
		return 0;

	/* UDP clients can only do GET */
	if ((udp) && (strcasecmp(arg[0], "GET") != 0))
		return 0;

	for (i = 0; new_netcmds[i].name != NULL; i++) {
		if (!strcasecmp(new_netcmds[i].name, arg[0])) {

			/* see if the flags are satisfied */
			if (!check_flags(client, new_netcmds[i].flags, arg[0], udp))
				return 1;

			if (!access_check(&client->sock, new_netcmds[i].level)) {
				send_err(client, NUT_ERR_ACCESS_DENIED);
				return 1;
			}

			/* may drop out with 0 in order to call old funcs */
			return new_netcmds[i].func(client, numarg - 1, &arg[1]);
		}
	}

	return 0;
}

/* parse requests from the network */
static void parse_net(ctype *client, int udp)
{
	int	i, len;
	char	*cmd, *arg2, *rest, *ptr, *buf;
	PCONF_CTX	ctx;

	len = client->rqpos;
	buf = client->rq;

	/* check for base access if this is a UDP client */
	if ((udp) && (!access_check(&client->sock, LEVEL_BASE)))
		return;

	/* FUTURE: maybe switch to pconf_char with per-client context */
	pconf_init(&ctx, NULL);

	/* paranoia */
	buf[len] = '\0';

	if (pconf_line(&ctx, buf)) {
		if (!pconf_parse_error(&ctx)) {
			if (new_net_parse(client, ctx.numargs, ctx.arglist, udp)) {
				pconf_finish(&ctx);
				return;
			}
		}
	}

	pconf_finish(&ctx);

	/* fallthrough: try parsing it the old way */

	arg2 = rest = NULL;

	/* kill any embedded cr/lf chars */
	for (i = 0; i < len; i++)
		if ((buf[i] == 13) || (buf[i] == 10))
			buf[i] = '\0';

	/* parse from buf: <cmd> [<arg2> [<rest>]] */

	cmd = buf;
	ptr = strchr(cmd, ' ');
	if (ptr) {
		*ptr++ = '\0';
		arg2 = ptr;
		ptr = strchr(arg2, ' ');
		if (ptr) {
			*ptr++ = '\0';
			rest = ptr;
		}
	}

	for (i = 0; netcmds[i].name != NULL; i++) {
		if (!strcasecmp(netcmds[i].name, cmd)) {

			/* see if the flags are satisfied */
			if (!check_flags(client, netcmds[i].flags, cmd, udp))
				return;

			if (!access_check(&client->sock, netcmds[i].level)) {
				send_err(client, NUT_ERR_ACCESS_DENIED);
				return;
			}

			netcmds[i].func(client, arg2, rest);
			return;
		}
	}

	upsdebugx(4, "Unknown command: %s", cmd);
	send_err(client, NUT_ERR_UNKNOWN_COMMAND);
}

/* scan the list of UPSes for sanity */
void check_every_ups(void)
{
	upstype *ups;

	ups = firstups;

	while (ups != NULL) {
		check_ups(ups);
		ups = ups->next;
	}
}

/* answer incoming tcp connections */
static void answertcp(void)
{
	int	acc;
	struct	sockaddr_in csock;
	ctype	*tmp, *last;
	socklen_t	clen;

	clen = sizeof(csock);
	acc = accept(listenfd, (struct sockaddr *) &csock, &clen);

	if (acc < 0)
		return;

	last = tmp = firstclient;

	while (tmp != NULL) {
		last = tmp;
		tmp = tmp->next;
	}

	tmp = xmalloc(sizeof(ctype));

	tmp->addr = xstrdup(inet_ntoa(csock.sin_addr));
	tmp->fd = acc;
	memcpy(&tmp->sock, &csock, sizeof(struct sockaddr_in));

	tmp->rqpos = 0;
	memset(tmp->rq, '\0', sizeof(tmp->rq));

	tmp->loginups = NULL;		/* for upsmon */
	tmp->username = NULL;
	tmp->password = NULL;

	tmp->ssl = NULL;
	tmp->ssl_connected = 0;

	tmp->next = NULL;

	if (last == NULL)
 		firstclient = tmp;
	else
		last->next = tmp;

	if (!access_check(&tmp->sock, LEVEL_BASE)) {
		upslogx(LOG_NOTICE, "Rejecting TCP connection from %s", 
			inet_ntoa(csock.sin_addr));

		delclient(tmp);
		return;
	}

	upslogx(LOG_INFO, "Connection from %s", inet_ntoa(csock.sin_addr));
}

/* read tcp messages and handle them */
static void readtcp(ctype *client)
{
	char	buf[SMALLBUF];
	int	i, ret;

	memset(buf, '\0', sizeof(buf));

	if (client->ssl)
		ret = ssl_read(client, buf, sizeof(buf));
	else
		ret = read(client->fd, buf, sizeof(buf));

	if (ret < 1) {
		upslogx(LOG_INFO, "Host %s disconnected (read failure)",
			client->addr);

		delclient(client);
		return;
	}

	/* if an overflow will happen, then clear the queue */
	if ((ret + client->rqpos) >= sizeof(client->rq)) {
		memset(client->rq, '\0', sizeof(client->rq));
		client->rqpos = 0;
	}

	/* fragment handling code */

	for (i = 0; i < ret; i++) {
		/* add to the receive queue one by one */
		client->rq[client->rqpos++] = buf[i];

		/* parse on linefeed ("blah blah\n") */
		if (buf[i] == 10) {	/* LF */
			currclient = client;	/* enable tcp replies */
			parse_net(client, 0);

			if (currclient == NULL) /* connection was closed */
				return;

			currclient = NULL;	/* disable */
			
			/* reset queue */
			client->rqpos = 0;
			memset(client->rq, '\0', sizeof(client->rq));
		}
	}

	return;
}

static void upsd_stop(void)
{
	ctype	*tmpcli;
	upstype	*ups, *unext;

	/* cleanup client fds */
	for (tmpcli = firstclient; tmpcli != NULL; tmpcli = tmpcli->next) {
		shutdown(tmpcli->fd, shutdown_how);
		close(tmpcli->fd);
	}

	if (strcmp(pidfn, "") != 0)
		unlink(pidfn);

	ups = firstups;

	while (ups) {
		unext = ups->next;

		if (ups->sock_fd != -1)
			close(ups->sock_fd);

		sstate_infofree(ups);
		sstate_cmdfree(ups);
		pconf_finish(&ups->sock_ctx);

		if (ups->fn)
			free(ups->fn);
		if (ups->name)
			free(ups->name);
		if (ups->desc)
			free(ups->desc);
		free(ups);

		ups = unext;
	}

	/* dump everything */

	acl_free();
	access_free();
	user_flush();
	desc_free();

	if (statepath)
		free(statepath);
	if (datapath)
		free(datapath);
	if (udpclient)
		free(udpclient);
	if (certfile)
		free(certfile);

	exit(0);
}

/* service requests and check on new data */
static void mainloop(void)
{
	fd_set	rfds;
	struct	timeval	tv;
	int	res, maxfd;
	ctype	*tmpcli, *tmpnext;
	upstype	*utmp, *unext;
	socklen_t	fromlen;

	if (stop_flag)
		upsd_stop();

	if (reload_flag) {
		reload_config();
		reload_flag = 0;
	}

	FD_ZERO(&rfds);
	FD_SET(udpfd, &rfds);
	FD_SET(listenfd, &rfds);
	
	tv.tv_sec = 2;
	tv.tv_usec = 0;

	maxfd = (udpfd > listenfd) ? udpfd : listenfd;

	/* scan through clients and add to FD_SET */
	for (tmpcli = firstclient; tmpcli != NULL; tmpcli = tmpcli->next) {
		FD_SET(tmpcli->fd, &rfds);
		if (tmpcli->fd > maxfd)
			maxfd = tmpcli->fd;
	}

	/* also add new driver sockets */
	for (utmp = firstups; utmp != NULL; utmp = utmp->next) {
		if (utmp->sock_fd != -1) {
			FD_SET(utmp->sock_fd, &rfds);

			if (utmp->sock_fd > maxfd)
				maxfd = utmp->sock_fd;
		}
	}
	
	res = select(maxfd + 1, &rfds, NULL, NULL, &tv);

	if (res > 0) {
		if (FD_ISSET(udpfd, &rfds)) {
			fromlen = sizeof(struct sockaddr_in);

			udpclient->fd = udpfd;
			memset(&udpclient->sock, '\0', fromlen);
			memset(&udpclient->rq, '\0', SMALLBUF);

			res = recvfrom(udpfd, udpclient->rq, SMALLBUF, 0, 
				(struct sockaddr *) &udpclient->sock,
				&fromlen);

			if (res > 0) {

				/* dummy setting, as opposed to NULL */
				currclient = udpclient;

				udpclient->rqpos = res;
				parse_net(udpclient, 1);

				currclient = NULL;
			}
		}

		if (FD_ISSET(listenfd, &rfds))
			answertcp();

		/* scan clients for activity */

		tmpcli = firstclient;

		while (tmpcli != NULL) {

			/* preserve for later since delclient may run */
			tmpnext = tmpcli->next;

			if (FD_ISSET(tmpcli->fd, &rfds))
				readtcp(tmpcli);

			tmpcli = tmpnext;
		}

		/* now scan ups sockets for activity */
		utmp = firstups;

		while (utmp) {
			unext = utmp->next;

			if (utmp->sock_fd != -1)
				if (FD_ISSET(utmp->sock_fd, &rfds))
					sstate_sock_read(utmp);

			utmp = unext;
		}
	}

	check_every_ups();
}

static void help(const char *progname) 
{
	printf("Network server for UPS data.\n\n");
	printf("usage: %s [OPTIONS]\n", progname);

	printf("\n");
	printf("  -c <command>	send <command> via signal to background process\n");
	printf("		commands:\n");
	printf("		 - reload: reread configuration files\n");
	printf("		 - stop: stop process and exit\n");
	printf("  -D		raise debugging level\n");           
	printf("  -f		stay in the foreground for testing\n");
	printf("  -h		display this help\n");
	printf("  -i <address>	binds to interface <address>\n");
	printf("  -p <port>	sets network port (UDP and TCP)\n");
	printf("  -r <dir>	chroots to <dir>\n");
	printf("  -u <user>	switch to <user> (if started as root)\n");
	printf("  -V		display the version of this software\n");

	exit(1);
}

static void set_reload_flag(int sig)
{
	reload_flag = 1;
}

static void set_stop_flag(int sig)
{
	upslogx(LOG_INFO, "Signal %d: exiting", sig);
	stop_flag = 1;
}

/* basic signal setup to ignore SIGPIPE */
static void setupsignals(void)
{
	sigemptyset(&upsd_sigmask);
	sa.sa_mask = upsd_sigmask;
	sa.sa_flags = 0;

	sa.sa_handler = SIG_IGN;
	sigaction(SIGPIPE, &sa, NULL);

	/* handle shutdown signals */
	sa.sa_handler = set_stop_flag;
	sigaction(SIGINT, &sa, NULL);
	sigaction(SIGQUIT, &sa, NULL);
	sigaction(SIGTERM, &sa, NULL);

	/* handle reloading */
	sa.sa_handler = set_reload_flag;
	sigaction(SIGHUP, &sa, NULL);
}

/* try not to exit before the DUMPDONE hits, so clients work on the first try */
static void initial_dump_wait(void)
{
	int	maxfd, ret, numups, numdone;
	fd_set	rfds;
	time_t	start, now;
	struct	timeval	tv;
	upstype	*utmp, *unext;

	printf("Synchronizing...");
	fflush(stdout);

	time(&start);
	time(&now);

	while (difftime(now, start) < INITIAL_WAIT_MAX) {
		maxfd = 0;
		numups = 0;
		numdone = 0;

		tv.tv_sec = 1;
		tv.tv_usec = 0;

		FD_ZERO(&rfds);

		for (utmp = firstups; utmp != NULL; utmp = utmp->next) {
			if (utmp->sock_fd != -1) {
				FD_SET(utmp->sock_fd, &rfds);

				if (utmp->sock_fd > maxfd)
					maxfd = utmp->sock_fd;

				numups++;
			}
                }

		ret = select(maxfd + 1, &rfds, NULL, NULL, &tv);

		if (ret < 1) {
			printf(".");
			fflush(stdout);

			time(&now);
			continue;
		}

		utmp = firstups;
		while (utmp) {
			unext = utmp->next;

			if (utmp->sock_fd != -1) {
				if (FD_ISSET(utmp->sock_fd, &rfds))
					sstate_sock_read(utmp);

				if (utmp->dumpdone == 1)
					numdone++;
			}

			utmp = unext;
		}

		/* if they're all done, then exit early */
		if (numdone >= numups) {
			printf("done\n");
			fflush(stdout);
			return;
		}

		time(&now);
        }

	printf(" giving up\n");
	fflush(stdout);
}

void check_perms(const char *fn)
{
	int	ret;
	struct stat	st;

	ret = stat(fn, &st);

	if (ret != 0)
		fatal("stat %s", fn);

	/* include the x bit here in case we check a directory */
	if (st.st_mode & (S_IROTH | S_IXOTH))
		upslogx(LOG_WARNING, "%s is world readable", fn);
}	

int main(int argc, char **argv)
{
	int	i, cmd = 0;
	const char *nut_statepath_env = getenv("NUT_STATEPATH");
	int	do_background = 1;
	const	char *user = NULL;
	char	*progname, *chroot_path = NULL;
	struct	passwd	*new_uid = NULL;

	progname = argv[0];

	/* pick up a default from configure --with-user */
	user = RUN_AS_USER;

	/* yes, xstrdup - the conf handlers call free on this later */
	statepath = xstrdup(STATEPATH);
	datapath = xstrdup(DATADIR);

	/* set up some things for later */

	listenaddr.s_addr = INADDR_ANY;
	snprintf(pidfn, sizeof(pidfn), "%s/upsd.pid", altpidpath());

	printf("Network UPS Tools upsd %s\n", UPS_VERSION);

	while ((i = getopt(argc, argv, "+hp:r:i:fu:Vc:D")) != EOF) {
		switch (i) {
			case 'h':
				help(progname);
				break;
			case 'p':
				net_port = atoi(optarg);
				break;
			case 'i':
				if (!inet_aton(optarg, &listenaddr))
					fatal("Invalid IP address");
				break;
			case 'r':
				chroot_path = optarg;
				break;
			case 'f':
				do_background = 0;
				break;
			case 'u':
				user = optarg;
				break;
			case 'V':

				/* do nothing - we already printed the banner */
				exit(0);

			case 'c':
				if (!strncmp(optarg, "reload", strlen(optarg)))
					cmd = SIGCMD_RELOAD;
				if (!strncmp(optarg, "stop", strlen(optarg)))
					cmd = SIGCMD_STOP;

				/* bad command given */
				if (cmd == 0)
					help(progname);
				break;

			case 'D':
				do_background = 0;
				nut_debug_level++;
				break;
			default:
				help(progname);
				break;
		}
	}

	if (cmd) {
		sendsignalfn(pidfn, cmd);
		exit(0);
	}

	argc -= optind;
	argv += optind;

	if (argc != 0)
		help(progname);

	/* provide a skeletal pseudo-client for all UDP-based requests */

	udpclient = xmalloc(sizeof(ctype));
	udpclient->ssl = NULL;			/* impossible */

	setupsignals();
	setupudp();
	setuptcp();

	open_syslog("upsd");

	/* send logging to the syslog pre-background for later use */
        xbit_set(&upslog_flags, UPSLOG_SYSLOG);

	if (nut_statepath_env)
		statepath = xstrdup(nut_statepath_env);

	/* do this here, since getpwnam() might not work in the chroot */
	if ((new_uid = get_user_pwent(user)) == NULL)
		fatal("getpwnam(%s)", user);

	if (chroot_path)
		chroot_start(chroot_path);

	become_user(new_uid);

	if (chdir(statepath))
		fatal("Can't chdir to %s", statepath);

	/* check statepath perms */
	check_perms(statepath);

	/* read in the ups definitions (required) */
	read_upsconf(1);

	/* ... then do something with them */
	upsconf_add(0);			/* 0 = initial */

	read_upsdconf(0);		/* 0 = initial */
	user_load();

	if (upsnum == 0)
		fatalx("Fatal error: you need to define at least one UPS in your ups.conf!\n");

	ssl_init();

	/* try to bring in the var/cmd descriptions */
	desc_load();

	/* try to get a full dump for each UPS before exiting */
	initial_dump_wait();

	/* this is after the uid change to detect permission problems */
	check_every_ups();

	if (do_background) {
		background();

		writepid(pidfn);
	} else {
		memset(&pidfn, '\0', sizeof(pidfn));
	}

	for (;;)
		mainloop();

	/* NOTREACHED */
	return 0;
}
