/* $Id: osutil.h,v 1.12 2004/08/19 23:34:31 graziano Exp $ */


#ifndef OSUTIL_H
#define OSUTIL_H


/*
** This package defines functions for working with operating system facilities.
*/

#include "config_portability.h"

#ifdef HAVE_SELECT_H
#	include <sys/select.h> 
#endif
#ifdef HAVE_UNISTD_H
#	include <unistd.h> 
#endif
#ifdef HAVE_SYS_TYPES_H
#	include <sys/types.h> 
#endif
#if TIME_WITH_SYS_TIME
#       include <sys/time.h>
#       include <time.h>
#else
#       if HAVE_SYS_TIME_H
#               include <sys/time.h>
#       else
#               include <time.h>
#       endif
#endif


#ifdef __cplusplus
extern "C" {
#endif

/*
** Returns a best-guess estimate of the number of CPUs on the system.
*/
int
CPUCount(void);

/**
 * Returns the system name as from uname(2). NULL in case of error.
 */
const char *
GetSystemName();

/**
 * Returns the release as from uname(2). NULL in case of error.
 */
const char *
GetRelease();

/**
 * Returns the machine as from uname(2). NULL in case of error.
 */
const char *
GetMachine();

/**
 * Returns the installed memory (in MB) and -1 on failure. (uses
 * sysconf so might not be accurate).
 */
double
InstalledMemory();

/**
 * Returns the currently available memory (in MB) and -1 on failure. (uses
 * sysconf so might not be accurate).
 */
double 
AvailableMemory();

/*
** Returns the current time in number of seconds since midnight 1/1/1970, GMT.
*/
double
CurrentTime(void);


/*
 * Returns the environment value for the variable #name#.  This value is
 * determined in one of three ways: if an environment variable of that name is
 * set, its value is used; otherwise, if a file named #rcName# in one of the
 * colon-delimited set of directories #rcDirs# contains a line of the form
 * #name#=value, then that value is used; otherwise #defaultValue# is used.
 * For convenience, a directory of "~" in #rcDirs# is expanded to the contents
 * of the HOME environment variable.
 *
 * The returned value need to be freed.
 */
char *
GetEnvironmentValue(const char *name,
                    const char *rcDirs,
                    const char *rcName,
                    const char *defaultValue);


/*
** Attempts to determine the login id of the user running the binary.  Returns
** 1 and copies the name into the #length#-long array name if successful, else
** returns 0.
*/
int
GetUserName(char *name,
            size_t length);


/*
** Prevents the O/S from raising the signal #sig# until a subsequent call to
** ReleaseSignal() is made.
*/
void
HoldSignal(int sig);

/**
 * Wait for childreen (call to waitpid with WNOHANG). Returns the pid of
 * the terminated child or 0 for no child left behind.
 */
pid_t
WaitChild();

/*
** Creates the directory specified by #path# with mode #mode#.  If
** #makeEntirePath# is non-zero, any missing subpaths in #path# will also be
** created.  Returns 1 if successful, else 0.
*/
int
MakeDirectory(const char *path,
              mode_t mode,
              int makeEntirePath);


/*
** Returns the number of microseconds that have elapsed since midnight, local
** time.
*/
double
MicroTime(void);


/*
** Allows the O/S to raise the signal #sig#.
*/
void
ReleaseSignal(int sig);


/*
 * Starts a timer that will raise a SIGALRM in #numberOfSecs# seconds. It
 * takes care of setting/resetting the signal handler.
 */
void
SetRealTimer(unsigned int numberOfSecs);
#define RESETREALTIMER SetRealTimer(0)

/**
 * Locks are handled in a peculiar way: #lock# is a void * passed in by
 * the module. If #lock# is NULL, GetNWSLock() will assign to it a mutex
 * and then it will lock it. It's up to the module to keep track of the
 * different locks.
 * Returns 1 if succesfull 0 otherwise.
 */
int
GetNWSLock(void **lock);

/**
 * release the NWS general lock. Returns 1 if succesful, 0 otherwise.
 */
int
ReleaseNWSLock(void **lock);

/**
 * defined to wrap select around a loop to avoid the EINTR case. If #err#
 * is not NULL it will contain the errno that select returned. Returns
 * whatever select returned.
 */
int
PortabilitySelect(	int n, 
			fd_set *readfs, 
			fd_set *writefs, 
			fd_set *exceptfs, 
			struct timeval *tv, 
			int *err);

/**
 * defined our own realloc because of the many problem we had with the
 * libraries one
 */
void *
PortabilityRealloc(void *ptr, size_t size);
#define REALLOC(ptr, size) PortabilityRealloc(ptr, size)

#ifdef __cplusplus
}
#endif

#endif
