/* $Id: timeouts.h,v 1.5 2004/04/26 16:19:02 graziano Exp $ */

#ifndef TIMEOUTS_H
#define TIMEOUTS_H

/*
 * This modules defines functions used to keep track of adaptive
 * timeouts. You can use these functions to (for example) keep track of
 * how long you should wait to receive a packet from a remote host.
 */

#include "dnsutil.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
 * We reserve timeouts for receive and send packets, connection time and
 * we leave space for 3 user definable timeouts.
 */
typedef enum {RECV, SEND, CONN, USER1, USER2, USER3} timeoutTypes;
#define TIMEOUT_TYPES_NUMBER 6

/**
 * Sets the bounds on a specific class of timeouts. #min# and #max# are
 * the respective minimum and maximum timeout allowed (defaults are 5 and
 * 120 seconds). #timedOut# is the value to use when the an operation
 * timedout and no data could have been moved (you don't want to have the
 * timeout set to the maximum if no connection can be made for example).
 */
void
SetDefaultTimeout(	timeoutTypes type, 
			double min, 
			double max,
			double timedOut);

/**
 * get the current defaults timeouts.
 */
void
GetDefaultTimeout(	timeoutTypes type, 
			double *min, 
			double *max,
			double *timedOut);


/**
 * Returns the current timeouts value for the specific class given the
 * IPAddress and the #size# of the data to be sent/received. If #size# is
 * <= 0 it will be ignored.
 */
double
GetTimeOut(	timeoutTypes type, 
		IPAddress addr, 
		long size);

/**
 * Reports how long it took to perform the action (say send or receive a
 * packet) last time and if it timed out and how many bytes were
 * sent/received (if applicable). If #size# is 0 it will be ignore (but
 * see the note in case of timeout).
 *
 * NOTE: If #size# is 0 and it was a timeout, we feed the forecaster with
 * the minimum timeout: this can be used to be more responsive when links
 * are broken and there is no answer (say firewall) to allow the caller
 * to not hang for too long.  If the network condition are temporary,
 * using #size# > 0 will bring the timeout back up quickly.
 */
void
SetTimeOut(	timeoutTypes type, 
		IPAddress addr, 
		double duration, 
		long size,  
		int timedOut);


#ifdef __cplusplus
}
#endif

#endif /* TIMEOUTS_H */
