#ifndef SCITBX_ISO_SURFACE_H
#define SCITBX_ISO_SURFACE_H

// Modified by Luc J. Bourhis
//
// Downloaded from http://local.wasp.uwa.edu.au/~pbourke/geometry/polygonise
//
// Original files: CIsoSurface.h and CIsoSurface.cpp
// Last modified: 5/8/2000
// Author: Raghavendra Chandrashekara (based on source code
// provided by Paul Bourke and Cory Gene Bloyd)
// Email: rc99@doc.ic.ac.uk, rchandrashekara@hotmail.com

#include <cmath>
#include <map>
#include <boost/static_assert.hpp>
#include <scitbx/vec3.h>
#include <scitbx/array_family/tiny.h>
#include <scitbx/array_family/tiny_algebra.h>
#include <scitbx/array_family/shared.h>
#include <scitbx/array_family/accessors/traits.h>
#include <scitbx/math/utils.h>
#include <scitbx/error.h>


namespace scitbx { namespace iso_surface {

namespace triangulation_detail {

  // Lookup table used in the construction of the isosurface.
  static const int edge_table[256] = {
    0x0  , 0x109, 0x203, 0x30a, 0x406, 0x50f, 0x605, 0x70c,
    0x80c, 0x905, 0xa0f, 0xb06, 0xc0a, 0xd03, 0xe09, 0xf00,
    0x190, 0x99 , 0x393, 0x29a, 0x596, 0x49f, 0x795, 0x69c,
    0x99c, 0x895, 0xb9f, 0xa96, 0xd9a, 0xc93, 0xf99, 0xe90,
    0x230, 0x339, 0x33 , 0x13a, 0x636, 0x73f, 0x435, 0x53c,
    0xa3c, 0xb35, 0x83f, 0x936, 0xe3a, 0xf33, 0xc39, 0xd30,
    0x3a0, 0x2a9, 0x1a3, 0xaa , 0x7a6, 0x6af, 0x5a5, 0x4ac,
    0xbac, 0xaa5, 0x9af, 0x8a6, 0xfaa, 0xea3, 0xda9, 0xca0,
    0x460, 0x569, 0x663, 0x76a, 0x66 , 0x16f, 0x265, 0x36c,
    0xc6c, 0xd65, 0xe6f, 0xf66, 0x86a, 0x963, 0xa69, 0xb60,
    0x5f0, 0x4f9, 0x7f3, 0x6fa, 0x1f6, 0xff , 0x3f5, 0x2fc,
    0xdfc, 0xcf5, 0xfff, 0xef6, 0x9fa, 0x8f3, 0xbf9, 0xaf0,
    0x650, 0x759, 0x453, 0x55a, 0x256, 0x35f, 0x55 , 0x15c,
    0xe5c, 0xf55, 0xc5f, 0xd56, 0xa5a, 0xb53, 0x859, 0x950,
    0x7c0, 0x6c9, 0x5c3, 0x4ca, 0x3c6, 0x2cf, 0x1c5, 0xcc ,
    0xfcc, 0xec5, 0xdcf, 0xcc6, 0xbca, 0xac3, 0x9c9, 0x8c0,
    0x8c0, 0x9c9, 0xac3, 0xbca, 0xcc6, 0xdcf, 0xec5, 0xfcc,
    0xcc , 0x1c5, 0x2cf, 0x3c6, 0x4ca, 0x5c3, 0x6c9, 0x7c0,
    0x950, 0x859, 0xb53, 0xa5a, 0xd56, 0xc5f, 0xf55, 0xe5c,
    0x15c, 0x55 , 0x35f, 0x256, 0x55a, 0x453, 0x759, 0x650,
    0xaf0, 0xbf9, 0x8f3, 0x9fa, 0xef6, 0xfff, 0xcf5, 0xdfc,
    0x2fc, 0x3f5, 0xff , 0x1f6, 0x6fa, 0x7f3, 0x4f9, 0x5f0,
    0xb60, 0xa69, 0x963, 0x86a, 0xf66, 0xe6f, 0xd65, 0xc6c,
    0x36c, 0x265, 0x16f, 0x66 , 0x76a, 0x663, 0x569, 0x460,
    0xca0, 0xda9, 0xea3, 0xfaa, 0x8a6, 0x9af, 0xaa5, 0xbac,
    0x4ac, 0x5a5, 0x6af, 0x7a6, 0xaa , 0x1a3, 0x2a9, 0x3a0,
    0xd30, 0xc39, 0xf33, 0xe3a, 0x936, 0x83f, 0xb35, 0xa3c,
    0x53c, 0x435, 0x73f, 0x636, 0x13a, 0x33 , 0x339, 0x230,
    0xe90, 0xf99, 0xc93, 0xd9a, 0xa96, 0xb9f, 0x895, 0x99c,
    0x69c, 0x795, 0x49f, 0x596, 0x29a, 0x393, 0x99 , 0x190,
    0xf00, 0xe09, 0xd03, 0xc0a, 0xb06, 0xa0f, 0x905, 0x80c,
    0x70c, 0x605, 0x50f, 0x406, 0x30a, 0x203, 0x109, 0x0
  };

  // Lookup table used in the construction of the isosurface.
  static const int tri_table[256][16] = {
    {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 8, 3, 9, 8, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 3, 1, 2, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {9, 2, 10, 0, 2, 9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {2, 8, 3, 2, 10, 8, 10, 9, 8, -1, -1, -1, -1, -1, -1, -1},
    {3, 11, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 11, 2, 8, 11, 0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 9, 0, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 11, 2, 1, 9, 11, 9, 8, 11, -1, -1, -1, -1, -1, -1, -1},
    {3, 10, 1, 11, 10, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 10, 1, 0, 8, 10, 8, 11, 10, -1, -1, -1, -1, -1, -1, -1},
    {3, 9, 0, 3, 11, 9, 11, 10, 9, -1, -1, -1, -1, -1, -1, -1},
    {9, 8, 10, 10, 8, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 7, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 3, 0, 7, 3, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 9, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 1, 9, 4, 7, 1, 7, 3, 1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {3, 4, 7, 3, 0, 4, 1, 2, 10, -1, -1, -1, -1, -1, -1, -1},
    {9, 2, 10, 9, 0, 2, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1},
    {2, 10, 9, 2, 9, 7, 2, 7, 3, 7, 9, 4, -1, -1, -1, -1},
    {8, 4, 7, 3, 11, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {11, 4, 7, 11, 2, 4, 2, 0, 4, -1, -1, -1, -1, -1, -1, -1},
    {9, 0, 1, 8, 4, 7, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1},
    {4, 7, 11, 9, 4, 11, 9, 11, 2, 9, 2, 1, -1, -1, -1, -1},
    {3, 10, 1, 3, 11, 10, 7, 8, 4, -1, -1, -1, -1, -1, -1, -1},
    {1, 11, 10, 1, 4, 11, 1, 0, 4, 7, 11, 4, -1, -1, -1, -1},
    {4, 7, 8, 9, 0, 11, 9, 11, 10, 11, 0, 3, -1, -1, -1, -1},
    {4, 7, 11, 4, 11, 9, 9, 11, 10, -1, -1, -1, -1, -1, -1, -1},
    {9, 5, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {9, 5, 4, 0, 8, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 5, 4, 1, 5, 0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {8, 5, 4, 8, 3, 5, 3, 1, 5, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, 9, 5, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {3, 0, 8, 1, 2, 10, 4, 9, 5, -1, -1, -1, -1, -1, -1, -1},
    {5, 2, 10, 5, 4, 2, 4, 0, 2, -1, -1, -1, -1, -1, -1, -1},
    {2, 10, 5, 3, 2, 5, 3, 5, 4, 3, 4, 8, -1, -1, -1, -1},
    {9, 5, 4, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 11, 2, 0, 8, 11, 4, 9, 5, -1, -1, -1, -1, -1, -1, -1},
    {0, 5, 4, 0, 1, 5, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1},
    {2, 1, 5, 2, 5, 8, 2, 8, 11, 4, 8, 5, -1, -1, -1, -1},
    {10, 3, 11, 10, 1, 3, 9, 5, 4, -1, -1, -1, -1, -1, -1, -1},
    {4, 9, 5, 0, 8, 1, 8, 10, 1, 8, 11, 10, -1, -1, -1, -1},
    {5, 4, 0, 5, 0, 11, 5, 11, 10, 11, 0, 3, -1, -1, -1, -1},
    {5, 4, 8, 5, 8, 10, 10, 8, 11, -1, -1, -1, -1, -1, -1, -1},
    {9, 7, 8, 5, 7, 9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {9, 3, 0, 9, 5, 3, 5, 7, 3, -1, -1, -1, -1, -1, -1, -1},
    {0, 7, 8, 0, 1, 7, 1, 5, 7, -1, -1, -1, -1, -1, -1, -1},
    {1, 5, 3, 3, 5, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {9, 7, 8, 9, 5, 7, 10, 1, 2, -1, -1, -1, -1, -1, -1, -1},
    {10, 1, 2, 9, 5, 0, 5, 3, 0, 5, 7, 3, -1, -1, -1, -1},
    {8, 0, 2, 8, 2, 5, 8, 5, 7, 10, 5, 2, -1, -1, -1, -1},
    {2, 10, 5, 2, 5, 3, 3, 5, 7, -1, -1, -1, -1, -1, -1, -1},
    {7, 9, 5, 7, 8, 9, 3, 11, 2, -1, -1, -1, -1, -1, -1, -1},
    {9, 5, 7, 9, 7, 2, 9, 2, 0, 2, 7, 11, -1, -1, -1, -1},
    {2, 3, 11, 0, 1, 8, 1, 7, 8, 1, 5, 7, -1, -1, -1, -1},
    {11, 2, 1, 11, 1, 7, 7, 1, 5, -1, -1, -1, -1, -1, -1, -1},
    {9, 5, 8, 8, 5, 7, 10, 1, 3, 10, 3, 11, -1, -1, -1, -1},
    {5, 7, 0, 5, 0, 9, 7, 11, 0, 1, 0, 10, 11, 10, 0, -1},
    {11, 10, 0, 11, 0, 3, 10, 5, 0, 8, 0, 7, 5, 7, 0, -1},
    {11, 10, 5, 7, 11, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {10, 6, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 3, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {9, 0, 1, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 8, 3, 1, 9, 8, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1},
    {1, 6, 5, 2, 6, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 6, 5, 1, 2, 6, 3, 0, 8, -1, -1, -1, -1, -1, -1, -1},
    {9, 6, 5, 9, 0, 6, 0, 2, 6, -1, -1, -1, -1, -1, -1, -1},
    {5, 9, 8, 5, 8, 2, 5, 2, 6, 3, 2, 8, -1, -1, -1, -1},
    {2, 3, 11, 10, 6, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {11, 0, 8, 11, 2, 0, 10, 6, 5, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 9, 2, 3, 11, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1},
    {5, 10, 6, 1, 9, 2, 9, 11, 2, 9, 8, 11, -1, -1, -1, -1},
    {6, 3, 11, 6, 5, 3, 5, 1, 3, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 11, 0, 11, 5, 0, 5, 1, 5, 11, 6, -1, -1, -1, -1},
    {3, 11, 6, 0, 3, 6, 0, 6, 5, 0, 5, 9, -1, -1, -1, -1},
    {6, 5, 9, 6, 9, 11, 11, 9, 8, -1, -1, -1, -1, -1, -1, -1},
    {5, 10, 6, 4, 7, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 3, 0, 4, 7, 3, 6, 5, 10, -1, -1, -1, -1, -1, -1, -1},
    {1, 9, 0, 5, 10, 6, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1},
    {10, 6, 5, 1, 9, 7, 1, 7, 3, 7, 9, 4, -1, -1, -1, -1},
    {6, 1, 2, 6, 5, 1, 4, 7, 8, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 5, 5, 2, 6, 3, 0, 4, 3, 4, 7, -1, -1, -1, -1},
    {8, 4, 7, 9, 0, 5, 0, 6, 5, 0, 2, 6, -1, -1, -1, -1},
    {7, 3, 9, 7, 9, 4, 3, 2, 9, 5, 9, 6, 2, 6, 9, -1},
    {3, 11, 2, 7, 8, 4, 10, 6, 5, -1, -1, -1, -1, -1, -1, -1},
    {5, 10, 6, 4, 7, 2, 4, 2, 0, 2, 7, 11, -1, -1, -1, -1},
    {0, 1, 9, 4, 7, 8, 2, 3, 11, 5, 10, 6, -1, -1, -1, -1},
    {9, 2, 1, 9, 11, 2, 9, 4, 11, 7, 11, 4, 5, 10, 6, -1},
    {8, 4, 7, 3, 11, 5, 3, 5, 1, 5, 11, 6, -1, -1, -1, -1},
    {5, 1, 11, 5, 11, 6, 1, 0, 11, 7, 11, 4, 0, 4, 11, -1},
    {0, 5, 9, 0, 6, 5, 0, 3, 6, 11, 6, 3, 8, 4, 7, -1},
    {6, 5, 9, 6, 9, 11, 4, 7, 9, 7, 11, 9, -1, -1, -1, -1},
    {10, 4, 9, 6, 4, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 10, 6, 4, 9, 10, 0, 8, 3, -1, -1, -1, -1, -1, -1, -1},
    {10, 0, 1, 10, 6, 0, 6, 4, 0, -1, -1, -1, -1, -1, -1, -1},
    {8, 3, 1, 8, 1, 6, 8, 6, 4, 6, 1, 10, -1, -1, -1, -1},
    {1, 4, 9, 1, 2, 4, 2, 6, 4, -1, -1, -1, -1, -1, -1, -1},
    {3, 0, 8, 1, 2, 9, 2, 4, 9, 2, 6, 4, -1, -1, -1, -1},
    {0, 2, 4, 4, 2, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {8, 3, 2, 8, 2, 4, 4, 2, 6, -1, -1, -1, -1, -1, -1, -1},
    {10, 4, 9, 10, 6, 4, 11, 2, 3, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 2, 2, 8, 11, 4, 9, 10, 4, 10, 6, -1, -1, -1, -1},
    {3, 11, 2, 0, 1, 6, 0, 6, 4, 6, 1, 10, -1, -1, -1, -1},
    {6, 4, 1, 6, 1, 10, 4, 8, 1, 2, 1, 11, 8, 11, 1, -1},
    {9, 6, 4, 9, 3, 6, 9, 1, 3, 11, 6, 3, -1, -1, -1, -1},
    {8, 11, 1, 8, 1, 0, 11, 6, 1, 9, 1, 4, 6, 4, 1, -1},
    {3, 11, 6, 3, 6, 0, 0, 6, 4, -1, -1, -1, -1, -1, -1, -1},
    {6, 4, 8, 11, 6, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {7, 10, 6, 7, 8, 10, 8, 9, 10, -1, -1, -1, -1, -1, -1, -1},
    {0, 7, 3, 0, 10, 7, 0, 9, 10, 6, 7, 10, -1, -1, -1, -1},
    {10, 6, 7, 1, 10, 7, 1, 7, 8, 1, 8, 0, -1, -1, -1, -1},
    {10, 6, 7, 10, 7, 1, 1, 7, 3, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 6, 1, 6, 8, 1, 8, 9, 8, 6, 7, -1, -1, -1, -1},
    {2, 6, 9, 2, 9, 1, 6, 7, 9, 0, 9, 3, 7, 3, 9, -1},
    {7, 8, 0, 7, 0, 6, 6, 0, 2, -1, -1, -1, -1, -1, -1, -1},
    {7, 3, 2, 6, 7, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {2, 3, 11, 10, 6, 8, 10, 8, 9, 8, 6, 7, -1, -1, -1, -1},
    {2, 0, 7, 2, 7, 11, 0, 9, 7, 6, 7, 10, 9, 10, 7, -1},
    {1, 8, 0, 1, 7, 8, 1, 10, 7, 6, 7, 10, 2, 3, 11, -1},
    {11, 2, 1, 11, 1, 7, 10, 6, 1, 6, 7, 1, -1, -1, -1, -1},
    {8, 9, 6, 8, 6, 7, 9, 1, 6, 11, 6, 3, 1, 3, 6, -1},
    {0, 9, 1, 11, 6, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {7, 8, 0, 7, 0, 6, 3, 11, 0, 11, 6, 0, -1, -1, -1, -1},
    {7, 11, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {7, 6, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {3, 0, 8, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 9, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {8, 1, 9, 8, 3, 1, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1},
    {10, 1, 2, 6, 11, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, 3, 0, 8, 6, 11, 7, -1, -1, -1, -1, -1, -1, -1},
    {2, 9, 0, 2, 10, 9, 6, 11, 7, -1, -1, -1, -1, -1, -1, -1},
    {6, 11, 7, 2, 10, 3, 10, 8, 3, 10, 9, 8, -1, -1, -1, -1},
    {7, 2, 3, 6, 2, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {7, 0, 8, 7, 6, 0, 6, 2, 0, -1, -1, -1, -1, -1, -1, -1},
    {2, 7, 6, 2, 3, 7, 0, 1, 9, -1, -1, -1, -1, -1, -1, -1},
    {1, 6, 2, 1, 8, 6, 1, 9, 8, 8, 7, 6, -1, -1, -1, -1},
    {10, 7, 6, 10, 1, 7, 1, 3, 7, -1, -1, -1, -1, -1, -1, -1},
    {10, 7, 6, 1, 7, 10, 1, 8, 7, 1, 0, 8, -1, -1, -1, -1},
    {0, 3, 7, 0, 7, 10, 0, 10, 9, 6, 10, 7, -1, -1, -1, -1},
    {7, 6, 10, 7, 10, 8, 8, 10, 9, -1, -1, -1, -1, -1, -1, -1},
    {6, 8, 4, 11, 8, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {3, 6, 11, 3, 0, 6, 0, 4, 6, -1, -1, -1, -1, -1, -1, -1},
    {8, 6, 11, 8, 4, 6, 9, 0, 1, -1, -1, -1, -1, -1, -1, -1},
    {9, 4, 6, 9, 6, 3, 9, 3, 1, 11, 3, 6, -1, -1, -1, -1},
    {6, 8, 4, 6, 11, 8, 2, 10, 1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, 3, 0, 11, 0, 6, 11, 0, 4, 6, -1, -1, -1, -1},
    {4, 11, 8, 4, 6, 11, 0, 2, 9, 2, 10, 9, -1, -1, -1, -1},
    {10, 9, 3, 10, 3, 2, 9, 4, 3, 11, 3, 6, 4, 6, 3, -1},
    {8, 2, 3, 8, 4, 2, 4, 6, 2, -1, -1, -1, -1, -1, -1, -1},
    {0, 4, 2, 4, 6, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 9, 0, 2, 3, 4, 2, 4, 6, 4, 3, 8, -1, -1, -1, -1},
    {1, 9, 4, 1, 4, 2, 2, 4, 6, -1, -1, -1, -1, -1, -1, -1},
    {8, 1, 3, 8, 6, 1, 8, 4, 6, 6, 10, 1, -1, -1, -1, -1},
    {10, 1, 0, 10, 0, 6, 6, 0, 4, -1, -1, -1, -1, -1, -1, -1},
    {4, 6, 3, 4, 3, 8, 6, 10, 3, 0, 3, 9, 10, 9, 3, -1},
    {10, 9, 4, 6, 10, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 9, 5, 7, 6, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 3, 4, 9, 5, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1},
    {5, 0, 1, 5, 4, 0, 7, 6, 11, -1, -1, -1, -1, -1, -1, -1},
    {11, 7, 6, 8, 3, 4, 3, 5, 4, 3, 1, 5, -1, -1, -1, -1},
    {9, 5, 4, 10, 1, 2, 7, 6, 11, -1, -1, -1, -1, -1, -1, -1},
    {6, 11, 7, 1, 2, 10, 0, 8, 3, 4, 9, 5, -1, -1, -1, -1},
    {7, 6, 11, 5, 4, 10, 4, 2, 10, 4, 0, 2, -1, -1, -1, -1},
    {3, 4, 8, 3, 5, 4, 3, 2, 5, 10, 5, 2, 11, 7, 6, -1},
    {7, 2, 3, 7, 6, 2, 5, 4, 9, -1, -1, -1, -1, -1, -1, -1},
    {9, 5, 4, 0, 8, 6, 0, 6, 2, 6, 8, 7, -1, -1, -1, -1},
    {3, 6, 2, 3, 7, 6, 1, 5, 0, 5, 4, 0, -1, -1, -1, -1},
    {6, 2, 8, 6, 8, 7, 2, 1, 8, 4, 8, 5, 1, 5, 8, -1},
    {9, 5, 4, 10, 1, 6, 1, 7, 6, 1, 3, 7, -1, -1, -1, -1},
    {1, 6, 10, 1, 7, 6, 1, 0, 7, 8, 7, 0, 9, 5, 4, -1},
    {4, 0, 10, 4, 10, 5, 0, 3, 10, 6, 10, 7, 3, 7, 10, -1},
    {7, 6, 10, 7, 10, 8, 5, 4, 10, 4, 8, 10, -1, -1, -1, -1},
    {6, 9, 5, 6, 11, 9, 11, 8, 9, -1, -1, -1, -1, -1, -1, -1},
    {3, 6, 11, 0, 6, 3, 0, 5, 6, 0, 9, 5, -1, -1, -1, -1},
    {0, 11, 8, 0, 5, 11, 0, 1, 5, 5, 6, 11, -1, -1, -1, -1},
    {6, 11, 3, 6, 3, 5, 5, 3, 1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, 9, 5, 11, 9, 11, 8, 11, 5, 6, -1, -1, -1, -1},
    {0, 11, 3, 0, 6, 11, 0, 9, 6, 5, 6, 9, 1, 2, 10, -1},
    {11, 8, 5, 11, 5, 6, 8, 0, 5, 10, 5, 2, 0, 2, 5, -1},
    {6, 11, 3, 6, 3, 5, 2, 10, 3, 10, 5, 3, -1, -1, -1, -1},
    {5, 8, 9, 5, 2, 8, 5, 6, 2, 3, 8, 2, -1, -1, -1, -1},
    {9, 5, 6, 9, 6, 0, 0, 6, 2, -1, -1, -1, -1, -1, -1, -1},
    {1, 5, 8, 1, 8, 0, 5, 6, 8, 3, 8, 2, 6, 2, 8, -1},
    {1, 5, 6, 2, 1, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 3, 6, 1, 6, 10, 3, 8, 6, 5, 6, 9, 8, 9, 6, -1},
    {10, 1, 0, 10, 0, 6, 9, 5, 0, 5, 6, 0, -1, -1, -1, -1},
    {0, 3, 8, 5, 6, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {10, 5, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {11, 5, 10, 7, 5, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {11, 5, 10, 11, 7, 5, 8, 3, 0, -1, -1, -1, -1, -1, -1, -1},
    {5, 11, 7, 5, 10, 11, 1, 9, 0, -1, -1, -1, -1, -1, -1, -1},
    {10, 7, 5, 10, 11, 7, 9, 8, 1, 8, 3, 1, -1, -1, -1, -1},
    {11, 1, 2, 11, 7, 1, 7, 5, 1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 3, 1, 2, 7, 1, 7, 5, 7, 2, 11, -1, -1, -1, -1},
    {9, 7, 5, 9, 2, 7, 9, 0, 2, 2, 11, 7, -1, -1, -1, -1},
    {7, 5, 2, 7, 2, 11, 5, 9, 2, 3, 2, 8, 9, 8, 2, -1},
    {2, 5, 10, 2, 3, 5, 3, 7, 5, -1, -1, -1, -1, -1, -1, -1},
    {8, 2, 0, 8, 5, 2, 8, 7, 5, 10, 2, 5, -1, -1, -1, -1},
    {9, 0, 1, 5, 10, 3, 5, 3, 7, 3, 10, 2, -1, -1, -1, -1},
    {9, 8, 2, 9, 2, 1, 8, 7, 2, 10, 2, 5, 7, 5, 2, -1},
    {1, 3, 5, 3, 7, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 7, 0, 7, 1, 1, 7, 5, -1, -1, -1, -1, -1, -1, -1},
    {9, 0, 3, 9, 3, 5, 5, 3, 7, -1, -1, -1, -1, -1, -1, -1},
    {9, 8, 7, 5, 9, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {5, 8, 4, 5, 10, 8, 10, 11, 8, -1, -1, -1, -1, -1, -1, -1},
    {5, 0, 4, 5, 11, 0, 5, 10, 11, 11, 3, 0, -1, -1, -1, -1},
    {0, 1, 9, 8, 4, 10, 8, 10, 11, 10, 4, 5, -1, -1, -1, -1},
    {10, 11, 4, 10, 4, 5, 11, 3, 4, 9, 4, 1, 3, 1, 4, -1},
    {2, 5, 1, 2, 8, 5, 2, 11, 8, 4, 5, 8, -1, -1, -1, -1},
    {0, 4, 11, 0, 11, 3, 4, 5, 11, 2, 11, 1, 5, 1, 11, -1},
    {0, 2, 5, 0, 5, 9, 2, 11, 5, 4, 5, 8, 11, 8, 5, -1},
    {9, 4, 5, 2, 11, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {2, 5, 10, 3, 5, 2, 3, 4, 5, 3, 8, 4, -1, -1, -1, -1},
    {5, 10, 2, 5, 2, 4, 4, 2, 0, -1, -1, -1, -1, -1, -1, -1},
    {3, 10, 2, 3, 5, 10, 3, 8, 5, 4, 5, 8, 0, 1, 9, -1},
    {5, 10, 2, 5, 2, 4, 1, 9, 2, 9, 4, 2, -1, -1, -1, -1},
    {8, 4, 5, 8, 5, 3, 3, 5, 1, -1, -1, -1, -1, -1, -1, -1},
    {0, 4, 5, 1, 0, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {8, 4, 5, 8, 5, 3, 9, 0, 5, 0, 3, 5, -1, -1, -1, -1},
    {9, 4, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 11, 7, 4, 9, 11, 9, 10, 11, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 3, 4, 9, 7, 9, 11, 7, 9, 10, 11, -1, -1, -1, -1},
    {1, 10, 11, 1, 11, 4, 1, 4, 0, 7, 4, 11, -1, -1, -1, -1},
    {3, 1, 4, 3, 4, 8, 1, 10, 4, 7, 4, 11, 10, 11, 4, -1},
    {4, 11, 7, 9, 11, 4, 9, 2, 11, 9, 1, 2, -1, -1, -1, -1},
    {9, 7, 4, 9, 11, 7, 9, 1, 11, 2, 11, 1, 0, 8, 3, -1},
    {11, 7, 4, 11, 4, 2, 2, 4, 0, -1, -1, -1, -1, -1, -1, -1},
    {11, 7, 4, 11, 4, 2, 8, 3, 4, 3, 2, 4, -1, -1, -1, -1},
    {2, 9, 10, 2, 7, 9, 2, 3, 7, 7, 4, 9, -1, -1, -1, -1},
    {9, 10, 7, 9, 7, 4, 10, 2, 7, 8, 7, 0, 2, 0, 7, -1},
    {3, 7, 10, 3, 10, 2, 7, 4, 10, 1, 10, 0, 4, 0, 10, -1},
    {1, 10, 2, 8, 7, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 9, 1, 4, 1, 7, 7, 1, 3, -1, -1, -1, -1, -1, -1, -1},
    {4, 9, 1, 4, 1, 7, 0, 8, 1, 8, 7, 1, -1, -1, -1, -1},
    {4, 0, 3, 7, 4, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 8, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {9, 10, 8, 10, 11, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {3, 0, 9, 3, 9, 11, 11, 9, 10, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 10, 0, 10, 8, 8, 10, 11, -1, -1, -1, -1, -1, -1, -1},
    {3, 1, 10, 11, 3, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 11, 1, 11, 9, 9, 11, 8, -1, -1, -1, -1, -1, -1, -1},
    {3, 0, 9, 3, 9, 11, 1, 2, 9, 2, 11, 9, -1, -1, -1, -1},
    {0, 2, 11, 8, 0, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {3, 2, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {2, 3, 8, 2, 8, 10, 10, 8, 9, -1, -1, -1, -1, -1, -1, -1},
    {9, 10, 2, 0, 9, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {2, 3, 8, 2, 8, 10, 0, 1, 8, 1, 10, 8, -1, -1, -1, -1},
    {1, 10, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 3, 8, 9, 1, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 9, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 3, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}
  };

} // namespace triangulation_detail

/// Triangulation of an isosurface of a scalar field
template <class CoordinatesType, class ValueType, class GridType>
class triangulation
{
public:
  BOOST_STATIC_ASSERT((af::may_be_padded<GridType>::value));

  typedef CoordinatesType coordinates_type;
  typedef ValueType value_type;
  typedef GridType grid_type;
  typedef typename grid_type::index_type index_type;
  typedef typename grid_type::index_value_type index_value_type;
  typedef af::const_ref<value_type, grid_type> map_const_ref_type;

  // geometrical types
  typedef scitbx::vec3<coordinates_type> point_3d;
  typedef point_3d vector_3d;

  // element-wise
  typedef af::tiny<coordinates_type, 3> extent_3d;
  typedef extent_3d position_3d;

  typedef af::tiny<index_value_type, 3> triangle;

  /** Construct the triangulation from a map of the scalar field and iso level.

  - the map must be filled differently depending on whether the argument
    periodic is true or false. Specifically, for each dimension, the
    coordinate corresponding to the last index stored in the map, i.e.
    index n-1 where n is the focus() component, differs depending on periodic.
    In both cases, the first index 0 correspond to the coordinate 0.

    + if not periodic, then index n-1 corresponds to the right bound
      of map_extent, i.e. there are n-1 voxels in that dimension;

    + if periodic, then map_extent is the unit_cell dimensions
      and index n-1 corresponds to the very last point before the right wall
      of the unit cell, i.e. there are n voxels in that dimension.

    That conventions is to be aware of mostly for the non-periodic case since
    in the periodic one, the map is most likely going to be produced by
    other layers of the scitbx which abide to that convention.

  - map_extent is the entire parallelepiped in which the scalar field is
    sampled on a grid

  - from_here and to_there are the diagonally opposite points defining the
    parallelepiped in which the iso-surface is computed, i.e. anything outside
    of that volume is ignored

  - the flag periodic tells how to deal with a volume (from_here, to_there)
    part of which lies outside of the volume defined by map_extent:

    + if not periodic, then the former volume is clipped to the intersection
      with the latter;

    + otherwise, the former volume is accepted as it is. The latter case
      is only valid if the map is periodic of course, i.e. its accessor is one
      of c_grid_periodic or c_grid_padded_periodic.

  - The flag lazy_normals specifies whether the normals
    shall be computed when the member function normals() is called for the first
    time or whether to compute them upfront.

  - The flag ascending_normal_direction specifies whether the normals
    orientation shall be from lower to higher values of the field
    (note that the normal direction induces the order of the associated
    triangle vertices)
  */
  triangulation(map_const_ref_type map,
                value_type iso_level,
                extent_3d const& map_extent,
                position_3d const& from_here,
                position_3d const& to_there,
                bool periodic=false,
                bool lazy_normals=true,
                bool ascending_normal_direction=true)
  : map_(map),
    iso_level_(iso_level),
    from_here_(from_here), to_there_(to_there),
    lazy_normals_(lazy_normals),
    ascending_normal_direction_(ascending_normal_direction)
  {
    SCITBX_ASSERT((from_here < to_there).all_eq(true));
    SCITBX_ASSERT(!periodic || af::may_be_periodic<GridType>::value);
    index_type voxel_numbers = map.accessor().focus();
    if (!periodic) {
      from_here_ = af::each_max(from_here_, 0.);
      to_there_ = af::each_min(to_there_, map_extent);
      voxel_numbers -= index_type(1,1,1);
    }
    voxel_lengths = map_extent/voxel_numbers;
    for(unsigned i=0;i<3;i++) {
      typedef math::float_int_conversions<
        coordinates_type,
        index_value_type> fic;
      first_grid_point[i] = fic::iceil(from_here_[i]/voxel_lengths[i]);
      last_grid_point[i] = fic::ifloor(to_there_[i]/voxel_lengths[i]);
    }
    n_grid_points = last_grid_point - first_grid_point + index_type(1,1,1);
    init();
  }

  /// The vertices of the triangulation
  af::shared<point_3d> vertices() { return vertices_; }

  /// The triangles
  /** Each triangle is a triplet of indices into vertices() */
  af::shared<triangle> triangles() { return triangles_; }

  /// The normals to the isosurface at each vertex
  /** Each normal is the mean of the area vectors of those triangles which
  share the vertex that normal emanates from. Each normal is either of norm 1
  or zero. The latter means that all those aformentioned triangles are
  degenerate and that they would not need to be drawn. */
  af::shared<vector_3d> normals() {
    if (lazy_normals_) {
      calculate_normals();
      lazy_normals_ = false;
    }
    return normals_;
  }

  /// The two diagonally opposite points of the parallelepiped within which
  /// The triangulation is built.
  point_3d from_here() { return from_here_; }
  point_3d to_there() { return to_there_; }

  /// The flag of same name passed to the constructor
  bool ascending_normal_direction() { return ascending_normal_direction_; }

protected:

  enum { X=0, Y=1, Z=2 };

  struct point_3d_id {
    point_3d_id(index_value_type idx, point_3d q) : new_id(idx), p(q) {}
    point_3d_id(point_3d q) : p(q) {}
    point_3d_id() {}
    index_value_type new_id;
    point_3d p;
  };

  typedef std::map<index_value_type, point_3d_id> id_to_point_3d_id;

  bool lazy_normals_;
  bool ascending_normal_direction_;

  // The map of the scalar field
  map_const_ref_type map_;

  // The iso level
  value_type iso_level_;

  // Bounding points of the region where the iso-surface is considered
  extent_3d from_here_, to_there_;

  // First and last grid point (close range)
  index_type first_grid_point, last_grid_point;

  // Number of points in that range
  index_type n_grid_points;

  // Cell length in x, y, and z directions.
  extent_3d voxel_lengths;

  // The vertices_ which make up the isosurface.
  af::shared<point_3d> vertices_;

  // The normals_.
  af::shared<vector_3d> normals_;

  // List of point_3d's which form the isosurface.
  id_to_point_3d_id id_to_vertex;

  // List of triangles which form the triangulation of the isosurface.
  af::shared<triangle> triangles_;

  // Initialisation
  void init();

  // Process one edge from the given point
  void process_edge(index_type grid_point_idx, int edge_no);

  // Returns the edge id.
  index_value_type get_edge_id(index_type n, int edge_no);

  // Calculates the intersection point of the isosurface with an edge.
  point_3d calculate_intersection(index_type n, int edge_no);

  // Interpolates between two grid points to produce the point at which
  // the isosurface intersects an edge.
  point_3d_id interpolate(point_3d p1, point_3d p2,
                          value_type val1, value_type val2);

  // Renames vertices and triangles so that they can be efficiently accessed.
  void rename_vertices_and_triangles();

  // Calculates the normals.
  void calculate_normals();

  // Vertex ID maker
  index_value_type get_vertex_id(index_type n) {
    index_value_type result = n[0] - first_grid_point[0];
    result *= n_grid_points[1];
    result += n[1] - first_grid_point[1];
    result *= n_grid_points[2];
    result += n[2] - first_grid_point[2];
    return 3*result;
  }

};

template <class CoordinatesType, class ValueType, class GridType>
void
triangulation<CoordinatesType, ValueType, GridType>::
init()
{
  using namespace triangulation_detail;

  // Generate isosurface.
  index_type index;
  for (index[0]=first_grid_point[0]; index[0] < last_grid_point[0]; index[0]++)
  for (index[1]=first_grid_point[1]; index[1] < last_grid_point[1]; index[1]++)
  for (index[2]=first_grid_point[2]; index[2] < last_grid_point[2]; index[2]++)
  {
    // Calculate table lookup index from those vertices below the isolevel.
    index_value_type x = index[0], y = index[1], z = index[2];
    index_value_type table_index = 0;
    if (map_(x, y, z)       < iso_level_) table_index |=   1;
    if (map_(x, y+1, z)     < iso_level_) table_index |=   2;
    if (map_(x+1, y+1, z)   < iso_level_) table_index |=   4;
    if (map_(x+1, y, z)     < iso_level_) table_index |=   8;
    if (map_(x, y, z+1)     < iso_level_) table_index |=  16;
    if (map_(x, y+1, z+1)   < iso_level_) table_index |=  32;
    if (map_(x+1, y+1, z+1) < iso_level_) table_index |=  64;
    if (map_(x+1, y, z+1)   < iso_level_) table_index |= 128;

    // Now create a triangulation of the isosurface in this voxel.
    int edge_flag = edge_table[table_index];

    if ( edge_flag == 0) continue;

    if (edge_flag &   8) process_edge(index, 3);
    if (edge_flag &   1) process_edge(index, 0);
    if (edge_flag & 256) process_edge(index, 8);
    if (x == last_grid_point[X] - 1) {
      if (edge_flag &    4) process_edge(index, 2);
      if (edge_flag & 2048) process_edge(index, 11);
    }
    if (y == last_grid_point[Y] - 1) {
      if (edge_flag &   2) process_edge(index, 1);
      if (edge_flag & 512) process_edge(index, 9);
    }
    if (z == last_grid_point[Z] - 1) {
      if (edge_flag &  16) process_edge(index, 4);
      if (edge_flag & 128) process_edge(index, 7);
    }
    if (x == last_grid_point[X] - 1 && y == last_grid_point[Y] - 1) {
      if (edge_flag & 1024) process_edge(index, 10);
    }
    if (x == last_grid_point[X] - 1 && z == last_grid_point[Z] - 1) {
      if (edge_flag & 64) process_edge(index, 6);
    }
    if (y == last_grid_point[Y] - 1 && z == last_grid_point[Z] - 1) {
      if (edge_flag & 32) process_edge(index, 5);
    }

    for (index_value_type i = 0; tri_table[table_index][i] != -1; i += 3) {
      index_value_type point_id0, point_id1, point_id2;
      point_id0 = get_edge_id(index, tri_table[table_index][i]);
      point_id1 = get_edge_id(index, tri_table[table_index][i+1]);
      point_id2 = get_edge_id(index, tri_table[table_index][i+2]);
      triangles_.push_back(triangle(point_id0, point_id1, point_id2));
    }
  }

  rename_vertices_and_triangles();
  if (!lazy_normals_) calculate_normals();
}

template <class CoordinatesType, class ValueType, class GridType>
inline
void
triangulation<CoordinatesType, ValueType, GridType>::
process_edge(index_type n, int edge_no)
{
  index_value_type idx = get_edge_id(n, edge_no);
  point_3d p = calculate_intersection(n, edge_no);
  id_to_vertex.insert(typename id_to_point_3d_id::value_type(idx, point_3d_id(p)));
}


template <class CoordinatesType, class ValueType, class GridType>
inline
typename triangulation<CoordinatesType, ValueType, GridType>::index_value_type
triangulation<CoordinatesType, ValueType, GridType>::
get_edge_id(index_type n, int edge_no)
{
  int c;
  switch (edge_no) {
    case 0:  c = 1;          ;          ;          ; break;
    case 1:  c = 0;          ; n[Y] += 1;          ; break;
    case 2:  c = 1; n[X] += 1;          ;          ; break;
    case 3:  c = 0;          ;          ;          ; break;
    case 4:  c = 1;          ;          ; n[Z] += 1; break;
    case 5:  c = 0;          ; n[Y] += 1; n[Z] += 1; break;
    case 6:  c = 1; n[X] += 1;          ; n[Z] += 1; break;
    case 7:  c = 0;          ;          ; n[Z] += 1; break;
    case 8:  c = 2;          ;          ;          ; break;
    case 9:  c = 2;          ; n[Y] += 1;          ; break;
    case 10: c = 2; n[X] += 1; n[Y] += 1;          ; break;
    case 11: c = 2; n[X] += 1;          ;          ; break;
    default:
      throw SCITBX_ERROR("Internal Error: Invalid edge no.");
  }
  index_value_type vertex_id = get_vertex_id(n);
  return vertex_id + c;
}

template <class CoordinatesType, class ValueType, class GridType>
inline
typename triangulation<CoordinatesType, ValueType, GridType>::point_3d
triangulation<CoordinatesType, ValueType, GridType>::
calculate_intersection(index_type n, int edge_no)
{
  // Find one point below and one point above the iso-surface
  index_type v1 = n, v2 = n;
  switch (edge_no)
  {
    case 0:            ;           ;           ;
             v2[Y] += 1;           ;           ; break;

    case 1:  v1[Y] += 1;           ;           ;
             v2[X] += 1; v2[Y] += 1;           ; break;

    case 2:  v1[X] += 1; v1[Y] += 1;           ;
             v2[X] += 1;           ;           ; break;

    case 3:  v1[X] += 1;           ;           ;
                       ;           ;           ; break;

    case 4:            ;           ; v1[Z] += 1;
                       ; v2[Y] += 1; v2[Z] += 1; break;

    case 5:            ; v1[Y] += 1; v1[Z] += 1;
             v2[X] += 1; v2[Y] += 1; v2[Z] += 1; break;

    case 6:  v1[X] += 1; v1[Y] += 1; v1[Z] += 1;
             v2[X] += 1;           ; v2[Z] += 1; break;

    case 7:  v1[X] += 1;           ; v1[Z] += 1;
                       ;           ; v2[Z] += 1; break;

    case 8:            ;           ;           ;
                       ;           ; v2[Z] += 1; break;

    case 9:            ; v1[Y] += 1;           ;
                       ; v2[Y] += 1; v2[Z] += 1; break;

    case 10: v1[X] += 1; v1[Y] += 1;           ;
             v2[X] += 1; v2[Y] += 1; v2[Z] += 1; break;
    case 11: v1[X] += 1;           ;           ;
             v2[X] += 1;           ; v2[Z] += 1; break;
  }

  // Interpolate between v1 and v2 to find an approximation of the intersection
  // with the iso-surface
  point_3d p1 = v1 * voxel_lengths;
  point_3d p2 = v2 * voxel_lengths;
  value_type val1 = map_(v1);
  value_type val2 = map_(v2);

  CoordinatesType mu = (iso_level_ - val1)/(val2 - val1);
  return p1 + mu*(p2 - p1);
}

template <class CoordinatesType, class ValueType, class GridType>
void
triangulation<CoordinatesType, ValueType, GridType>::
rename_vertices_and_triangles()
{
  // Rename vertices.
  index_value_type next_id = 0;
  for (typename id_to_point_3d_id::iterator iter = id_to_vertex.begin();
       iter != id_to_vertex.end(); iter++)
  {
    (*iter).second.new_id = next_id++;
  }

  // Now rename triangles
  for (typename af::shared<triangle>::iterator iter = triangles_.begin();
       iter != triangles_.end(); iter++)
  {
    for (index_value_type i = 0; i < 3; i++) {
      (*iter)[i] = id_to_vertex[ (*iter)[i] ].new_id;
    }
    if(!ascending_normal_direction_) std::swap((*iter)[0], (*iter)[2]);
  }

  // Copy all the vertices into an array so that they
  // can be efficiently accessed.
  vertices_ = af::shared<point_3d>(id_to_vertex.size());
  next_id = 0;
  for (typename id_to_point_3d_id::iterator iter = id_to_vertex.begin();
       iter != id_to_vertex.end(); iter++)
  {
    vertices_[next_id++] = (*iter).second.p;
  }

  // Release the std::map we don't need anymore
  id_to_vertex.clear();
}

template <class CoordinatesType, class ValueType, class GridType>
void
triangulation<CoordinatesType, ValueType, GridType>::
calculate_normals()
{
  normals_ = af::shared<vector_3d>(vertices_.size());

  // Set all normals to 0.
  for (index_value_type i = 0; i < normals_.size(); i++) normals_[i] = 0;

  // Calculate normals_.
  for (index_value_type i = 0; i < triangles_.size(); i++) {
    index_value_type id0 = triangles_[i][0];
    index_value_type id1 = triangles_[i][1];
    index_value_type id2 = triangles_[i][2];
    vector_3d vec1 = vertices_[id1] - vertices_[id0];
    vector_3d vec2 = vertices_[id2] - vertices_[id0];
    vector_3d triangle_area_vec = vec1.cross(vec2);
    normals_[id0] += triangle_area_vec;
    normals_[id1] += triangle_area_vec;
    normals_[id2] += triangle_area_vec;
  }

  // Normalize normals
  for (index_value_type i=0; i < normals_.size(); i++) {
    value_type norm = abs(normals_[i]);
    if (norm != 0) normals_[i] /= norm;
  }
}

}} // end namespace scitbx::iso_surface

#endif // SCITBX_ISO_SURFACE_H
