/*
 * Copyright 2007 Samuel Mimram
 *
 * This file is part of ocaml-ogg.
 *
 * ocaml-ogg is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ocaml-ogg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with ocaml-ogg; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GNU Library General Public License, you may
 * link, statically or dynamically, a "work that uses the Library" with a publicly
 * distributed version of the Library to produce an executable file containing
 * portions of the Library, and distribute that executable file under terms of
 * your choice, without any of the additional requirements listed in clause 6
 * of the GNU Library General Public License.
 * By "a publicly distributed version of the Library", we mean either the unmodified
 * Library as distributed by INRIA, or a modified version of the Library that is
 * distributed under the conditions defined in clause 3 of the GNU Library General
 * Public License. This exception does not however invalidate any other reasons why
 * the executable file might be covered by the GNU Library General Public License.
 *
 */


#include <caml/alloc.h>
#include <caml/callback.h>
#include <caml/custom.h>
#include <caml/fail.h>
#include <caml/memory.h>
#include <caml/mlvalues.h>

#include <caml/signals.h>

#include <ogg/ogg.h>

#include <string.h>
#include <assert.h>

#include "ocaml-ogg.h"

value ocaml_ogg_page_create(value unit);

/***** Helper functions ******/

static value caml_copy_buffer(char *buf, int buflen)
{
  value ans;

  ans = caml_alloc_string(buflen);
  memcpy(String_val(ans), buf, buflen);

  return ans;
}

value string_of_page(ogg_page *op)
{
  value ans = caml_alloc_string(op->header_len + op->body_len);

  memcpy(String_val(ans), op->header, op->header_len);
  memcpy(String_val(ans) + op->header_len, op->body, op->body_len);

  return ans;
}

/***** Sync state *****/

static void finalize_sync_state(value s)
{
  ogg_sync_state *oy = Sync_state_val(s);
  ogg_sync_clear(oy);
  free(oy);
}

static struct custom_operations sync_state_ops =
{
  "ocaml_ogg_sync_state",
  finalize_sync_state,
  custom_compare_default,
  custom_hash_default,
  custom_serialize_default,
  custom_deserialize_default
};

CAMLprim value ocaml_ogg_sync_init(value unit)
{
  CAMLparam0();
  CAMLlocal1(ans);
  ogg_sync_state *oy = malloc(sizeof(ogg_sync_state));

  ogg_sync_init(oy);
  ans = caml_alloc_custom(&sync_state_ops, sizeof(ogg_sync_state*), 1, 0);
  Sync_state_val(ans) = oy;

  CAMLreturn(ans);
}

/***** Stream state ******/

static void finalize_stream_state(value s)
{
  ogg_stream_state *os = Stream_state_val(s);
  ogg_stream_clear(os);
  free(os);
}

static struct custom_operations stream_state_ops =
{
  "ocaml_ogg_stream_state",
  finalize_stream_state,
  custom_compare_default,
  custom_hash_default,
  custom_serialize_default,
  custom_deserialize_default
};

CAMLprim value ocaml_ogg_stream_init(value serial)
{
  CAMLparam0();
  CAMLlocal1(ans);
  ogg_stream_state *os = malloc(sizeof(ogg_stream_state));

  ogg_stream_init(os, Int_val(serial));
  ans = caml_alloc_custom(&stream_state_ops, sizeof(ogg_stream_state*), 1, 0);
  Stream_state_val(ans) = os;

  CAMLreturn(ans);
}

CAMLprim value ocaml_ogg_stream_pageout(value o_stream_state)
{
  ogg_stream_state *os = Stream_state_val(o_stream_state);
  ogg_page og;

  if(!ogg_stream_pageout(os, &og))
    caml_raise_constant(*caml_named_value("ogg_exn_not_enough_data"));

  return string_of_page(&og);
}

CAMLprim value ocaml_ogg_flush_stream(value o_stream_state)
{
  CAMLparam1(o_stream_state);
  CAMLlocal1(ans);
  ogg_stream_state *os = Stream_state_val(o_stream_state);
  ogg_page og;
  char *buf = NULL;
  int buflen = 0;
  int ret;

  while((ret = ogg_stream_flush(os, &og)) > 0)
  {
    buf = realloc(buf, buflen + og.header_len + og.body_len);
    memcpy(buf + buflen, og.header, og.header_len);
    buflen += og.header_len;
    memcpy(buf + buflen, og.body, og.body_len);
    buflen += og.body_len;
  }

  /* TODO: exception: internal ogg library error. */
  assert(ret == 0);

  ans = caml_copy_buffer(buf, buflen);
  free(buf);

  CAMLreturn(ans);
}

CAMLprim value ocaml_ogg_stream_pagesout(value o_stream_state)
{
  CAMLparam1(o_stream_state);
  CAMLlocal1(ans);
  ogg_stream_state *os = Stream_state_val(o_stream_state);
  ogg_page og;
  char *buf = NULL;
  int buflen = 0;
  int ret;

  while((ret = ogg_stream_pageout(os, &og)) > 0)
  {
    buf = realloc(buf, buflen + og.header_len + og.body_len);
    memcpy(buf + buflen, og.header, og.header_len);
    buflen += og.header_len;
    memcpy(buf + buflen, og.body, og.body_len);
    buflen += og.body_len;
  }

  /* TODO: exception: internal ogg library error. */
  assert(ret == 0);

  ans = caml_copy_buffer(buf, buflen);
  free(buf);

  CAMLreturn(ans);
}

CAMLprim value ocaml_ogg_stream_pagesout_eos(value o_stream_state)
{
  CAMLparam1(o_stream_state);
  CAMLlocal1(ans);
  ogg_stream_state *os = Stream_state_val(o_stream_state);
  ogg_page og;
  char *buf = NULL;
  int buflen = 0;
  int eos = 0;

  while(!eos)
  {
    if(!ogg_stream_pageout(os, &og)) break;

    buf = realloc(buf, buflen + og.header_len + og.body_len);
    memcpy(buf + buflen, og.header, og.header_len);
    buflen += og.header_len;
    memcpy(buf + buflen, og.body, og.body_len);
    buflen += og.body_len;

    if(ogg_page_eos(&og)) eos = 1;
  }

  ans = caml_copy_buffer(buf, buflen);
  free(buf);

  CAMLreturn(ans);
}
