(* Perl DBD database interface for mod_caml programs.
 * Copyright (C) 2003 Merjis Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: dbi_perl.ml,v 1.5 2004/03/06 16:06:34 ChriS Exp $
 *)

open Printf
open Perl

let _ = eval "use DBI"

class statement dbh sv =
object (self)
  inherit Dbi.statement dbh

  method execute args =
    let args = List.map (function
			     `Null ->
			       sv_undef ()
			   | `Int i ->
			       sv_of_int i
			   | `Float f ->
			       sv_of_float f
			   | `String s ->
			       sv_of_string s
			   | `Bool b ->
			       sv_of_bool b
			   | _ ->
			       failwith ("Dbi_perl: unknown argument "^
					 "type in execute")
			) args in
    call_method_void sv "execute" args

  method fetch1 () =
    let avref = call_method sv "fetchrow_arrayref" [] in
    if sv_is_undef avref then raise Not_found;
    prerr_endline "fetch1 - getting fields";
    let fields = list_of_av (deref_array avref) in
    prerr_endline "fetch1 - decoding types";
    let types = list_of_av (deref_array (hv_get (deref_hash sv) "TYPE")) in
    prerr_endline "fetch1 - creating row";
    let row = List.map2 (fun sv typ ->
			   let typ = int_of_sv typ in
			   prerr_endline ("typ = " ^ string_of_int typ);
			   prerr_endline ("sv = " ^ string_of_sv sv);
			   `String (string_of_sv sv)
			) fields types in
    row

  method names =
    failwith "Dbi_perl.statement#names: NOT IMPLEMENTED" (* FIXXXME *)

  method serial seq =
    failwith "sth#serial cannot be implemented for Perl DBD drivers"

  method finish () =
    call_method_void sv "finish" []

end

and connection ?host ?port ?user ?password database =

  (* XXX This should be configurable. *)
  let attrs =
    let hv = hv_empty () in
    hv_set hv "PrintError" (sv_of_int 0);
    hv_set hv "RaiseError" (sv_of_int 1);
    hv_set hv "AutoCommit" (sv_of_int 0);
    hv
  in
  let args = [sv_of_string database;
	      sv_of_string (match user with
				None -> ""
			      | Some user -> user);
	      sv_of_string (match password with
				None -> ""
			      | Some password -> password);
	      hashref attrs] in
  let sv = call_class_method "DBI" "connect" args in

object (self)
  inherit Dbi.connection ?host ?port ?user ?password database as super

  method host = host
  method port = port
  method user = user
  method password = password
  method database = database

  method database_type = "perl"

  (* This is a very literal mapping of DBI methods. In particular we ignore
   * the "closed" flag and debugging, because Perl DBI already supports
   * that for us. We also use Perl DBI statement caching.
   *)

  method prepare query =
    let stmt_sv = call_method sv "prepare" [sv_of_string query] in
    new statement (self : #Dbi.connection :> Dbi.connection) stmt_sv
  method prepare_cached query =
    let stmt_sv = call_method sv "prepare_cached" [sv_of_string query] in
    new statement (self : #Dbi.connection :> Dbi.connection) stmt_sv
  method commit () =
    super#commit ();
    call_method_void sv "commit" []
  method rollback () =
    call_method_void sv "rollback" [];
    super#rollback ()
  method close () =
    call_method_void sv "disconnect" [];
    super#close ()
  method ping () =
    bool_of_sv (call_method sv "ping" [])
end

(* Register with Dbi.Factory. *)
let () = Dbi.Factory.register "perl" (new connection)
