(* $Id: netsendmail.ml,v 1.1 2002/02/02 23:52:43 stolpmann Exp $
 * ----------------------------------------------------------------------
 *
 *)

open Netchannels
open Netmime

let sendmail_program = "/usr/lib/sendmail" ;;


let only_usascii_re = Pcre.regexp "^[\001-\127]*$";;

let specials_re = 
  Pcre.regexp "[\\<\\>\\\"\\\\\\,\\(\\)\\@\\;\\:\\.\\[\\]\\/\\=\\?]"

let exists rex s =
  try 
    ignore(Pcre.exec ~rex s); true
  with
      Not_found -> false
;;

let make_address 
    ?(in_charset = `Enc_utf8) ?(out_charset = `Enc_utf8) (hr_addr, formal_addr)=
  (* Generates addresses like "Gerd Stolpmann <gerd@gerd-stolpmann.de>".
   * hr_addr is the "human readable" part, and formal_addr is the formal
   * address. hr_addr must be encoded by [charset].
   *)
  let hr_addr = 
    Netconversion.recode_string ~in_enc:in_charset ~out_enc:out_charset hr_addr in
  let hr_words =
    if Pcre.pmatch ~rex:only_usascii_re hr_addr then begin
      (* Use double quotes to protect meta characters *)
      if exists specials_re hr_addr then
	[Mimestring.QString hr_addr]
      else 
	[Mimestring.Atom hr_addr]
    end
    else
      [Mimestring.EncodedWord((Netconversion.string_of_encoding out_charset,""),
			      "Q", 
			      hr_addr)]
  in
  (* TODO: Check syntax of formal_addr *)
  let formal_words =
    [ Mimestring.Special '<'; 
      Mimestring.Atom formal_addr; 
      Mimestring.Special '>'
    ] in
  (hr_words @ [ Mimestring.Special ' ' ] @ formal_words)
;;


let make_address_list ?in_charset ?out_charset fieldname addrs =
  let rec map addrs =
    match addrs with
	[] -> []
      | addr :: (addr' :: _ as addrs') ->
	  make_address ?in_charset ?out_charset addr @ 
	  [ Mimestring.Special ','; Mimestring.Special ' ' ] @
	  map addrs'
      | [ addr ] ->
	  make_address ?in_charset ?out_charset addr
  in

  let val_buf = Buffer.create 80 in
  let val_ch = new output_buffer val_buf in
  let maxlen = 78 in
  let hardmaxlen = 998 in
  let initlen = String.length fieldname + 2 in  (* String.length ": " = 2 *)
  Mimestring.write_value 
    ~maxlen1:(maxlen - initlen)
    ~maxlen
    ~hardmaxlen1:(hardmaxlen - initlen)
    ~hardmaxlen
    val_ch
    (map addrs);
  (fieldname, Buffer.contents val_buf)
;;


let make_unstructured_field 
      ?(in_charset = `Enc_utf8) ?(out_charset = `Enc_utf8) fieldname value =
  let value =
    Netconversion.recode_string ~in_enc:in_charset ~out_enc:out_charset value in
  let words =
    if Pcre.pmatch ~rex:only_usascii_re value then
      [ Mimestring.Atom value ]
    else
      [ Mimestring.EncodedWord((Netconversion.string_of_encoding out_charset,""), 
			       "Q", 
			       value) ]
  in
  let val_buf = Buffer.create 80 in
  let val_ch = new output_buffer val_buf in
  let maxlen = 78 in
  let hardmaxlen = 998 in
  let initlen = String.length fieldname + 2 in  (* String.length ": " = 2 *)
  Mimestring.write_value 
    ~maxlen1:(maxlen - initlen)
    ~maxlen
    ~hardmaxlen1:(hardmaxlen - initlen)
    ~hardmaxlen
    val_ch
    words;
  (fieldname, Buffer.contents val_buf)
;;


let compose
      ?(in_charset = `Enc_iso88591)
      ?(out_charset = `Enc_iso88591)
      ?from_addr ?(cc_addrs = []) ?(bcc_addrs = []) 
      ?(attachments = ([] : complex_mime_message list))
      ~to_addrs ~subject body : complex_mime_message =

  let body = 
    Netconversion.recode_string 
      ~in_enc:in_charset ~out_enc:out_charset body in

  let addr_hdr_fields =
    ( match from_addr with
	  None -> []
	| Some a -> [ make_address_list ~in_charset ~out_charset "From" [ a ] ]
    ) @
    [ make_address_list ~in_charset ~out_charset "To" to_addrs;
      make_unstructured_field ~in_charset ~out_charset "Subject" subject;
      "MIME-Version", "1.0";
      "X-Mailer", "OcamlNet (ocamlnet.sourceforge.net)";
      "Date", Netdate.mk_mail_date ~zone:Netdate.localzone (Unix.time());
    ] @
    ( if cc_addrs = [] then
	[]
      else
	[ make_address_list ~in_charset ~out_charset "Cc" cc_addrs ]
    ) @
    ( if bcc_addrs = [] then
	[]
      else
	[ make_address_list ~in_charset ~out_charset "Bcc" bcc_addrs ]
    ) 
  in
  if attachments = [] then begin
    let hdr_fields =
      addr_hdr_fields @ 
      [ "Content-Type", ("text/plain; charset=" ^ 
			 Netconversion.string_of_encoding out_charset);
	"Content-Transfer-Encoding", "quoted-printable";
      ]
    in
    let hdr = new basic_mime_header hdr_fields in
    (hdr, `Body (new memory_mime_body body))
  end
  else begin
    let outer_hdr_fields =
      addr_hdr_fields @
      [ "Content-Type", "multipart/mixed";  (* boundary is added later *)
      ] 
    in
    let inner_hdr_fields =
      [ "Content-Type", ("text/plain; charset=" ^ 
			 Netconversion.string_of_encoding out_charset);
	"Content-Transfer-Encoding", "quoted-printable";
      ]
    in
    let outer_hdr = new basic_mime_header outer_hdr_fields in
    let inner_hdr = new basic_mime_header inner_hdr_fields in
    (outer_hdr, 
     `Parts 
       ( ( inner_hdr, 
	   `Body (new memory_mime_body body) 
	 ) :: attachments 
       )
    )
  end
;;

let sendmail ?(mailer = sendmail_program) message =
  let cmd = mailer ^ " -B8BITMIME -t -i" in
  with_out_obj_channel
    (new output_command cmd)
    (fun ch ->
       write_mime_message ch message;
    )
;;

(* ======================================================================
 * History:
 * 
 * $Log: netsendmail.ml,v $
 * Revision 1.1  2002/02/02 23:52:43  stolpmann
 * 	Initial revision.
 *
 * 
 *)
