(* $Id: netstream.mli,v 2.4 2002/10/25 21:29:56 stolpmann Exp $
 * ----------------------------------------------------------------------
 *
 *)


(* A netstream is an input channel that is read block by block. The 
 * fragment of the channel currently loaded into memory is called the
 * current window of the netstream. The window can be thought as
 * look-ahead buffer.
 *
 * PICTURE:
 *
 * 0                pos_in        pos_in + window_length            EOF
 * +------------------+-------------------+--------------------------+
 *                    ====================
 *                     The current window
 *
 * You can use a netstream like an in_obj_channel, and read characters
 * and strings from the stream. The bytes come first from the look-ahead
 * buffer, and if there are not enough bytes, they are loaded from the
 * underlying channel. After every read operation it is tried to enlarge
 * the window such that it contains at least one block.
 *
 * If you want that the window becomes larger, you can call 'want' (to
 * enlarge the window to a certain size) or 'want_another_block' (to load
 * just another block from the underlying channel). Note that this affects only
 * the current window and not the future size of the window.
 *
 * Note Buffer_underrun: netstreams can cope with underruns of underlying
 * channels. An underrun happens when it is not possible to ensure the
 * minimum window size. However, it is possible that the window size
 * sinks under the minimum, but the Buffer_underrun is deferred until the
 * next call of an input method. Furthermore, there is a problem in the [skip]
 * method which may only be partially executed, i.e. the method skips some
 * bytes and then raises Buffer_underrun. 
 *)


class type in_obj_stream =
object
  inherit Netchannels.in_obj_channel
    (* The normal input operations work as usual. The window is moved after
     * every read sequence of bytes by exactly the number of bytes, and 
     * if the window length becomes smaller than the block size, it will
     * be ensured that the window will be enlarged to the block size (or
     * to the rest of the stream until EOF, whatever is smaller).
     *)

  method block_size : int
    (* The block size of the stream *)

  method window : Netbuffer.t
    (* The look-ahead window. The first byte of the window is the byte that
     * would be read next by input_char. The length of the window is returned
     * by the method [window_length]. This length may be smaller than the
     * current length of the netbuffer, i.e. the netbuffer may contain 
     * additional data that must be ignored.
     *)

  method want : int -> unit
    (* Increases the length of the window such that the length is at least
     * the passed number of bytes or that the window reaches EOF (whatever
     * happens first).
     *)

  method want_another_block : unit -> unit
    (* The same as: want block_size *)

  method window_length : int
    (* Returns the length of the window *)

  method window_at_eof : bool
    (* Whether the window is at eof *)

  method skip : int -> unit
    (* Skip the n bytes of the stream. It is not an error to skip more bytes
     * than available in the remaining stream.
     *)

end


class input_stream : 
        ?len:int -> 
	?block_size:int -> 
	Netchannels.in_obj_channel -> 
	  in_obj_stream
  (* Make an [in_obj_stream] on top of an [in_obj_channel]. The [block_size]
   * can be specified; it defaults to 4096. 
   *
   * If ~len is passed, this parameter limits the length of the channel:
   * Only the first ~len bytes are read from the input channel, then an EOF
   * is simulated even if the input channel is longer.
   *)


class sub_stream :
        ?len:int ->             (* default: no maximum length *)
	?delimiter:string ->    (* default: no delimiter *)
	in_obj_stream ->
	  in_obj_stream
  (* A sub stream is the part of the whole stream from the current position
   * to an arbitrary other position that is determined by ~len and
   * ~delimiter. ~len specifies the maximum length of the sub stream.
   * ~delimiter is an arbitrary string that indicates the end of the
   * sub stream (the delimiter is not part of the sub stream; i.e. the
   * sub stream ends immediately before the delimiter).
   *
   * While reading from the sub stream, not only the current position of
   * the sub stream moves, but also the current position of the main
   * stream. This means that it must be avoided to read data from the
   * main stream while the sub stream is in use. The typical pattern
   * is:
   * - Read from the main stream until the beginning of a section is
   *   recognized
   * - Create a sub stream at this point
   * - Read from the sub stream until EOF
   * - Continue reading the main stream. The next character of the main
   *   stream is exactly the character following the EOF of the sub stream
   *)

val print_in_obj_stream : Format.formatter -> in_obj_stream -> unit
  (* A top-loop printer for streams *)


(**********************************************************************)

type t

module Deprecated : sig

  (* This is the old interface. It should not be used in new programs. 
   * 
   * IMPORTANT NOTE: The deprecated interface differs from the new interface
   * in that the window is tried to enlarge to TWICE the block size.
   *)

  val create_from_channel : in_channel -> int option -> int -> t
    (* create_from_channel ch maxlength blocksize:
     * The new netstream reads from the channel 'ch'. If maxlength = None,
     * the channel is read until EOF. If maxlength = Some n, at most n bytes
     * are read; i.e. the netstream reads until n bytes have been read or
     * until EOF has been reached, whatever comes first. The blocksize 
     * specifies the number of bytes to read at once.
     *)

  val create_from_string : string -> t
    (* Creates a new netstream from a string. The initial window of this
     * netstream is a copy of the passed string.
     *)

  val block_size : t -> int
   (* Returns the (immutable) block size. *)

  val current_length : t -> int
    (* Returns the number of bytes read so far. *)

  val at_eos : t -> bool
    (* True iff EOS (end of stream) is reached, i.e. the last byte of the
     * window is the last byte of the stream.
     *)

  val window_position : t -> int
    (* Returns the absolute position of the current window. *)

  val window_length : t -> int
    (* Returns the length of the current window. *)

  val window : t -> Netbuffer.t
    (* Returns the current window. *)

  val move : t -> int -> unit
    (* move s n:
     * Moves the window: The first n bytes of the current window are 
     * discarded. If the window would become smaller than twice the
     * blocksize and if the end of the stream is not yet reached, another
     * block is read from the input channel and appended to the window.
     * 
     * PRECONDITION:
     * - n <= window_length
     *)

  val want : t -> int -> unit
    (* want s n:
     * If the window is smaller than n bytes, it is tried to enlarge
     * the window such that it is at least n bytes long. The enlargement
     * is not possible if the stream is not long enough; in this case
     * the window becomes as large as possible.
     *)

  val want_another_block : t -> unit
    (* Enlarges the window by another block (if possible i.e. if the stream
     * is long enough).
     *)

  val print_stream : t -> unit
end

(* ======================================================================
 * History:
 * 
 * $Log: netstream.mli,v $
 * Revision 2.4  2002/10/25 21:29:56  stolpmann
 * 	Fixed: Buffer_underrun exceptions are taken into account.
 *
 * Revision 2.3  2002/01/05 19:38:30  stolpmann
 * 	New: class [sub_stream].
 * 	The method [delete] has been renamed into [skip].
 *
 * Revision 2.2  2001/12/28 21:09:24  stolpmann
 * 	New OO interface. The type in_obj_stream extends in_obj_channel,
 * and adds the look-ahead window. The class input_stream is a layer on top
 * of an arbitrary in_obj_channel.
 * 	The implementation of the OO interface looks much more elegant
 * than the old implementation.
 * 	The old interface is still available in the sub module
 * "Deprecated".
 *
 * Revision 2.1  2001/09/14 14:22:34  stolpmann
 * 	Initial revision (sourceforge)
 *
 *
 * ======================================================================
 * Revision 1.2  2000/06/24 20:20:33  gerd
 * 	Added the toploop printer.
 *
 * Revision 1.1  2000/04/15 13:07:48  gerd
 * 	Initial revision.
 *
 * 
 *)
