/* OpenCP Module Player
 * copyright (c) '94-'05 Niklas Beisert <nbeisert@physik.tu-muenchen.de>
 *
 * PMain - main module (loads and inits all startup modules)
 *
 * revision history: (please note changes here)
 *  -nb980510   Niklas Beisert <nbeisert@physik.tu-muenchen.de>
 *    -first release
 *  -kb980717   Tammo Hinrichs <opencp@gmx.net>
 *    - plScreenChanged variable to notify the interfaces when the
 *      screen mode has changed
 *    - added command line help
 *    - replaced INI link symbol reader with _dllinfo reader
 *    - added screen mode check for avoiding redundant mode changes
 *    - various minor changes
 *  -fd981016   Felix Domke    <tmbinc@gmx.net>
 *    - Win32-Port
 *  -doj981213  Dirk Jagdmann  <doj@cubic.org>
 *    - added the nice end ansi
 *  -fd981220   Felix Domke    <tmbinc@gmx.net>
 *    - added stack dump and fault-in-faultproc-check
 *  -kb981224   Tammo Hinrichs <kb@ms.demo.org>
 *    - cleaned up dos shell code a bit (but did not help much)
 *  -doj990421  Dirk Jagdmann  <doj@cubic.org>
 *    - changed conSave(), conRestore, conInit()
 *  -fd990518   Felix Domke <tmbinc@gmx.net>
 *    - clearscreen now works in higher-modes too. dos shell now switches
 *      to mode 3
 *  -ss040613   Stian Skjelstad <stian@nixia.no>
 *    - rewritten for unix
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "types.h"

#include "pmain.h"
#include "stuff/err.h"
#include "stuff/poutput.h"
#include "binfile/binfpak.h"
#include "plinkman.h"
#include "psetting.h"

static struct initcloseregstruct *plInitClose = 0;

static int plRegisterInitClose(struct initcloseregstruct *r)
{
	if (r->Init)
	{
		int ret=r->Init();
		if (ret<0)
			return ret;
	}
	if (r->Close)
	{
		r->next=plInitClose;
		plInitClose=r;
	}
	return 0;
}

static void plCloseAll(void)
{
	while (plInitClose)
	{
		plInitClose->Close();
		plInitClose=plInitClose->next;
	}
}

static int cmdhlp(void)
{
	if (cfGetProfileString("commandline", "h", 0) || cfGetProfileString("commandline", "?", 0))
	{
		printf("\nopencp command line help\n");
		printf("Usage: ocp [<options>]* [@<playlist>]* [<modulename>]* \n");
		printf("\nOptions:\n");
		printf("-h                : show this help\n");
		printf("-c<name>          : use specific configuration\n");
		printf("-f : fileselector settings\n");
		printf("     r[0|1]       : remove played files from module list\n");
		printf("     o[0|1]       : don't scramble module list order\n");
		printf("     l[0|1]       : loop modules\n");
		printf("-v : sound settings\n");
		printf("     a{0..800}    : set amplification\n");
		printf("     v{0..100}    : set volume\n");
		printf("     b{-100..100} : set balance\n");
		printf("     p{-100..100} : set panning\n");
		printf("     r{-100..100} : set reverb\n");
		printf("     c{-100..100} : set chorus\n");
		printf("     s{0|1}       : set surround on/off\n");
		printf("     f{0..2}      : set filter (0=off, 1=AOI, 2=FOI)\n");
		printf("-s : device settings\n");
		printf("     p<name>      : use specific player device\n");
		printf("     s<name>      : use specific sampler device\n");
		printf("     w<name>      : use specific wavetable device\n");
		printf("     r{0..64000}  : sample at specific rate\n");
		printf("     8            : play/sample/mix as 8bit\n");
		printf("     m            : play/sample/mix mono\n");
		printf("-p                : quit when playlist is empty\n");
		printf("\nExample : ocp -fl0,r1 -vp75,f2 -spdevpdisk -sr48000 ftstar.xm\n");
		printf("          (for nice HD rendering of modules)\n");
		return errHelpPrinted;
	}
	return errOk;
}


extern char compiledate[], compiletime[]/*, compiledby[]*/;

static int init_modules(int argc, char *argv[])
{
	char regname[50];
	char regsline[1024];
	char const *regs;

	void *reg;
	struct mainstruct *regmain;
	int ret;

	if ((ret=cmdhlp()))
		return ret;

	lnkInit();

	fprintf(stderr, "linking default objects...\n");

	cfConfigSec="defaultconfig";

	if (cfGetProfileInt("version", "epoch", 0, 10)!=20060815)
	{

		if (isatty(2))
		{
			fprintf(stderr,"\n\n\033[1m\033[31mWARNING, ocp.ini has changed, please update or remove by doing this:\033[0m\nrm -f ~/.ocp/ocp.ini\n");

		} else {
			fprintf(stderr,"\n\nWARNING, ocp.ini has changed, please update or remove by doing this:\n\nrm -f ~/.ocp/ocp.ini\n");
		}
		sleep(5);
	}

	cfScreenSec=cfGetProfileString(cfConfigSec, "screensec", "screen");
	cfSoundSec=cfGetProfileString(cfConfigSec, "soundsec", "sound");

	lnkLink(cfGetProfileString2(cfConfigSec, "defaultconfig", "prelink", ""));
	lnkLink(cfGetProfileString("general", "prelink", ""));

	if ((lnkLink(cfGetProfileString("general", "link", ""))<0))
	{
		fprintf(stderr, "could not link default objects!\n");
		return -1;
	}

	if ((lnkLink(cfGetProfileString2(cfConfigSec, "defaultconfig", "link", ""))<0))
	{
		fprintf(stderr, "could not link default objects!\n");
		return -1;
	}

	if (pakfInit())
		return -1;

	fprintf(stderr, "running initializers...\n");

	strcpy(regsline, _lnkReadInfoReg("preinitclose"));
	regs=regsline;
	while (cfGetSpaceListEntry(regname, &regs, 49))
	{
#ifdef LD_DEBUG
		fprintf(stderr, "symbol: %s\n", regname);
#endif
		reg=_lnkGetSymbol(regname);
		ret=errSymSym;
		if (reg)
			ret=plRegisterInitClose((struct initcloseregstruct *)reg);
		else
			fprintf(stderr, "Error finding symbol %s\n", regname);
		if (ret<0)
			return errGen;
	}

	strcpy(regsline, _lnkReadInfoReg("initclose"));
	regs=regsline;
	while (cfGetSpaceListEntry(regname, &regs, 49))
	{
#ifdef LD_DEBUG
		fprintf(stderr, "symbol: %s\n", regname);
#endif
		reg=_lnkGetSymbol(regname);
		ret=errSymSym;
		if (reg)
			ret=plRegisterInitClose((struct initcloseregstruct *)reg);
		else
			fprintf(stderr, "Error finding symbol %s\n", regname);

		if (ret<0)
			return errGen;
	}

	strcpy(regsline, _lnkReadInfoReg("initcloseafter"));
	regs=regsline;
	while (cfGetSpaceListEntry(regname, &regs, 49))
	{
#ifdef LD_DEBUG
		fprintf(stderr, "symbol: %s\n", regname);
#endif
		reg=_lnkGetSymbol(regname);
		ret=errSymSym;
		if (reg)
			ret=plRegisterInitClose((struct initcloseregstruct *)reg);
		else
			fprintf(stderr, "Error finding symbol %s\n", regname);
		if (ret<0)
			return errGen;
	}
	if (!geteuid())
		if (getuid())
		{
			fprintf(stderr, "Changing user to non-root\n");
			if (seteuid(getuid()))
			{
				perror("seteuid()");
				return errGen;
			}
		}

	strcpy(regsline, _lnkReadInfoReg("main"));
	regs=regsline;
	if (!strlen(regs))
	{
		fprintf(stderr, "ERROR - No main specified in libraries\n");
		return 99999;
	}
	if (strchr(regs,' '))
	{
		fprintf(stderr, "WARNING - multiple main functions found, using first one\n");
		*strchr(regs,' ')=0;
	}

	reg=_lnkGetSymbol(regs);
	if (!reg)
		return errSymSym;

	regmain=reg;
	if (regmain->main(argc, argv)<0)
		return errGen;

	plSetTextMode(255);

	return 0;
}

void done_modules(void)
{
	plCloseAll();
	pakfClose();
	lnkFree(0);
}

static int _bootup(int argc, char *argv[])
{
	int result;
	if (isatty(2))
	{
		fprintf(stderr, "\033[33m\033[1mOpen Cubic Player for Unix \033[32mv" VERSION "\033[33m, compiled on %s, %s\n", compiledate, compiletime);
		fprintf(stderr, "\033[31m\033[22mPorted to \033[1m\033[32mUnix \033[31m\033[22mby \033[1mStian Skjelstad\033[0m\n");
	} else {
		fprintf(stderr, "Open Cubic Player for Unix v" VERSION ", compiled on %s, %s\n", compiledate, compiletime);
		fprintf(stderr, "Ported to Unix by Stian Skjelstad\n");
	}

	if (cfGetConfig(argc, argv))
		return -1;

	result=init_modules(argc, argv);
	if (result)
		if (result!=errHelpPrinted)
			fprintf(stderr, "%s\n", errGetLongString(result));

    	done_modules();

	cfCloseConfig();

	return 0;
}

struct mainstruct bootup = { _bootup };


