/*  GNU Ocrad - Optical Character Recognition program
    Copyright (C) 2003, 2004, 2005 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <algorithm>
#include <cstdio>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "block.h"
#include "blockmap.h"


// Return the total filled area of this block (no recursive)
int Block::area() const throw()
  {
  int a = 0;

  for( int row = top(); row <= bottom(); ++row )
    for( int col = left(); col <= right(); ++col )
      if( _blockmap->id( row, col ) == _id ) ++a;

  return a;
  }


// Return the central octagon filled area of this block (no recursive)
int Block::area_octagon() const throw()
  {
  int a = 0;
  int bevel = ( 29 * std::min( height(), width() ) ) / 100;
  int l =  left() + bevel;
  int r = right() - bevel;

  for( int i = 0; i < bevel; ++i )
    for( int row = top() + i, col = l - i; col <= r + i; ++col )
      if( _blockmap->id( row, col ) == _id ) ++a;

  for( int row = top() + bevel; row <= bottom() - bevel; ++row )
    for( int col = left(); col <= right(); ++col )
      if( _blockmap->id( row, col ) == _id ) ++a;

  for( int i = bevel - 1; i >= 0; --i )
    for( int row = bottom() - i, col = l - i; col <= r + i; ++col )
      if( _blockmap->id( row, col ) == _id ) ++a;

  return a;
  }


// Return the size of the central octagon of this block
int Block::size_octagon() const throw()
  {
  int bevel = ( 29 * std::min( height(), width() ) ) / 100;
  return size() - ( 2 * bevel * ( bevel + 1 ) );
  }


const Block & Block::block( int i ) const throw()
  {
  if( i < 0 || i >= blocks() )
    Ocrad::internal_error( "block, index out of bounds" );
  return _block_vector[i];
  }


bool Block::compare_id( int id, bool recursive ) const throw()
  {
  if( id == _id ) return true;
  if( recursive )
    {
    for( int i = 0; i < blocks(); ++i )
      if( _block_vector[i].compare_id( id, recursive ) ) return true;
    }
  return false;
  }


bool Block::escape_left( int row, int col ) const throw()
  {
  const Blockmap & bm = *_blockmap;
  if( bm.id( row, col ) == _id ) return false;
  int u, d;

  for( u = row; u > top() + 1; --u )
    if( bm.id( u - 1, col ) == _id ) break;
  for( d = row; d < bottom() - 1; ++d )
    if( bm.id( d + 1, col ) == _id ) break;
  while( u <= d && --col >= left() )
    {
    if( u > top() + 1 && bm.id( u, col ) != _id ) --u;
    if( d < bottom() - 1 && bm.id( d, col ) != _id ) ++d;
    while( u <= d && bm.id( u, col ) == _id ) ++u;
    while( u <= d && bm.id( d, col ) == _id ) --d;
    }
  return ( col < left() );
  }


bool Block::escape_top( int row, int col ) const throw()
  {
  const Blockmap & bm = *_blockmap;
  if( bm.id( row, col ) == _id ) return false;
  int l, r;

  for( l = col; l > left() + 1; --l )
    if( bm.id( row, l - 1 ) == _id ) break;
  for( r = col; r < right() - 1; ++r )
    if( bm.id( row, r + 1 ) == _id ) break;
  while( l <= r && --row >= top() )
    {
    if( l > left() + 1 && bm.id( row, l ) != _id ) --l;
    if( r < right() - 1 && bm.id( row, r ) != _id ) ++r;
    while( l <= r && bm.id( row, l ) == _id ) ++l;
    while( l <= r && bm.id( row, r ) == _id ) --r;
    }
  return ( row < top() );
  }


bool Block::escape_right( int row, int col ) const throw()
  {
  const Blockmap & bm = *_blockmap;
  if( bm.id( row, col ) == _id ) return false;
  int u, d;

  for( u = row; u > top() + 1; --u )
    if( bm.id( u - 1, col ) == _id ) break;
  for( d = row; d < bottom() - 1; ++d )
    if( bm.id( d + 1, col ) == _id ) break;
  while( u <= d && ++col <= right() )
    {
    if( u > top() + 1 && bm.id( u, col ) != _id ) --u;
    if( d < bottom() - 1 && bm.id( d, col ) != _id ) ++d;
    while( u <= d && bm.id( u, col ) == _id ) ++u;
    while( u <= d && bm.id( d, col ) == _id ) --d;
    }
  return ( col > right() );
  }


bool Block::escape_bottom( int row, int col ) const throw()
  {
  const Blockmap & bm = *_blockmap;
  if( bm.id( row, col ) == _id ) return false;
  int l, r;

  for( l = col; l > left() + 1; --l )
    if( bm.id( row, l - 1 ) == _id ) break;
  for( r = col; r < right() - 1; ++r )
    if( bm.id( row, r + 1 ) == _id ) break;
  while( l <= r && ++row <= bottom() )
    {
    if( l > left() + 1 && bm.id( row, l ) != _id ) --l;
    if( r < right() - 1 && bm.id( row, r ) != _id ) ++r;
    while( l <= r && bm.id( row, l ) == _id ) ++l;
    while( l <= r && bm.id( row, r ) == _id ) --r;
    }
  return ( row > bottom() );
  }


int Block::follow_top( int row, int col ) const throw()
  {
  const Blockmap & bm = *_blockmap;
  if( bm.id( row, col ) != _id ) return row;
  bool array[width()];

  for( int c = col; c >= left(); --c )
    {
    if( bm.id( row, c ) == _id ) array[c-left()] = true;
    else { for( c -= left(); c >= 0; --c ) array[c] = false; break; }
    }
  for( int c = col + 1; c <= right(); ++c )
    {
    if( bm.id( row, c ) == _id ) array[c-left()] = true;
    else { for( c -= left(); c < width(); ++c ) array[c] = false; break; }
    }

  while( --row >= top() )
    {
    bool alive = false;
    for( int i = 0; i < width(); ++i ) if( array[i] )
      { if( bm.id( row, left() + i ) != _id ) array[i] = false;
      else alive = true; }
    if( !alive ) break;

    for( int i = 1; i < width(); ++i )
      if( array[i-1] && !array[i] &&
          bm.id( row, left() + i ) == _id ) array[i] = true;
    for( int i = width() - 2; i >= 0; --i )
      if( array[i+1] && !array[i] &&
          bm.id( row, left() + i ) == _id ) array[i] = true;
    }
  return row + 1;
  }


int Block::follow_bottom( int row, int col ) const throw()
  {
  const Blockmap & bm = *_blockmap;
  if( bm.id( row, col ) != _id ) return row;
  bool array[width()];

  for( int c = col; c >= left(); --c )
    {
    if( bm.id( row, c ) == _id ) array[c-left()] = true;
    else { for( c -= left(); c >= 0; --c ) array[c] = false; break; }
    }
  for( int c = col + 1; c <= right(); ++c )
    {
    if( bm.id( row, c ) == _id ) array[c-left()] = true;
    else { for( c -= left(); c < width(); ++c ) array[c] = false; break; }
    }

  while( ++row <= bottom() )
    {
    bool alive = false;
    for( int i = 0; i < width(); ++i ) if( array[i] )
      { if( bm.id( row, left() + i ) != _id ) array[i] = false;
      else alive = true; }
    if( !alive ) break;

    for( int i = 1; i < width(); ++i )
      if( array[i-1] && !array[i] &&
          bm.id( row, left() + i ) == _id ) array[i] = true;
    for( int i = width() - 2; i >= 0; --i )
      if( array[i+1] && !array[i] &&
          bm.id( row, left() + i ) == _id ) array[i] = true;
    }
  return row - 1;
  }


int Block::seek_left( int row, int col, bool black ) const throw()
  {
  const Blockmap & bm = *_blockmap;

  for( ; col > left(); --col )
    if( ( bm.id( row, col - 1 ) == _id ) == black ) break;
  return col;
  }


int Block::seek_top( int row, int col, bool black ) const throw()
  {
  const Blockmap & bm = *_blockmap;

  for( ; row > top(); --row )
    if( ( bm.id( row - 1, col ) == _id ) == black ) break;
  return row;
  }


int Block::seek_right( int row, int col, bool black ) const throw()
  {
  const Blockmap & bm = *_blockmap;

  for( ; col < right(); ++col )
    if( ( bm.id( row, col + 1 ) == _id ) == black ) break;
  return col;
  }


int Block::seek_bottom( int row, int col, bool black ) const throw()
  {
  const Blockmap & bm = *_blockmap;

  for( ; row < bottom(); ++row )
    if( ( bm.id( row + 1, col ) == _id ) == black ) break;
  return row;
  }


// Looks for an inverted-U-shaped curve near the top, then tests which of
// the vertical bars goes deeper
bool Block::top_hook( int *hdiff ) const throw()
  {
  const Blockmap & bm = *_blockmap;
  int row, lcol = 0, rcol = 0, black_section = 0, wmax = 0;

  for( row = top() + 1; row < vcenter(); ++row )
    {
    int l = -1, r = -2;
    bool prev_black = false;
    black_section = 0;
    for( int col = left(); col <= right(); ++col )
      {
      bool black = ( bm.id( row, col ) == _id );
      if( black )
        {
        if( !prev_black && ++black_section == 2 ) rcol = col;
        r = col; if( l < 0 ) l = col;
        }
      else if( prev_black && black_section == 1 ) lcol = col - 1;
      prev_black = black;
      }
    r = r - l + 1;
    if( 10 * r <= 9 * wmax ) return false;
    if( r > wmax ) wmax = r ;
    if( black_section >= 2 ) break;
    }

  if( black_section != 2 ) return false;
  if( escape_top( row, lcol + 1 ) ) return false;
  int lrow = follow_bottom( row, lcol ), rrow = follow_bottom( row, rcol );
  if( lrow <= row || rrow <= row ) return false;
  if( hdiff ) *hdiff = lrow - rrow;
  return true;
  }


// Looks for an U-shaped curve near the bottom, then tests which of
// the vertical bars is taller
bool Block::bottom_hook( int *hdiff ) const throw()
  {
  const Blockmap & bm = *_blockmap;
  int row, lcol = 0, rcol = 0, black_section = 0, wmax = 0;

  for( row = bottom(); row > vpos( 80 ); --row )
    {
    int l, r;
    for( l = left(); l <= right(); ++l ) if( bm.id( row, l ) == _id ) break;
    for( r = right(); r > l; --r ) if( bm.id( row, r ) == _id ) break;
    const int w = r - l + 1;
    if( w > wmax ) wmax = w;
    if( 4 * w >= width() )
      {
      int i;
      for( i = l + 1; i < r; ++i ) if( bm.id( row, i ) != _id ) break;
      if( i >= r ) break;
      }
    }

  if( row > vpos( 80 ) ) while( --row > vcenter() )
    {
    int l = -1, r = -2;
    bool prev_black = false;
    black_section = 0;
    for( int col = left(); col <= right(); ++col )
      {
      bool black = ( bm.id( row, col ) == _id );
      if( black )
        {
        if( !prev_black && ++black_section == 2 ) rcol = col;
        r = col; if( l < 0 ) l = col;
        }
      else if( prev_black && black_section == 1 ) lcol = col - 1;
      prev_black = black;
      }
    const int w = r - l + 1;
    if( black_section > 2 || 10 * w <= 8 * wmax ) break;
    if( w > wmax ) wmax = w;
    if( black_section == 2 && rcol - lcol >= 2 )
      {
      if( escape_bottom( row, lcol + 1 ) ) break;
      if( hdiff ) *hdiff = follow_top( row, lcol ) - follow_top( row, rcol );
      return true;
      }
    }
  return false;
  }


void Block::print( FILE * outfile, int sp ) const throw()
  {
  char ch = (_id > 0) ? 'O' : '-';
  for( int row = top(); row <= bottom(); ++row)
    {
    for( int i = 0; i < sp; ++i ) std::putc( ' ', outfile );
    for( int col = left(); col <= right(); ++col)
      {
      if( _blockmap->id( row, col ) == _id )
        std::fprintf( outfile, " %c", ch );
      else std::fprintf( outfile, " " );
      }
    std::fputs( "\n", outfile );
    }
  std::fputs( "\n", outfile );

  if( sp >= 0 )
    for( int i = 0; i < blocks(); ++i )
      _block_vector[i].print( outfile, sp + 8 );
  }


// Returns true if height is or can be adjusted
bool Block::adjust_height() throw()
  {
  int row1, row2;

  for( row1 = top(); row1 <= bottom(); ++row1 )
    for( int col = left(); col <= right(); ++col )
      if( _blockmap->id( row1, col ) == _id ) goto L1;
  L1:
  for( row2 = bottom(); row2 >= row1; --row2 )
    for( int col = left(); col <= right(); ++col )
      if( _blockmap->id( row2, col ) == _id ) goto L2;
  L2:
  if( row1 > top() || row2 < bottom() )
    { if( row1 <= row2 ) { top( row1 ); bottom( row2 ); } else return false; }
  return true;
  }


void Block::delete_block( int i ) throw()
  {
  if( i < 0 || i >= blocks() )
    Ocrad::internal_error( "delete_block, index out of bounds" );
  _block_vector.erase( _block_vector.begin() + i );
  }


void Block::shift_block( const Block & b ) throw()
  {
  int i = blocks() - 1;
  for( ; i >= 0; --i )
    {
    Block & bi = _block_vector[i];
    if( b.top() > bi.top() ) break;
    if( b.top() == bi.top() && b.left() >= bi.left() ) break;
    }
  _block_vector.insert( _block_vector.begin() + i + 1, b );
  }


void Block::hierarchize_blocks( std::vector< Block > & block_vector ) throw()
  {
  int blocks = block_vector.size();
  for( int i1 = blocks - 2; i1 >= 0; --i1 )
    {
    for( int i2 = i1 + 1; i2 < blocks; )
      {
      Block & b1 = block_vector[i1];
      Block & b2 = block_vector[i2];
      if( b2.top() >= b1.bottom() ) break;
      bool shift = false;
      if( b1.strictly_includes( b2 ) )
        {
        if( ( b1.id() > 0 && b2.id() < 0 ) || ( b1.id() < 0 && b2.id() > 0 ) )
          { shift = true; b1.shift_block( b2 ); }
        else
          {
          for( int i3 = 0; i3 < b1.blocks(); ++i3 )
            {
            Block & b3 = b1._block_vector[i3];
            if( b3.strictly_includes( b2 ) )
              { shift = true; b3.shift_block( b2 ); }
            }
          }
        }
      if( !shift ) ++i2;
      else { --blocks; block_vector.erase( block_vector.begin() + i2 ); }
      }
    }
  }
