// Copyright (C) 2005 Shai Ayal <shaiay@users.sourceforge.net>
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  

#include "surface.h"
#include "axes.h"
#include "figure.h"
#include "line_plotter.h"
#include "mathutils.h"

#define OBJ_NAME "Surface"

Surface::Surface(ocpl::Handle Parent, 
	     Matrix& XData, 
	     Matrix& YData, 
	     Matrix& ZData) : Object(Parent)
{
  Properties["XData"]     = new Matrix(XData,1);
  Properties["YData"]     = new Matrix(YData,1);
  Properties["ZData"]     = new Matrix(ZData,1);
  //Properties["CData"]     = new Matrix(ZData,1);
  
  SET_TYPE;
  
  COPY_DEFAULT(Clipping,Radio);
  COPY_DEFAULT(LineWidth,Scalar);
  COPY_DEFAULT(FaceColor,ColorRadio);
  COPY_DEFAULT(EdgeColor,ColorRadio);
  COPY_DEFAULT(LineStyle,Radio);
  COPY_DEFAULT(Marker,Radio);
  COPY_DEFAULT(MarkerFaceColor,Color);
  COPY_DEFAULT(MarkerEdgeColor,Color);
  COPY_DEFAULT(MarkerSize,Scalar);
  COPY_DEFAULT(MeshStyle,Radio);
  COPY_DEFAULT(FaceAlpha,Scalar);
  // update limits
  FindMinMax(XData,min[0],max[0],lmin[0],lmax[0],ZData.nc());
  FindMinMax(YData,min[1],max[1],lmin[1],lmax[1],ZData.nr());
  FindMinMax(ZData,min[2],max[2],lmin[2],lmax[2]);

  // init display list
  display_list = 0;
}

void Surface::draw()
{
  IS_VISIBLE;

  // if we have a list, use it
//   if(display_list) {
//     SET_CLIPPING;
//     glCallList(display_list);
//     UNSET_CLIPPING;
//     return;
//   }

  MAKE_REF(xdata,Matrix);
  MAKE_REF(ydata,Matrix);
  MAKE_REF(zdata,Matrix);
  MAKE_REF(clipping,Radio);
  MAKE_REF(linewidth,Scalar);
  MAKE_REF(linestyle,Radio);
  MAKE_REF(facecolor,ColorRadio); 
  MAKE_REF(edgecolor,ColorRadio); 
  MAKE_REF(markerfacecolor,Color);
  MAKE_REF(markeredgecolor,Color);
  MAKE_REF(marker,Radio);
  MAKE_REF(markersize,Scalar);
  MAKE_REF(meshstyle,Radio);
  MAKE_REF(facealpha,Scalar);

  Axes* axes  = dynamic_cast<Axes*>(FindParentOfType("axes"));
  Figure* fig = dynamic_cast<Figure*>(FindParentOfType("Figure"));
  Radio& xscale = ::Get<Radio>(axes,"xscale");
  Radio& yscale = ::Get<Radio>(axes,"yscale");
  Matrix& clim  = ::Get<Matrix>(axes,"clim");
  Matrix& cmap  = ::Get<Matrix>(fig,"ColorMap");
  Radio None("|{none}|");

  SET_CLIPPING;
  
  //
  // 1---2
  // |   |
  // |   |
  // 0---3
  //


//   display_list = glGenLists (1);
//   glNewList(display_list, GL_COMPILE_AND_EXECUTE);


  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  //glClearColor (0.0, 0.0, 0.0, 0.0);

  if (facecolor.val()=="flat")
    glShadeModel(GL_FLAT);
  else
    glShadeModel(GL_SMOOTH);

  double x0,y0,x1,y1,x2,y2,x3,y3,z0,z1,z2,z3;
  double midpx,midpy,midpz;
  double cv[4]={1.0,1.0,1.0,1.0};
  double face_alpha_val = facealpha.GetVal(); 
  bool face_color_interp = (facecolor.val()=="interp");
  bool face_color_flat = (facecolor.val()=="flat");
  bool face_color_none = (facecolor.val()=="none");
  bool interp_facet = !facecolor.IsColor()&&face_color_interp;
  for(long xi = 0 ; xi < zdata.nc()-1 ; xi++) {
    for(long yi = 0 ; yi < zdata.nr()-1 ; yi++) {

      if(xdata.len()==0) {
	x0 = x1 = xi+1;
	x2 = x3 = xi+2;
      }
      else if(xdata.len()==zdata.nc()) {
	x0 = x1 = xdata(xi);
	x2 = x3 = xdata(xi+1);
      }
      else {
	x0 = xdata(yi,xi);
	x1 = xdata(yi+1,xi);
	x2 = xdata(yi+1,xi+1);
	x3 = xdata(yi,xi+1);
      }

      if(ydata.len()==0) {
	y0 = y3 = yi+1.;
	y1 = y2 = yi+2.;
      }
      else if(ydata.len()==zdata.nr()) {
	y0 = y3 = ydata(yi);
	y1 = y2 = ydata(yi+1);
      }
      else {
	y0 = ydata(yi,xi);
	y1 = ydata(yi+1,xi);
	y2 = ydata(yi+1,xi+1);
	y3 = ydata(yi,xi+1);
      }
      z0 = zdata(yi,xi);
      z1 = zdata(yi+1,xi);
      z2 = zdata(yi+1,xi+1);
      z3 = zdata(yi,xi+1);
      midpx = (x0+x1+x2+x3)/4.;
      midpy = (y0+y1+y2+y3)/4.;
      midpz = (z0+z1+z2+z3)/4.;
      glBegin(GL_TRIANGLE_FAN);
      
      if (facecolor.IsColor()) {
	facecolor.SetAlpha(face_alpha_val);
        facecolor.GetRGBA(cv);
      }
      else if (face_color_flat) {
        cv[3]=face_alpha_val;
        InterpColor(cmap, clim, z0, cv);
      }
      else if (face_color_none) {
        // draw white facets, cv4 is already {1,1,1,1}
        cv[3]=1.0;
        facecolor.SetRGBA(cv);
      }
      else if (face_color_interp) {
        cv[3]=face_alpha_val;
	InterpColor(cmap, clim, z0, cv);
      }
      else {
        // should not happen
      }
      glColor4dv(cv);
      glVertex3d(x0,y0,0);
      if (interp_facet) {
        InterpColor(cmap, clim, z1, cv);
      }
      glColor4dv(cv);
      glVertex3d(x1,y1,0);
      if (interp_facet) {
        InterpColor(cmap, clim, z2, cv);
      }
      glColor4dv(cv);
      glVertex3d(x2,y2,0);
      if (interp_facet) {
        InterpColor(cmap, clim, z3, cv);
      }
      glColor4dv(cv);
      glVertex3d(x3,y3,0);
      glEnd();
    }
  }
 
  if (edgecolor.IsColor()) {
    NEXT_LAYER;
    //plot vertical edges ||
    if (meshstyle()=="both"||meshstyle()=="column") {
      if (ydata.len()==0) {
        y0 = 1.0;
        y1 = zdata.nr();
      }
      else {
        y0 = ydata(0,0);
        y1 = ydata(zdata.nr()-1,0);
      }
      for(unsigned long xi = 0 ; xi < zdata.nc()-1 ; xi++) {
        if (xdata.len()==0)
          x0 = xi + 1.0;
        else
          x0 = xdata(0,xi);
	line_plotter::Instance().plot(*Real2Matrix(x0,x0),
                                *Real2Matrix(y0,y1),
                                linestyle,
                                &edgecolor,
                                linewidth(),
                                markerfacecolor,
                                markeredgecolor,
                                marker,
                                markersize(),
                                axes,
                                None,
                                0,0,0,
                                printing,
                                false);
      }
    }
    //plot horizontal edges --
    if (meshstyle()=="both"||meshstyle()=="row") {
      if (xdata.len()==0) {
        x0 = 1.0;
        x1 = zdata.nc();
      }
      else {
        x0 = xdata(0,0);
        x1 = xdata(0,zdata.nc()-1);
      }
      for(unsigned long yi = 0 ; yi < zdata.nr()-1 ; yi++) {
        if (ydata.len()==0)
          y0 = yi + 1.0;
        else
          y0 = ydata(yi,0);
	line_plotter::Instance().plot(*Real2Matrix(x0,x1),
                                *Real2Matrix(y0,y0),
                                linestyle,
                                &edgecolor,
                                linewidth(),
                                markerfacecolor,
                                markeredgecolor,
                                marker,
                                markersize(),
                                axes,
                                None,
                                0,0,0,
                                printing,
                                false);
      }
    }
  }
//   glEndList();
  
  UNSET_CLIPPING;
}
