/***************************************************************************
                          converter.h  -  description
                             -------------------
    begin                : Tue Dec 4 2001
    copyright            : (C) 2001 by Michael von Mengershausen
    email                : mengers@cns.mpg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CONVERTER_H
#define CONVERTER_H

#include <limits>
#include <assert.h>
#include <stdint.h>

#include <tjutils/tjcomplex.h>
#include <tjutils/tjtypes.h>
#include <tjutils/tjlog.h>

/**
  * @addtogroup odindata
  * @{
  */

// helper class used for debugging the odindata component
class OdinData {
 public:
  static const char* get_compName();
};

////////////////////////////////////////////////////////////////////

/**
  * Scaling strategy when converting data to integer types:
  * - noscale: No scaling
  * - autoscale: scale/offset values so they will fit into the numeric limits of the destination (0 < scaling factor < inv)
  * - noupscale: dont upscale values to fit them into the destination (0 < scaling factor <=1)
  * Note: if destination is non integer no scaling is done at all
  */
enum autoscaleOption {noscale, autoscale,noupscale};

////////////////////////////////////////////////////////////////////

/**
  * Helper class for conversion between numbers
  */
class Converter {

public:



//////////////////////////////////////////////////////////
// get_elements function overloading

//  specializations for complex type
static unsigned int get_elements(const STD_complex&) {
  return 2;
}

/**
  * Returns number of scalar numbers in this number type, e.g. complex has 2
  */
template<typename T>
static unsigned int get_elements(const T&) {
  return 1;
}


//////////////////////////////////////////////////////////


/**
  * Converts array 'src' with 'srcsize' elements to array 'dst' with 'dstsize' elements
  * Will scale "around 0" if 0 is part of the source value range. Elsewise values will be offset towards 0.
  * If destination is unsigned values will be offset to be positive domain if necessary.
  * If autoscaleOption is "noscale" or if destination is floating point no scaling is done at all.
  */
template<typename Src, typename Dst>
static void convert_array(const Src* src, Dst* dst, unsigned int srcsize, unsigned int dstsize, autoscaleOption scaleopt=autoscale) {
  Log<OdinData> odinlog("Converter","convert_array");
  init();
  unsigned int srcstep=get_elements(*dst);
  unsigned int dststep=get_elements(*src);
  bool doScale = (scaleopt!=noscale && std::numeric_limits<Dst>::is_integer);

  if(dststep*srcsize != srcstep*dstsize) {
    ODINLOG(odinlog,warningLog) << "size mismatch: dststep(" << dststep << ") * srcsize(" << srcsize << ") != srcstep(" << srcstep << ") * dstsize(" << dstsize << ")" << STD_endl;
  }

  double scale=1.0;
  double offset=0.0;
  if(doScale) {
    const double domain_minus=creal(std::numeric_limits<Dst>::min());//negative value domain of this dst
    const double domain_plus =creal(std::numeric_limits<Dst>::max());//positive value domain of this dst
    double minval=std::numeric_limits<double>::min();
    double maxval=std::numeric_limits<double>::max();
    if(srcsize>0) minval=maxval=creal(src[0]);
    for(unsigned int i=1; i<srcsize; i++) {
      if(src[i]<minval) minval=creal(src[i]);
      if(src[i]>maxval) maxval=creal(src[i]);
    }

    ODINLOG(odinlog,normalDebug) << "src Range:" << minval << "=>" << maxval << STD_endl;
    ODINLOG(odinlog,normalDebug) << "dst Domain:" << domain_minus << "=>" << domain_plus << STD_endl;

    assert(domain_minus<=0 && domain_plus>=0); //I think we can assume this
    assert(domain_minus<domain_plus);//we also should assume this

    //set offset for src
    //if all src is completly on positive domain, or if there is no negative domain use minval
    //else if src is completly on negative dmain, or if there is no positive domain use maxval
    //elsewise leave it at 0 and scale both sides
    if(minval>0 || !domain_minus)offset=-minval;
    else if(maxval<0 || !domain_plus)offset=-maxval;

    //calculate range of values which will be on postive/negative domain when offset is applied
    const double range_plus =maxval+offset; //allways >=0
    const double range_minus=minval+offset; //allways <=0

    //set scaling factor to fit src-range into dst domain
    //some compilers dont make x/0 = inf, so we use std::numeric_limits<double>::max() instead, in this case
    const double scale_plus =
        range_plus ? domain_plus/range_plus :
        std::numeric_limits<double>::max();
    const double scale_minus=
        range_minus ? domain_minus/ range_minus:
        std::numeric_limits<double>::max();
    ODINLOG(odinlog,normalDebug) << "scale_minus/scale_plus=" << scale_minus << "/" << scale_plus << STD_endl;

    scale = STD_min(scale_plus,scale_minus);//get the smaller scaling factor so the bigger range will fit into his domain
    if(scale<1){
      ODINLOG(odinlog,normalDebug) << "Downscaling your values by Factor " << scale << " you might lose information."<< STD_endl;
    } else if(scaleopt==noupscale){
      if(scale>1) {
        ODINLOG(odinlog,normalDebug) << "upscale not given, clamping scale " << scale << " to 1"<< STD_endl;
      }
      scale=1;
    }
    doScale=(scale!=1. || offset);
    offset*=scale;//calc offset for dst
  }
  if(doScale)ODINLOG(odinlog,normalDebug) << "converting with scale/offset=" << scale << "/" << offset << STD_endl;
  else ODINLOG(odinlog,normalDebug) << "converting without scaling" << STD_endl;

  if(srcstep==dststep)//use common convert for data of same complexity
  {
     if(doScale)
      convert_array_impl(src,dst, srcsize < dstsize?srcsize:dstsize,scale,offset);
    else
      convert_array_impl(src,dst, srcsize < dstsize?srcsize:dstsize);
  }
  else //do generic convert for data of different complexity
  {
    unsigned int srcindex=0, dstindex=0;
    while(srcindex<srcsize && dstindex<dstsize) {
      convert(src+srcindex, dst+dstindex,scale,offset);
      srcindex+=srcstep;
      dstindex+=dststep;
    }
  }
}

static void init();

private:

//////////////////////////////////////////////////////////
// convert

/**
  * simple convert just do "inc" on both pointers and "dec" on the counter
  */
template<typename Src, typename Dst> static void convert_array_impl(const Src* src, Dst* dst, unsigned int count,double scale=1,double offset=0) {
	Log<OdinData> odinlog("Converter","convert_array_impl(generic)");
	ODINLOG(odinlog,normalDebug) << "generic convert " << TypeTraits::type2label(*src) << "=>" << TypeTraits::type2label(*dst) << STD_endl;
	ODINLOG(odinlog,normalDebug) << "scale/offset=" << scale << "/" << offset << STD_endl;
	while(count--)
		convert(src++, dst++,scale,offset);
}

#ifdef USE_OIL
#define DECL_CONVERT(SRC_TYPE,DST_TYPE,SRC_KEY,DST_KEY) static void convert_array_impl(const SRC_TYPE  *src,DST_TYPE *dst, unsigned int count);
#define DECL_SCALED_CONVERT(SRC_TYPE,DST_TYPE,SRC_KEY,DST_KEY) static void convert_array_impl(const SRC_TYPE  *src,DST_TYPE *dst, unsigned int count,const double scale,const double offset);

//>>s32
DECL_CONVERT(float,int32_t,f32,s32)
DECL_CONVERT(double,int32_t,f64,s32)
DECL_CONVERT(uint32_t,int32_t,u32,s32)
DECL_CONVERT(int16_t,int32_t,s16,s32)
DECL_CONVERT(uint16_t,int32_t,u16,s32)
DECL_CONVERT(int8_t,int32_t,s8,s32)
DECL_CONVERT(uint8_t,int32_t,u8,s32)

//>>u32
//DECL_CONVERT(float,uint32_t,f32,u32) conversion to u32 is broken (https://bugs.freedesktop.org/show_bug.cgi?id=16524)
//DECL_CONVERT(double,uint32_t,f64,u32)
DECL_CONVERT(int32_t,uint32_t,s32,u32)
//DECL_CONVERT(int16_t,uint32_t,s16,u32) ** Not available in liboil - but should be imho
DECL_CONVERT(uint16_t,uint32_t,u16,u32)
//DECL_CONVERT(int8_t,uint32_t,s8,u32) ** Not available in liboil - but should be imho
DECL_CONVERT(uint8_t,uint32_t,u8,u32)

//>>s16
DECL_CONVERT(float,int16_t,f32,s16)
DECL_CONVERT(double,int16_t,f64,s16)
DECL_CONVERT(int32_t,int16_t,s32,s16)
DECL_CONVERT(uint32_t,int16_t,u32,s16)
DECL_CONVERT(uint16_t,int16_t,u16,s16)
DECL_CONVERT(int8_t,int16_t,s8,s16)
DECL_CONVERT(uint8_t,int16_t,u8,s16)

//>>u16
DECL_CONVERT(float,uint16_t,f32,u16)
DECL_CONVERT(double,uint16_t,f64,u16)
DECL_CONVERT(int32_t,uint16_t,s32,u16)
DECL_CONVERT(uint32_t,uint16_t,u32,u16)
DECL_CONVERT(int16_t,uint16_t,s16,u16)
//DECL_CONVERT(int8_t,uint16_t,s8,u16) ** Not available in liboil - but should be imho
DECL_CONVERT(uint8_t,uint16_t,u8,u16)

//>>s8
DECL_CONVERT(float,int8_t,f32,s8)
DECL_CONVERT(double,int8_t,f64,s8)
DECL_CONVERT(int32_t,int8_t,s32,s8)
DECL_CONVERT(uint32_t,int8_t,u32,s8)
DECL_CONVERT(int16_t,int8_t,s16,s8)
DECL_CONVERT(uint16_t,int8_t,u16,s8)
DECL_CONVERT(uint8_t,int8_t,u8,s8)

//>>u8
DECL_CONVERT(float,uint8_t,f32,u8)
DECL_CONVERT(double,uint8_t,f64,u8)
DECL_CONVERT(int32_t,uint8_t,s32,u8)
DECL_CONVERT(uint32_t,uint8_t,u32,u8)
DECL_CONVERT(int16_t,uint8_t,s16,u8)
DECL_CONVERT(uint16_t,uint8_t,u16,u8)
DECL_CONVERT(int8_t,uint8_t,s8,u8)

//>>f32
DECL_CONVERT(double,float,f64,f32)
DECL_CONVERT(int32_t,float,s32,f32)
DECL_CONVERT(uint32_t,float,u32,f32)
DECL_CONVERT(int16_t,float,s16,f32)
DECL_CONVERT(uint16_t,float,u16,f32)
DECL_CONVERT(int8_t,float,s8,f32)
DECL_CONVERT(uint8_t,float,u8,f32)

//>>f64
DECL_CONVERT(float,double,f32,f64)
DECL_CONVERT(int32_t,double,s32,f64)
DECL_CONVERT(uint32_t,double,u32,f64)
DECL_CONVERT(int16_t,double,s16,f64)
DECL_CONVERT(uint16_t,double,u16,f64)
DECL_CONVERT(int8_t,double,s8,f64)
DECL_CONVERT(uint8_t,double,u8,f64)

//scale>>s32
DECL_SCALED_CONVERT(float,int32_t,f32,s32)
DECL_SCALED_CONVERT(double,int32_t,f64,s32)

//scale>>u32
//DECL_SCALED_CONVERT(float,uint32_t,f32,u32) conversion to u32 is broken (https://bugs.freedesktop.org/show_bug.cgi?id=16524)
//DECL_SCALED_CONVERT(double,uint32_t,f64,u32)

//scale>>s16
DECL_SCALED_CONVERT(float,int16_t,f32,s16)
DECL_SCALED_CONVERT(double,int16_t,f64,s16)

//scale>>u16
DECL_SCALED_CONVERT(float,uint16_t,f32,u16)
DECL_SCALED_CONVERT(double,uint16_t,f64,u16)

//scale>>s8
DECL_SCALED_CONVERT(float,int8_t,f32,s8)
DECL_SCALED_CONVERT(double,int8_t,f64,s8)

//scale>>u8
DECL_SCALED_CONVERT(float,uint8_t,f32,u8)
DECL_SCALED_CONVERT(double,uint8_t,f64,u8)

//scale>>f32
DECL_SCALED_CONVERT(int32_t,float,s32,f32)
DECL_SCALED_CONVERT(uint32_t,float,u32,f32)
DECL_SCALED_CONVERT(int16_t,float,s16,f32)
DECL_SCALED_CONVERT(uint16_t,float,u16,f32)
DECL_SCALED_CONVERT(int8_t,float,s8,f32)
DECL_SCALED_CONVERT(uint8_t,float,u8,f32)

//scale>>f64
DECL_SCALED_CONVERT(int32_t,double,s32,f64)
DECL_SCALED_CONVERT(uint32_t,double,u32,f64)
DECL_SCALED_CONVERT(int16_t,double,s16,f64)
DECL_SCALED_CONVERT(uint16_t,double,u16,f64)
DECL_SCALED_CONVERT(int8_t,double,s8,f64)
DECL_SCALED_CONVERT(uint8_t,double,u8,f64)

#undef DECL_CONVERT
#undef DECL_SCALED_CONVERT
#endif //USE_OIL


// Implementing our own round functions since lround/lroundf do not work for unsigned numbers, and they are flawed in MinGW
template<typename T> static T round(double x) {
  return (T)(x < 0 ? x - 0.5 : x + 0.5);
}

// specialized converter functions

// to complex
static void convert(const s8bit* src, STD_complex* dst,float scale,float offset) {
  (*dst)=STD_complex(src[0]*scale+offset,src[1]*scale);
}
static void convert(const u8bit* src, STD_complex* dst,float scale,float offset) {
  (*dst)=STD_complex(src[0]*scale+offset,src[1]*scale);
}
static void convert(const u16bit* src, STD_complex* dst,float scale,float offset) {
  (*dst)=STD_complex(src[0]*scale+offset,src[1]*scale);
}
static void convert(const s16bit* src, STD_complex* dst,float scale,float offset) {
  (*dst)=STD_complex(src[0]*scale+offset,src[1]*scale);
}
static void convert(const u32bit* src, STD_complex* dst,float scale,float offset) {
  (*dst)=STD_complex(src[0]*scale+offset,src[1]*scale);
}
static void convert(const s32bit* src, STD_complex* dst,float scale,float offset) {
  (*dst)=STD_complex(src[0]*scale+offset,src[1]*scale);
}
static void convert(const float *src, STD_complex* dst,float scale,float offset) {
  dst[0]=STD_complex(src[0]*scale+offset,src[1]*scale);
}
static void convert(const double *src, STD_complex* dst,float scale,float offset) {
  dst[0]=STD_complex(src[0]*scale+offset,src[1]*scale);
}

// from complex
static void convert(const STD_complex* src, s8bit* dst,float scale,float offset) {
  dst[0]=round<s8bit>(src->real()*scale+offset);
  dst[1]=round<s8bit>(src->imag()*scale);
}
static void convert(const STD_complex* src, u8bit* dst,float scale,float offset) {
  dst[0]=round<u8bit>(src->real()*scale+offset);
  dst[1]=round<u8bit>(src->imag()*scale);
}
static void convert(const STD_complex* src, s16bit* dst,float scale,float offset) {
  dst[0]=round<s16bit>(src->real()*scale+offset);
  dst[1]=round<s16bit>(src->imag()*scale);
}
static void convert(const STD_complex* src, u16bit* dst,float scale,float offset) {
  dst[0]=round<u16bit>(src->real()*scale+offset);
  dst[1]=round<u16bit>(src->imag()*scale);
}
static void convert(const STD_complex* src, s32bit* dst,float scale,float offset) {
  dst[0]=round<s32bit>(src->real()*scale+offset);
  dst[1]=round<s32bit>(src->imag()*scale);
}
static void convert(const STD_complex* src, u32bit* dst,float scale,float offset) {
  dst[0]=round<u32bit>(src->real()*scale+offset);
  dst[1]=round<u32bit>(src->imag()*scale);
}
static void convert(const STD_complex* src, float* dst,float scale,float offset) {
  dst[0]=src->real()*scale+offset;
  dst[1]=src->imag()*scale;
}
static void convert(const STD_complex* src, double* dst,float scale,float offset) {
  dst[0]=src->real()*scale+offset;
  dst[1]=src->imag()*scale;
}

// from float to integer
static void convert(const float* src, s8bit* dst,float scale,float offset) {
  dst[0]=round<s8bit>(src[0]*scale+offset);
}
static void convert(const float* src, u8bit* dst,float scale,float offset) {
  dst[0]=round<u8bit>(src[0]*scale+offset);
}
static void convert(const float* src, s16bit* dst,float scale,float offset) {
  dst[0]=round<s16bit>(src[0]*scale+offset);
}
static void convert(const float* src, u16bit* dst,float scale,float offset) {
  dst[0]=round<u16bit>(src[0]*scale+offset);
}
static void convert(const float* src, s32bit* dst,float scale,float offset) {
  dst[0]=round<s32bit>(src[0]*scale+offset);
}
static void convert(const float* src, u32bit* dst,float scale,float offset) {
  dst[0]=round<u32bit>(src[0]*scale+offset);
}

// from double to integer
static void convert(const double* src, s8bit* dst,float scale,float offset) {
  dst[0]=round<s8bit>(src[0]*scale+offset);
}
static void convert(const double* src, u8bit* dst,float scale,float offset) {
  dst[0]=round<u8bit>(src[0]*scale+offset);
}
static void convert(const double* src, s16bit* dst,float scale,float offset) {
  dst[0]=round<s16bit>(src[0]*scale+offset);
}
static void convert(const double* src, u16bit* dst,float scale,float offset) {
  dst[0]=round<u16bit>(src[0]*scale+offset);
}
static void convert(const double* src, s32bit* dst,float scale,float offset) {
  dst[0]=round<s32bit>(src[0]*scale+offset);
}
static void convert(const double* src, u32bit* dst,float scale,float offset) {
  dst[0]=round<u32bit>(src[0]*scale+offset);
}



//default
template<typename Src, typename Dst>
    static void convert(const Src* src, Dst* dst,float scale,float offset) {
      dst[0]=(Dst)(src[0]*scale+offset);
}

//////////////////////////////////////////////////////////

Converter() {} // Do not allow instances

};

/** @}
  */

#endif
