/*
 *
 *  oFono - Open Source Telephony
 *
 *  Copyright (C) 2013 Canonical Ltd.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <stdio.h>
#include <assert.h>
#include <glib.h>
#include <errno.h>

#include <ofono/modem.h>
#include <ofono/gprs-context.h>
#include <ofono/types.h>

#include "grilreply.h"

/*
 * TODO: It may make sense to split this file into
 * domain-specific files ( eg. test-grilreply-gprs-context.c )
 * once more tests are added.
 */

static const struct ril_msg reply_data_call_invalid_1 = {
	.buf = "",
	.buf_len = 0,
};

/*
 * The following hexadecimal data equates te the following
 * RIL_REQUEST_SETUP_DATA_CALL reply parameters:
 *
 * {version=2,num=2 [status=0,retry=-1,cid=0,active=2,type=IP}
 * Parcel is truncated, as num=2 should trigger a failure.
 */
static const guchar reply_data_call_invalid_parcel2[36] = {
	0x07, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff,	0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
	0x02, 0x00, 0x00, 0x00, 0x49, 0x00, 0x50, 0x00,	0x00, 0x00, 0x00, 0x00
};

static const struct ril_msg reply_data_call_invalid_2 = {
	.buf = (gchar *) &reply_data_call_invalid_parcel2,
	.buf_len = 36,
};

/*
 * The following hexadecimal data is a binary representation of
 * a parcel containing an invalid RIL_REQUEST_SETUP_DATA_CALL reply
 * with a NULL string specified cfor 'type':
 *
 * {version=7,num=1 [status=0,retry=-1,cid=0,active=2,type=NULL
 * ifname=rmnet_usb0,address=10.181.235.154/30,
 * dns=172.16.145.103 172.16.145.103,gateways=10.181.235.153]}
 */
static const guchar reply_data_call_invalid_parcel3[196] = {
	0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff,	0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff,
	0x0a, 0x00, 0x00, 0x00, 0x72, 0x00, 0x6d, 0x00, 0x6e, 0x00, 0x65, 0x00,
	0x74, 0x00, 0x5f, 0x00, 0x75, 0x00, 0x73, 0x00, 0x62, 0x00, 0x30, 0x00,
	0x00, 0x00, 0x00, 0x00,	0x11, 0x00, 0x00, 0x00, 0x31, 0x00, 0x30, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x38, 0x00, 0x31, 0x00,	0x2e, 0x00, 0x32, 0x00,
	0x33, 0x00, 0x35, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x35, 0x00, 0x34, 0x00,
	0x2f, 0x00, 0x33, 0x00, 0x30, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00,
	0x31, 0x00, 0x37, 0x00,	0x32, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x36, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x34, 0x00, 0x35, 0x00,	0x2e, 0x00, 0x31, 0x00,
	0x30, 0x00, 0x33, 0x00, 0x20, 0x00, 0x31, 0x00, 0x37, 0x00, 0x32, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x34, 0x00,
	0x35, 0x00, 0x2e, 0x00,	0x31, 0x00, 0x30, 0x00, 0x33, 0x00, 0x00, 0x00,
	0x0e, 0x00, 0x00, 0x00, 0x31, 0x00, 0x30, 0x00,	0x2e, 0x00, 0x31, 0x00,
	0x38, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x33, 0x00, 0x35, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x35, 0x00, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const struct ril_msg reply_data_call_invalid_3 = {
	.buf = (gchar *) &reply_data_call_invalid_parcel3,
	.buf_len = 196,
};

/*
 * The following hexadecimal data is a binary representation of
 * a parcel containing an invalid RIL_REQUEST_SETUP_DATA_CALL reply
 * with a NULL string specified for 'ifname':
 *
 * {version=7,num=1 [status=0,retry=-1,cid=0,active=2,type=IP
 * ifname=NULL,address=10.181.235.154/30,
 * dns=172.16.145.103 172.16.145.103,gateways=10.181.235.153]}
 */
static const guchar reply_data_call_invalid_parcel4[190] = {
	0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff,	0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
	0x02, 0x00, 0x00, 0x00, 0x49, 0x00, 0x50, 0x00,	0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff,	0x11, 0x00, 0x00, 0x00, 0x31, 0x00, 0x30, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x38, 0x00, 0x31, 0x00,	0x2e, 0x00, 0x32, 0x00,
	0x33, 0x00, 0x35, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x35, 0x00, 0x34, 0x00,
	0x2f, 0x00, 0x33, 0x00, 0x30, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00,
	0x31, 0x00, 0x37, 0x00,	0x32, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x36, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x34, 0x00, 0x35, 0x00,	0x2e, 0x00, 0x31, 0x00,
	0x30, 0x00, 0x33, 0x00, 0x20, 0x00, 0x31, 0x00, 0x37, 0x00, 0x32, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x34, 0x00,
	0x35, 0x00, 0x2e, 0x00,	0x31, 0x00, 0x30, 0x00, 0x33, 0x00, 0x00, 0x00,
	0x0e, 0x00, 0x00, 0x00, 0x31, 0x00, 0x30, 0x00,	0x2e, 0x00, 0x31, 0x00,
	0x38, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x33, 0x00, 0x35, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x35, 0x00, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const struct ril_msg reply_data_call_invalid_4 = {
	.buf = (gchar *) &reply_data_call_invalid_parcel4,
	.buf_len = 190,
};

/*
 * The following hexadecimal data is a binary representation of
 * a parcel containing an invalid RIL_REQUEST_SETUP_DATA_CALL reply
 * with a NULL string specified for 'address':
 *
 * {version=7,num=1 [status=0,retry=-1,cid=0,active=2,type=IP
 * ifname=rmnet_usb0,address=NULL,
 * dns=172.16.145.103 172.16.145.103,gateways=10.181.235.153]}
 */
static const guchar reply_data_call_invalid_parcel5[168] = {
	0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff,	0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
	0x02, 0x00, 0x00, 0x00, 0x49, 0x00, 0x50, 0x00,	0x00, 0x00, 0x00, 0x00,
	0x0a, 0x00, 0x00, 0x00, 0x72, 0x00, 0x6d, 0x00, 0x6e, 0x00, 0x65, 0x00,
	0x74, 0x00, 0x5f, 0x00, 0x75, 0x00, 0x73, 0x00, 0x62, 0x00, 0x30, 0x00,
	0x00, 0x00, 0x00, 0x00,	0xff, 0xff, 0xff, 0xff, 0x1d, 0x00, 0x00, 0x00,
	0x31, 0x00, 0x37, 0x00,	0x32, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x36, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x34, 0x00, 0x35, 0x00,	0x2e, 0x00, 0x31, 0x00,
	0x30, 0x00, 0x33, 0x00, 0x20, 0x00, 0x31, 0x00, 0x37, 0x00, 0x32, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x34, 0x00,
	0x35, 0x00, 0x2e, 0x00,	0x31, 0x00, 0x30, 0x00, 0x33, 0x00, 0x00, 0x00,
	0x0e, 0x00, 0x00, 0x00, 0x31, 0x00, 0x30, 0x00,	0x2e, 0x00, 0x31, 0x00,
	0x38, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x33, 0x00, 0x35, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x35, 0x00, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const struct ril_msg reply_data_call_invalid_5 = {
	.buf = (gchar *) &reply_data_call_invalid_parcel5,
	.buf_len = 168,
};

/*
 * The following hexadecimal data represents a serialized Binder parcel
 * instance containing an invalid RIL_REQUEST_SETUP_DATA_CALL reply
 * with a NULL string specified for 'gateways':
 *
 * {version=7,num=1 [status=0,retry=-1,cid=0,active=2,type=IP
 * ifname=rmnet_usb0,address=10.181.235.154/30,
 * dns=172.16.145.103 172.16.145.103,gateways=NULL]}
 */
static const guchar reply_data_call_invalid_parcel6[180] = {
	0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff,	0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
	0x02, 0x00, 0x00, 0x00, 0x49, 0x00, 0x50, 0x00,	0x00, 0x00, 0x00, 0x00,
	0x0a, 0x00, 0x00, 0x00, 0x72, 0x00, 0x6d, 0x00, 0x6e, 0x00, 0x65, 0x00,
	0x74, 0x00, 0x5f, 0x00, 0x75, 0x00, 0x73, 0x00, 0x62, 0x00, 0x30, 0x00,
	0x00, 0x00, 0x00, 0x00,	0x11, 0x00, 0x00, 0x00, 0x31, 0x00, 0x30, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x38, 0x00, 0x31, 0x00,	0x2e, 0x00, 0x32, 0x00,
	0x33, 0x00, 0x35, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x35, 0x00, 0x34, 0x00,
	0x2f, 0x00, 0x33, 0x00, 0x30, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00,
	0x31, 0x00, 0x37, 0x00,	0x32, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x36, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x34, 0x00, 0x35, 0x00,	0x2e, 0x00, 0x31, 0x00,
	0x30, 0x00, 0x33, 0x00, 0x20, 0x00, 0x31, 0x00, 0x37, 0x00, 0x32, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x34, 0x00,
	0x35, 0x00, 0x2e, 0x00,	0x31, 0x00, 0x30, 0x00, 0x33, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff
};

static const struct ril_msg reply_data_call_invalid_6 = {
	.buf = (gchar *) &reply_data_call_invalid_parcel6,
	.buf_len = 180,
};

/*
 * The following hexadecimal data represents a serialized Binder parcel
 * instance containing an invalid RIL_REQUEST_SETUP_DATA_CALL reply with
 * with a NULL string specified for 'dns':
 *
 * {version=7,num=1 [status=0,retry=-1,cid=0,active=2,type=IP
 * ifname=rmnet_usb0,address=10.181.235.154/30,
 * dns=NULL,gateways=10.181.235.153]}
 */
static const guchar reply_data_call_invalid_parcel7[144] = {
	0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff,	0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
	0x02, 0x00, 0x00, 0x00, 0x49, 0x00, 0x50, 0x00,	0x00, 0x00, 0x00, 0x00,
	0x0a, 0x00, 0x00, 0x00, 0x72, 0x00, 0x6d, 0x00, 0x6e, 0x00, 0x65, 0x00,
	0x74, 0x00, 0x5f, 0x00, 0x75, 0x00, 0x73, 0x00, 0x62, 0x00, 0x30, 0x00,
	0x00, 0x00, 0x00, 0x00,	0x11, 0x00, 0x00, 0x00, 0x31, 0x00, 0x30, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x38, 0x00, 0x31, 0x00,	0x2e, 0x00, 0x32, 0x00,
	0x33, 0x00, 0x35, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x35, 0x00, 0x34, 0x00,
	0x2f, 0x00, 0x33, 0x00, 0x30, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0x0e, 0x00, 0x00, 0x00, 0x31, 0x00, 0x30, 0x00,	0x2e, 0x00, 0x31, 0x00,
	0x38, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x33, 0x00, 0x35, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x35, 0x00, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const struct ril_msg reply_data_call_invalid_7 = {
	.buf = (gchar *) &reply_data_call_invalid_parcel7,
	.buf_len = 144,
};

/*
 * The following hexadecimal data represents a serialized Binder parcel
 * instance containing a valid RIL_REQUEST_SETUP_DATA_CALL reply with the
 * following parameters:
 *
 * {version=7,num=1 [status=0,retry=-1,cid=0,active=2,type=IP,
 * ifname=rmnet_usb0,address=10.181.235.154/30,
 * dns=172.16.145.103 172.16.145.103,gateways=10.181.235.153]}
 */
static const guchar reply_data_call_valid_parcel1[204] = {
	0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff,	0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
	0x02, 0x00, 0x00, 0x00, 0x49, 0x00, 0x50, 0x00,	0x00, 0x00, 0x00, 0x00,
	0x0a, 0x00, 0x00, 0x00, 0x72, 0x00, 0x6d, 0x00, 0x6e, 0x00, 0x65, 0x00,
	0x74, 0x00, 0x5f, 0x00, 0x75, 0x00, 0x73, 0x00, 0x62, 0x00, 0x30, 0x00,
	0x00, 0x00, 0x00, 0x00,	0x11, 0x00, 0x00, 0x00, 0x31, 0x00, 0x30, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x38, 0x00, 0x31, 0x00,	0x2e, 0x00, 0x32, 0x00,
	0x33, 0x00, 0x35, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x35, 0x00, 0x34, 0x00,
	0x2f, 0x00, 0x33, 0x00, 0x30, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00,
	0x31, 0x00, 0x37, 0x00,	0x32, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x36, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x34, 0x00, 0x35, 0x00,	0x2e, 0x00, 0x31, 0x00,
	0x30, 0x00, 0x33, 0x00, 0x20, 0x00, 0x31, 0x00, 0x37, 0x00, 0x32, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x34, 0x00,
	0x35, 0x00, 0x2e, 0x00,	0x31, 0x00, 0x30, 0x00, 0x33, 0x00, 0x00, 0x00,
	0x0e, 0x00, 0x00, 0x00, 0x31, 0x00, 0x30, 0x00,	0x2e, 0x00, 0x31, 0x00,
	0x38, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x33, 0x00, 0x35, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x35, 0x00, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const struct ril_msg reply_data_call_valid_1 = {
	.buf = (gchar *) &reply_data_call_valid_parcel1,
	.buf_len = 204,
	.unsolicited = FALSE,
	.req = RIL_REQUEST_SETUP_DATA_CALL,
	.serial_no = 0,
	.error = 0,
};

static void test_reply_data_call_invalid(gconstpointer data)
{
	/* TODO: fix de-const cast... */
	struct ril_msg *message = (struct ril_msg *) data;
	struct ofono_error error;
	struct reply_setup_data_call *reply;

	reply = g_ril_reply_parse_data_call(NULL, message, &error);
	g_assert(reply != NULL);
	g_ril_reply_free_setup_data_call(reply);

	g_assert(error.type == OFONO_ERROR_TYPE_FAILURE &&
			error.error == -EINVAL);
}

static void test_reply_data_call_valid(gconstpointer data)
{
	/* TODO: fix de-const cast... */
	struct ril_msg *message = (struct ril_msg *) data;
	struct ofono_error error;
	struct reply_setup_data_call *reply;

	reply = g_ril_reply_parse_data_call(NULL, message, &error);
	g_assert(reply != NULL);
	g_ril_reply_free_setup_data_call(reply);

	g_assert(error.type == OFONO_ERROR_TYPE_NO_ERROR &&
			error.error == 0);
}

int main(int argc, char **argv)
{
	g_test_init(&argc, &argv, NULL);

/*
 * As all our architectures are little-endian except for
 * PowerPC, and the Binder wire-format differs slightly
 * depending on endian-ness, the following guards against test
 * failures when run on PowerPC.
 */
#if BYTE_ORDER == LITTLE_ENDIAN

	g_test_add_data_func("/testgrilreply/gprs-context: "
				"invalid SETUP_DATA_CALL Test 1",
				&reply_data_call_invalid_1,
				test_reply_data_call_invalid);

	g_test_add_data_func("/testgrilreply/gprs-context: "
				"invalid SETUP_DATA_CALL Test 2",
				&reply_data_call_invalid_2,
				test_reply_data_call_invalid);

	g_test_add_data_func("/testgrilreply/gprs-context: "
				"invalid SETUP_DATA_CALL Test 3",
				&reply_data_call_invalid_3,
				test_reply_data_call_invalid);

	g_test_add_data_func("/testgrilreply/gprs-context: "
				"invalid SETUP_DATA_CALL Test 4",
				&reply_data_call_invalid_4,
				test_reply_data_call_invalid);

	g_test_add_data_func("/testgrilreply/gprs-context: "
				"invalid SETUP_DATA_CALL Test 5",
				&reply_data_call_invalid_5,
				test_reply_data_call_invalid);

	g_test_add_data_func("/testgrilreply/gprs-context: "
				"invalid SETUP_DATA_CALL Test 6",
				&reply_data_call_invalid_6,
				test_reply_data_call_invalid);

	g_test_add_data_func("/testgrilreply/gprs-context: "
				"invalid SETUP_DATA_CALL Test 7",
				&reply_data_call_invalid_7,
				test_reply_data_call_invalid);

	g_test_add_data_func("/testgrilreply/gprs-context: "
				"valid SETUP_DATA_CALL Test 1",
				&reply_data_call_valid_1,
				test_reply_data_call_valid);

#endif

	return g_test_run();
}
