/***

  Olive - Non-Linear Video Editor
  Copyright (C) 2022 Olive Team

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.

***/

#ifndef SHADERJOB_H
#define SHADERJOB_H

#include <QMatrix4x4>
#include <QVector>

#include "generatejob.h"
#include "render/colorprocessor.h"
#include "render/texture.h"

namespace olive {

class ShaderJob : public GenerateJob {
public:
  ShaderJob()
  {
    iterations_ = 1;
    iterative_input_ = nullptr;
    will_change_image_size_ = true;
  }

  const QString& GetShaderID() const
  {
    return shader_id_;
  }

  void SetShaderID(const QString& id)
  {
    shader_id_ = id;
  }

  void SetIterations(int iterations, const NodeInput& iterative_input)
  {
    SetIterations(iterations, iterative_input.input());
  }

  void SetIterations(int iterations, const QString& iterative_input)
  {
    iterations_ = iterations;
    iterative_input_ = iterative_input;
  }

  int GetIterationCount() const
  {
    return iterations_;
  }

  const QString& GetIterativeInput() const
  {
    return iterative_input_;
  }

  Texture::Interpolation GetInterpolation(const QString& id) const
  {
    return interpolation_.value(id, Texture::kDefaultInterpolation);
  }

  const QHash<QString, Texture::Interpolation> &GetInterpolationMap() const
  {
    return interpolation_;
  }

  void SetInterpolation(const NodeInput& input, Texture::Interpolation interp)
  {
    interpolation_.insert(input.input(), interp);
  }

  void SetInterpolation(const QString& id, Texture::Interpolation interp)
  {
    interpolation_.insert(id, interp);
  }

  void SetVertexCoordinates(const QVector<float> &vertex_coords)
  {
    vertex_overrides_ = vertex_coords;
  }

  const QVector<float>& GetVertexCoordinates()
  {
    return vertex_overrides_;
  }

  bool GetWillChangeImageSize() const { return will_change_image_size_; }
  void SetWillChangeImageSize(bool e) { will_change_image_size_ = e; }

private:
  QString shader_id_;

  int iterations_;

  QString iterative_input_;

  QHash<QString, Texture::Interpolation> interpolation_;

  QVector<float> vertex_overrides_;

  bool will_change_image_size_;

};

}

Q_DECLARE_METATYPE(olive::ShaderJob)

#endif // SHADERJOB_H
