# -*- coding: utf-8 -*-

# Copyright (C) 2004-2005 Johan Svedberg <johan@svedberg.com>

# This file is part of ontv.

# ontv is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# ontv is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with ontv; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

from gettext import gettext as _

import gtk
import gobject
import gnomeapplet

TIMEOUT = 60000

class ProgramTable(gtk.Frame):
    def __init__(self, listings, title):
        gtk.Frame.__init__(self, title)
        self.listings = listings
        self.tooltips = gtk.Tooltips()

        self.table = gtk.Table()
        self.table.set_border_width(6)
        self.table.props.row_spacing = 2
        self.table.props.column_spacing = 6

        label = self.get_label_widget()
        label.set_use_markup(True)
        self.set_border_width(3)
        self.set_label_align(0.5, 0.5)
        self.set_shadow_type(gtk.SHADOW_NONE)
        self.add(self.table)

        self.filled_programs = []

        gobject.timeout_add(TIMEOUT, self.__refresh)

    def __refresh(self):
        self.update()
        return True

    def update(self, force_update=False):
        if len(self.listings.selected_channels) == 0:
            self.__resize(1)
            self.__clear()

            label = gtk.Label(_("No channels selected"))
            label.set_alignment(0.0, 0.5)
            label.show()
            self._attach(label, 0, 0, xoptions=gtk.SHRINK|gtk.FILL)

            self.filled_programs = None
        else:
            self.set_programs()
            if self.programs != self.filled_programs or force_update:
                self.__resize(len(self.programs))
                self.__clear()

                for i in xrange(len(self.programs)):
                    program = self.programs[i]
                    self.__add_channel(program.channel_name, 0, i)
                    self.__add_program(program, 1, i)
                    self.add_other_bar(program, 2, i)

                self.filled_programs = self.programs

    def __resize(self, rows, columns=3):
        if rows == 0:
            rows = 1
        self.table.resize(rows, columns)

    def __clear(self):
        cells = []
        self.table.foreach(cells.append)
        for cell in cells:
            self.table.remove(cell)

        self.filled_programs = []

    def __add_channel(self, channel_name, column, row):
        channel = self.listings.channels[channel_name]
        if channel.logo_small:
            widget = gtk.Image()
            widget.set_from_pixbuf(channel.logo_small)
        else:
            widget = gtk.Label()
            widget.set_markup("<b>" + channel_name + "</b>")
            widget.set_alignment(0.0, 0.5)

        widget.show()
        self._attach(widget, column, row, xoptions=gtk.SHRINK|gtk.FILL)

    def __add_program(self, program, column, row):
        label = gtk.Label()
        label.set_label(program.title)
        label.set_alignment(0.0, 0.5)
        label.show()

        eb = gtk.EventBox()
        eb.add(label)
        eb.show()
        if program.description:
            self.tooltips.set_tip(eb, program.description)

        self._attach(eb, column, row, xoptions=gtk.SHRINK|gtk.FILL)

    def _attach(self, widget, column, row, xoptions=gtk.FILL|gtk.EXPAND):
        self.table.attach(widget, column, column + 1, row, row + 1,
                          xoptions=xoptions)

class ProgramBar:
    def get_readable_time(self, duration):
        descs = []

        hours = duration.seconds / 3600
        if hours > 0:
            s = str(hours) + _(" hour")
            if not hours is 1:
                s = str(hours) + _(" hours")
            descs.append(s)

        minutes = (duration.seconds / 60) % 60
        if minutes > 0:
            s = str(minutes) + _(" minute")
            if not minutes is 1:
                s = str(minutes) + _(" minutes")
            descs.append(s)

        if len(descs) == 0:
            s = str(duration.seconds) + _(" second")
            if not duration.seconds is 1:
                s = str(duration.seconds) + _(" seconds")
            descs.append(s)

        return _(" and ").join(descs) + _(" left")

class ProgramProgressBar(ProgramBar, gtk.EventBox):
    def __init__(self, program):
        gtk.EventBox.__init__(self)
        self.program = program
        self.tooltips = gtk.Tooltips()
        self.pb = gtk.ProgressBar()
        self.pb.show()
        self.add(self.pb)
        self.show()

        gobject.timeout_add(TIMEOUT, self.update)

    def __time_delta_to_seconds(self, delta):
        return delta.days * 3600 * 24 + delta.seconds

    def update(self):
        time_from_start = self.program.time_span.time_from_start()
        played = self.__time_delta_to_seconds(time_from_start)
        duration = self.program.time_span.duration()
        total = self.__time_delta_to_seconds(duration)
        ratio = float(played) / float(total)
        if ratio > 1 or ratio < 0:
            return False
        time_until_stop = self.program.get_time_until_stop()
        readable_time = self.get_readable_time(time_until_stop)
        self.tooltips.set_tip(self, readable_time)
        self.pb.set_fraction(ratio)
        return True

class CurrentProgramTable(ProgramTable):
    def __init__(self, listings):
        ProgramTable.__init__(self, listings, "<span size=\"larger\" weight=\"bold\">%s</span>" % _("Current programs"))

    def set_programs(self):
        self.programs = []
        for channel_name in self.listings.selected_channels:
            if self.listings.channels.has_key(channel_name):
                channel = self.listings.channels[channel_name]
                current_program = channel.get_current_program()
                if current_program:
                    self.programs.append(current_program)

    def add_other_bar(self, program, column, row):
        ppb = ProgramProgressBar(program)
        ppb.update()

        self._attach(ppb, column, row)

class ProgramTimeLeftBar(ProgramBar, gtk.Label):
    def __init__(self, program):
        gtk.Label.__init__(self)
        self.program = program
        self.set_alignment(1.0, 0.5)
        self.show()

        gobject.timeout_add(TIMEOUT, self.update)

    def update(self):
        time_until_start = self.program.get_time_until_start()
        readable_time = self.get_readable_time(time_until_start)
        self.set_label(readable_time)
        return True

class UpcomingProgramTable(ProgramTable):
    def __init__(self, listings):
        ProgramTable.__init__(self, listings, "<span size=\"larger\" weight=\"bold\">%s</span>" % _("Upcoming programs"))

    def set_programs(self):
        self.programs = []
        for channel_name in self.listings.selected_channels:
            if self.listings.channels.has_key(channel_name):
                channel = self.listings.channels[channel_name]
                upcoming_program = channel.get_upcoming_program()
                if upcoming_program:
                    self.programs.append(upcoming_program)

    def add_other_bar(self, program, column, row):
        ptlb = ProgramTimeLeftBar(program)
        ptlb.update()

        self._attach(ptlb, column, row)

class ProgramWindow(gtk.Window):
    def __init__(self, xmltvfile, get_orient):
        gtk.Window.__init__(self, gtk.WINDOW_TOPLEVEL)
        xmltvfile.connect("loaded", self.__xmltvfile_loaded)
        self.get_orient = get_orient

        self.cpt = CurrentProgramTable(xmltvfile.listings)
        self.upt = UpcomingProgramTable(xmltvfile.listings)

        self.set_decorated(False)
        self.set_resizable(False)

        vbox = gtk.VBox()
        vbox.set_border_width(6)
        vbox.pack_start(self.cpt)
        vbox.pack_start(self.upt)

        frame = gtk.Frame()
        frame.add(vbox)
        frame.set_shadow_type(gtk.SHADOW_IN)

        self.add(frame)

    def position_window(self, widget):
        self.realize()
        gtk.gdk.flush()

        self.stick()
        self.set_keep_above(True)
        self.set_skip_pager_hint(True)
        self.set_skip_taskbar_hint(True)
        self.set_type_hint (gtk.gdk.WINDOW_TYPE_HINT_DOCK)

        widget.realize()
        (x, y) = widget.window.get_origin()

        (w, h) = self.get_size()
        (w, h) = self.size_request()

        button_w = widget.allocation.width
        button_h = widget.allocation.height

        screen = self.get_screen()

        found_monitor = False
        n = screen.get_n_monitors()
        for i in range(0, n):
            monitor = screen.get_monitor_geometry(i)
            if (x >= monitor.x and x <= monitor.x + monitor.width and
                y >= monitor.y and y <= monitor.y + monitor.height):
                    found_monitor = True
                    break

        if not found_monitor:
            screen_width = screen.get_width()
            monitor = gtk.gdk.Rectangle(0, 0, screen_width, screen_width)

        orient = self.get_orient()

        if orient == gnomeapplet.ORIENT_RIGHT:
            x += button_w

            if ((y + h) > monitor.y + monitor.height):
                y -= (y + h) - (monitor.y + monitor.height)

            if ((y + h) > (monitor.height / 2)):
                gravity = gtk.gdk.GRAVITY_SOUTH_WEST
            else:
                gravity = gtk.gdk.GRAVITY_NORTH_WEST
        elif orient == gnomeapplet.ORIENT_LEFT:
            x -= w

            if ((y + h) > monitor.y + monitor.height):
                y -= (y + h) - (monitor.y + monitor.height)

            if ((y + h) > (monitor.height / 2)):
                gravity = gtk.gdk.GRAVITY_SOUTH_EAST
            else:
                gravity = gtk.gdk.GRAVITY_NORTH_EAST
        elif orient == gnomeapplet.ORIENT_DOWN:
            y += button_h

            if ((x + w) > monitor.x + monitor.width):
                x -= (x + w) - (monitor.x + monitor.width)

            gravity = gtk.gdk.GRAVITY_NORTH_WEST
        elif orient == gnomeapplet.ORIENT_UP:
            y -= h

            if ((x + w) > monitor.x + monitor.width):
                x -= (x + w) - (monitor.x + monitor.width)

            gravity = gtk.gdk.GRAVITY_SOUTH_WEST

        self.move(x, y)
        self.set_gravity(gravity)
        self.show()

    def __xmltvfile_loaded(self, xmltvfile, listings):
        if listings:
            self.cpt.listings = listings
            self.upt.listings = listings
            self.update()

    def update(self):
        self.cpt.update(True)
        self.upt.update(True)

# vim: set sw=4 et sts=4 tw=79 fo+=l:
