# -*- coding: utf-8 -*-

# Copyright (C) 2004-2006 Johan Svedberg <johan@svedberg.com>

# This file is part of ontv.

# ontv is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# ontv is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with ontv; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA

from gettext import gettext as _

import gtk
import gconf
import gobject
import pango

import Configuration
import GUI
from Reminders import Reminder

class PreferencesDialog:
    def __init__(self, ontvapplet):
        self.ontvapplet = ontvapplet
        self.config = ontvapplet.config
        self.xmltvfile = ontvapplet.xmltvfile
        self.reminders = ontvapplet.reminders
        self.kb = ontvapplet.kb
        self.pw = ontvapplet.pw
        self.sd = ontvapplet.sd

        self.xmltvfile.connect("loaded", self.__xmltvfile_loaded)

        self.__get_widgets()
        self.__make_channels_treeview()
        self.__make_reminders_treeview()
        self.sd.reminders_model = self.reminders_treeview.get_model()

        channels_combobox_model = gtk.ListStore(str)
        channels_combobox_model.set_sort_func(0,
                                              self.__channels_model_sort_func,
                                              False)
        channels_combobox_model.set_sort_column_id(0, gtk.SORT_ASCENDING)
        self.channels_comboboxentry.set_model(channels_combobox_model)
        self.channels_comboboxentry.set_text_column(0)
        self.channels_comboboxentry.child.set_text(_("All"))
        self.sd.channels_combobox.set_model(channels_combobox_model)

        self.__connect_widgets()

        self.__add_config_notifications()

        self.dialog.set_icon(GUI.icon)
        self.xmltv_file_entry.set_text(self.config.xmltv_file)
        self.reload_xmltv_file_checkbutton.set_active(self.config.auto_reload_xmltv_file)
        self.reload_xmltv_file_spinbutton.set_value(self.config.auto_reload_xmltv_file_rate)
        self.enable_hotkeys_checkbutton.set_active(self.config.enable_hotkeys)
        self.show_window_hotkey_entry.set_text(self.config.show_window_hotkey)
        self.show_search_program_hotkey_entry.set_text(self.config.show_search_program_hotkey)
        self.current_programs_checkbutton.set_active(self.config.display_current_programs)
        self.upcoming_programs_checkbutton.set_active(self.config.display_upcoming_programs)
        self.channels_image.set_from_stock(gtk.STOCK_DIALOG_INFO,
                                           gtk.ICON_SIZE_DIALOG)

        self.__auto_reload_id = None

    def __get_widgets(self):
        self.dialog = GUI.xml.get_widget("preferences_dialog")
        self.closebutton = GUI.xml.get_widget("closebutton")
        self.notebook = GUI.xml.get_widget("notebook")

        # General tab
        self.xmltv_file_entry = GUI.xml.get_widget("xmltv_file_entry")
        self.xmltv_browse_button = GUI.xml.get_widget("xmltv_browse_button")
        self.reload_xmltv_file_checkbutton = GUI.xml.get_widget("reload_xmltv_file_checkbutton")
        self.reload_xmltv_file_spinbutton = GUI.xml.get_widget("reload_xmltv_file_spinbutton")
        self.enable_hotkeys_checkbutton = GUI.xml.get_widget("enable_hotkeys_checkbutton")
        self.show_window_hotkey_entry = GUI.xml.get_widget("show_window_hotkey_entry")
        self.show_search_program_hotkey_entry = GUI.xml.get_widget("show_search_program_hotkey_entry")
        self.current_programs_checkbutton = GUI.xml.get_widget("current_programs_checkbutton")
        self.upcoming_programs_checkbutton = GUI.xml.get_widget("upcoming_programs_checkbutton")

        # Channels tab
        self.channels_treeview = GUI.xml.get_widget("channels_treeview")
        self.channels_image = GUI.xml.get_widget("channels_image")

        # Reminders tab
        self.reminders_treeview = GUI.xml.get_widget("reminders_treeview")
        self.program_entry = GUI.xml.get_widget("program_entry")
        self.channels_comboboxentry = GUI.xml.get_widget("channels_comboboxentry")
        self.notify_spinbutton = GUI.xml.get_widget("notify_spinbutton")
        self.add_reminder_button = GUI.xml.get_widget("add_reminder_button")
        self.update_reminder_button = GUI.xml.get_widget("update_reminder_button")
        self.remove_reminder_button = GUI.xml.get_widget("remove_reminder_button")

    def __make_channels_treeview(self):
        channels_model = gtk.ListStore(object)
        channels_model.set_sort_func(0, self.__channels_model_sort_func)
        channels_model.set_sort_column_id(0, gtk.SORT_ASCENDING)
        self.channels_treeview.set_model(channels_model)

        crtoggle = gtk.CellRendererToggle()
        crtoggle.props.xpad = 6
        crtoggle.props.ypad = 3
        crtoggle.props.activatable = True
        crtoggle.connect("toggled", self.__channels_treeview_toggled,
                         channels_model)
        selected_column = gtk.TreeViewColumn("Selected", crtoggle)
        selected_column.set_cell_data_func(crtoggle,
                                           self.__crtoggle_cell_data_func)
        self.channels_treeview.append_column(selected_column)

        crpixbuf = gtk.CellRendererPixbuf()
        crpixbuf.props.xpad = 3
        crpixbuf.props.ypad = 3
        logo_column = gtk.TreeViewColumn("Logo", crpixbuf)
        logo_column.set_cell_data_func(crpixbuf, self.__crpixbuf_cell_data_func)
        self.channels_treeview.append_column(logo_column)

        crtext = gtk.CellRendererText()
        crtext.props.xpad = 3
        crtext.props.ypad = 3
        channel_column = gtk.TreeViewColumn("Channel", crtext)
        channel_column.set_cell_data_func(crtext, self.__crtext_cell_data_func)
        self.channels_treeview.append_column(channel_column)
        self.channels_treeview.set_search_column(2)
        self.channels_treeview.set_search_equal_func(self.__channels_treeview_search_equal)

        self.channels_treeview_menu = gtk.Menu()
        self.add_logo_imi = gtk.ImageMenuItem(gtk.STOCK_ADD)
        self.add_logo_imi.child.set_markup_with_mnemonic(_("_Add logo"))
        self.add_logo_imi.show()
        self.channels_treeview_menu.append(self.add_logo_imi)

        self.remove_logo_imi = gtk.ImageMenuItem(gtk.STOCK_REMOVE)
        self.remove_logo_imi.child.set_markup_with_mnemonic(_("_Remove logo"))
        self.remove_logo_imi.show()
        self.channels_treeview_menu.append(self.remove_logo_imi)

    def __channels_model_sort_func(self, model, iter1, iter2, object=True):
        channel = model.get_value(iter1, 0)
        other_channel = model.get_value(iter2, 0)
        if object:
            return cmp(channel.name.lower(), other_channel.name.lower())
        return cmp(channel.lower(), other_channel.lower())

    def __channels_treeview_toggled(self, cell, path, model):
        channel = model[path][0]
        channel.selected = not channel.selected
        if channel.selected:
            self.listings.selected_channels.append(channel.name)
        else:
            self.listings.selected_channels.remove(channel.name)

        self.pw.update()
        self.listings.save()

    def __channels_treeview_search_equal(self, model, column, key, iter):
        channel = model.get_value(iter, 0)
        return key.lower() not in channel.name.lower()

    def __crtoggle_cell_data_func(self, column, cell, model, iter):
        channel = model.get_value(iter, 0)
        cell.props.active = channel.selected

    def __crpixbuf_cell_data_func(self, column, cell, model, iter):
        channel = model.get_value(iter, 0)
        cell.props.pixbuf = channel.logo

    def __crtext_cell_data_func(self, column, cell, model, iter):
        channel = model.get_value(iter, 0)
        markup = "<b>%s</b>"
        cell.props.markup = markup % channel.markup_escaped_name

    def __make_reminders_treeview(self):
        reminders_model = gtk.ListStore(object)
        self.reminders_treeview.set_model(reminders_model)

        program_crt = gtk.CellRendererText()
        program_crt.props.xpad = 3
        program_crt.props.ypad = 3
        program_column = gtk.TreeViewColumn(_("Program"), program_crt)
        program_column.set_cell_data_func(program_crt,
                                          self.__program_crt_cell_data_func)
        self.reminders_treeview.append_column(program_column)

        channel_crt = gtk.CellRendererText()
        channel_crt.props.xpad = 3
        channel_crt.props.ypad = 3
        channel_column = gtk.TreeViewColumn(_("Channel"), channel_crt)
        channel_column.set_cell_data_func(channel_crt,
                                          self.__channel_crt_cell_data_func)
        self.reminders_treeview.append_column(channel_column)

        notify_crt = gtk.CellRendererText()
        notify_crt.props.xpad = 3
        notify_crt.props.ypad = 3
        notify_column = gtk.TreeViewColumn(_("Notification time"), notify_crt)
        notify_column.set_cell_data_func(notify_crt,
                                         self.__notify_crt_cell_data_func)
        self.reminders_treeview.append_column(notify_column)

        for reminder in self.reminders.reminders:
            reminders_model.append([reminder])

    def __program_crt_cell_data_func(self, column, cell, model, iter):
        reminder = model.get_value(iter, 0)
        cell.props.text = reminder.program

    def __channel_crt_cell_data_func(self, column, cell, model, iter):
        reminder = model.get_value(iter, 0)
        cell.props.markup = reminder.channel

    def __notify_crt_cell_data_func(self, column, cell, model, iter):
        reminder = model.get_value(iter, 0)
        cell.props.text = reminder.notify_time

    def __connect_widgets(self):
        self.dialog.connect("close", self.__close_dialog)
        self.closebutton.connect("clicked", self.__close_dialog)

        # General tab
        self.xmltv_file_entry.connect("activate",
                                      self.__xmltv_file_entry_activate)
        self.xmltv_file_entry.connect("changed",
                                      self.__xmltv_file_entry_changed)
        self.xmltv_browse_button.connect("clicked",
                                         self.__xmltv_browse_button_clicked)
        self.reload_xmltv_file_checkbutton.connect("toggled", self.__reload_xmltv_file_checkbutton_toggled)
        self.reload_xmltv_file_spinbutton.connect("value_changed", self.__reload_xmltv_file_spinbutton_value_changed)
        self.enable_hotkeys_checkbutton.connect("toggled", self.__enable_hotkeys_checkbutton_toggled)
        self.show_window_hotkey_entry.connect("changed", self.__show_window_hotkey_entry_changed)
        self.show_search_program_hotkey_entry.connect("changed", self.__show_search_program_hotkey_entry_changed)
        self.current_programs_checkbutton.connect("toggled", self.__current_programs_checkbutton_toggled)
        self.upcoming_programs_checkbutton.connect("toggled", self.__upcoming_programs_checkbutton_toggled)

        # Channels tab
        self.channels_treeview.connect("row-activated",
                                       self.__channels_treeview_row_activated)
        self.channels_treeview.connect("button-press-event",
                                       self.__channels_treeview_button_press,
                                       self.channels_treeview_menu)
        self.add_logo_imi.connect("activate", self.__add_logo_imi_activate)
        self.remove_logo_imi.connect("activate",
                                     self.__remove_logo_imi_activate)

        # Reminders tab
        self.reminders_treeview.connect("key-press-event", self.__reminders_treeview_key_press_event)
        selection = self.reminders_treeview.get_selection()
        selection.connect("changed",
                          self.__reminders_treeview_selection_changed)
        self.program_entry.connect("changed", self.__program_entry_changed)
        self.channels_comboboxentry.connect("changed", self.__channels_comboboxentry_changed)
        self.notify_spinbutton.connect("value_changed",
                                       self.__notify_spinbutton_value_changed)
        self.add_reminder_button.connect("clicked",
                                         self.__add_reminder_button_clicked)
        self.update_reminder_button.connect("clicked", self.__update_reminder_button_clicked)
        self.remove_reminder_button.connect("clicked", self.__remove_reminder_button_clicked)

    def __close_dialog(self, widget):
        self.dialog.hide()

    def __xmltv_file_entry_activate(self, entry):
        self.config.xmltv_file = entry.get_text()

    def __xmltv_file_entry_changed(self, entry):
        if len(entry.get_text()) > 0:
            self.reload_xmltv_file_checkbutton.set_sensitive(True)
            self.current_programs_checkbutton.set_sensitive(True)
            self.upcoming_programs_checkbutton.set_sensitive(True)
        else:
            self.reload_xmltv_file_checkbutton.set_sensitive(False)
            self.current_programs_checkbutton.set_sensitive(False)
            self.upcoming_programs_checkbutton.set_sensitive(False)

    def __xmltv_browse_button_clicked(self, button):
        xml_filter = gtk.FileFilter()
        xml_filter.set_name(_("XML files"))
        xml_filter.add_mime_type("text/xml")
        fcd = GUI.FileChooserDialog(_("Select XMLTV file..."), (xml_filter,))
        response = fcd.run()
        if response == gtk.RESPONSE_OK:
            self.config.xmltv_file = fcd.get_filename()
            fcd.hide()
        elif response == gtk.RESPONSE_CANCEL:
            fcd.hide()
        fcd.hide()

    def __reload_xmltv_file_checkbutton_toggled(self, checkbutton):
        self.config.auto_reload_xmltv_file = checkbutton.get_active()

    def __reload_xmltv_file_spinbutton_value_changed(self, spinbutton):
        self.config.auto_reload_xmltv_file_rate = spinbutton.get_value_as_int()

    def __enable_hotkeys_checkbutton_toggled(self, checkbutton):
        self.config.enable_hotkeys = checkbutton.get_active()

    def __show_window_hotkey_entry_changed(self, entry):
        self.config.show_window_hotkey = entry.get_text()

    def __show_search_program_hotkey_entry_changed(self, entry):
        self.config.show_search_program_hotkey = entry.get_text()

    def __current_programs_checkbutton_toggled(self, checkbutton):
        self.config.display_current_programs = checkbutton.get_active()

    def __upcoming_programs_checkbutton_toggled(self, checkbutton):
        self.config.display_upcoming_programs = checkbutton.get_active()

    def __channels_treeview_row_activated(self, treeview, path, column):
        channels_model = treeview.get_model()
        self.__channels_treeview_toggled(None, path, channels_model)

    def __channels_treeview_button_press(self, treeview, event, menu):
        if event.type == gtk.gdk.BUTTON_PRESS and event.button == 3:
            channels_model = treeview.get_model()
            path = treeview.get_path_at_pos(int(event.x), int(event.y))[0]
            channel = channels_model[path][0]
            if channel.logo:
                self.add_logo_imi.set_sensitive(False)
                self.remove_logo_imi.set_sensitive(True)
            else:
                self.add_logo_imi.set_sensitive(True)
                self.remove_logo_imi.set_sensitive(False)

            menu.popup(None, None, None, event.button, event.time)

    def __add_logo_imi_activate(self, menuitem):
        selection = self.channels_treeview.get_selection()
        (model, iter) = selection.get_selected()
        path = model.get_path(iter)
        channel = model.get_value(iter, 0)

        png_filter = gtk.FileFilter()
        png_filter.set_name(_("PNG images"))
        png_filter.add_mime_type("image/png")

        jpeg_filter = gtk.FileFilter()
        jpeg_filter.set_name(_("JPEG images"))
        jpeg_filter.add_mime_type("image/jpeg")

        gif_filter = gtk.FileFilter()
        gif_filter.set_name(_("GIF images"))
        gif_filter.add_mime_type("image/gif")

        fcd = GUI.FileChooserDialog(_("Select logo image..."),
                                (png_filter, gif_filter, jpeg_filter))

        response = fcd.run()
        if response == gtk.RESPONSE_OK:
            channel.logo_file = fcd.get_filename()
            channel.set_logo(channel.logo_file)
            model.row_changed(path, iter)
            self.pw.update()
            fcd.hide()
            self.listings.save()
        elif response == gtk.RESPONSE_CANCEL:
            fcd.hide()
        fcd.hide()

    def __remove_logo_imi_activate(self, menuitem):
        selection = self.channels_treeview.get_selection()
        (model, iter) = selection.get_selected()
        path = model.get_path(iter)
        channel = model.get_value(iter, 0)
        channel.logo_file = ""
        channel.logo = channel.logo_small = None
        model.row_changed(path, iter)
        logo_column = self.channels_treeview.get_column(1)
        logo_column.queue_resize()
        self.pw.update()
        self.listings.save()

    def __reminders_treeview_selection_changed(self, selection):
        (reminders_model, reminders_iter) = selection.get_selected()
        if reminders_iter:
            self.remove_reminder_button.set_sensitive(True)
            reminder = reminders_model.get_value(reminders_iter, 0)
            self.program_entry.set_text(reminder.program)
            channel = pango.parse_markup(reminder.channel, u"\x00")[1]
            self.channels_comboboxentry.child.set_text(channel)
            self.notify_spinbutton.set_value(reminder.notify_time)
        else:
            self.program_entry.set_text("")
            self.channels_comboboxentry.child.set_text("")
            self.notify_spinbutton.set_value(5)
            self.add_reminder_button.set_sensitive(False)
            self.update_reminder_button.set_sensitive(False)
            self.remove_reminder_button.set_sensitive(False)

    def __reminders_treeview_key_press_event(self, treeview, event):
        if event.keyval == gtk.keysyms.Delete:
            self.remove_reminder_button.emit("clicked")

    def __program_entry_changed(self, entry):
        if len(entry.get_text()) > 0:
            self.__set_reminder_buttons_sensitivity()
        else:
            self.add_reminder_button.set_sensitive(False)
            self.update_reminder_button.set_sensitive(False)

    def __channels_comboboxentry_changed(self, comboboxentry):
        self.__set_reminder_buttons_sensitivity()

    def __notify_spinbutton_value_changed(self, spinbutton):
        self.__set_reminder_buttons_sensitivity()

    def __set_reminder_buttons_sensitivity(self):
        if not self.__get_reminder() in self.reminders.reminders:
            self.add_reminder_button.set_sensitive(True)
            if GUI.has_selection(self.reminders_treeview):
                self.update_reminder_button.set_sensitive(True)
        else:
            self.add_reminder_button.set_sensitive(False)
            self.update_reminder_button.set_sensitive(False)

    def __add_reminder_button_clicked(self, button):
        reminder = self.__get_reminder()
        if self.reminders.add(reminder):
            reminders_model = self.reminders_treeview.get_model()
            reminders_model.append([reminder])
            GUI.queue_resize(self.reminders_treeview)
            self.reminders.save()
            self.add_reminder_button.set_sensitive(False)
            self.sd.search_treeview.get_selection().emit("changed")
            self.pw.update()

    def __get_reminder(self):
        program = self.program_entry.get_text()
        channel = self.channels_comboboxentry.child.get_text()
        channel = gobject.markup_escape_text(channel)
        notify_time = self.notify_spinbutton.get_value_as_int()
        return Reminder(program, channel, notify_time)

    def __update_reminder_button_clicked(self, button):
        selection = self.reminders_treeview.get_selection()
        (reminders_model, reminders_iter) = selection.get_selected()
        current_reminder = reminders_model.get_value(reminders_iter, 0)
        new_reminder = self.__get_reminder()
        if not current_reminder is new_reminder:
            self.reminders.update(current_reminder, new_reminder)
            reminders_model.set_value(reminders_iter, 0, new_reminder)
            GUI.queue_resize(self.reminders_treeview)
            self.reminders.save()
            self.add_reminder_button.set_sensitive(False)
            self.update_reminder_button.set_sensitive(False)
            self.sd.search_treeview.get_selection().emit("changed")
            self.pw.update()

    def __remove_reminder_button_clicked(self, button):
        selection = self.reminders_treeview.get_selection()
        (reminders_model, reminders_iter) = selection.get_selected()
        if reminders_iter:
            reminder = reminders_model.get_value(reminders_iter, 0)
            if self.reminders.remove(reminder):
                reminders_model.remove(reminders_iter)
                GUI.queue_resize(self.reminders_treeview)
                self.reminders.save()
                self.sd.search_treeview.get_selection().emit("changed")

    def __add_config_notifications(self):
        self.config.add_notify(Configuration.KEY_XMLTV_FILE,
                               self.__xmltv_file_key_changed)
        self.config.add_notify(Configuration.KEY_AUTO_RELOAD_XMLTV_FILE,
                               self.__auto_reload_xmltv_file_key_changed)
        self.config.add_notify(Configuration.KEY_AUTO_RELOAD_XMLTV_FILE_RATE,
                               self.__auto_reload_xmltv_file_rate_key_changed)
        self.config.add_notify(Configuration.KEY_DISPLAY_CURRENT_PROGRAMS,
                               self.__display_current_programs_key_changed)
        self.config.add_notify(Configuration.KEY_DISPLAY_UPCOMING_PROGRAMS,
                               self.__display_upcoming_programs_key_changed)
        self.config.add_notify(Configuration.KEY_ENABLE_HOTKEYS,
                               self.__enable_hotkeys_key_changed)
        self.config.add_notify(Configuration.KEY_SHOW_WINDOW_HOTKEY,
                               self.__show_window_hotkey_key_changed)
        self.config.add_notify(Configuration.KEY_SHOW_SEARCH_PROGRAM_HOTKEY,
                               self.__show_search_program_hotkey_key_changed)

    def __xmltv_file_key_changed(self, client, cnxn_id, entry, data):
        if not entry.value:
            self.xmltv_file_entry.set_text("")
        elif entry.value.type == gconf.VALUE_STRING:
            filename = entry.value.to_string()
            self.xmltv_file_entry.set_text(filename)
            self.xmltvfile.props.path = filename
        else:
            self.xmltv_file_entry.set_text("!type error!")

    def __auto_reload_xmltv_file_key_changed(self, client, cnxn_id, entry, data):
        if not entry.value:
            self.reload_xmltv_file_checkbutton.set_active(True)
        elif entry.value.type == gconf.VALUE_BOOL:
            reload = entry.value.get_bool()
            self.reload_xmltv_file_checkbutton.set_active(reload)
            self.reload_xmltv_file_spinbutton.set_sensitive(reload)
            if reload:
                self.start_auto_reload(self.config.auto_reload_xmltv_file_rate)
            else:
                self.stop_auto_reload()
        else:
            self.reload_xmltv_file_checkbutton.set_active(True)

    def __auto_reload_xmltv_file_rate_key_changed(self, client, cnxn_id, entry, data):
        if not entry.value:
            self.reload_xmltv_file_spinbutton.set_value(12)
        elif entry.value.type == gconf.VALUE_INT:
            rate = entry.value.get_int()
            self.reload_xmltv_file_spinbutton.set_value(rate)
            if self.config.auto_reload_xmltv_file:
                self.stop_auto_reload()
                self.start_auto_reload(rate)
        else:
            self.reload_xmltv_file_spinbutton.set_value(12)

    def start_auto_reload(self, rate):
        if rate >= 1:
            timeout = rate * 3600 * 1000
            self.__auto_reload_id = gobject.timeout_add(timeout, self.xmltvfile.load, (None, True))
            if self.config.debug:
                print("[Added timeout %s]: %sh" % (str(self.__auto_reload_id),
                                                   str(rate)))

    def stop_auto_reload(self):
        if self.__auto_reload_id:
            gobject.source_remove(self.__auto_reload_id)
            if self.config.debug:
                print("[Removed timeout %s]" % (str(self.__auto_reload_id)))

    def __enable_hotkeys_key_changed(self, client, cnxn_id, entry, data):
        if not entry.value:
            self.enable_hotkeys_checkbutton.set_active(False)
        elif entry.value.type == gconf.VALUE_BOOL:
            enable = entry.value.get_bool()
            self.enable_hotkeys_checkbutton.set_active(enable)
            self.show_window_hotkey_entry.set_sensitive(enable)
            self.show_search_program_hotkey_entry.set_sensitive(enable)
            if enable:
                self.kb.props.show_window_hotkey = self.config.show_window_hotkey
                self.kb.props.show_search_program_hotkey = self.config.show_search_program_hotkey
            else:
                self.kb.unbind(self.config.show_window_hotkey)
                self.kb.unbind(self.config.show_search_program_hotkey)
        else:
            self.current_programs_checkbutton.set_active(True)

    def __show_window_hotkey_key_changed(self, client, cnxn_id, entry, data):
        if not entry.value:
            self.show_window_hotkey_entry.set_text("")
        elif entry.value.type == gconf.VALUE_STRING:
            value = entry.value.to_string()
            self.show_window_hotkey_entry.set_text(value)
            if self.config.enable_hotkeys:
                if self.kb.props.show_window_hotkey != "":
                    self.kb.unbind(self.kb.props.show_window_hotkey)
                self.kb.props.show_window_hotkey = value
        else:
            self.show_window_hotkey_entry.set_text("!type error!")

    def __show_search_program_hotkey_key_changed(self, client, cnxn_id, entry, data):
        if not entry.value:
            self.show_search_program_hotkey_entry.set_text("")
        elif entry.value.type == gconf.VALUE_STRING:
            value = entry.value.to_string()
            self.show_search_program_hotkey_entry.set_text(value)
            if self.config.enable_hotkeys:
                if self.kb.props.show_search_program_hotkey != "":
                    self.kb.unbind(self.kb.props.show_search_program_hotkey)
                self.kb.props.show_search_program_hotkey = value
        else:
            self.show_search_program_hotkey_entry.set_text("!type error!")

    def __display_current_programs_key_changed(self, client, cnxn_id, entry, data):
        if not entry.value:
            self.current_programs_checkbutton.set_active(True)
        elif entry.value.type == gconf.VALUE_BOOL:
            value = entry.value.get_bool()
            self.current_programs_checkbutton.set_active(value)
            if value:
                self.pw.cpt.show()
            else:
                self.pw.cpt.hide()
        else:
            self.current_programs_checkbutton.set_active(True)

    def __display_upcoming_programs_key_changed(self, client, cnxn_id, entry, data):
        if not entry.value:
            self.upcoming_programs_checkbutton.set_active(True)
        elif entry.value.type == gconf.VALUE_BOOL:
            value = entry.value.get_bool()
            self.upcoming_programs_checkbutton.set_active(value)
            if value:
                self.pw.upt.show()
            else:
                self.pw.upt.hide()
        else:
            self.upcoming_programs_checkbutton.set_active(True)

    def __xmltvfile_loaded(self, xmltvfile, listings):
        if listings:
            self.listings = listings
            channels_model = self.channels_treeview.get_model()
            channels_model.clear()
            channels_combobox_model = self.channels_comboboxentry.get_model()
            channels_combobox_model.clear()
            channels_combobox_model.append([_("All")])
            for channel in self.listings.channels.values():
                channels_model.append([channel])
                channels_combobox_model.append([channel])
            self.sd.set_all_as_combo_active(self.sd.channels_combobox)

    def show(self, uicomponent=None, verb=None):
        self.dialog.show()

# vim: set sw=4 et sts=4 tw=79 fo+=l:
