/***************************************************************************
 *   Copyright (C) 2006 by Christian Nitschkowski   *
 *   christian.nitschkowski@kdemail.net   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include <qevent.h>
#include <qtimer.h>
#include <qtooltip.h>

#include <dcopclient.h>
#include <kaction.h>
#include <kapplication.h>
#include <kconfig.h>
#include <khelpmenu.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>

#include "oooqs2.h"
#include "oooqs2configdialog.h"
#include "oooqs2detector.h"

OOoQs2::OOoQs2( KAboutData* aboutData )
   : DCOPObject( "OOoQs2Iface" ), KSystemTray( 0, "oooqs2" ),
   p_processQuickstart( this ), p_bFirstPreloadRun( true ),
   p_bRunPreloader( false )
{
   KPopupMenu* menu = contextMenu();

   setPixmap( loadIcon( "oooqs2" ) );

   connect( &p_processQuickstart, SIGNAL( processExited( KProcess* ) ),
      this, SLOT( preloadInstance() )
   );

   p_actionBase = new KAction( i18n( "OpenOffice.org Base" ),
      "ooo-base", 0, this, SLOT( startBase() ),
      actionCollection(), "start_1"
   );
   p_actionCalc = new KAction( i18n( "OpenOffice.org Calc" ),
      "ooo-calc", 0, this, SLOT( startCalc() ),
      actionCollection(), "start_2"
   );
   p_actionDraw = new KAction( i18n( "OpenOffice.org Draw" ),
      "ooo-draw", 0, this, SLOT( startDraw() ),
      actionCollection(), "start_3"
   );
   p_actionImpress = new KAction( i18n( "OpenOffice.org Impress" ),
      "ooo-impress", 0, this, SLOT( startImpress() ),
      actionCollection(), "start_4"
   );
   p_actionMath = new KAction( i18n( "OpenOffice.org Math" ),
      "ooo-math", 0, this, SLOT( startMath() ),
      actionCollection(), "start_5"
   );
   p_actionTemplate = new KAction( i18n( "Start with template" ),
      "ooo-template", 0, this, SLOT( startTemplate() ),
      actionCollection(), "start_0"
   );
   p_actionWriter = new KAction( i18n( "OpenOffice.org Writer" ),
      "ooo-writer", 0, this, SLOT( startWriter() ),
      actionCollection(), "start_6"
   );
   p_actionPreferences = KStdAction::preferences( this, SLOT( preferences() ),
      actionCollection()
   );
   p_menuHelp = new KHelpMenu( this, aboutData, false );

   p_actionTemplate->plug( menu, -1 );
   menu->insertSeparator();
   p_actionBase->plug( menu, -1 );
   p_actionCalc->plug( menu, -1 );
   p_actionDraw->plug( menu, -1 );
   p_actionImpress->plug( menu, -1 );
   p_actionMath->plug( menu, -1 );
   p_actionWriter->plug( menu, -1 );
   menu->insertSeparator();
   p_actionPreferences->plug( menu, -1 );
   menu->insertSeparator();
   menu->insertItem( i18n( "&Help" ), p_menuHelp->menu() );

   setup();
}

OOoQs2::~OOoQs2()
{
}

QString OOoQs2::preloadMessage()
{
   return i18n( "By loading the office suite in the background after login, "
      "the quickstart mode can greatly reduce OpenOffice.org's start-up time, "
      "but permanently requires about 30MB of RAM. If you use OpenOffice.org "
      "frequently and have 128MB of RAM or more, "
      "'Yes' might be a good choice.\n"
      "\n"
      "Do you want to enable the OpenOffice.org 2 quickstart mode now?" );
}

void OOoQs2::mouseDoubleClickEvent( QMouseEvent* e )
{
   p_actionQuickAccess->activate();

   e->accept();
}

void OOoQs2::setup()
{
   KConfig* config = KGlobal::config();

   config->setGroup( "General" );
   if( config->readBoolEntry( "FirstStart", true ) )
   {
      if( KMessageBox::questionYesNo( this,
         OOoQs2::preloadMessage() +
         "\n" +
         i18n( "Note: You may change this setting anytime in the "
         "Quickstarter configuration dialog." ) )
         == KMessageBox::Yes )
      {
         config->setGroup( "Advanced" );
         config->writeEntry( "EnablePreload", true );
      }
      config->setGroup( "General" );
      config->writeEntry( "FirstStart", false );
      config->sync();
   }
   p_actionQuickAccess = actionCollection()->action(
         QString( "start_%1" ).arg( config->readNumEntry( "QuickAccess", 0 ) )
   );
   QToolTip::remove( this );
   if( p_actionQuickAccess )
   {
      QToolTip::add( this, p_actionQuickAccess->text() );
   }

   config->setGroup( "Advanced" );
   p_bRunPreloader = config->readBoolEntry( "EnablePreload", false );
   if( config->readBoolEntry( "OverrideAutodetection", false ) )
   {
      p_strBinary = config->readEntry( "Binary" );
   }
   else
   {
      p_strBinary = OOoQs2Detector::detectBinary();
   }

   if( p_strBinary.isEmpty() )
   {
      if( KMessageBox::warningYesNo( this,
         i18n( "There was no version of OpenOffice.org 2 "
         "found on your system. Please make sure you have OpenOffice.org 2 "
         "or later installed on your system.\n\nIf you have installed "
         "OpenOffice.org 2 already chances are that it can't be detected "
         "automatically. In this case please select your installation manually."
         "\n\nDo you want the preferences dialog to be shown?" ) )
         == KMessageBox::Yes )
      {
         preferences( true );
         return;
      }
      else
      {
         return;
      }
   }

   p_processQuickstart.clearArguments();
   p_processQuickstart << p_strBinary;
   p_processQuickstart << "-nologo" << "-nodefault";

   preloadInstance();
}

void OOoQs2::startInstance( const QString& app )
{
   if( p_strBinary.isEmpty() )
   {
      KMessageBox::error( this,
         i18n( "Can't start OpenOffice.org 2 as there is no version "
         "installed on this system or it can't be found automatically.\n"
         "Please check your configuration." )
      );
      return;
   }

   KProcess proc;
   proc << p_strBinary << app;
   proc.start( KProcess::DontCare );
}

bool OOoQs2::preloading()
{
   return p_bRunPreloader;
}

void OOoQs2::enablePreloading( bool state )
{
   KConfig* config = KGlobal::config();

   config->setGroup( "Advanced" );
   config->writeEntry( "EnablePreload", state );
   config->sync();

   if( state )
   {
      preloadInstance();
   }
}

bool OOoQs2::startOnLogin()
{
   KConfig* config = KGlobal::config();

   config->setGroup( "General" );
   return config->readBoolEntry( "Autostart", false );
}

void OOoQs2::enableStartOnLogin( bool state )
{
   KConfig* config = KGlobal::config();

   config->setGroup( "General" );
   config->writeEntry( "Autostart", state );
}

bool OOoQs2::overrideDetection()
{
   KConfig* config = KGlobal::config();

   config->setGroup( "Advanced" );
   return config->readBoolEntry( "OverrideAutodetection", false );
}

void OOoQs2::setOverrideDetection( bool state )
{
   KConfig* config = KGlobal::config();

   config->setGroup( "Advanced" );
   config->writeEntry( "OverrideAutodetection", state );
   config->sync();

   if( state )
   {
      p_strBinary = config->readEntry( "Binary" );
   }
}

QString OOoQs2::overrideBinary()
{
   KConfig* config = KGlobal::config();

   config->setGroup( "Advanced" );
   return config->readEntry( "Binary" );
}

void OOoQs2::setOverrideBinary( QString path )
{
   KConfig* config = KGlobal::config();

   config->setGroup( "Advanced" );
   config->writeEntry( "Binary", path );
   config->sync();

   if( overrideBinary() )
   {
      p_strBinary = path;
   }
}

QString OOoQs2::binary()
{
   return p_strBinary;
}

int OOoQs2::instanceOOo()
{
   return OOoQs2Detector::detectInstance();
}

int OOoQs2::quickAccess()
{
   KConfig* config = KGlobal::config();

   config->setGroup( "General" );
   return config->readNumEntry( "QuickAccess", 0 );
}

void OOoQs2::setQuickAccess( int index )
{
   if( index < 0 || index > 6)
   {
      return;
   }

   KConfig* config = KGlobal::config();

   config->setGroup( "General" );
   config->writeEntry( "QuickAccess", index );
   config->sync();
}

void OOoQs2::preferences( bool onError )
{
   OOoQs2ConfigDialog* dlg = new OOoQs2ConfigDialog( onError, this );

   if( dlg->exec() == false )
   {
      delete dlg;
      return;
   }

   KConfig* config = KGlobal::config();

   config->setGroup( "General" );
   config->writeEntry( "QuickAccess", dlg->quickAccess() );
   config->writeEntry( "Autostart", dlg->autostart() );

   config->setGroup( "Advanced" );
   config->writeEntry( "EnablePreload", dlg->preload() );
   config->writeEntry( "OverrideAutodetection", dlg->overrideAutodetection() );
   config->writeEntry( "Binary", dlg->binary() );

   config->sync();

   delete dlg;

   setup();
}

void OOoQs2::preloadInstance()
{
   if( p_bRunPreloader == false )
   {
      return;
   }

   if( p_bFirstPreloadRun )
   {
      KConfig smConfig( "ksmserverrc" );
      smConfig.setGroup( "General" );

      QStringList listExclusions = smConfig.readListEntry( "excludeApps" );
      if( listExclusions.contains( "soffice.bin" ) == false )
      {
         if( KMessageBox::warningYesNo( this,
            i18n( "I've detected Session Management for OpenOffice.org 2 "
            "isn't disabled. Should I disable it for you now? (Recommended)" ) )
            == KMessageBox::Yes )
         {
            listExclusions.append( "soffice.bin" );
            smConfig.writeEntry( "excludeApps", listExclusions );
            smConfig.sync();
            KApplication::dcopClient()->send( "ksmserver",
               "MainApplication-Interface", "reparseConfiguration()",
               ""
            );
         }
      }

      p_bFirstPreloadRun = false;
   }

   // Check if there's an instance of OOo running
   // If there's no instance, start the quickstart instance
   // If there's already an instance try again in 60 seconds.
   if( OOoQs2Detector::detectInstance() == 0 )
   {
      p_processQuickstart.start();
   }
   else
   {
      QTimer::singleShot( 60000, this, SLOT( preloadInstance() ) );
   }
}

void OOoQs2::startBase()
{
   startInstance( "-base" );
}

void OOoQs2::startCalc()
{
   startInstance( "-calc" );
}

void OOoQs2::startDraw()
{
   startInstance( "-draw" );
}

void OOoQs2::startImpress()
{
   startInstance( "-impress" );
}

void OOoQs2::startMath()
{
   startInstance( "-math" );
}

void OOoQs2::startTemplate()
{
   startInstance( "slot:5500" );
}

void OOoQs2::startWriter()
{
   startInstance( "-writer" );
}

#include "oooqs2.moc"
