<?php
/* 	OpenDb - Open Lending Database Project
	Copyright (C) 2001,2002 by Jason Pell

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

// This must be first - includes config.php
require_once("./include/begin.inc.php");

include_once("./include/language.php");
include_once("./include/theme.php");

include_once("./functions/email.php");
include_once("./functions/http.php");
include_once("./functions/function.php");
include_once("./functions/utils.php");
include_once("./functions/borrowed_item.php");
include_once("./functions/item.php");
include_once("./functions/datetime.php");
include_once("./functions/item_attribute.php");
include_once("./functions/item_type.php");
include_once("./functions/widgets.php");
include_once("./functions/review.php");
include_once("./functions/listings.php");
include_once("./functions/status_type.php");
include_once("./functions/Listing.class.inc");

/**
*/
function handle_reserve($item_id, $instance_no, $borrower_id, &$errors)
{
	global $LANG_VARS;
	global $HTTP_SESSION_VARS;
	global $CONFIG_VARS;
	
	$status_type_r = fetch_status_type_r(fetch_item_s_status_type($item_id, $instance_no));
	if($status_type_r['borrow_ind'] != 'Y')
	{
		$errors = replace_lang_var('s_status_type_desc', $status_type_r['description'], $LANG_VARS['s_status_type_items_cannot_be_borrowed']);
		return FALSE;
	}
	else if(strlen($status_type_r['min_display_user_type'])>0 && !in_array($status_type_r['min_display_user_type'], get_min_user_type_r($HTTP_SESSION_VARS['user_type'])))
	{
		$errors = replace_lang_var('s_status_type_desc', $status_type_r['description'], $LANG_VARS['s_status_type_items_cannot_be_borrowed']);
		return FALSE;
	}
	else if(is_user_owner_of_item($item_id, $instance_no, $borrower_id))
	{
		$errors = $LANG_VARS['cannot_reserve_items_you_own'];
		return FALSE;
	}
	else if(is_item_borrowed_by_user($item_id, $instance_no, $borrower_id))
	{
		$errors = $LANG_VARS['you_have_it_borrowed'];
		return FALSE;
	}
	else if(is_item_reserved_by_user($item_id, $instance_no, $borrower_id))
	{
		$errors = $LANG_VARS['you_have_reservation'];
		return FALSE;
	}
	else if($CONFIG_VARS['borrow.allow_reserve_if_borrowed'] ===FALSE  && is_item_borrowed($item_id, $instance_no))
	{
		$errors = $LANG_VARS['item_is_already_checked_out'];
		return FALSE;
	}
	else if($CONFIG_VARS['borrow.allow_multi_reserve'] === FALSE && is_item_reserved($item_id, $instance_no))
	{
		$errors = $LANG_VARS['item_is_already_reserved'];
		return FALSE;
	}
	else
	{
		// All but the actual reservation of item should occur, when this variable is set to TRUE.
		if($CONFIG_VARS['borrow.reserve_email_only'] !== TRUE)
		{
			$new_borrowed_item_id = reserve_item($item_id, $instance_no, $borrower_id);
			if($new_borrowed_item_id!==FALSE)
			{
				return new_borrowed_item_id;
			}
			else
			{
				return FALSE;
			}
		}
		else
		{
			return TRUE;
		}
	}		
}

/**
* A handle_quick_checkout will only ever handle one request at a time, because
* they are initiated from item_display and not listings.php or borrow.php
* 
* @param $uid is always currently logged in user
*/
function handle_quick_checkout($item_id, $instance_no, $borrower_id, $borrow_duration, &$errors)
{
	global $LANG_VARS;
	global $HTTP_SESSION_VARS;
	
	if(!is_user_valid($borrower_id))
	{
		$errors = replace_lang_var("user_id", $borrower_id, $LANG_VARS['invalid_borrower_user']);
		return FALSE;
	}
	else if(!is_user_allowed_to_borrow($borrower_id))
	{
		$errors = replace_lang_var("user_id", $borrower_id, $LANG_VARS['user_must_be_borrower']);
		return FALSE;
	}
	else if(!is_user_owner_of_item($item_id, $instance_no, $HTTP_SESSION_VARS['user_id']))
	{
		$errors = $LANG_VARS['not_owner_of_item'];
		return FALSE;
	}
	else if(is_item_borrowed($item_id, $instance_no))
	{
		$errors = $LANG_VARS['item_is_already_checked_out'];
		return FALSE;
	}
	
	$status_type_r = fetch_status_type_r(fetch_item_s_status_type($item_id, $instance_no));
	if($status_type_r['borrow_ind'] == 'Y' || $status_type_r['borrow_ind'] == 'N')//Owner operation
	{
		$new_borrowed_item_id = quick_check_out_item($item_id, $instance_no, $borrower_id, $borrow_duration);
		if($new_borrowed_item_id!==FALSE)
		{
			return $new_borrowed_item_id;
		}
		else
		{
			return FALSE;
		}
	}		
	else
	{
		$errors = replace_lang_var('s_status_type_desc', $status_type_r['description'], $LANG_VARS['s_status_type_items_cannot_be_borrowed']);
		return FALSE;
	}
}

/**
	$uid is always currently logged in user
*/
function handle_cancelreserve($sequence_number, &$errors)
{
	global $LANG_VARS;
	global $HTTP_SESSION_VARS;

	$borrowed_item_r = fetch_borrowed_item_r($sequence_number);
	if ($borrowed_item_r['borrower_id'] !== $HTTP_SESSION_VARS['user_id'] && !is_user_owner_of_item($borrowed_item_r['item_id'], $borrowed_item_r['instance_no'], $HTTP_SESSION_VARS['user_id']))
	{
		$errors = $LANG_VARS['not_allowed_cancel_reserve'];
		return FALSE;
	}
	else if($borrowed_item_r['status'] == 'X')
	{
		$errors = $LANG_VARS['already_cancelled'];
		return FALSE;
	}
	else if($borrowed_item_r['status'] != 'R')
	{
		$errors = $LANG_VARS['borrowed_item_not_found'];
		return FALSE;
	}
	else
	{
		if(cancel_reserve_item($sequence_number))
		{
			return TRUE;
		}
		else
		{
			return FALSE;
		}
	}
}

/**
	$uid is always currently logged in user
*/
function handle_checkout($sequence_number, $borrow_duration, &$errors)
{
	global $LANG_VARS;
	global $HTTP_SESSION_VARS;

	$borrowed_item_r = fetch_borrowed_item_r($sequence_number);
	if(!is_user_owner_of_item($borrowed_item_r['item_id'], $borrowed_item_r['instance_no'], $HTTP_SESSION_VARS['user_id']))
	{
		$errors = $LANG_VARS['not_owner_of_item'];
		return FALSE;
	}
	else if(is_item_borrowed($borrowed_item_r['item_id'], $borrowed_item_r['instance_no']))
	{
		$errors = $LANG_VARS['item_is_already_checked_out'];
		return FALSE;
	}
	else if ($borrowed_item_r['status'] != 'R')
	{
		$errors = $LANG_VARS['borrowed_item_not_found'];
		return FALSE;
	}

	// Once a reservation exists, the s_status_type of the item can be changed to a borrow_ind=N type,
	// and the item can still be checked out.  The change of status to a borrow_ind=N, only prevents
	// new reservations from being recorded.
	$status_type_r = fetch_status_type_r(fetch_item_s_status_type($borrowed_item_r['item_id'], $borrowed_item_r['instance_no']));
	if($status_type_r['borrow_ind'] == 'Y' || $status_type_r['borrow_ind'] == 'N')//Existing reservation
	{
		if(check_out_item($sequence_number, $borrow_duration))
		{
			return TRUE;
		}
		else
		{
			return FALSE;
		}
	}
	else
	{
		$errors = replace_lang_var('s_status_type_desc', $status_type_r['description'], $LANG_VARS['s_status_type_items_cannot_be_borrowed']);
		return FALSE;
	}
}

/**
	$uid is always currently logged in user
*/
function handle_checkin($sequence_number, &$errors)
{
	global $LANG_VARS;
	global $HTTP_SESSION_VARS;

	$borrowed_item_r = fetch_borrowed_item_r($sequence_number);
	if(!is_user_owner_of_item($borrowed_item_r['item_id'], $borrowed_item_r['instance_no'], $HTTP_SESSION_VARS['user_id']))
	{
		$errors = $LANG_VARS['not_owner_of_item'];
		return FALSE;
	}
	else if ($borrowed_item_r['status'] == 'C')
	{
		$errors = $LANG_VARS['already_checked_in'];
		return FALSE;
	}
	else if ($borrowed_item_r['status'] != 'B')
	{
		$errors = $LANG_VARS['borrowed_item_not_found'];
		return FALSE;
	}
	else
	{
		if(check_in_item($sequence_number))
		{
			return TRUE;
		}
		else
		{
			return FALSE;
		}
	}
}

/**
	$uid is always currently logged in user
*/
function handle_reminder($sequence_number, &$errors)
{
	global $LANG_VARS;
	global $HTTP_SESSION_VARS;

	$borrowed_item_r = fetch_borrowed_item_r($sequence_number);
	if(!is_user_owner_of_item($borrowed_item_r['item_id'], $borrowed_item_r['instance_no'], $HTTP_SESSION_VARS['user_id']))
	{
		$errors = $LANG_VARS['not_owner_of_item'];
		return FALSE;
	}
	else if ($borrowed_item_r['status'] == 'C')
	{
		$errors = $LANG_VARS['already_checked_in'];
		return FALSE;
	}
	else if ($borrowed_item_r['status'] != 'B')
	{
		$errors = $LANG_VARS['borrowed_item_not_found'];
		return FALSE;
	}
	else
		return TRUE;
}

function handle_extension($sequence_number, $borrow_extension, &$errors)
{
	global $LANG_VARS;
	global $HTTP_SESSION_VARS;

	$borrowed_item_r = fetch_borrowed_item_r($sequence_number);
	if(!is_user_owner_of_item($borrowed_item_r['item_id'], $borrowed_item_r['instance_no'], $HTTP_SESSION_VARS['user_id']))
	{
		$errors = $LANG_VARS['not_owner_of_item'];
		return FALSE;
	}
	else if ($borrowed_item_r['status'] == 'C')
	{
		$errors = $LANG_VARS['already_checked_in'];
		return FALSE;
	}
	else if ($borrowed_item_r['status'] != 'B')
	{
		$errors = $LANG_VARS['borrowed_item_not_found'];
		return FALSE;
	}
	else
	{
		if(item_borrow_duration_extension($sequence_number, $borrow_extension))
			return TRUE;
		else
			return FALSE;
	}
}

/**
* @param $borrowed_item_rs Items that this action will be performed against.  It may actually
* 						be an array of 'sequence_number' values, in which case the borrow
* 						record for the sequence_number will be fetched.
* @param $HTTP_VARS
* @param $uid Should always the current user id from $HTTP_SESSION_VARS['user_id']
* @param $duration_attr_type
* @param $borrow_duration
* @param $borrower_id_field	If TRUE, this form will display a borrower_id field LOV, minus the
* 		 $HTTP_SESSION_VARS['user_id']
* */
function more_information_form($op, $borrowed_item_rs, $HTTP_VARS, $email_notification=TRUE)
{
	global $PHP_SELF;
	global $LANG_VARS;
	global $CONFIG_VARS;
	global $HTTP_SESSION_VARS;
	
	$duration_attr_type=NULL;
	$default_borrow_duration=NULL;

	echo("\n<table cellspacing=1 border=0>");
	echo("\n<form action=\"$PHP_SELF\" method=\"post\">");
	// In case no detail is required.
	echo("\n<input type=\"hidden\" name=\"more_info_requested\" value=\"true\">");

	// Pass all http variables onto next instance...
	// Includes empty fields...
	echo get_url_fields($HTTP_VARS, NULL, NULL);

	// Display the items to be operated on.
	if(is_not_empty_array($borrowed_item_rs))
	{
		$field = "";
		
		//initialise
		$max_overdue_duration = NULL;
		$default_borrow_duration = NULL;
		
		reset($borrowed_item_rs);
		while(list(,$borrowed_item_r) = each($borrowed_item_rs))
		{
			// If only a sequence_number, we need to fetch the borrow record.
			if(!is_array($borrowed_item_r) && is_numeric($borrowed_item_r))
			{
				$borrowed_item_r = fetch_borrowed_item_r($borrowed_item_r);
			}
			
			$item_r = fetch_item_instance_r($borrowed_item_r['item_id'], $borrowed_item_r['instance_no']);
			
			// Expand title mask.
			$item_r['title'] = expand_item_title_mask($item_r['title'], $item_r['s_item_type'], $item_r['item_id'], $item_r['instance_no'], $item_r['s_status_type'], $CONFIG_VARS['item_display.title_display_mask']);
			$field .= replace_lang_vars(array('display_title'=>$item_r['title'],'item_id'=>$item_r['item_id'],'instance_no'=>$item_r['instance_no']),$LANG_VARS['borrow_item_title_listing'])."<br>";
			
			// While we are here, we are going to calculate the default duration value.
			// We want to choose the least duration value.  If any of the items use a
			// different s_attribute_type for duration, then we should not try to get
			// a default value.  This is indicated by the $duration_attr_type===FALSE
			// if we have encountered a difference.
			if($CONFIG_VARS['borrow.duration_support'] && $duration_attr_type!==FALSE)
			{
				$new_duration_attr_type = fetch_sfieldtype_item_attribute_type($item_r['s_item_type'], 'DURATION');
				if($duration_attr_type == NULL)
					$duration_attr_type = $new_duration_attr_type;
				else if($duration_attr_type !== $new_duration_attr_type)
				{
					// Different s_attribute_type's for DURATION, so cannot display Duration chooser.
					$duration_attr_type=FALSE;
				}

				if($duration_attr_type!==FALSE)
				{
					if($op == 'check_out' || $op == 'quick_check_out')
					{
						// The default borrow duration should be the least amount of days or undefined
						// if no records have a borrow duration.
						if($default_borrow_duration === NULL)
						{
							$default_borrow_duration = $item_r['borrow_duration'];
						}
						else if($default_borrow_duration !== '') // Undefined empty value.
						{
							if(is_numeric($default_borrow_duration) && $item_r['borrow_duration'] < $default_borrow_duration)
								$default_borrow_duration = $item_r['borrow_duration'];
						}
					}
					else if($op == 'extension')
					{
						if($borrowed_item_r['total_duration']>$borrowed_item_r['borrow_duration'])
							$tmp_overdue_duration = $borrowed_item_r['total_duration'] - $borrowed_item_r['borrow_duration'];
				
						// We want to get the max overdue duration, so we can give the User granting the extension a
						// default, that will bring all selected items back into non-overdue status.
						if(!is_numeric($max_overdue_duration) || $max_overdue_duration < $tmp_overdue_duration)
							$max_overdue_duration = $tmp_overdue_duration;
					}
				}
			}
		}
		
		echo(format_field($LANG_VARS['item(s)'],
						NULL,
						$field));
	}
	
	// Do not display this more information section unless email is enabled.
	if($CONFIG_VARS['email.use_php_mail']!==FALSE && $email_notification!==FALSE)
	{
		echo get_input_field("more_information",
						NULL, 
						$LANG_VARS['more_information'], 
						"textarea(50,10)",
						"N", //compulsory
						NULL,
						TRUE);
	}
	
	// Include a Borrower ID select, minus the current user.
	if($op == 'quick_check_out')
	{
		$results = fetch_user_rs(get_borrower_user_types_r(), "fullname", "ASC", FALSE, $HTTP_SESSION_VARS['user_id']);
		if($results)
		{
			echo(
				format_field($LANG_VARS['borrower'], 
						NULL, 
						custom_select('borrower_id', $results, '%fullname% (%user_id%)', 1, NULL, 'user_id')
					)
				);
		}
		else
		{
			echo(
				format_field(
					$LANG_VARS['borrower'],
					NULL,
					$LANG_VARS['no_records_found'])
				);
				
		}
	}
	
	//Only for check_out/quick_check_out operations - makes no sense otherwise!
	if(($op == 'check_out' || $op == 'quick_check_out' || $op == 'extension') && $CONFIG_VARS['borrow.duration_support'])
	{
		// Display default borrow duration.
		if(strlen($duration_attr_type)>0)
		{
			$duration_attr_type_r = fetch_attribute_type_r($duration_attr_type);
			
			// We have to find the matching DURATION lookup value, which is at least
			// as many days as the max_overdue value, or the highest possible
			// duration value, if none found as large as the $max_overdue_duration
			if($op == 'extension')
			{
				$default_borrow_duration = NULL;
				$result = fetch_attribute_type_lookup_rs($duration_attr_type_r['s_attribute_type'], 'value', 'asc');
				if($result)
				{
					while($lookup_r = mysql_fetch_array($result, MYSQL_ASSOC))
					{	
						if(is_numeric($lookup_r['value']) && (!is_numeric($max_overdue_duration) || (is_numeric($max_overdue_duration) && $max_overdue_duration <= $lookup_r['value'])))
						{
							$default_borrow_duration = $lookup_r['value'];
							break;
						}
						
						// backup, in case we need to use outside while loop
						$lookup_r2 = $lookup_r;					
					}
					mysql_free_result($result);
					
					// If still null, then set to the largest option
					if($default_borrow_duration == NULL)
					{
						$default_borrow_duration = $lookup_r2['value'];
					}
				}
			}
			
			if($op == 'extension')
				$prompt = $duration_attr_type_r['prompt'];
			else if(strlen($LANG_VARS['override_duration'])>0 && is_array($borrowed_item_rs) && count($borrowed_item_rs)>1)
				$prompt = $LANG_VARS['override_duration'];
			else
				$prompt = $duration_attr_type_r['prompt'];
									
			echo get_input_field("default_borrow_duration",
							$duration_attr_type_r['s_attribute_type'],
							$prompt, 
							$duration_attr_type_r['input_type'],
							"N",//not compulsory, as we already have default!
							$default_borrow_duration,
							TRUE);
		
			// Not appropriate for extension operation
			if($op == 'check_out' || $op == 'quick_check_out')
			{
				// No point unless more than one item, as the default_borrow_duration will actually
				// be the same as the single item.
				if(is_array($borrowed_item_rs) && count($borrowed_item_rs)>1)
				{
					echo get_input_field("use_item_duration",
								NULL, 
								$LANG_VARS['use_item_duration'], 
								"simple_checkbox()",
								"N",//not compulsory, as we already have default!
								"Y",
								TRUE);
				}
			}
		}
		else//otherwise tell checkout to use item_instane borrow duration instead.
		{
			if($op == 'check_out' || $op == 'quick_check_out')
				echo("\n<input type=\"hidden\" name=\"use_item_duration\" value=\"Y\">");
		}
	}
					
	echo("<tr><td colspan=2 align=left><br><input type=submit value=\"".$LANG_VARS['continue']."\"></td></tr>");
	echo("</form>");
	echo("</table>");

	// Display some help information for Duration support functionality.
	if(($op == 'check_out' || $op == 'quick_check_out') && $CONFIG_VARS['borrow.duration_support']!==FALSE && strlen($duration_attr_type)>0 && is_array($borrowed_item_rs) && count($borrowed_item_rs)>1)
	{	
		echo(format_help_block($LANG_VARS['check_out_borrow_duration_moreinfo_help']));
	}
}

/**
	@param cannot be about particular items!
*/
function display_borrow_results($op, $heading, $success_intro, $failure_intro, $more_information, $success_item_rs, $failure_item_rs, $email_notification=TRUE)
{
	global $LANG_VARS;
	global $CONFIG_VARS;
	global $HTTP_SESSION_VARS;

	if(is_not_empty_array($success_item_rs))
	{
		echo("\n<div class=\"success\">".$success_intro."</div>");
		
		// Sort the items by user, so we can send emails for multiple 
		// items, instead of individually.
		$borrowed_item_user_r = array();
		while(list(,$borrowed_item_r) = each($success_item_rs))
		{
			$item_r = fetch_item_instance_r($borrowed_item_r['item_id'], $borrowed_item_r['instance_no']);

			// Expand title mask.
			$item_r['title'] = expand_item_title_mask($item_r['title'], $item_r['s_item_type'], $item_r['item_id'], $item_r['instance_no'], $item_r['s_status_type'], $CONFIG_VARS['item_display.title_display_mask']);

			// A array of item_entries.
			$item_entry_r['item'] = $item_r;
			$item_entry_r['detail'] = get_borrow_details($op, $item_r, $borrowed_item_r);

			if($HTTP_SESSION_VARS['user_id'] == $borrowed_item_r['borrower_id'])
				$to_user = $item_r['owner_id'];
			else
				$to_user = $borrowed_item_r['borrower_id'];

			// Now add an entry to this user array.
			$borrowed_item_user_r[$to_user][] = $item_entry_r;
		}

		// Now process the user->item structure and send emails.
		$errors = null;
		while(list($to_user,$item_entry_rs) = each($borrowed_item_user_r))
		{
			echo("\n<ul>");
			while(list(,$item_entry_r) = each($item_entry_rs))
			{
				echo("<li>".replace_lang_vars(array('display_title'=>$item_entry_r['item']['title'],'item_id'=>$item_entry_r['item']['item_id'],'instance_no'=>$item_entry_r['item']['instance_no']),$LANG_VARS['borrow_item_title_listing'])."</li>");
			}
			echo("\n</ul>");

			if($CONFIG_VARS['email.use_php_mail']!==FALSE && $email_notification!==FALSE)
			{
				// How can the from user be anything but the currently logged in user!
				if(send_notification_email($to_user, $HTTP_SESSION_VARS['user_id'], $heading, $success_intro, $more_information, $item_entry_rs, $errors))
					echo("\n<div class=\"smsuccess\">".$LANG_VARS['notication_email_sent']."</div>");
				else
				{
					echo("\n<div class=\"smerror\">".$LANG_VARS['notication_email_not_sent']);
					if(count($errors)>0)
					{
						echo ("\n<br>".$LANG_VARS['errors']);
						for($i=0; $i<count($errors); $i++)
							echo("\n<br>&nbsp;&nbsp;&nbsp;&nbsp;<i>".$errors[$i]."</i>");
					}
					echo("\n</div>");
				}
			}
		}
		echo("\n<br><br>");
	}

	// No emails here.
	if(is_not_empty_array($failure_item_rs))
	{
		echo("\n<div class=\"success\">".$failure_intro."</div>");
		echo("\n<ul>");
		while(list(,$borrowed_item_r) = each($failure_item_rs))
		{
			$item_r = fetch_item_instance_r($borrowed_item_r['item_id'], $borrowed_item_r['instance_no']);
			
			// Expand title mask.
			$item_r['title'] = expand_item_title_mask($item_r['title'], $item_r['s_item_type'], $item_r['item_id'], $item_r['instance_no'], $item_r['s_status_type'], $CONFIG_VARS['item_display.title_display_mask']);
			
			echo("\n<li>".replace_lang_vars(array('display_title'=>$item_r['title'],'item_id'=>$item_r['item_id'],'instance_no'=>$item_r['instance_no']),$LANG_VARS['borrow_item_title_listing']));
			// Now display any errors if present.
			if(strlen($borrowed_item_r['errors'])>0)
				echo("\n<br><div class=\"smerror\">".replace_lang_var("error", $borrowed_item_r['errors'], $LANG_VARS['borrow_error_detail'])."</div>");
			echo("\n</li>");
		}
		echo("\n</ul>");
	}
}

/**
	This will format a status line for the borrowed_item_r passed as parameter.

	$op == check_in
		$LANG_VARS['check_in_detail'] = 'Total/Overdue: {total_duration} day(s)/{overdue_duration} day(s)';
	$op == check_out || quick_check_out
		$LANG_VARS['due_date_detail'] = 'Due: {date} ({borrow_duration} day(s))';
	$op == reminder
		$LANG_VARS['due_date_detail'] = 'Due: {date} ({borrow_duration} day(s))';
		$LANG_VARS['check_in_detail'] = 'Total/Overdue: {total_duration} day(s)/{overdue_duration} day(s)';
*/
function get_borrow_details($op, $item_r, $borrowed_item_r)
{
	global $CONFIG_VARS;
	global $LANG_VARS;

	if($op == "check_out" || $op == "quick_check_out" || $op == "reminder" || $op == "extension")
	{
		if($borrowed_item_r['due_date']>0 && is_numeric($borrowed_item_r['borrow_duration']) && $borrowed_item_r['borrow_duration']>0)
		{
			// Disabled 0.60-dev28 - So that borrow duration extension logic works!
			//$duration_attr_type_r = fetch_sfieldtype_item_attribute_type_r($item_r['s_item_type'], 'DURATION', TRUE);
			//$borrow_duration = get_display_field($duration_attr_type_r['s_attribute_type'], NULL, $duration_attr_type_r['display_type'], $borrowed_item_r['borrow_duration'], FALSE);
			$details_r[] = replace_lang_vars(array('date'=>get_localised_timestamp($CONFIG_VARS['borrow.date_mask'], $borrowed_item_r['due_date']),'borrow_duration'=>$borrowed_item_r['borrow_duration']),$LANG_VARS['due_date_detail']);
		}
	}

	if($op == "check_in" || $op == "reminder")
	{
		if($borrowed_item_r['total_duration']>0 && is_numeric($borrowed_item_r['borrow_duration']) && $borrowed_item_r['borrow_duration']>0)
		{
			if($borrowed_item_r['total_duration']>$borrowed_item_r['borrow_duration'])
				$overdue_duration = $borrowed_item_r['total_duration'] - $borrowed_item_r['borrow_duration'];
			else
				$overdue_duration = 0;
			$details_r[] = replace_lang_vars(array('total_duration'=>$borrowed_item_r['total_duration'],'overdue_duration'=>$overdue_duration),$LANG_VARS['check_in_detail']);
		}
	}

	return $details_r;
}

/**
	The $borrow_item_r array format:
		$item_r
		$details
*/
function send_notification_email($to_user, $from_user, $heading, $introduction, $more_information, $item_entry_rs, &$errors)
{
	global $LANG_VARS;
	global $CONFIG_VARS;

	// Get the address information.
	$to_user_r = fetch_user_r($to_user);
	$from_user_r = fetch_user_r($from_user);

	// Format the entire message.
	$message =  replace_lang_var("fullname", $to_user_r['fullname'], expand_langvar_newlines($LANG_VARS['to_user_email_intro'])).
				"\n\n".
				$introduction;
	
	while(list(,$item_entry_r) = each($item_entry_rs))
	{
		// Only one email at a time initially, but we will improve logic eventually.
		$message .= "\n*    ".replace_lang_vars(array('display_title'=>$item_entry_r['item']['title'],'item_id'=>$item_entry_r['item']['item_id'],'instance_no'=>$item_entry_r['item']['instance_no']), $LANG_VARS['borrow_item_title_listing']);

		// Add any item Borrow (overdue,due,reminder,etc) details here.
		while(list(,$detail) = @each($item_entry_r['detail']))
		{
			$message .= "\n     - ".$detail;
		}
	}

	if(strlen($more_information)>0)
	{
		$message .= "\n\n\n\n\n".
				$LANG_VARS['more_information'].
				"\n";

		// Add title underline...
		for($i=0;$i<strlen($LANG_VARS['more_information']);$i++)
			$message .= "-";

		$message .= "\n".
					$more_information.
					"\n\n";
	}

	// Send the mail!
	$results = opendb_email($to_user_r['email'],
					$to_user_r['fullname'],
					$from_user_r['email'],
					$from_user_r['fullname'],
					$heading,
					$message);

	if($results === TRUE)
		return TRUE;
	else
	{
		$errors = $results;
		return FALSE;
	}			
}

function add_errors_to_borrowed_item_r($borrowed_item_r, $errors)
{
	$borrowed_item_r['errors'] = $errors;
	return $borrowed_item_r;
}

session_start();
if (is_opendb_valid_session())
{
	if($CONFIG_VARS['borrow.enable']!==FALSE)
	{
		if(is_user_allowed_to_borrow($HTTP_SESSION_VARS['user_id'], $HTTP_SESSION_VARS['user_type'])) // don't allow guests in!
		{
			// So we can get errors back!
			$errors = null;
			
			if ( ( $HTTP_VARS['op'] == 'reserve_all' && is_not_empty_array($HTTP_SESSION_VARS['reserve_basket_listing']) ) || 
						($HTTP_VARS['op'] == 'reserve' &&
							(is_not_empty_array($HTTP_VARS['item_id_instance_no']) || 
							strlen(trim($HTTP_VARS['checked_item_id_instance_no_list']))>0 || 
							(is_numeric($HTTP_VARS['item_id']) && is_numeric($HTTP_VARS['instance_no']))) ) ) 
			{	
				echo _theme_header($LANG_VARS['item_reservation']);
				echo("<h2>".$LANG_VARS['item_reservation']."</h2>");

				// If $op == 'reserve_all', then the reserve_basket_listing session variable should be taken into account, 
				// otherwise it should be ignored.
				if($HTTP_VARS['op'] == 'reserve_all')
				{
					$HTTP_VARS['item_id_instance_no'] = $HTTP_SESSION_VARS['reserve_basket_listing'];
					
					// Reset - so we will not process the session list again.
					$HTTP_VARS['op'] == 'reserve';
				}
				else if(is_not_empty_array($HTTP_VARS['checked_item_id_instance_no']) || strlen(trim($HTTP_VARS['checked_item_id_instance_no_list']))>0)
				{
					$listingObject = new Listing();
					$listingObject->setCheckBoxColumn('item_id_instance_no');
					
					// The checked_item_id_instance_no and checked_item_id_instance_no_list variables
					// will be unset by the this call.
					$listingObject->setCheckedList(NULL, $HTTP_VARS);
					
					$HTTP_VARS['item_id_instance_no'] = $listingObject->getCheckedList();
				
					// Don't need the Listings object anymore.
					unset($listingObject);
				}
				
				if(is_not_empty_array($HTTP_VARS['item_id_instance_no']))
				{
					// Format of $item_id_instance_no {item_id}_{instance_no}
					while ( list(,$item_id_instance_no) = each($HTTP_VARS['item_id_instance_no']))
					{
						$item_id_instance_no_r = get_item_id_and_instance_no($item_id_instance_no);
						if(is_not_empty_array($item_id_instance_no_r))
							$reserve_item_rs[] = $item_id_instance_no_r;
					}
				}
				else if(is_numeric($HTTP_VARS['item_id']) && is_numeric($HTTP_VARS['instance_no']))
					$reserve_item_rs[] = array('item_id'=>$HTTP_VARS['item_id'],'instance_no'=>$HTTP_VARS['instance_no']);
				
				// There is no point in providing a More Information form, unless we either have use of php email,
				// or we are in checkout mode.
				if($CONFIG_VARS['borrow.reserve_more_information'] && $HTTP_VARS['more_info_requested'] != "true")//even if $message_done is empty it is still set!
				{
					more_information_form('reserve', 
										$reserve_item_rs, 
										$HTTP_VARS,
										$CONFIG_VARS['borrow.reserve_email_notification']);
				}
				else
				{
					// Now get rid of the items that have been processed.
					if(is_not_empty_array($HTTP_SESSION_VARS['reserve_basket_listing']))
					{
						// Remove the items which have been processed.
						$HTTP_SESSION_VARS['reserve_basket_listing'] = 
														minus_array_values(
															$HTTP_SESSION_VARS['reserve_basket_listing'], 
															$HTTP_VARS['item_id_instance_no']);

						// If still items left, better update the array.						
						if(is_not_empty_array($HTTP_SESSION_VARS['reserve_basket_listing']))
						{
							if(is_register_globals_enabled())
							{
								session_register('reserve_basket_listing');
								$reserve_basket_listing = $HTTP_SESSION_VARS['reserve_basket_listing']; 
							}
						}
						else
						{	
							if(is_register_globals_enabled())
							{
								session_unregister('reserve_basket_listing');
							}
						}
					}
					
					while ( list(,$reserve_item_r) = @each($reserve_item_rs))
					{
						// In case someone is trying to pass invalid item_id/instance_no combo's
						if(is_exists_item_instance($reserve_item_r['item_id'],$reserve_item_r['instance_no']))
						{
							if(($new_borrowed_item_id = handle_reserve($reserve_item_r['item_id'], $reserve_item_r['instance_no'], $HTTP_SESSION_VARS['user_id'], $errors)) !== FALSE)// This allows reserve to support calls from borrow.php, item_display.php or listings.php
								$success_items_rs[] = array(item_id=>$reserve_item_r['item_id'],instance_no=>$reserve_item_r['instance_no'],borrower_id=>$HTTP_SESSION_VARS['user_id']);	
							else
								$failure_items_rs[] = array(item_id=>$reserve_item_r['item_id'],instance_no=>$reserve_item_r['instance_no'],borrower_id=>$HTTP_SESSION_VARS['user_id'],errors=>$errors);
						}
					}
					
					display_borrow_results($HTTP_VARS['op'],
									$LANG_VARS['item_reservation'],
									$LANG_VARS['items_have_been_registered'], 
									$LANG_VARS['items_have_not_been_registered'],
 								    $HTTP_VARS['more_information'],
									$success_items_rs, 
									$failure_items_rs,
									$CONFIG_VARS['borrow.reserve_email_notification']);
				}
			}
			else if($HTTP_VARS['op'] == "cancel_reserve" && 
						( (is_numeric($HTTP_VARS['sequence_number']) || 
							is_not_empty_array($HTTP_VARS['sequence_number'])) || 
							is_not_empty_array($HTTP_VARS['item_id_instance_no']) || 
							(is_numeric($HTTP_VARS['item_id']) && is_numeric($HTTP_VARS['instance_no'])) ) )
			{
				echo _theme_header($LANG_VARS['item_cancel_reservation']);
				echo("<h2>".$LANG_VARS['item_cancel_reservation']."</h2>");

				// So we can process only sequence_numbers
				if(is_not_empty_array($HTTP_VARS['item_id_instance_no']))
				{
					// Format of $item_id_instance_no {item_id}_{instance_no}
					reset($HTTP_VARS['item_id_instance_no']);
					while ( list(,$item_id_instance_no) = each($HTTP_VARS['item_id_instance_no']))
					{
						$item_id_instance_no_r = get_item_id_and_instance_no($item_id_instance_no);
						if(is_not_empty_array($item_id_instance_no_r))
							$sequence_number_r[] = fetch_borrowed_item_seq_no($item_id_instance_no_r['item_id'], $item_id_instance_no_r['instance_no'], $HTTP_SESSION_VARS['user_id'], 'R');
					}
				}
				else if(is_not_empty_array($HTTP_VARS['sequence_number']))
					$sequence_number_r = $HTTP_VARS['sequence_number'];		
				else if(is_numeric($HTTP_VARS['sequence_number']))
					$sequence_number_r[] = $HTTP_VARS['sequence_number']; //convert to array here.
				else if(is_numeric($HTTP_VARS['item_id']) && is_numeric($HTTP_VARS['instance_no']))//In this case the $borrower_id has to be the current user, no one else can cancel using this function!
					$sequence_number_r[] = fetch_borrowed_item_seq_no($HTTP_VARS['item_id'], $HTTP_VARS['instance_no'], $HTTP_SESSION_VARS['user_id'], 'R');

				// There is no point in providing a More Information form, unless we either have use of php email,
				// or we are in checkout mode.
				if($CONFIG_VARS['borrow.cancel_more_information'] && $HTTP_VARS['more_info_requested'] != "true")
				{
					more_information_form('cancel_reserve', 
										$sequence_number_r, 
										$HTTP_VARS,
										$CONFIG_VARS['borrow.cancel_email_notification']);
				}
				else
				{
					while ( list(,$sequence_number) = @each($sequence_number_r))
					{
						// This allows cancel-reserve to support calls from borrow.php, item_display.php or listings.php
						if(handle_cancelreserve($sequence_number, $errors))
							$success_items_rs[] = fetch_borrowed_item_r($sequence_number);
						else
							$failure_items_rs[] = add_errors_to_borrowed_item_r(fetch_borrowed_item_pk_r($sequence_number), $errors);
					}
			
					display_borrow_results($HTTP_VARS['op'],
									$LANG_VARS['item_cancel_reservation'],
									$LANG_VARS['reserve_items_have_been_cancelled'], 
									$LANG_VARS['reserve_items_have_not_been_cancelled'],
 								    $HTTP_VARS['more_information'],
									$success_items_rs, 
									$failure_items_rs,
									$CONFIG_VARS['borrow.cancel_email_notification']);
				}
			}
			else if($HTTP_VARS['op'] == "check_out" && ( is_numeric($HTTP_VARS['sequence_number']) || is_not_empty_array($HTTP_VARS['sequence_number']) ) )
			{
				echo _theme_header($LANG_VARS['item_check_out']);
				echo("<h2>".$LANG_VARS['item_check_out']."</h2>");

				// It is easier to assume an array in all cases.	
				if(is_not_empty_array($HTTP_VARS['sequence_number']))
					$sequence_number_r = $HTTP_VARS['sequence_number'];
				else if(is_numeric($HTTP_VARS['sequence_number']))//is_numeric!
					$sequence_number_r[] = $HTTP_VARS['sequence_number']; //convert to array here.

				// There is no point in providing a More Information form, unless we either have use of php email,
				// or we are in checkout mode.
				if($CONFIG_VARS['borrow.checkout_more_information'] && $HTTP_VARS['more_info_requested'] != "true")
				{	
					more_information_form('check_out', 
									$sequence_number_r, 
									$HTTP_VARS,
									$CONFIG_VARS['borrow.checkout_email_notification']);
				}
				else
				{
					while ( list(,$sequence_number) = @each($sequence_number_r))
					{
						$borrow_duration = NULL;
						
						// The More Information form was not presented
						// So we need to get the default duration from the item table.
						if($HTTP_VARS['more_info_requested'] != "true")
						{
							$borrowed_item_r = fetch_borrowed_item_pk_r($sequence_number);
							if(is_not_empty_array($borrowed_item_r))
							{
								$item_r = fetch_item_instance_r($borrowed_item_r['item_id'], $borrowed_item_r['instance_no']);
								if(is_not_empty_array($item_r))
								{
									$borrow_duration = $item_r['borrow_duration'];
								}
							}
						}
						else // else more information form presented, so we have to factor in overriding borrow duration.
						{
							// In this case a duration of '' is supported
							if($HTTP_VARS['use_item_duration'] == 'Y')
							{
								$borrowed_item_r = fetch_borrowed_item_pk_r($sequence_number);
								if(is_not_empty_array($borrowed_item_r))
								{
									$item_r = fetch_item_instance_r($borrowed_item_r['item_id'], $borrowed_item_r['instance_no']);
									if(is_not_empty_array($item_r))
									{
										$borrow_duration = $item_r['borrow_duration'];
									}
								}
								$borrow_duration = $item_r['borrow_duration'];
							}
							
							if(!is_numeric($borrow_duration))
							{
								$borrow_duration = $HTTP_VARS['default_borrow_duration'];
							}
						}

						if(handle_checkout($sequence_number, $borrow_duration, $errors))
							$success_items_rs[] = fetch_borrowed_item_r($sequence_number,TRUE);
						else
							$failure_items_rs[] = add_errors_to_borrowed_item_r(fetch_borrowed_item_pk_r($sequence_number), $errors);
					}

					display_borrow_results($HTTP_VARS['op'],
										$LANG_VARS['item_check_out'],
										$LANG_VARS['items_have_been_checked_out'], 
										$LANG_VARS['items_have_not_been_checked_out'],
 									    $HTTP_VARS['more_information'],
										$success_items_rs, 
										$failure_items_rs,
										$CONFIG_VARS['borrow.checkout_email_notification']);
				}
			}
			else if($HTTP_VARS['op'] == "quick_check_out" && $CONFIG_VARS['borrow.quick_checkout'] && is_numeric($HTTP_VARS['item_id']) && is_numeric($HTTP_VARS['instance_no']))
			{
				echo _theme_header($LANG_VARS['item_quick_check_out']);
				echo("<h2>".$LANG_VARS['item_quick_check_out']."</h2>");

				$checkout_item_rs[] = array('item_id'=>$HTTP_VARS['item_id'],'instance_no'=>$HTTP_VARS['instance_no']);
				
				// In the case of quick check, we always want a more information form, because this is
				// where the 'borrower_id' is selected.
				if($HTTP_VARS['more_info_requested'] != "true")
				{	
					more_information_form(
								'quick_check_out',
								$checkout_item_rs,
								$HTTP_VARS,
								$CONFIG_VARS['borrow.quick_checkout_email_notification']);
				}
				else
				{
					while ( list(,$checkout_item_r) = @each($checkout_item_rs))
					{
						// In case someone is trying to pass invalid item_id/instance_no combo's
						if(is_exists_item_instance($checkout_item_r['item_id'],$checkout_item_r['instance_no']))
						{
							$borrow_duration = NULL;
							if($HTTP_VARS['use_item_duration'] == 'Y' && is_numeric($item_r['borrow_duration']))
								$borrow_duration = $item_r['borrow_duration'];
							else
								$borrow_duration = $HTTP_VARS['default_borrow_duration'];	
							
							// This allows reserve to support calls from borrow.php, item_display.php or listings.php
							$new_borrowed_item_id = handle_quick_checkout($checkout_item_r['item_id'], $checkout_item_r['instance_no'], $HTTP_VARS['borrower_id'], $borrow_duration, $errors);
							if($new_borrowed_item_id!==FALSE)
								$success_items_rs[] = fetch_borrowed_item_r($new_borrowed_item_id,TRUE);
							else
								$failure_items_rs[] = array(item_id=>$checkout_item_r['item_id'],instance_no=>$checkout_item_r['instance_no'],borrower_id=>$HTTP_VARS['borrower_id'],errors=>$errors);
						}							
					}

					display_borrow_results($HTTP_VARS['op'],
									$LANG_VARS['item_quick_check_out'],
									$LANG_VARS['items_have_been_checked_out'], 
									$LANG_VARS['items_have_not_been_checked_out'],
 								    $HTTP_VARS['more_information'],
									$success_items_rs, 
									$failure_items_rs,
									$CONFIG_VARS['borrow.quick_checkout_email_notification']);
				}
			}
			else if($HTTP_VARS['op'] == "check_in" && ( is_numeric($HTTP_VARS['sequence_number']) || is_not_empty_array($HTTP_VARS['sequence_number']) ) )
			{
				echo _theme_header($LANG_VARS['item_check_in']);		
				echo("<h2>".$LANG_VARS['item_check_in']."</h2>");

				// It is easier to assume an array in all cases.	
				if(is_not_empty_array($HTTP_VARS['sequence_number']))
					$sequence_number_r = $HTTP_VARS['sequence_number'];
				else if(is_numeric($HTTP_VARS['sequence_number']))//is_numeric
					$sequence_number_r[] = $HTTP_VARS['sequence_number']; //convert to array here.
						
				// There is no point in providing a More Information form, unless we either have use of php email,
				// or we are in checkout mode.
				if($CONFIG_VARS['borrow.checkin_more_information'] && $HTTP_VARS['more_info_requested'] != "true")
				{
					more_information_form('check_in', 
									$sequence_number_r, 
									$HTTP_VARS,
									$CONFIG_VARS['borrow.checkin_email_notification']);
				}
				else
				{
					while ( list(,$sequence_number) = @each($sequence_number_r))
					{
						if(handle_checkin($sequence_number, $HTTP_SESSION_VARS['user_id'], $errors))
							$success_items_rs[] = fetch_borrowed_item_r($sequence_number);
						else
							$failure_items_rs[] = add_errors_to_borrowed_item_r(fetch_borrowed_item_pk_r($sequence_number), $errors);
					}

					display_borrow_results($HTTP_VARS['op'],
									$LANG_VARS['item_check_in'],
									$LANG_VARS['items_have_been_checked_in'], 
									$LANG_VARS['items_have_not_been_checked_in'],
 								    $HTTP_VARS['more_information'],
									$success_items_rs, 
									$failure_items_rs,
									$CONFIG_VARS['borrow.checkin_email_notification']);
				}
			}
			else if($HTTP_VARS['op'] == "reminder" && ( is_numeric($HTTP_VARS['sequence_number']) || is_not_empty_array($HTTP_VARS['sequence_number']) ) )
			{
				echo _theme_header($LANG_VARS['check_in_reminder']);
				echo("<h2>".$LANG_VARS['check_in_reminder']."</h2>");

				// It is easier to assume an array in all cases.	
				if(is_not_empty_array($HTTP_VARS['sequence_number']))
					$sequence_number_r = $HTTP_VARS['sequence_number'];
				else if(is_numeric($HTTP_VARS['sequence_number']))//is_numeric
					$sequence_number_r[] = $HTTP_VARS['sequence_number']; //convert to array here.
						
				// If we are providing for custom email to go along with the message then we need to take care of it here.
				if($CONFIG_VARS['borrow.reminder_more_information'] && $HTTP_VARS['more_info_requested'] != "true")
				{
					more_information_form('reminder', 
									$sequence_number_r, 
									$HTTP_VARS,
									TRUE);
				}
				else
				{
					while ( list(,$sequence_number) = @each($sequence_number_r))
					{
						if(handle_reminder($sequence_number, $errors))
							$success_items_rs[] = fetch_borrowed_item_r($sequence_number,TRUE);
						else
							$failure_items_rs[] = add_errors_to_borrowed_item_r(fetch_borrowed_item_pk_r($sequence_number), $errors);
					}

					display_borrow_results($HTTP_VARS['op'],
									$LANG_VARS['check_in_reminder'],
									$LANG_VARS['check_in_reminder_for_items'],
									$LANG_VARS['check_in_reminder_not_for_items'],
 								    $HTTP_VARS['more_information'],
									$success_items_rs, 
									$failure_items_rs,
									TRUE);
				}
			}
			else if($HTTP_VARS['op'] == "extension" && $CONFIG_VARS['borrow.duration_support']!==FALSE && ( is_numeric($HTTP_VARS['sequence_number']) || is_not_empty_array($HTTP_VARS['sequence_number']) ) )
			{
				echo _theme_header($LANG_VARS['borrow_duration_extension']);
				echo("<h2>".$LANG_VARS['borrow_duration_extension']."</h2>");

				// It is easier to assume an array in all cases.	
				if(is_not_empty_array($HTTP_VARS['sequence_number']))
					$sequence_number_r = $HTTP_VARS['sequence_number'];
				else if(is_numeric($HTTP_VARS['sequence_number']))//is_numeric
					$sequence_number_r[] = $HTTP_VARS['sequence_number']; //convert to array here.
						
				// If we are providing for custom email to go along with the message then we need to take care of it here.
				if(!is_numeric($HTTP_VARS['default_borrow_duration']) || $HTTP_VARS['more_info_requested'] != "true")
				{
					// If more info has already been requested, then the 
					// default_borrow_duration must have not been specified
					if($HTTP_VARS['more_info_requested'] == "true")
					{
						echo format_error_block($LANG_VARS['borrow_duration_extension_must_be_specified']);
					}
					more_information_form('extension', 
									$sequence_number_r, 
									$HTTP_VARS,
									TRUE);
				}
				else
				{
					while ( list(,$sequence_number) = @each($sequence_number_r))
					{
						if(handle_extension($sequence_number, $HTTP_VARS['default_borrow_duration'], $errors))
							$success_items_rs[] = fetch_borrowed_item_r($sequence_number,TRUE);
						else
							$failure_items_rs[] = add_errors_to_borrowed_item_r(fetch_borrowed_item_pk_r($sequence_number), $errors);
					}

					display_borrow_results($HTTP_VARS['op'],
								$LANG_VARS['borrow_duration_extension'],
								$LANG_VARS['borrow_duration_extension_for_items'],
								$LANG_VARS['borrow_duration_extension_not_for_items'],
							    $HTTP_VARS['more_information'],
								$success_items_rs, 
								$failure_items_rs,
								TRUE);
				}
			}
			else
			{
				echo _theme_header($LANG_VARS['operation_not_available']);
				echo("<h2>".$LANG_VARS['operation_not_available']."</h2>");
				echo("<div class=\"error\">".$LANG_VARS['operation_not_available']."</div>");
			}
		}//no guests allowed!
		else
		{
			echo _theme_header($LANG_VARS['not_authorized_to_page']);
			echo _theme_error($LANG_VARS['not_authorized_to_page']);
		}
	}//borrow functionality disabled.
	else
	{
		echo _theme_header($LANG_VARS['borrow_not_supported']);
		echo _theme_error($LANG_VARS['borrow_not_supported']);
	}

	// Include a link no matter what, because they might have initiated the action by accident.
	if($HTTP_VARS['item_link'] === "y" && is_numeric($HTTP_VARS['item_id']) && is_numeric($HTTP_VARS['instance_no']))
	{
		$footer_links_r[] = array(url=>"item_display.php?item_id=".$HTTP_VARS['item_id']."&instance_no=".$HTTP_VARS['instance_no']."&listing_link=".$HTTP_VARS['listing_link'],text=>$LANG_VARS['back_to_item']);
	}
	if($HTTP_VARS['listing_link'] === "y" && is_not_empty_array($HTTP_SESSION_VARS['listing_url_vars']))
	{
		$footer_links_r[] = array(url=>"listings.php?".get_url_string($HTTP_SESSION_VARS['listing_url_vars']),text=>$LANG_VARS['back_to_listing']);
	}

	echo format_footer_links($footer_links_r);
	echo _theme_footer();
}
else
{
	include("./include/invalidsession.php");
}

// Cleanup after begin.inc.php
require_once("./include/end.inc.php");
?>
