<?php
/* 	OpenDb - Open Media Lending Database
	Copyright (C) 2001,2002 by Jason Pell

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

include_once("./functions/item_attribute.php");
include_once("./functions/item_type.php");
include_once("./functions/item.php");
include_once("./functions/http.php");
include_once("./functions/fileutils.php");
include_once("./functions/utils.php");
include_once("./functions/parseutils.php");
include_once("./functions/datetime.php");
include_once("./functions/email.php");
include_once("./functions/status_type.php");
include_once("./functions/theme.php");
// -------------------------------------------------------------
// This set of functions is used to provide input/display fields for
// item_input.php and possibly in the future user_admin.php, etc.
// -------------------------------------------------------------

// Will be set to true by any input field function that uses a type=file
$_OPENDB_FILE_UPLOAD_WIDGET=FALSE;

function get_popup_javascript($wrap = TRUE)
{
	$script = 
	"function popup(url, width, height)
	{
		window.open(url, '_blank', 'resizable=yes,toolbar=no,scrollbars=yes,location=no,menubar=no,status=no,width='+width+',height='+height);
	}";
	
	if($wrap)
	{
		$script = "\n<script language=\"JavaScript\">\n<!-- // hide from stupid browsers\n".
				$script.
				"\n// -->\n</script>\n";
	}	
	return $script;
}

function get_setcheckboxes_javascript($wrap = TRUE)
{
	$script = 
	"function setCheckboxes(form, elname, checked)".
	"\n{".
	"\n	for (var i=0; i < form.length; i++)".
	"\n	{".
	"\n		if (form.elements[i].type.toLowerCase() == 'checkbox' && ".
	"\n				(elname == null || form.elements[i].name == elname || form.elements[i].name == elname+'[]' || (form.elements[i].name.substring(0,elname.length+1) == elname+'[' && form.elements[i].name.substring(form.elements[i].name.length-1) == ']')))".
	"\n		{".
	"\n			form.elements[i].checked = checked;".
	"\n		}".
	"\n	}".
	"\n}";
	
	if($wrap)
	{
		$script = "\n<script language=\"JavaScript\">\n<!-- // hide from stupid browsers\n".
				$script.
				"\n// -->\n</script>\n";
	}	
	return $script;
}
					
/*
 This function will return a <script> block of all javascript
 validation functions used by any of the input_fields.
*/
function get_validation_javascript()
{
	return '<script src="./include/validation.js" language="JavaScript" type="text/javascript"></script>'.
			'<script src="./include/date.js" language="JavaScript" type="text/javascript"></script>'.
			"\n<script language=\"JavaScript\">\n<!-- // hide from stupid browsers\n".
			get_popup_javascript(FALSE).
			"\n// -->\n</script>\n";
}

/*
	Supported input widgets:
    	
  	hidden										Display a hidden field
	readonly									Display a hidden field and a readonly text field
  	text(length,maxlength,field_mask)			Display a text field
	email(length,maxlength,field_mask)			Display a email field.  Will validate that email
												address is valid.
	filtered(length,maxlength,legalchars,field_mask)
												A filtered text field.  Only characters specified
												in legalchars will be allowed, any others will be
												removed when onChange event fires.  You can specify
												?-? ranges, and include '-', using '\-'.
	date(mask) [NOT IMPLEMENTED]				This field will validate an entered date against
												a specified mask.
	format(mask)[NOT IMPLEMENTED]				This field will validate input against a mask.  Where
												the mask values are: 9 = numbers; X = uppercase letters
												x = lowercase letters.  No other numeral or number is 
												allowed.  Any punctuation, such as '.', '-' will be
												included in the final value sent to the database.

  	simple_checkbox(CHECKED,field_mask)			A really simple little input field, that does not do much
												of use.  Used to be known as 'checkbox', but we added a 
												much more powerful field of the same name with different
												parameters.  Not used in OpenDb anymore - kept for backwards
												compatibility, although this is hardly necessary.
												
	checkbox(checked-val,					    This is a new much more powerful checkbox solution.
			unchecked-val,					    This one will be used when the checkbox is called with
			field_mask)							more than two parameters.  You can still use this version
												without specifying the display_mask, by including an extra
												','.  This will only work from 0.50-dev26 onwards as the
												prc_function_spec function has been updated to support this.

	textarea(cols,rows,field_mask)				Displays a textarea with specified 
  	number(length,field_mask)					Display a text field, which can only have numeric input.
  												Numeric fields also have a maxlength exactly the same as 
  												their length.
  	check_boxes(display_mask, orientation)		A checkbox for each value in lookup table
  	radio_group(display_mask, orientation)		A set of radio buttons, one for each lookup value.
	
	url(length,maxlength,"ext,ext2,etc"[,viewbutton])	Similiar to 'saveurl', except this does not provide the
	 											ability to save locally.

	upload(length,maxlength,"ext,ext2,etc")		Upload field, complete with extension validation.  It is
												up to the item_input.php script to ensure that the 
												validations for extensions are performed, in case the
												javascript validations have been bypassed.
												
	saveurl(length,
	 		maxlength,
			"ext,ext2,etc",
			viewbutton,
			saveurl-checked)					As of O.51-dev3 this field serves as a plain text field
												to store a URL value.  This url value may be used in the
												case of IMAGEURL to display a Cover image.  A 'Save'
												checkbox is also included with the field.  If this checkbox
												is checked, and there is a valid external url (The url and
												specified upload directory must accessible by the server 
												running this application), the external URL will be downloaded
												and saved locally.  The attribute value will be updated to
												point at this local copy.  In order to be considered valid,
												the url must end in one of the extensions specified for the 
												first argument.	When item_input.php executes, it will display 
												an error if the file cannot be downloaded.  The URL itself
  												will still be saved to the database if its extension is valid.
	
	upload_or_saveurl(
	 		length,
	 		maxlength,
	 		"ext,ext2,etc",
			viewbutton,
			saveurl-checked)					A combination of both upload/saveurl widgets.  It will include
												both the saveurl and upload form input fields.  The upload
												part of the field will be checked first (as a normal 'upload'
												widget would be) and if a valid FILE_UPLOAD is found, it will
												be processed as normal.  Otherwise the 'saveurl' portion will
												be accessed, and processed as per 'saveurl' functionality.  If
												neither of the fields are processed the default functionality is
												to restore the previous value of the field, or in the case of
												'saveurl' (if the URL has a valid extension), the URL itself
												will be saved.

	For check_boxes and radio_group versions with a vertical_ or horizontal_ prefix and no orientation will
	still resolve correctly.

	**** Special case - only used in item_review.php
	review_options(display_mask, orientation)	Displays a list of options, with stars beside.  Replaces
												the logic in item_review.php.

	Note: For all the above orientation corresponds to HORIZONTAL or VERTICAL

	radio_grid(display_mask,columns,border)		Display radio group in a grid of columns wide.
	checkbox_grid(display_mask,columns,border)	Display checkboxes in a grid of columns wide.
	
  	single_select(display_mask, length)			A single select list
	multi_select(display_mask, length, size)	A multi select list

	value_select(values, size)	A select widget which will generate an lov based on the comma
												delimited list of values specified for the first argument.
												In order to get this to work, you will need to enclose the
												values argument in double quotes.  If $size>1, a MULTIPLE 
												select object will be generated.  The fieldname will be 
												modified so that it returns an array via HTTP.

	Note: The s_attribute_type is for generating the lookup records.

	@param $dowrap 			Specify whether the field should be wrapped using format_field before
							returning.  Fields of type hidden will ignore this variable even if true.
	@param $promp_mask		The %prompt% variable will be replaced with the actual prompt.
							This parameter will be ignored if $dowrap is FALSE.
   	@param $onchange_event	Specify extra javascript for onchange event handler.  	
	 						The onchange event is passed to widgets:
								'textarea', 'text', 'url', 'saveurl', 'filtered', 'number', 'email',
								'single_select', 'multi_select','value_select'

							For compatibility with older browsers the widgets:
								'simple_checkbox', 'checkbox' use the 'onclick' event instead.
							
	The compulsory indicator is only of use for straight text input fields.  The lookups already enforce
	entering a value, by allowing the selection of a default in the s_attribute_type_lookup table.
    
	Note:
	-----
		The display_mask argument for single_select and multi_select only supports the %value% and %display% specifiers.

		The other functions support all three %img%, %value% and %display%.  If the %img% column has no value, then
		the %value% is used instead.  However if the %img% has a value of "none", then the image tag will be replaced
		with an empty string.

*/  
function get_input_field($fieldname, $s_attribute_type, $prompt, $input_type, $compulsory_ind="N", $value=NULL, $dowrap=TRUE, $prompt_mask=NULL, $onchange_event=NULL)
{
	global $LANG_VARS;
	global $CONFIG_VARS;
	
	// an array will be a lookup value
	if(!is_array($value))
	{
		// Escape all html entities so they are displayed correctly!
		if(strlen($value)>0)
		{
			$value = htmlspecialchars($value);
		}
	}
	
    //Process the input_type to get the widget type and arguments.
    $widget = prc_function_spec($input_type);
	
	// Now we have to work out how to parse the input_type
	if($widget['type'] == 'hidden')
	{
		return hidden_field($fieldname, $value);
	}
	else if($widget['type'] == 'readonly')// arg[0] = field_mask
	{
		return format_input_field($prompt, $widget['args']['0'], readonly_field($fieldname, $value), $dowrap, $prompt_mask);
	}
	else if($widget['type'] == 'textarea' || $widget['type'] == 'htmlarea') // arg[0] = rows, arg[1] = cols, arg[2] = length, arg[3] = field_mask
	{
		return format_input_field($prompt, $widget['args']['3'], textarea_field($fieldname, $prompt, $widget['args']['0'], $widget['args']['1'], $widget['widget']['2'], $compulsory_ind, $value, $onchange_event), $dowrap, $prompt_mask, $compulsory_ind);
	}
	else if($widget['type'] == 'text') // arg[0] = length of field, arg[1] = maxlength of field, arg[2] = field_mask
	{
		return format_input_field($prompt, $widget['args']['2'], text_field($fieldname, $prompt, $widget['args']['0'], $widget['args']['1'], $compulsory_ind, $value, $onchange_event), $dowrap, $prompt_mask, $compulsory_ind);
	}
	else if($widget['type'] == 'password') // arg[0] = length of field, arg[1] = maxlength of field, arg[2] = field_mask
	{
		return format_input_field($prompt, $widget['args']['2'], password_field($fieldname, $prompt, $widget['args']['0'], $widget['args']['1'], $compulsory_ind, $value, $onchange_event), $dowrap, $prompt_mask, $compulsory_ind);
	}
	else if($widget['type'] == 'email') // arg[0] = length of field, arg[1] = maxlength of field, arg[2] = field_mask
	{
		return format_input_field($prompt, $widget['args']['2'], email_field($fieldname, $prompt, $widget['args']['0'], $widget['args']['1'], $compulsory_ind, $value, $onchange_event), $dowrap, $prompt_mask, $compulsory_ind);
	}
	else if($widget['type'] == 'filtered') // arg[0] = length of field, arg[1] = maxlength of field, arg[2] = legalChars, arg[3] = field_mask
	{
		return format_input_field($prompt, $widget['args']['3'], filtered_field($fieldname, $prompt, $widget['args']['0'], $widget['args']['1'], $widget['args']['2'], $compulsory_ind, $value, $onchange_event), $dowrap, $prompt_mask, $compulsory_ind);
    }
	else if($widget['type'] == 'datetime') // arg[0] = datetime mask, arg[1] = auto_datetime, arg[2] = field_mask
	{
		return format_input_field($prompt, $widget['args']['2'], datetime_field($fieldname, $prompt, ifempty($widget['args']['0'],'DD/MM/YYYY'), $widget['args']['1'], $compulsory_ind, $value, $onchange_event), $dowrap, $prompt_mask, $compulsory_ind);
    }
	else if($widget['type'] == 'number') // arg[0] = length of field, arg[0] = maxlength of field, arg[1] = field_mask
	{
		return format_input_field($prompt, $widget['args']['1'], number_field($fieldname, $prompt, $widget['args']['0'], $widget['args']['0'], $compulsory_ind, $value, $onchange_event), $dowrap, $prompt_mask, $compulsory_ind);
	}
	else if($widget['type'] == 'simple_checkbox') // arg[0] = checked, arg[1] = field_mask
	{
		return format_input_field($prompt, $widget['args']['1'], checkbox_field($fieldname, $prompt, strcasecmp(trim($widget['args']['0']), 'CHECKED')===0, $value, $onchange_event), $dowrap, $prompt_mask, $compulsory_ind);
	}
	else if($widget['type'] == 'checkbox') // arg[0] = checked, arg[1] = unchecked, arg[2] = field_mask
	{
		return format_input_field($prompt, $widget['args']['2'], enhanced_checkbox_field($fieldname, $prompt, $widget['args']['0'], $widget['args']['1'], $value, $onchange_event), $dowrap, $prompt_mask, $compulsory_ind);
	}
	else if($widget['type'] == 'checkbox_grid' || $widget['type'] == 'check_boxes' || $widget['type'] == 'vertical_check_boxes' || $widget['type'] == 'horizontal_check_boxes')
	{
		$lookup_results = fetch_attribute_type_lookup_rs($s_attribute_type, get_lookup_order_by($widget['args']['0']), 'asc');
		if($lookup_results)//arg[0] = display_mask, arg[1] = columns, arg[2] = border
		{
			//backwards compatible
			if($widget['type'] == 'vertical_check_boxes')
				$widget['args']['1'] = '1';
			else if($widget['type'] == 'horizontal_check_boxes')
				$widget['args']['1'] = '*';
			else if($widget['type'] == 'check_boxes')
			{
				if(strcasecmp($widget['args']['1'], 'VERTICAL')===0)
					$widget['args']['1'] = '1';
				else
					$widget['args']['1'] = '*';
			}

			return format_input_field(
						$prompt, 
						NULL, 
						checkbox_grid($fieldname, $lookup_results, $widget['args']['0'], $widget['args']['1'], $widget['args']['2'], $value),
						$dowrap,
						$prompt_mask,
						$compulsory_ind); 	
		}
	}
	else if($widget['type'] == 'radio_grid' || $widget['type'] == 'radio_group' || $widget['type'] == 'vertical_radio_group' || $widget['type'] == 'horizontal_radio_group')
	{
		$lookup_results = fetch_attribute_type_lookup_rs($s_attribute_type, get_lookup_order_by($widget['args']['0']), 'asc');
		if($lookup_results)//arg[0] = display_mask, arg[1] = columns, arg[2] = border
		{
			//backwards compatible
			if($widget['type'] == "vertical_radio_group")
				$widget['args']['1'] = '1';
			else if($widget['type'] == "horizontal_radio_group")
				$widget['args']['1'] = '*';
			else if($widget['type'] == "radio_group")
			{
				if(strcasecmp($widget['args']['1'], 'VERTICAL')===0)
					$widget['args']['1'] = '1';
				else
					$widget['args']['1'] = '*';
			}
				
			return format_input_field($prompt, NULL, radio_grid($fieldname, $lookup_results, $widget['args']['0'], $widget['args']['1'], $widget['args']['2'], $value), $dowrap, $prompt_mask, $compulsory_ind);
		}
	}
	else if($widget['type'] == 'value_radio_grid')//arg[0] = "comma delimited list of values"; arg[1] = number of visible rows (Defaults to single select
	{
		return format_input_field($prompt, NULL, value_radio_grid($fieldname, explode(',', $widget['args']['0']), $widget['args']['1'], $widget['args']['2'], $value), $dowrap, $prompt_mask, $compulsory_ind);
	}
	else if($widget['type'] == 'single_select')
	{
		$lookup_results = fetch_attribute_type_lookup_rs($s_attribute_type, get_lookup_order_by($widget['args']['0']), 'asc');
		if($lookup_results){//arg[0] = display mask, arg[1] = max value length
			return format_input_field($prompt, NULL, single_select($fieldname, $lookup_results, $widget['args']['0'], $widget['args']['1'], $value, $onchange_event), $dowrap, $prompt_mask, $compulsory_ind);
		}
	}
	else if($widget['type'] == 'multi_select')
	{
		$lookup_results = fetch_attribute_type_lookup_rs($s_attribute_type, get_lookup_order_by($widget['args']['0']), 'asc');
		if($lookup_results){//arg[0] = display mask, arg[1] = max value length, arg[2] = select box number of visible rows
			return format_input_field($prompt, NULL, multi_select($fieldname, $lookup_results, $widget['args']['0'], $widget['args']['1'], $widget['args']['2'], $value, $onchange_event), $dowrap, $prompt_mask, $compulsory_ind);
		}
	}
	else if($widget['type'] == 'value_select')//arg[0] = "comma delimited list of values"; arg[1] = number of visible rows (Defaults to single select
	{
		return format_input_field($prompt, NULL, value_select($fieldname, explode(',', $widget['args']['0']), $widget['args']['1'], $value, $onchange_event), $dowrap, $prompt_mask, $compulsory_ind);
	}
	else if($widget['type'] == 'review_options')//arg[1] = display_mask, arg[1] = orientation
	{
		$lookup_results = fetch_attribute_type_lookup_rs($s_attribute_type, 'value', 'desc');//We want the rows highest value first.
		if($lookup_results){
			return format_input_field($prompt, NULL, review_options($fieldname, $lookup_results, $widget['args']['0'], $widget['args']['1'], $value), $dowrap, $prompt_mask, $compulsory_ind);
		}
	}
	else if($widget['type'] == 'url')//arg[0] = length of field, arg[1] = maxlength of field, arg[2] = extensions, $arg[3] = viewbutton (default: Y)
	{
		return format_input_field($prompt, NULL, url($fieldname, $prompt, $widget['args']['0'], $widget['args']['1'], $widget['args']['2'], $widget['args']['3'], $value, $onchange_event), $dowrap, $prompt_mask, $compulsory_ind);
	}
	else if($widget['type'] == 'upload')//arg[0] = length of field, arg[1] = maxlength of field, arg[2] = extensions
	{
		return format_input_field($prompt, NULL, upload($fieldname, $prompt, $widget['args']['0'], $widget['args']['1'], $widget['args']['2']), $dowrap, $prompt_mask, $compulsory_ind);
	}
	else if($widget['type'] == 'saveurl')//arg[0] = length of field, arg[1] = maxlength of field, arg[2] = extensions, $arg[3] = viewbutton (default: Y), $arg[4] = checked (default: N)
	{
		return format_input_field($prompt, NULL, saveurl($fieldname, $prompt, $widget['args']['0'], $widget['args']['1'], $widget['args']['2'], $widget['args']['3'], $widget['args']['4'], $value, $onchange_event), $dowrap, $prompt_mask, $compulsory_ind);
	}
	else if($widget['type'] == 'upload_or_saveurl')//arg[0] = length of field, arg[1] = maxlength of field, arg[2] = extensions, $arg[3] = viewbutton (default: Y), $arg[4] = checked (default: N)
	{
		return format_input_field($prompt, NULL, upload_or_saveurl($fieldname, $prompt, $widget['args']['0'], $widget['args']['1'], $widget['args']['2'], $widget['args']['3'], $widget['args']['4'], $value), $dowrap, $prompt_mask, $compulsory_ind);
	}
	
    //else
    return format_input_field($prompt, NULL, ">>> ERROR (input_type = $input_type) <<<", $dowrap, $prompt_mask, $compulsory_ind);
}

/*
* A display mask consists of %value%, %display% and %img% mask variables, the
* first of %value% or %display% encountered will effect the order by chosen.
*/
function get_lookup_order_by($display_mask)
{
	// if display mask is empty return default of 'value'
	if(strlen($display_mask)==0)
		return 'value'; //default orderby is 'value'
	else
	{
		$displayPos = strpos($display_mask, '%display%');
		$valuePos = strpos($display_mask, '%value%');
		if($displayPos!==FALSE && ($valuePos===FALSE || $valuePos > $displayPos))
			return 'display';
		else
			return 'value';
	} 
}

/*
* Returns true if the $input_type function spec is a recognised
* file widget.  A file widget is a widget which supports saving
* files locally, such as saveurl and upload
*/
function is_file_widget_input_type($input_type)
{
	$function_type = get_function_type($input_type);
	if($function_type == 'saveurl' || $function_type == 'upload' || $function_type == 'upload_or_saveurl')
		return TRUE;
	else
		return FALSE;
}

function is_lookup_widget_input_type($input_type)
{
	$lookup_input_type_funcs = array('vertical_check_boxes',
								'horizontal_check_boxes',
								'vertical_radio_group',
								'horizontal_radio_group',
								'radio_grid',
								'checkbox_grid',
								'single_select',
								'multi_select'
							);
}

/*
* This function assumes that the widgets has been used to create a single form only.
*/
function is_file_upload_form()
{
	global $_OPENDB_FILE_UPLOAD_WIDGET;
	return $_OPENDB_FILE_UPLOAD_WIDGET;
}

/*
* Validate all input fields and return error(s) to caller, if failed.
* 
* This is a basic attempt to prevent users bypassing javascript validation, and causing integrity
* problems in the database.  In future releases this function will be further augmented for other
* widget types.
*/
function validate_input_field($s_attribute_type, $prompt, $input_type, $compulsory_ind="N", $value, &$errors)
{
	global $LANG_VARS;

	$is_lookup_attribute_type = is_lookup_attribute_type($s_attribute_type);
	
	if($compulsory_ind == 'Y' && 
		( ($is_lookup_attribute_type && is_empty_or_not_array($value)) || (!$is_lookup_attribute_type && strlen(trim($value))==0) ))
	{
		$error = array('error'=>replace_lang_var("prompt", $prompt, $LANG_VARS['prompt_must_be_specified']),'detail'=>'');
		if(is_array($errors))
			$errors[] = $error;
		else
			$errors = $error;
		return FALSE;
	}
	
	// If not compulsory, no point validating if $value is empty!
	if(($is_lookup_attribute_type && is_not_empty_array($value)) || (!$is_lookup_attribute_type && strlen(trim($value))>0))
	{
	    //Process the input_type to get the widget type and arguments.
	    $widget = prc_function_spec($input_type);

		// Now we have to work out how to parse the input_type
		switch($widget['type'])
		{
			case 'hidden':
			case 'readonly':
			case 'textarea':
			case 'htmlarea':
			case 'text':
			case 'password':
			case 'simple_checkbox':
			case 'checkbox':
			case 'check_boxes': // deprecated
			case 'vertical_check_boxes': // deprecated
			case 'horizontal_check_boxes': // deprecated
			case 'radio_group': // deprecated
			case 'vertical_radio_group': // deprecated
			case 'horizontal_radio_group': // deprecated
			case 'radio_grid':
			case 'value_radio_grid':
			case 'checkbox_grid':
			case 'single_select':
			case 'multi_select':
			case 'value_select':
				return TRUE;
				break;
				
			case 'url':
			case 'saveurl':
			case 'upload':
			case 'upload_or_saveurl':
				$extension = get_valid_extension($value, $widget['args']['2']);
				if($extension!==FALSE)
					return TRUE;
				else
				{
					$error = array('error'=>replace_lang_vars(array('prompt'=>$prompt,'extensions'=>$widget['args']['2']), $LANG_VARS['url_is_not_valid']),'detail'=>'');
					if(is_array($errors))
						$errors[] = $error;
					else
						$errors = $error;
					return FALSE;
				}
				break;
		
			case 'email':
				if(is_valid_email_addr($value))
					return TRUE;
				else
				{
					$error = array('error'=>replace_lang_var("prompt", $prompt, $LANG_VARS['email_is_not_valid']),'detail'=>'');
					if(is_array($errors))
						$errors[] = $error;
					else
						$errors = $error;					
					return FALSE;
				}
				break;
			
			case 'datetime':
				$timestamp = get_timestamp_for_datetime($value, $widget['args']['0']);
				if($timestamp!==FALSE)
				{
					return TRUE;
				}
				else
				{
					//else perhaps it is a timestamp value already.
					$timestamp = get_timestamp_for_datetime($value, 'YYYYMMDDHH24MISS');
					if($timestamp!==FALSE)
						return TRUE;
					else
					{
						$error = array('error'=>replace_lang_vars(array('prompt'=>$prompt,'format_mask'=>$widget['args']['0']), $LANG_VARS['datetime_is_not_valid']),'detail'=>'');
						if(is_array($errors))
							$errors[] = $error;
						else
							$errors = $error;
						return FALSE;
					}
				}
				break;
			
			case 'filtered':
				$value = trim($value);
				$legalChars = expand_chars_exp($widget['args']['2']);
				for($i=0; $i<strlen($value); $i++)
				{
					if(strstr($legalChars, substr($value,$i,1)) === FALSE)
					{
						$error = array('error'=>replace_lang_vars(array('prompt'=>$prompt,'format'=>'['.$widget['args']['2'].']'), $LANG_VARS['prompt_must_be_format']),'detail'=>'');
						if(is_array($errors))
							$errors[] = $error;
						else
							$errors = $error;
						return FALSE;
					}
				}
				//else
				return TRUE;
			
				break;
			
			case 'number':
				if(is_numeric($value))
					return TRUE;
				else
				{
					$error = array('error'=>replace_lang_vars(array('prompt'=>$prompt,'format'=>'[0-9]'), $LANG_VARS['prompt_must_be_format']),'detail'=>'');
					if(is_array($errors))
						$errors[] = $error;
					else
						$errors = $error;
					return FALSE;
				}
				break;
			
			default:
				return TRUE;
				break;
		}
	}
	else
		return TRUE;		
}

function remove_illegal_chars($value, $legalChars)
{
	$buffer = '';
	for($i=0; $i<strlen($value); $i++)
	{
		if(strstr($legalChars, substr($value,$i,1)) !== FALSE)
		{
			$buffer .= substr($value,$i,1);
		}
	}
					
	return $buffer;
}

/*
* Will filter input field according to the input_type widget.  In some cases there will be no filtering
* performed.  This filter will also do things like remove HTML, and replace windows/mac newlines with
* unix ones.
*/
function filter_input_field($s_attribute_type, $input_type, $value)
{
	global $CONFIG_VARS;

	$is_lookup_attribute_type = is_lookup_attribute_type($s_attribute_type);
	
	// FALSE is not understood as a value, but it means it is not found, so
	// set to NULL which is pretty much the same thing.
	if($value === FALSE)
	{
		return NULL;
	}
	
	if($is_lookup_attribute_type)
	{
		// if lookup attribute and not already an array, convert to an array for rest of process.
		if(!is_array($value) && strlen(trim($value))>0)
		{
			$tmpval = $value;
			unset($value);
			$value[] = $tmpval;
		}
	}
	else
	{
		$value = replace_newlines($value);
	}
	
	// If not compulsory, no point validating if $value is empty!
	if(($is_lookup_attribute_type && is_not_empty_array($value)) || (!$is_lookup_attribute_type && strlen(trim($value))>0))
	{
		//Process the input_type to get the widget type and arguments.
    	$widget = prc_function_spec($input_type);
	
		// Now we have to work out how to parse the input_type
		switch($widget['type'])
		{
			case 'hidden':
			case 'readonly':
			case 'text':
			case 'password':
			case 'textarea':
				return strip_tags($value);

			case 'htmlarea':
				return strip_tags($value, $CONFIG_VARS['widgets.legal_html_tags']);
			
			case 'check_boxes':// deprecated
			case 'vertical_check_boxes':// deprecated
			case 'horizontal_check_boxes':// deprecated
			case 'radio_group':// deprecated
			case 'vertical_radio_group':// deprecated
			case 'horizontal_radio_group':// deprecated
			case 'simple_checkbox':
			case 'checkbox':
			case 'radio_grid':
			case 'checkbox_grid':
			case 'single_select':
			case 'multi_select':
			case 'value_radio_grid':
			case 'value_select':
				return $value;
				
			case 'url':
			case 'saveurl':
			case 'upload':
			case 'upload_or_saveurl':
				return $value;
		
			case 'email':
				return $value;
				
			case 'datetime':
				$components = get_timestamp_components_for_datetime($value, $widget['args']['0']);
				if($components !== FALSE)
				{
					// This is the 'YYYYMMDDHH24MISS' mask.
					$value = 
						str_pad($components['year'],4,'0', STR_PAD_LEFT)
						.str_pad($components['month'],2,'0', STR_PAD_LEFT)
						.str_pad($components['day'],2,'0', STR_PAD_LEFT)
						.str_pad($components['hour'],2,'0', STR_PAD_LEFT)
						.str_pad($components['minute'],2,'0', STR_PAD_LEFT)
						.str_pad($components['second'],2,'0', STR_PAD_LEFT);
						
					return $value;
				}
				else
				{
					$timestamp = get_timestamp_for_datetime($value, 'YYYYMMDDHH24MISS');
					if($timestamp!==FALSE)
					{
						// already a timestamp value.
						return $value;
					}
					else
					{
					
						return $value; // as a last resort
					}
				}

			case 'number':
				return remove_illegal_chars($value, expand_chars_exp('0-9'));
			case 'filtered':
				return remove_illegal_chars($value, expand_chars_exp($widget['args']['2']));
			default:
				return $value;
				break;
		}
	}
	else
		return $value;
}

/*
 Will format onchange event check if $compulsory_ind == 'Y'
*/
function compulsory_ind_check($prompt, $compulsory_ind)
{
	global $LANG_VARS;

	if($compulsory_ind == "Y")
		return "if(this.value.length==0){alert('".replace_lang_var("prompt", $prompt, $LANG_VARS['prompt_must_be_specified'])."'); this.focus(); return false;} ";
	else
		return false;		
}

/**
*/
function hidden_field($name, $value)
{
	return "\n<input type=\"hidden\" name=\"$name\" value=\"$value\">";
}

/**
*/
function readonly_field($name, $value)
{
	return 	$value.
			"\n<input type=\"hidden\" name=\"$name\" value=\"$value\">";
}

/**
  If compulsory_ind = 'Y', we need to provide an onchange event to check
  for this.
*/
function text_field($name, $prompt, $length, $maxlength, $compulsory_ind, $value, $onchange_event=NULL)
{
	global $LANG_VARS;
	global $CONFIG_VARS;
	
	// Default size.
	$size = $length;
	if($size >= 50 || $size<=0)
		$size = 50;

	// Workaround for Mozilla 0.9x problem with input field.	
	$size++;
	
	if($CONFIG_VARS['widgets.enable_javascript_validation']!==FALSE)
	{
		$onchange = "onchange=\"".compulsory_ind_check($prompt, $compulsory_ind)." $onchange_event return true;\"";
	}
	else
	{
		$onchange = "onchange=\"$onchange_event\"";
	}
	
	return "\n<input type=\"text\" name=\"".$name."\" $onchange size=\"".$size."\" ".(is_numeric($maxlength)?"maxlength=\"".$maxlength."\"":"")." value=\"".$value."\">";
}

/**
  If compulsory_ind = 'Y', we need to provide an onchange event to check
  for this.
*/
function password_field($name, $prompt, $length, $maxlength, $compulsory_ind, $value, $onchange_event=NULL)
{
	global $LANG_VARS;
	global $CONFIG_VARS;
	
	// Default size.
	$size = $length;
	if($size >= 50 || $size<=0)
		$size = 50;

	// Workaround for Mozilla 0.9x problem with input field.	
	$size++;
	
	if($CONFIG_VARS['widgets.enable_javascript_validation']!==FALSE)
	{
		$onchange = "onchange=\"".compulsory_ind_check($prompt, $compulsory_ind)." $onchange_event return true;\"";
	}
	else
		$onchange = "onchange=\"$onchange_event\"";

	return "\n<input type=\"password\" name=\"".$name."\" $onchange size=\"".$size."\" ".(is_numeric($maxlength)?"maxlength=\"".$maxlength."\"":"")." value=\"".$value."\">";
}

/**
  If compulsory_ind = 'Y', we need to provide an onchange event to check
  for this.
*/
function email_field($name, $prompt, $length, $maxlength, $compulsory_ind, $value, $onchange_event=NULL)
{
	global $LANG_VARS;
	global $CONFIG_VARS;
	
	// Default size.
	$size = $length;
	if($size >= 50 || $size<=0)
		$size = 50;

	// Workaround for Mozilla 0.9x problem with input field.	
	$size++;

	if($CONFIG_VARS['widgets.enable_javascript_validation']!==FALSE)
		$onchange = "onchange=\"".compulsory_ind_check($prompt, $compulsory_ind)."if(!checkEmail(this.value)){alert('".replace_lang_var("prompt", $prompt, $LANG_VARS['email_is_not_valid'])."');return false;} $onchange_event return true;\"";
	else
		$onchange = "onchange=\"$onchange_event\"";
	
	return "\n<input type=\"text\" name=\"".$name."\" $onchange size=\"".$size."\" ".(is_numeric($maxlength)?"maxlength=\"".$maxlength."\"":""). " value=\"".$value."\">";
}

/**
  If compulsory_ind = 'Y', we need to provide an onchange event to check
  for this.
  
  A special field, which  
*/
function filtered_field($name, $prompt, $length, $maxlength, $legalCharsExp, $compulsory_ind, $value, $onchange_event=NULL)
{
	global $LANG_VARS;
	global $CONFIG_VARS;
	
	// Default size.
	$size = $length;
	if($size >= 50 || $size<=0)
		$size = 50;

	// Workaround for Mozilla 0.9x problem with input field.	
	$size++;

	if($CONFIG_VARS['widgets.enable_javascript_validation']!==FALSE)
	{
		// Get list of legal characters.
		$legalChars = expand_chars_exp($legalCharsExp);
		if(strlen($legalChars)==0)//Default if not defined.
		{
			$legalChars	= expand_chars_exp('a-zA-Z0-9_.');
		}
			
		$onchange = "onchange=\"this.value=legalCharFilter(this.value, '".$legalChars."'); ".compulsory_ind_check($prompt, $compulsory_ind)." $onchange_event return true;\"";
	}
	else
		$onchange = "onchange=\"$onchange_event\"";
		
	return "\n<input type=\"text\" name=\"".$name."\" $onchange size=\"".$size."\" ".(is_numeric($maxlength)?"maxlength=\"".$maxlength."\"":""). " value=\"".$value."\">";
}


/**
  If compulsory_ind = 'Y', we need to provide an onchange event to check
  for this.
  
  The @param display_mask, supports one format specifier of %field% which specifies where the
  &lt;input ...&gt; field is going, the rest is text that will be included in the returned field 
  verbatim.  If display_mask is empty, then format will will a default of the input fiel by itself.
*/
function number_field($name, $prompt, $length, $maxlength, $compulsory_ind, $value, $onchange_event=NULL)
{
	global $LANG_VARS;
	global $CONFIG_VARS;
	
	// Default size.
	$size = $length;
	if($size > 50 || $size<=0)
		$size = 50;
	
	// Workaround for Mozilla 0.94 problem with input field.	
	$size++;

	if($CONFIG_VARS['widgets.enable_javascript_validation']!==FALSE)
	{
		$onchange = "onchange=\"this.value=numericFilter(this.value);".compulsory_ind_check($prompt, $compulsory_ind)." $onchange_event return true;\"";
	}
	else
		$onchange = "onchange=\"$onchange_event\"";
	
	return "\n<input type=\"text\" name=\"".$name."\" $onchange size=\"".$size."\" ".(is_numeric($maxlength)?"maxlength=\"".$maxlength."\"":""). " value=\"".$value."\">";
}

/**
 * Format mask should correspond to a mask using the following components:
 * 
 * 	Mask components supported are:
 *		DD - Days (01 - 31)
 *		MM - Months (01 -12)
 *		YYYY - Years
 *		HH24 - Hours (00 - 23)
 *		HH - Hours (01 - 12)
 *		MI - Minutes (00 - 59)
 *		SS - Seconds (00 - 59)
*/
function datetime_field($fieldname, $prompt, $format_mask, $auto_datetime, $compulsory_ind, $value, $onchange_event)
{
	global $LANG_VARS;
	global $CONFIG_VARS;
	
	if($CONFIG_VARS['widgets.enable_javascript_validation']!==FALSE)
	{
		$onchange = "onchange=\"".compulsory_ind_check($prompt, $compulsory_ind)."if(this.value.length > 0 && !is_datetime(this.value, '".$format_mask."')){alert('".replace_lang_vars(array('prompt'=>$prompt,'format_mask'=>$format_mask), $LANG_VARS['datetime_is_not_valid'])."');return false;} $onchange_event return true;\"";
	}
	else
		$onchange = "onchange=\"$onchange_event\"";
	
	if(strlen($value)>0)
	{
		// the timestamp is stored in the database with the format YYYYMMDDHH24MISS
		$timestamp = get_timestamp_for_datetime($value, 'YYYYMMDDHH24MISS');
		if($timestamp !== FALSE)
		{
			if(strlen($format_mask)==0)
				$format_mask = 'DD/MM/YYYY';
			
			$datetime = get_localised_timestamp($format_mask, $timestamp);
			if($datetime === FALSE)
			{
				$datetime = $value; // as a last resort
			}
		}
		else
		{
			$datetime = $value; // as a last resort
		}
	}
	else
	{
		if($value === NULL && strcasecmp($auto_datetime, 'Y') === 0)
		{
			$datetime = get_localised_timestamp($format_mask); // current date
		}
		else
		{
			$datetime = '';
		}
	}
	return "\n<input type=text name=\"".$fieldname."\" value=\"".$datetime."\" ".$onchange.">";
}

/*
	If compulsory_ind = 'Y', we need to provide an onchange event to check
	for this.
*/
function textarea_field($name, $prompt, $cols, $rows, $length, $compulsory_ind, $value, $onchange_event=NULL)
{
	global $LANG_VARS;
	global $CONFIG_VARS;
	
	if($CONFIG_VARS['widgets.enable_javascript_validation']!==FALSE)
	{
		$onchange = "onchange=\"".compulsory_ind_check($prompt, $compulsory_ind)." $onchange_event return true;\"";
	}
	else
		$onchange = "onchange=\"$onchange_event\"";
	
	return "\n<textarea name=\"$name\" wrap=virtual $onchange cols=\"$cols\" rows=\"$rows\">".
			$value.
			"</textarea>";
}

/**
*/
function checkbox_field($name, $prompt, $checked, $value, $onclick_event=NULL)
{
	global $CONFIG_VARS;
	
	if($CONFIG_VARS['widgets.enable_javascript_validation']!==FALSE)
	{
		$onclick = "onclick=\"$onclick_event return true;\"";
	}
	else
		$onchange = "onchange=\"$onclick_event\"";
	
	return "\n<input type=\"checkbox\" name=\"$name\" value=\"$value\" $onclick ".($checked?"CHECKED":"").">";
}

/**
	@param The $value parameter is never actually used to set the value of the parameter,
	but only to work out which of the $checked and $unchecked values to use.

	Note: CASE INSENSITIVE match performed.
*/
function enhanced_checkbox_field($name, $prompt, $checked_value, $unchecked_value, $value, $onclick_event=NULL)
{
	global $CONFIG_VARS;
	
	// Work out whether checked or not...
	if($value!==NULL && strcasecmp($value, $checked_value)===0)
		$is_checked = TRUE;
	else
		$is_checked = FALSE;
	
	if($CONFIG_VARS['widgets.enable_javascript_validation']!==FALSE)
	{
		$onclick = "onclick=\"$onclick_event return true;\"";
	}
	else
		$onchange = "onchange=\"$onclick_event\"";
	
	return "\n<input type=hidden name=\"$name\" value=\"".($is_checked?$checked_value:$unchecked_value)."\">"
			."\n\t<input type=checkbox name=\"".$name."_cbox\" onclick=\"if (this.checked){this.form['$name'].value='$checked_value';}else{this.form['$name'].value='$unchecked_value';}\" $onclick ".($is_checked?"CHECKED":"").">";
}

/*
*/
function url($name, $prompt, $length, $maxlength, $extensions, $view_button, $value, $onchange_event=NULL)
{
	global $LANG_VARS;
	global $CONFIG_VARS;
	
	// Default size.
	$size = $length;
	if($size >= 50 || $size<=0)
		$size = 50;

	// Workaround for Mozilla 0.9x problem with input field.	
	$size++;
	
	if($CONFIG_VARS['widgets.enable_javascript_validation']!==FALSE)
	{
		$onchange = "onchange=\"";
		if(strlen(trim($extensions))>0)
		{	
			$extensions_r = prc_args($extensions);
			$url_is_not_valid_message = replace_lang_vars(array('prompt'=>$prompt,'extensions'=>$extensions), $LANG_VARS['url_is_not_valid']);
			$onchange .= "if(!isValidExtension(this.value, ".get_javascript_array($extensions_r).")){alert('".$url_is_not_valid_message."'); this.focus(); return false;}";
		}
		$onchange .= " $onchange_event return true;\"";
	}
	else
		$onchange = "onchange=\"$onchange_event\"";
		
	$field = "\n<input type=\"text\" name=\"$name\" value=\"$value\" $onchange size=\"".$size."\" ".(is_numeric($maxlength)?"maxlength=\"".$maxlength."\"":"").">";
	
	// Do not show 'View' button, if explicitly set to 'N'
	if($view_button!='N')
		$field .= "<input type=button onclick=\"if(this.form['$name'].value.length>0){popup(this.form['$name'].value,'400','300');}else{alert('".replace_lang_var("prompt", $prompt, $LANG_VARS['prompt_must_be_specified'])."');}\" value=\"".$LANG_VARS['view']."\">";

	return $field;
}

/**
*/
function upload($name, $prompt, $length, $maxlength, $extensions)
{
	global $LANG_VARS;
	global $CONFIG_VARS;

	// Set to true to indicate we are using a type="FILE" input field.
	global $_OPENDB_FILE_UPLOAD_WIDGET;
	$_OPENDB_FILE_UPLOAD_WIDGET=TRUE;
	
	// Default size.
	$size = $length;
	if($size >= 50 || $size<=0)
		$size = 50;

	// Workaround for Mozilla 0.9x problem with input field.	
	$size++;

	if(strlen(trim($extensions))>0 && $CONFIG_VARS['widgets.enable_javascript_validation']!==FALSE)
	{
		$extensions_r = prc_args($extensions);
		$onchange = "onchange=\"if(!isValidExtension(this.value, ".get_javascript_array($extensions_r).")){alert('".replace_lang_vars(array('prompt'=>$prompt,'extensions'=>$extensions), addslashes($LANG_VARS['filename_is_not_valid']))."'); this.focus(); return false;} return true;\"";
	}

	$field = "<input type=\"FILE\" name=\"$name\" $onchange size=\"".$size."\">";

	return $field;
}

/**
	This is how it is going to work:

		The url field will be displayed as normal text field.  A checkbox will be
		included above it, with a comment of 'Save locally', or something similiar.

		If the file has already been saved, then the location of the local file
		will be stored in a hidden field only, where the '[View]' operation can
		find it.

		This allows a user to overwrite the local image with a external URL, which
		they have the option of saving locally by checking the checkbox.

		The checkbox will be ignored if the text field is empty, or the same value
		as the hidden field (when it is local)
*/
function saveurl($name, $prompt, $length, $maxlength, $extensions, $view_button, $saveurl_checked, $value, $onchange_event=NULL)
{
	global $LANG_VARS;
	global $CONFIG_VARS;
	
	// Default size.
	$size = $length;
	if($size >= 50 || $size<=0)
		$size = 50;

	// Workaround for Mozilla 0.9x problem with input field.	
	$size++;
		
	if(is_url_absolute($value))
	{
		// File is not saved locally.
		$url = $value;
		$local_url = "";
	}
	else if (file_exists($value))
	{
		// File is local.
		$url = $value;
		$local_url = $url;
	}
	else
	{
		// File is not external url and no longer exists
		// $url = "";
		$url = $value;
		$local_url = "";
	}

	if($CONFIG_VARS['widgets.enable_javascript_validation']!==FALSE)
	{
		if(strlen(trim($extensions))>0)
		{
			$extensions_r = prc_args($extensions);
			$url_is_not_valid_message = replace_lang_vars(array('prompt'=>$prompt,'extensions'=>$extensions), addslashes($LANG_VARS['url_is_not_valid']));
			$onchange = "onchange=\"if(!isValidExtension(this.value, ".get_javascript_array($extensions_r).")){alert('".$url_is_not_valid_message."'); this.focus(); return false;} $onchange_event return true;\"";
		}
	
		// If we already have a saved image locally, we can check before the user tries to resave it.  Only works if they
		// try to save the EXACT local image filename. But other checking will pick it up anyway.
		if(strlen($local_url)>0)
			$onclick = "onclick=\"if(this.checked && this.form['".$name."_local_url'].value == this.form['$name'].value){alert('".replace_lang_var("prompt", $prompt, $LANG_VARS['url_already_saved_local'])."'); this.checked = false; this.form['$name'].focus(); return false;}else{return true;}\"";
		else
			$onclick = "onclick=\"if(this.checked && isempty(this.form['$name'].value)){alert('".replace_lang_var("prompt", $prompt, $LANG_VARS['prompt_must_be_specified'])."'); this.checked = false; this.form['$name'].focus(); return false;}else{return true;}\"";
	}
	else
		$onchange = "onchange=\"$onchange_event\"";

	// Whether to check saveurl box by default	
	if($saveurl_checked == 'Y' && is_url_absolute($value))
		$saveurl_checked = 'CHECKED';
	else
		$saveurl_checked = '';
	
	$field = "\n<input type=hidden name=\"".$name."_local_url\" value=\"$local_url\">".
				"\n<table border=0 cellspacing=0 cellpadding=0>".
				"\n\t<tr><td class=\"data\" colspan=2>".replace_lang_var("prompt", $prompt, $LANG_VARS['saveurl_prompt']).":".
					"\n\t<input type=checkbox name=\"".$name."_saveurl\" value=\"y\" $onclick $saveurl_checked>".$LANG_VARS['save']."</td></tr>".
				"\n\t<tr><td class=\"data\"><input type=\"text\" name=\"$name\" value=\"$url\" $onchange size=\"".$size."\" ".(is_numeric($maxlength)?"maxlength=\"".$maxlength."\"":"")."></td><td>";
				
	// Do not show 'View' button, if explicitly set to 'N'
	if($view_button!='N')
	{
		$field .= "<input type=button onclick=\"if(this.form['$name'].value.length>0){popup(this.form['$name'].value,'400','300');}else{alert('".replace_lang_var("prompt", $prompt, $LANG_VARS['prompt_must_be_specified'])."');}\" value=\"".$LANG_VARS['view']."\">";
	}
	$field .= "\n</td></tr></table>";

	return $field;
}

/**
*/
function upload_or_saveurl($name, $prompt, $length, $maxlength, $extensions, $view_button, $saveurl_checked, $value)
{
	global $LANG_VARS;

	return  "\n<table border=0 cellspacing=0 cellpadding=0>".
				"\n\t<tr><td class=\"data\">".replace_lang_var("prompt", $prompt, $LANG_VARS['upload_prompt']).":</td></tr>".
				"\n\t<tr><td class=\"data\">".
				upload($name."_upload", $prompt, $length, $maxlength, $extensions).
				 "</td></tr>".
				"\n</table>".
			saveurl($name, $prompt, $length, $maxlength, $extensions, $view_button, $saveurl_checked, $value);
}

/**
* @param $lookup_rs - array of values
*/
function value_radio_grid($name, $lookup_rs, $columns, $border, $value)
{
	$count=0;
	$grid_row=0;
	$field = "";

	$is_checked = FALSE;
	while(list(,$val) = each($lookup_rs))
	{	
		if($count==0)
		{
			$field .= "<tr>";
			$grid_row++;
		}
		
		if((strlen($value)>0 && strcasecmp(trim($value), $val)===0) || (strlen($value)==0 && !$is_checked))
		{
			$field .= format_data(NULL, "\n<input type=\"radio\" name=\"$name\" value=\"$val\" CHECKED>$val&nbsp;&nbsp;");
			$is_checked=TRUE;
		}
		else
			$field .= format_data(NULL, "\n<input type=\"radio\" name=\"$name\" value=\"$val\">$val&nbsp;&nbsp;");
		
    	$count++;
		
		if(is_numeric($columns) && $count == $columns)
		{
			$count=0;
			$field .= "</tr>";
		}
	}
	
	// Now close current row if required.
	if(is_numeric($columns) && $count!=0 && $grid_row>1)
	{
		for ($i=0; $i<($columns-$count); $i++)
		{
			$field.= format_data(NULL, "&nbsp;");
		}
		$field .= "</tr>";
	}
	
	// Now return complete table.
	return "<table cellpadding=1 cellspacing=0 border=\"".(is_numeric($border)?$border:0)."\">".
			$field.
			"</table>";
}

/**
	Special function, only used by item_review script.
*/
function review_options($name, $lookup_results, $mask, $orientation, $value)
{
	if(isset($orientation))
		$orientation=trim(strtolower($orientation));
	else
		$orientation="horizontal";

	$total_count = 0;
	$is_first_value=TRUE;
	$value_found=FALSE;

	$value = trim($value);
	
	$var = "<tr>";
	while($lookup_r = mysql_fetch_array($lookup_results, MYSQL_ASSOC))
	{
		if($is_first_value === TRUE)
		{
			$is_first_value = FALSE;
			$total_count = (int)$lookup_r['value'];
		}

		$field = "";
		if($value===NULL && $lookup_r['checked_ind']=='Y')
			$field .= "\n<input type=\"radio\" name=\"$name\" value=\"".$lookup_r['value']."\" CHECKED>";
		else
		{
			// Case insensitive!
			if($value!==NULL && strcasecmp($value, $lookup_r['value'])===0)
			{
				$value_found=TRUE;
				$field .= "\n<input type=\"radio\" name=\"$name\" value=\"".$lookup_r['value']."\" CHECKED>";
			}
			else
			{
				$field .= "\n<input type=\"radio\" name=\"$name\" value=\"".$lookup_r['value']."\">";
			}
		}

		// Now display the images.
		for ($i=0; $i<(int)$lookup_r['value']; $i++)
			$field .= _theme_image("rs.gif");
		for ($i=0; $i<(int)($total_count-(int)$lookup_r['value']); $i++)
			$field .= _theme_image("gs.gif");

		// now the display value.
		$field .= "&nbsp;".format_display_value($mask, $lookup_r['img'], $lookup_r['value'], $lookup_r['display']);
		$var .= format_data(NULL, $field);

        if ($orientation == "vertical")
			$var .= "</tr><tr>\n";
	}
	mysql_free_result($lookup_results);
	
	$var .= "</tr>";
	
	// Now return complete table.
	return "<table cellpadding=1 border=0 cellspacing=0>$var</table>";
}

/*
	Will format a complete table grid with the number of specified columns.  Will fill out the
	last columns in the last row with &nbsp;
 
	@param $columns 1 for VERTICAL, * for HORIZONTAL one row, otherwise a numeric column value
	 		will build a table.

	@param $value - will not be array.
*/
function radio_grid($name, $lookup_results, $mask, $columns, $border, $value)
{
	$count=0;
	$grid_row=0;
	$value_found=FALSE;
	
	// sanity check
	if(is_array($value))// convert single element array, to string
		$value = $value[0];
	
	while($lookup_r = mysql_fetch_array($lookup_results, MYSQL_ASSOC))
	{
		if($count==0)
		{
			$var .= "<tr>";
			$grid_row++;
		}
		
		$field = "";
		if($value === NULL && $lookup_r['checked_ind'] == 'Y')
		{
			$field .= "\n<input type=\"radio\" name=\"$name\" value=\"".$lookup_r['value']."\" CHECKED>";
		}
		else
		{
			// Case insensitive!
			if( $value !== NULL && strcasecmp($value, $lookup_r['value'])===0)
			{
				$value_found=TRUE;
				$field .= "\n<input type=\"radio\" name=\"$name\" value=\"".$lookup_r['value']."\" CHECKED>";
			}
			else
			{
				$field .= "\n<input type=\"radio\" name=\"$name\" value=\"".$lookup_r['value']."\">";
			}
		}
		
		// now the display value.
		$field .= format_display_value($mask, $lookup_r['img'], $lookup_r['value'], $lookup_r['display'])."&nbsp;&nbsp;\n";
		$var .= format_data(NULL, $field);
    	$count++;
		
		if(is_numeric($columns) && $count == $columns)
		{
			$count=0;
			$var .="</tr>";
		}
	}
	
	// Add the value to the list of options and select it.
	if(!$value_found && $value !== NULL)
	{
		if($count==0)
			$var .= "<tr>";
		$var .= format_data(NULL, "<input type=\"radio\" name=\"$name\" value=\"".$value."\" CHECKED>".$value);
		$count++;
	}
	
	// Now close current row if required.
	if(is_numeric($columns) && $count!=0 && $grid_row>1)
	{
		for ($i=0; $i<($columns-$count); $i++)
		{
			$var.= format_data(NULL, "&nbsp;");
		}
		$var.="</tr>";
	}
	mysql_free_result($lookup_results);
	
	// Now return complete table.
	return "<table cellpadding=1 cellspacing=0 border=\"".(is_numeric($border)?$border:0)."\">$var</table>";
}

/*
	Will format a complete table grid with the number of specified columns.  Will fill out the
	last columns in the last row with &nbsp;
	
	@param $columns 1 for VERTICAL, * for HORIZONTAL one row, otherwise a numeric column value
	 		will build a table.
	
	@param $value - array of values.
*/
function checkbox_grid($name, $lookup_results, $mask, $columns, $border, $value)
{
	$count=0;
	$grid_row=0;
	
	// sanity check
	if(is_array($value) && count($value)>0)
		$values_r = $value;
	else if(!is_array($value) && $value !== NULL)// if a single string value, convert to single element array.
		$values_r[] = $value;
	else // is_empty_array!
		$values_r = NULL;

	while($lookup_r = mysql_fetch_array($lookup_results, MYSQL_ASSOC))
	{
		if($count==0)
		{
			$var .= "<tr>";
			$grid_row++;
		}
		
		$field = "";
		if(!is_array($values_r) && $lookup_r['checked_ind'] == 'Y')
			$field .= "\n<input type=\"checkbox\" name=\"".$name."[]\" value=\"".$lookup_r['value']."\" CHECKED>";
		else
		{
			// Case insensitive!
			if(is_array($values_r) && ($lookup_key = array_search2($lookup_r['value'], $values_r, TRUE)) !== FALSE)
			{
				// Remove the matched element
				array_splice($values_r, $lookup_key, 1);

				$field .= "\n<input type=\"checkbox\" name=\"".$name."[]\" value=\"".$lookup_r['value']."\" CHECKED>";
			}
			else
			{
				$field .= "\n<input type=\"checkbox\" name=\"".$name."[]\" value=\"".$lookup_r['value']."\">";
			}
		}
		
		// now the display value.
		$field .= format_display_value($mask, $lookup_r['img'], $lookup_r['value'], $lookup_r['display'])."&nbsp;&nbsp;\n";
		$var .= format_data(NULL, $field);
    	$count++;
		
		if(is_numeric($columns) && $count == $columns)
		{
			$count=0;
			$var .="</tr>";
		}
	}

	if(is_array($values_r))
	{
		// Add the value to the list of options and select it.
		reset($values_r);
		while(list(,$val) = each($values_r))
		{
			if(strlen($val)>0)
			{
				if($count==0)
					$var .= "<tr>";
				$var .= format_data(NULL, "<input type=\"checkbox\" name=\"".$name."[]\" value=\"$val\" CHECKED>$val");
				$count++;
			}
		}
	}
	
	// Now close current row if required.
	if(is_numeric($columns) && $count!=0 && $grid_row>1)
	{
		for ($i=0; $i<($columns-$count); $i++)
		{
			$var.= format_data(NULL, "&nbsp;");
		}
		$var.="</tr>";
	}
	mysql_free_result($lookup_results);
	
	// Now return complete table.
	return "<table cellpadding=1 cellspacing=0 border=\"".(is_numeric($border)?$border:0)."\">$var</table>";
}

/**
	$length restricts the value part to a specified length.  If $length===FALSE or 0 (zero) then no restriction is placed,
	damn those who pass $length as a negative number, because you get what you deserve.
*/
function single_select($name, $lookup_results, $mask, $length, $value, $onchange_event=NULL)
{
	global $CONFIG_VARS;

	if($CONFIG_VARS['widgets.enable_javascript_validation']!==FALSE)
		$onchange = "onchange=\"$onchange_event\"";
	else
		$onchange = "onchange=\"$onchange_event\"";

	// sanity check
	if(is_array($value))// convert single element array, to string
		$value = $value[0];

	$var = "\n<select name=\"$name\" $onchange>";
			
	$value_found=FALSE;
	while($lookup_r = mysql_fetch_array($lookup_results, MYSQL_ASSOC))
	{
        // Now get the display value.
		$display = format_display_value($mask, NULL, $lookup_r['value'], $lookup_r['display']);
        
		// Ensure any length restriction is enforced.
		if($length>0 && strlen($display)>$length)
		{
			$display = substr($display, 0, $length);
		}

		if($value===NULL && $lookup_r['checked_ind']=='Y')
		{
			$var .= "\n<option value=\"".$lookup_r['value']."\" SELECTED>$display";
		}
		else
		{
			// Case insensitive!
			if($value!==NULL && strcasecmp($value,$lookup_r['value'])===0 )
			{
				$value_found=TRUE;
				$var .= "\n<option value=\"".$lookup_r['value']."\" SELECTED>$display";
			}
			else
			{
				$var .= "\n<option value=\"".$lookup_r['value']."\">$display";
			}
		}
	}
	mysql_free_result($lookup_results);
	
	// Add the value to the list of options and select it.
	if(!$value_found && $value !== NULL)
	{
		$var .= "\n<option value=\"".$value."\" SELECTED>".$value;
	}
	
	$var.="\n</select>";
	return $var;
}

/**
	Will generate a SELECT option that allows for multiple selection, the values will be passed back to the server
	as an array matching the $name.
*/
function multi_select($name, $lookup_results, $mask, $length, $size, $value, $onchange_event=NULL)
{
	global $CONFIG_VARS;
	
	if($CONFIG_VARS['widgets.enable_javascript_validation']!==FALSE)
		$onchange = "onchange=\"$onchange_event\"";
	else
		$onchange = "onchange=\"$onchange_event\"";
	
	if(is_numeric($size) && $size>1)
		$var="\n<select multiple name=\"".$name."[]\" size=\"$size\" $onchange>";
	else
		$var = "\n<select name=\"$name\" $onchange>";

	// sanity check
	if(is_array($value) && count($value)>0)
		$values_r = $value;
	else if(!is_array($value) && $value !== NULL)// if a single string value, convert to single element array.
		$values_r[] = $value;
	else // is_empty_array!
		$values_r = NULL;
		
	while($lookup_r = mysql_fetch_array($lookup_results, MYSQL_ASSOC))
	{
        // Now get the display value.
		$display = format_display_value($mask, NULL, $lookup_r['value'], $lookup_r['display']);
		
		// Ensure any length restriction is enforced.
		if($length>0 && strlen($display)>$length)
		{
			$display = substr($display, 0, $length);
		}
		
		if($value_found!==TRUE && is_empty_or_not_array($values_r) && $lookup_r['checked_ind']=='Y')
		{
			$var .= "\n<option value=\"".$lookup_r['value']."\" SELECTED>$display";
		}
		else
		{
			// Case insensitive!
			if(is_not_empty_array($values_r) && ($lookup_key = array_search2($lookup_r['value'], $values_r, TRUE)) !== FALSE)
			{
				// Remove the matched element
				array_splice($values_r, $lookup_key, 1);

				$var .= "\n<option value=\"".$lookup_r['value']."\" SELECTED>$display";
			}
			else
			{
				$var .= "\n<option value=\"".$lookup_r['value']."\">$display";
			}
		}
	}
	mysql_free_result($lookup_results);
	
	if(is_not_empty_array($values_r))
	{
		// Add the value to the list of options and select it.
		reset($values_r);
		while(list(,$value) = each($values_r))
		{
			if(strlen($value)>0)
			{
				$var .= "\n<option value=\"$value\" SELECTED>$value";
			}
		}
	}
	
	$var.="\n</select>";
	return $var;
}

/**
*/
function value_select($name, $values_r, $size, $value, $onchange_event=NULL)
{
	global $CONFIG_VARS;
	
	if($CONFIG_VARS['widgets.enable_javascript_validation']!==FALSE)
		$onchange = "onchange=\"$onchange_event\"";
	else
		$onchange = "onchange=\"$onchange_event\"";
			
	if(is_numeric($size) && $size>1)
		$var="\n<select multiple name=\"".$name."[]\" size=\"$size\" $onchange>";
	else
		$var = "\n<select name=\"$name\" $onchange>";

	while(list(,$val) = each($values_r))
	{
		if($value!==NULL && strcasecmp(trim($value), $val)===0)
			$var .= "\n<option value=\"$val\" SELECTED>$val";
		else
			$var .= "\n<option value=\"$val\">$val";
	}

	$var .= "\n</select>";

	return $var;
}

/**
*/
function next_array_record(&$results)
{
	if(is_not_empty_array($results))
	{
		//ignore any error.
		return @each($results);
	}
	else
	{
		//ignore any error.
		return @mysql_fetch_array($results, MYSQL_ASSOC);
	}
}

/*
* Work out where in the $lookup_r the value for the $value_column
* is actually located.
* 
* NO SUPPORT FOR NUMERIC COLUMNS.  WILL NAVIGATE ARRAYS RETURNED
* FROM each() 
*/
function get_array_variable_value($lookup_r, $value_column)
{
	// Work out what to return, based on value_column specifier.
	if($value_column == 'key')
		return $lookup_r['key'];
	else if($value_column == 'valkey')// key is actual value, but not if numeric.
	{
		// Use value, if 'key' column is auto generated numeric index.
		if(!is_array($lookup_r['value']) && is_numeric($lookup_r['key']))
			return $lookup_r['value'];
		else
			return $lookup_r['key'];
	}
	else if(!is_array($lookup_r['value']) && $value_column == 'value')
		return $lookup_r['value'];
	else if(is_array($lookup_r['value']) && isset($lookup_r['value'][$value_column]))
		return $lookup_r['value'][$value_column];
	else if(isset($lookup_r[$value_column]))
		return $lookup_r[$value_column];
	else
		return $value;
}

/**
	This is a simple mask processor (Used by widgets.php::custom_select(...)).  It 
	is not as advanced as the parse_title_mask functionality, because it does not 
	support mask functions (if, ifdef, elsedef), or the special mask options '.img', etc.

	@param $display_mask	The display mask with variables delimited by $variable_char.
							The variable_name must exist as a keyname in $values_r.
	@param $values_r
	@param $variable_char
*/
function expand_display_mask($display_mask, $values_r, $variable_char="%")
{
	$i = 0;
	$inside_variable = FALSE;
	$variable="";
	$value = $display_mask;

	for ($i=0; $i<strlen($display_mask); $i++)
	{
		if($inside_variable)
		{
			// If closing bracket
			if($display_mask[$i] == $variable_char && ($i==0 || $display_mask[$i-1]!= '\\'))
			{
				// Indicate close of reference.
				$inside_variable = FALSE;

				if(strlen($variable)>0)
				{
					$replace = get_array_variable_value($values_r, $variable);
					$value = str_replace($variable_char.$variable.$variable_char, $replace, $value);
					$variable="";
				}
			}
			else
				$variable .= $display_mask[$i];
		}
		else if ($display_mask[$i] == $variable_char && ($i==0 || $display_mask[$i-1]!= '\\'))
		{
			$inside_variable = TRUE;	
		}
	}

	return $value;
}

/**
	@param $name
	@param $lookup_results	This can be a MySQL $results reference or an PHP associative
							array.  Even if you index your array numerically, the 'value'
							and index still get set inside this function.
	@param $display_mask
			This mask can be anything.  Any %??????% will be interpreted as a database
			column name and will be selected from there as appropriate.
	@param $lookup_results	Database results mysql reference
	@param $size			Size of SELECT object.  Specify a size of 'NA' to stop the 
	 						<select ...> and </select> tags being generated.  This will allow
	 						things like empty options to be included, while still taking
	 						advantage of the generation of database/array options.
	@param $value			Checked Ind value.  If the value param is found in the current
							'value' column, that record will be SELECTED.
	@param $value_column	Specifies the value column, or 'value' as a default.
	@param $checked_ind 	Specifies the checked_ind column to select a default record, or
							'' as default, which will mean the first record will be selected
							by the browser when building the generated select.
	@param $include_ind_func
							If defined, will call the function with the current lookup array
 						    as argument.  If the function returns TRUE, the record will be
							included, otherwise it will be skipped.
	
	However the $lookup_results DO have to include a 'value' column for this
	process to work, as this will be used for the value of each select option.

	You can however set the $value_column to a value indicating a column name to
	be used instead of 'value'
*/
function custom_select($name, $lookup_results, $display_mask, $size=1, $value=NULL, $value_column='value', $include_ind_func=NULL, $checked_ind='', $onchange_event='')
{
	global $CONFIG_VARS;
	
	if($size !== 'NA')
	{
		if(is_numeric($size) && $size>1)
			$var = "\n<select name=\"".$name."[]\" multiple size=\"$size\" onchange=\"$onchange_event\">";
		else
			$var = "\n<select name=\"$name\" onchange=\"$onchange_event\">";
	}
	else
		$var = '';
	
	// Must reset, otherwise 'each' will fail.
	if(is_array($lookup_results))
		reset($lookup_results);

	$value_found=FALSE;
	while($lookup_r = next_array_record($lookup_results))
	{
		// Check if this record should be included in list of values.
		if(!function_exists($include_ind_func) || $include_ind_func($lookup_r))
		{
			$lookup_value = get_array_variable_value($lookup_r, $value_column);
			
			// Now get the display value.
			$display = expand_display_mask($display_mask, $lookup_r, "%");

			if(!$value_found && $value==NULL && $lookup_r[$checked_ind]=="Y")
				$var .= "\n<option value=\"".$lookup_value."\" SELECTED>$display";
			else
			{
				// Case insensitive!
				if(strcasecmp(trim($value), $lookup_value) === 0)
				{
					$value_found=TRUE;
					$var .= "\n<option value=\"".$lookup_value."\" SELECTED>$display";
				}
				else
				{
					$var .= "\n<option value=\"".$lookup_value."\">$display";
				}
			}
		}
	}

	if($lookup_results && !is_array($lookup_results))
		mysql_free_result($lookup_results);
	
	if($size !== 'NA')
	{
		$var.="\n</select>";
	}
	return $var;	
}

function status_type_input_field($fieldname, $lookup_results, $value=NULL)
{
	$field = "<table border=0 cellpadding=1 cellspacing=0>\n<tr>";
			
	$is_checked=FALSE;
	while($lookup_r = mysql_fetch_array($lookup_results, MYSQL_ASSOC))
	{
		if(!$is_checked && ((strlen($value)==0 && $lookup_r['default_ind'] == 'Y') || $lookup_r['value'] == $value))
		{
			$lookup_r['checked_ind'] = 'Y';
			$is_checked=TRUE;
		}	
		$lookup_rs[] = $lookup_r;
	}
	mysql_free_result($lookup_results);
			
	// Otherwise enforce the first value as checked.
	if(!$is_checked)
	{
		$lookup_rs[0]['checked_ind'] = 'Y';
	}
			
	$no_columns = 4;
	$columns = 0;
	while(list(,$lookup_r) = each($lookup_rs))
	{
		if($columns>=$no_columns)
		{
			$columns=0;
			$field .= "</tr>\n<tr>";
		}
			
		$field .= "\n<td><input type=radio name=\"$fieldname\" value=\"".$lookup_r['value']."\" ".($lookup_r['checked_ind'] == 'Y'?"CHECKED":"").">".
				format_display_value('%img%', $lookup_r['img'], $lookup_r['value'], $lookup_r['display']).
				"\n</td>";
										
		$columns++;
	}						

	// Now fill remaining columns with &nbsp!!!
	for(; $columns<$no_columns;$columns++)
		$field .= "<td>&nbsp;</td>";
											
	$field .= "</tr></table>";
									
	return $field;
}

/**
	Used to format the display of the input/display field.  You specify the %field%
	which represents the actual $field parameter, and then it will be positioned
	within context of the rest of the $display_mask.  It all gets placed into a
	TD table cell.
*/
function format_data($field_mask, $field, $align = NULL)
{
	// If $display_mask is defined, then format $field to include it.
	if(strlen($field_mask)>0 && strpos($field_mask,"%field%")!==FALSE)
	{
		$field = str_replace("%field%", $field, $field_mask);
	}
	
	//Ensure a valid table value.
	if(strlen($field)==0)
		$field = "&nbsp;";

	return "<td class=\"data\" ".(strlen($align)>0?"align=\"$align\"":"").">$field</td>";
}

/**
*/
function format_prompt($prompt, $prompt_mask = NULL)
{
	// If $prompt_mask is defined, then format $prompt to include it.
	if(strlen($prompt_mask)>0 && strpos($prompt_mask,"%prompt%")!==FALSE)
	{
		$prompt = str_replace("%prompt%", $prompt, $prompt_mask);
	}

	return "<td nowrap class=\"prompt\" align=right>$prompt:&nbsp;</td>";
}

/**
	Formats field within the $display_mask specified.
*/
function format_field($prompt, $field_mask, $field, $dowrap=TRUE, $prompt_mask=NULL)
{
	// stub for old functionality.
	return format_field2($prompt, $field_mask, $field, NULL, $dowrap, $prompt_mask);
}

/*
* A second version of the format_field function to use in get_display_field, but we
* avoid the impact of changing every call to format_field.
*/
function format_field2($prompt, $field_mask, $field, $align = NULL, $dowrap=TRUE, $prompt_mask=NULL)
{
	if($dowrap)
		return "\n<tr>".format_prompt($prompt,$prompt_mask).format_data($field_mask, $field, $align)."</tr>";
	else
		return $field;
}

function format_input_field($prompt, $field_mask, $field, $dowrap=TRUE, $prompt_mask=NULL, $compulsory_ind = 'N')
{
	global $CONFIG_VARS;
	
	if($dowrap)
	{
		// If $prompt_mask is defined, then format $prompt to include it.
		if(strlen($prompt_mask)>0 && strpos($prompt_mask,"%prompt%")!==FALSE)
		{
			$prompt = str_replace("%prompt%", $prompt, $prompt_mask);
		}
		
		return "<tr><td nowrap class=\"prompt\" align=right>".$prompt.($CONFIG_VARS['widgets.show_prompt_compulsory_ind']!==FALSE && $compulsory_ind=='Y'?_theme_image("compulsory.gif", NULL, $LANG_VARS['compulsory_field'], 'top'):"").":&nbsp;</td>".
				format_data($field_mask, $field, $align)."</tr>";
	}
	else
	{
		return $field;
	}
}

/**
	This script will contain all the functions associated with the item attributes
	This will include accessors for the s_item_type, s_attribute_type, s_attribute_type_lookup
	and s_item_attribute_type tables.
	* 
	This function will also support specifying a $order_no of FALSE, so we can get a fieldname
	without the _$order_no
*/
function get_field_name($s_attribute_type, $order_no=NULL)
{
	if(is_numeric($order_no))
		return strtolower($s_attribute_type)."_".$order_no;
	else
		return strtolower($s_attribute_type);
}

/**
	Given a fieldname, it will return an associative array containing
	type=>
	order_no=>
*/
function get_attribute_and_order_no($fieldname)
{
	// We have to look for the last "_", and check if the value
    // after this is a number.  For now we will not bother checking that it
    // is a valid order_no, but just that it is a number.
    $idx = strrpos($fieldname, "_");
    if($idx !== FALSE)
    {
    	$type = substr($fieldname, 0, $idx);
        $order_no = substr($fieldname, $idx+1, strlen($fieldname)-$idx);
	    if(!is_numeric($order_no))
        	$order_no = null;
    }
    else
    	$type = $fieldname;
    
    // Return array now.
    return array("type"=>$type, "order_no"=>$order_no);
}

/**
	Will split the $value into separate array elements for each line encountered.  As of
	0.50-dev7 empty lines within the text of the $value will be added to the array as 
	well.  (Because this function is called from get_display_field, lines which are on the
	start or end of the $value will be trimmed as before).

	A line is considered to be terminated by any one of a line feed ('\n'), a carriage
	return ('\r'), or a carriage return followed immediately by a linefeed.
*/
function explode_lines($value)
{
	$count = 0;
	$star = 0;
	while($count < strlen($value))
	{
		if($value[$count] == "\r" || $value[$count] == "\n")
		{
			if($value[$count] == "\r" && ($count+1)<strlen($value) && $value[$count+1] == "\n")
				$count++;//Skip dos extra end-of-line character.

			$line = trim(substr($value, $start, $count-$start));
			
			//Even if an empty line, still count it!
			$lines[] = $line;
				
			$start = $count;
		}
		$count++;
	}
	
	// Get last line.
	if($count > $start)
	{
		$line = trim(substr($value, $start, $count-$start));
		// But does not include the last line break if empty.
		if(strlen($line)>0)
			$lines[] = $line;
	}
			 
	return $lines;
}

/**
	Convert $args array into equivalent Javascript array statement.
*/
function get_javascript_array($args)
{
	$buf="";
	for ($i=0; $i<count($args); $i++)
	{
		if(strlen($buf)>0)
			$buf .= ", '".$args[$i]."'";
		else
			$buf = "'".$args[$i]."'";
	}
	
	return "new Array($buf)";			
}

/**
	Will expand any ?-? expressions into their actual
	range.  If you want to include '-' as an option escape
	it with \
*/	
function expand_chars_exp($exp)
{
	$retval="";
	$i=0;
	while($i<strlen($exp))
	{
		if(substr($exp, $i, 1) == '-' && $i>0 && substr($exp, $i-1, 1) != '\\')
		{
			$start = ord(substr($exp, $i-1, 1));
			$end = ord(substr($exp, ++$i, 1));
			if($start < $end)
			{
				for($j=($start+1); $j<=$end; $j++)
					$retval .= chr($j);
			}
			//else - invalid range,ignore
		}
		else if(substr($exp, $i, 1) == '\\')
		{
			// If this is escaping a character other than  '\'
			// then do not include.  The test will still look
			// at the original exp, for the '\', so getting rid
			// of it here will be alright!
			if($i>0 && substr($exp, $i-1, 1) == '\\')
				$retval .= '\\'; 
		}
		else
			$retval .= substr($exp, $i, 1);
		$i++;
	}
	
	return $retval;
}

function expand_range($left, $right)
{
	$retval = '';
	for($i=$left; $i<=$right; $i++)
	{
		if(strlen($retval)>0)
			$retval .= ',';
		
		$retval .= $i;
	}
	
	return $retval;
}

/**
* Specify a range of characters in the following format:
* 	1-15,10,1,12,423,312312,123-124.  If you specify
* a range, that is not valid, that portion will be ignored.
*/
function expand_number_range($range)
{
	$retval='';
	$i=0;

	$number = '';
	$left_number = '';
	$right_number = '';
	while($i<strlen($range))
	{
		if(is_numeric($range{$i}))
		{
			 if(is_numeric($left_number))
				$right_number .= $range{$i};
			else
				$number .= $range{$i};
		}
		else if($range{$i} == '-') // end of left range number
		{
			$left_number = $number;
			
			//reset
			$number = '';
		}
		else if($range{$i} == ',') // end of right range number, or lone number
		{
			if(is_numeric($left_number) && is_numeric($right_number))
			{
				$retval .= expand_range($left_number, $right_number);
				
				//reset
				$left_number = '';
				$right_number = '';
			}
			else
			{
				$retval .= $number;
				
				//reset
				$number = '';
			}
			
			$retval .= ',';
		}
		
		$i++;
	}
	
	if(is_numeric($left_number) && is_numeric($right_number))
	{
		$retval .= expand_range($left_number, $right_number);
	}
	else
	{
		$retval .= $number;
	}
	
	// get rid of last character, if a comma.
	if($retval{strlen($retval)-1} == ',')
		$retval = substr($retval, 0, strlen($retval)-1);
		
	return $retval;
}



/**
	$display_type can have the following values:
		list(list_type [,delimiter][, list-link])
						Will split text according to delimiter (or newline of not specified)
						and format based on list_type.  If the 'list-link' is specified,
						then will add a s_attribute_type listing link to each item.

								*** START DEPRECATED LIST ****
		**** The following options (split, nl2br,ordered_list,unordered_list,ticks_list) are
				deprecated.
		****

		split(delimiter [, list-link])
						Will split text up based on first argument 'delimiter'.  If the 
						'list-link' is specified, then will add a s_attribute_type 
						listing link to each item.

		nl2br			Will convert all newlines to <br>.  Will trim the string first
						so that any newlines on either end of the string are not converted.

		ordered_list	Will generate a numbered list of all lines, will trim the string
						first...

		unordered_list	Will generate a bulleted list of all lines, will trim the string
						first...
		
		ticks_list		Will generate a ticks (A image of a red tick) list of all lines, 
						will trim the string first...
		
								**** END DEPRECATED LIST *****

		urlpopup([theme_img][,width][,height]) 
						This widget, will display a URL value, with a popup link to display
						it in a new window.  The width & height arguments are optional, to
						control the dimensions of the window.  The window will be opened
						with 640x480 dimensions by default. 
		
		datetime(datetime_mask)
		 				Format the value entered via the 'datetime' input field according to the
		 				display mask, which supports the following mask elements:
							Month - Month name
					 		Mon - Abbreviated month, Initcap.
 					 		MON - Abreviated month UPPERCASE
			 				Day	- Weekday name
			 				DDth - Day of the month with English suffix (1st, 2nd, 3rd)
			 				DD - Days (01 - 31)
			 				MM - Months (01 -12)
			 				YYYY - Years
			 				HH24 - Hours (00 - 23)
			 				HH - Hours (01 - 12)
			 				MI - Minutes (00 - 59)
			 				SS - Seconds (00 - 59)
			 				AM/PM - Meridian indicator  (Will be replaced with the actual context meridian value!)		

		format_mins(display_mask)		
						Expects a int value of total minutes, and will format it according to 
						the specified mask.  The default mask is: "%h %H %m %M"
				
							%h - hour value only
						    %H - text "hour" or "hours"
						    %m - minute value only
						    %M - text "minute" or "minutes"
	
						In the case where an hour value is not available, because the total 
						minutes is less than 60, then everything in the mask before the %m
						or %M will be ignored.
		
		display(display_mask [,list-link])
		display(display_mask [,align][,list-link])
						This is a new function to allow the inclusion of a list-link for what would
						have originally been specified as a %display% or %value% type mask, without
						functional reference.  Now you can specify the display mask and an optional
						list-link argument.  This is the preferred means of specifying a display
						type for a field, even if the list-link argument is not used.

		category(display_mask [, list-link])
						Special 'CATEGORY' display_type.  Will split the category values according to 
						matches in s_attribute_type_lookup table and display as 'category / category' etc.

		review([display_mask])
						Display a single Review value, which will consist of a set of stars, and any
						s_attribute_type_lookup columns for the matching value.  This option is not
						available for use with s_attribute_type's
		 
		hidden			Allows users to disable display of fields that were, either previously automatically
						populated by a site plugin, or were originally display attributes that are now
						no longer required.

		For display_mask:
			Options are %img%, %value%, %display% variables.

		Options for list_type are:
				plain
				nl2br
				ordered
				unordered
				ticks

		Note:
		-----

			If you have any display attributes, that should only ever display %value%, ensure that the 
			s_attribute_type.display_type column is explicitly set to '%value%' as this will provide a performance
			improvement.

			The default display_type is %value% which will be processed with nl2br($value) to ensure that newlines
			are maintained in the attribute value displayed.
*/
function get_display_field($s_attribute_type, $prompt, $display_type, $value, $dowrap=TRUE, $prompt_mask=NULL)
{
	global $LANG_VARS;
	global $CONFIG_VARS;
	
	$widget = prc_function_spec($display_type);

	//
	// ############### Backwards compatibility for lists & split #################
	//
	if($widget['type'] == 'split')
	{
		// set the delimiter
		$widget['args']['1'] = $widget['args']['0'];
		$widget['args']['0'] = 'plain';
		$widget['type'] = 'list';
	}
	else if($widget['type'] == 'nl2br' || $widget['type'] == 'ordered_list' || $widget['type'] == 'unordered_list' || $widget['type'] == 'ticks_list')
	{
		// No delimiter - use newline.
		$widget['args']['1'] = '';

		// Is list-link specified?
		$widget['args']['2'] = $widget['args']['0'];

		// Get the list_type from the original name.
		$widget['args']['0'] = substr($widget['type'], 0,strpos($widget['type'], '_list'));
		$widget['type'] = 'list';
	}
	// ############################################################################

	if($widget['type'] == 'hidden' || $widget['type'] == 'popupsize') // popupsize: item_display reserved widget.
	{
		// Do nothing.
		return '';
	}
	else if($widget['type'] == 'urlpopup' || $widget['type'] == 'urlpopup2') 
	{
		$value = trim($value);
		
		// by default
		$text = $value;

		if(is_numeric($widget['args']['0']))
		{
			$width = ifempty($widget['args']['0'], '640');
			$height = ifempty($widget['args']['1'], '480');
		}
		else 
		{
			if(strlen($widget['args']['0'])>0)
			{
				$mask_elements = parse_field_mask($widget['args']['0']);
				// if mask code is present, to programatically work out the 
				// image to display based on the file extension, we must validate it
				if(is_array($mask_elements))
				{
					$values_rs['extension'] = get_file_ext($value);
					$src = expand_field_mask($values_rs, $widget['args']['0'], $mask_elements);
				}
				else
				{
					$src = $widget['args']['0'];
				}

				$image_src = _theme_image_src($src);
				if($image_src !== FALSE)
				{
					$text = '<img src="'.$image_src.'" title="'.$value.'" border=0>';
				}
			}
			$width = ifempty($widget['args']['1'], '640');
			$height = ifempty($widget['args']['2'], '480');
		}
		
		if($widget['type'] == 'urlpopup')
			return format_field($prompt, NULL, "<a href=\"javascript:popup('external.php?url=".urlencode($value)."&title=".urlencode($prompt)."','".($width+20)."', '".($height+25)."')\" title=\"$prompt\" class=\"popuplink\">$text</a>", $dowrap, $prompt_mask);
		else //if($widget['type'] == 'urlpopup2')
			return format_field($prompt, NULL, "<a href=\"javascript:popup('".urlencode($value)."','".($width+20)."', '".($height+25)."')\" title=\"$prompt\" class=\"popuplink\">$text</a>", $dowrap, $prompt_mask);
	}
	else if($widget['type'] == 'list')	//list(list_type [,delimiter][, list-link])
	{
		if(!is_array($value))
		{
			$value = trim($value);
		
			if(strlen($widget['args']['1'])==0) // Use newline!
			{
				$values = explode_lines($value);
				$attr_match = 'partial';
			}
			else
			{
				$values = explode($widget['args']['1'], $value);
	
				// If delimiter is a 'space', this indicates that
				// the attribute_type is most likely a multi-value
				// type, so we should do a word match.
				if(strlen(trim($widget['args']['1']))===0)
					$attr_match = 'word';
				else // otherwise do a partial, because we are trying to match part of a single value attribute (Such as ACTORS)
					$attr_match = 'partial';	// for the purpose of listing links alone.
			}
		}
		else
		{
			$values = $value;
			$attr_match = 'word';
		}
			
		switch($widget['args']['0'])
		{
			case 'nl2br':
			case 'ordered':
			case 'unordered':
			case 'ticks':
				return format_field($prompt, NULL, format_list_from_array($values, $widget['args']['0'], $prompt, $s_attribute_type, strcasecmp($widget['args']['2'],"list-link")===0?$attr_match:FALSE), $dowrap, $prompt_mask);
			case 'plain':
				return format_field($prompt, NULL, format_csv_line($values, $widget['args']['1'], $prompt, $s_attribute_type, strcasecmp($widget['args']['2'],"list-link")===0?$attr_match:FALSE), $dowrap, $prompt_mask);
			default:
				return format_field($prompt, NULL, format_list_from_array($values, 'nl2br', $prompt, $s_attribute_type, strcasecmp($widget['args']['2'],"list-link")===0?$attr_match:FALSE), $dowrap, $prompt_mask);
		}
	}
	else if($widget['type'] == 'datetime')
	{
		$value = trim($value);
		
		$timestamp = get_timestamp_for_datetime($value, 'YYYYMMDDHH24MISS');
		if($timestamp !== FALSE)
		{
			if(strlen($widget['args']['0'])==0)
				$widget['args']['0'] = 'DD/MM/YYYY';
			
			$datetime = get_localised_timestamp($widget['args']['0'], $timestamp);
			if($datetime!==FALSE)
				return format_field($prompt, NULL, $datetime, $dowrap, $prompt_mask);
			else
				return format_field($prompt, NULL, $value, $dowrap, $prompt_mask);
		}
		else
		{
			return format_field($prompt, NULL, $value, $dowrap, $prompt_mask);
		}
	}
	else if($widget['type'] == 'format_mins')
	{
		$time_value=trim($value);// time display
		if( is_numeric($time_value) )
		{
			// Ensure we have a mask to work with.
			$display_mask = $widget['args']['0'];
			if(strlen($display_mask)==0)
				$display_mask = '%h %H %m %M';

			$hrs = floor($time_value/60); // hours
			$mins = $time_value%60;	// minutes

			// Process display_mask and remove any bits that are not needed because the hour/minute is zero.
			if($mins == 0 && $hrs > 0) // only get rid of minutes if $hrs is a value.
			{
				$index = strpos($display_mask, '%H');
				if($index !== FALSE)
					$display_mask = substr($display_mask, 0, $index+2);//include the %H
				else
				{
					$index = strpos($display_mask, '%m');
					if($index!=FALSE)
						$display_mask = substr($display_mask, 0, $index);//include the %H
				}
			}
			else if($hrs == 0)
			{
				$index = strpos($display_mask, '%m');
				if($index!=FALSE)
					$display_mask = substr($display_mask, $index);//include the %H
			}

			// Unfortunately we need to do $mins>0 and $hrs>0 if's twice, because otherwise once we 
			// replace the %h and %H the test for $mins>0 would not be able to cut the display_mask,
			// based on the %h/%H...
			if($hrs>0)
			{			
				// Now do all replacements.
				$display_mask = str_replace('%h',$hrs,$display_mask);
				if($hrs!=1)
					$display_mask = str_replace('%H',$LANG_VARS['hours'],$display_mask);
				else
					$display_mask = str_replace('%H',$LANG_VARS['hour'],$display_mask);
			}
			
			if($mins>=0 || ($hrs===0 && $mins===0))
			{
				// Now do minute replacements only.
				$display_mask = str_replace('%m',$mins,$display_mask);
				if($mins!=1)
					$display_mask = str_replace('%M',$LANG_VARS['minutes'],$display_mask);
				else
					$display_mask = str_replace('%M',$LANG_VARS['minute'],$display_mask);
			}
			
			// Now return mask with parts of value inserted.
			return format_field($prompt, NULL, $display_mask, $dowrap, $prompt_mask);
		}
		else
		{
			// what else can we do here?!
			return format_field($prompt, NULL, $time_value, $dowrap, $prompt_mask);
		}			
	}
	else if($widget['type'] == 'review')
	{
		$value = trim($value);
		// no point unless numeric
		if(is_numeric($value))
		{
			$total_count = fetch_attribute_type_cnt('S_RATING');
			if(is_numeric($total_count))
			{
				$field = '';
				$j = $value;
				for($i=0;$i<$total_count;++$i)
				{
					if($j >= 0.75)
						$field .= _theme_image('rs.gif');
					else if ($j >=0.25)
						$field .= _theme_image('rgs.gif');
					else
						$field .= _theme_image('gs.gif');
					$j = $j - 1;
				}
			
				// If a mask is defined, format the display value.	
				if(strlen($widget['args']['0'])>0)
				{
					$lookup_r = fetch_attribute_type_lookup_r('S_RATING', $value);
					if(is_not_empty_array($lookup_r))
					{
						$field .= '&nbsp;'.
								format_display_value($widget['args']['0'], $lookup_r['img'], $lookup_r['value'], $lookup_r['display']);
					}
				}
				return $field; // this is only used in a few places.
			}
		}
		else
		{
			return ''; // nothing to do!
		}
	}
	else if($widget['type'] == 'star_rating') // arg[0] = rating range
	{
		$value = trim($value);
		
		// no point unless numeric
		if(is_numeric($value))
		{
			$total_count = $widget['args']['0'];
			if(is_numeric($total_count))
			{
				$field = '';
				$j = $value;
				for($i=0;$i<$total_count;++$i)
				{
					if($j >= 0.75)
						$field .= _theme_image('rs.gif');
					else if ($j >=0.25)
						$field .= _theme_image('rgs.gif');
					else
						$field .= _theme_image('gs.gif');
					$j = $j - 1;
				}
				
				$ratingmask = NULL;
				if(strcasecmp($widget['args']['1'], 'list-link')===0)
				{
					$listlink = TRUE;
					$ratingmask = $widget['args']['2'];
				}
				else if(strcasecmp($widget['args']['2'], 'list-link')===0)
				{
					$listlink = TRUE;
					$ratingmask = $widget['args']['1'];
				}
				else
				{
					$ratingmask = $widget['args']['1'];
				}
				
				if(strlen($ratingmask)>0)
				{
					$field = str_replaces(
								array('%value%', '%maxrange%', '%starrating%'), 
								array($value, $total_count, $field),
								$ratingmask);
				}
				
				if($listlink)
					$field = format_listing_link($value, $field, $prompt, $s_attribute_type, $attr_match);
				
				return format_field2($prompt, NULL, $field, $align, $dowrap, $prompt_mask);
			}
		}
		else
		{
			return ''; // nothing to do!
		}
	}
	else if(!is_array($value) && $widget['type'] == 'display' && $widget['args']['0'] == '%value%')
	{
		// Support newline formatting by default.
		$value = nl2br(trim($value));
		
		// display(mask, list-link) or display(mask, align, list-link)
		
		$align = NULL;
		$listlink = FALSE;
		
		if(count($widget['args'])>1)
		{
			if(strcasecmp($widget['args']['1'], 'list-link')===0)
			{
				$listlink = TRUE;
			}
			else // list-link is not second argument
			{
				if(strcasecmp($widget['args']['2'], 'list-link')===0)
					$listlink = TRUE;
		
				if(strcasecmp($widget['args']['1'], 'LEFT')===0)
					$align = 'left';
				else if(strcasecmp($widget['args']['1'], 'RIGHT')===0)
					$align = 'right';
				else if(strcasecmp($widget['args']['1'], 'CENTER')===0 || strcasecmp($widget['args']['1'], 'CENTRE')===0)
					$align = 'center';
				else
					$align = strtolower($widget['args']['1']);
			}
		}
		
		if($listlink)
			return format_field2($prompt, NULL, format_listing_links($value, $prompt, $s_attribute_type, 'word'), $align, $dowrap, $prompt_mask);
		else
			return format_field2($prompt, NULL, $value, $align, $dowrap, $prompt_mask);
	}
	else if($widget['type'] == 'category' || $widget['type'] == '%display%' || $widget['type'] == 'display')
	{
		$field = '';
		
		if(is_array($value))
		{
			$value_array = $value;
		}
		else if(strlen($value)>0) // should not be required, but will leave in anyway.
		{
			$value_array = explode(' ', $value);
		}
		
		$results = fetch_value_match_attribute_type_lookup_rs($s_attribute_type, $value_array, get_lookup_order_by($widget['args']['0']), 'asc');
		if($results)
		{
			while($lookup_r = mysql_fetch_array($results, MYSQL_ASSOC))
			{
				$lookup_key = array_search2($lookup_r['value'], $value_array, TRUE);
				if($lookup_key !== FALSE)
				{
					// Remove the matched element
					array_splice($value_array, $lookup_key, 1);
					
					// Replace any blank spaces with no breaking spaces in the case of genre.
					if($widget['type'] == 'category' && strlen($display)>0)
					{
						$lookup_r['display'] = str_replace(' ', '&nbsp;', $lookup_r['display']);
					}
					
					$field = format_lookup_display_field(
											$prompt, 
											$s_attribute_type, 
											$widget, 
											$field, 
											$lookup_r['value'], 
											$lookup_r['display'], 
											$lookup_r['img']);
				}
			}
			mysql_free_result($results);
		}
		
		if(is_not_empty_array($value_array))
		{
			reset($value_array);
			while(list(,$value) = each($value_array))
			{
				if(strlen(trim($value))>0) // In case there are extra spaces
				{
					$field = format_lookup_display_field(
									$prompt, 
									$s_attribute_type, 
									$widget, 
									$field, 
									$value,
									$value); // By default if no display lookup, then value is used.
				}
			}
		}
			
		// If we have never gone into while or inner if statement, then let default return at end of function
		if(strlen($field)>0)
		{
			// $var would be empty, if we had not been inside while and inner if!
			return format_field($prompt, NULL, $field, $dowrap, $prompt_mask);
		}
	}

   	//else -- no display type match.  -- Assumes $widget['type'] == "%value%"
	// Support multiline formatting.
    	return format_field($prompt, NULL, nl2br($value), $dowrap, $prompt_mask);
}

/*
* Used exclusively by get_display_field
*/
function format_lookup_display_field($prompt, $s_attribute_type, $widget, $field, $value, $display=NULL, $img=NULL)
{
	// Format the display value, based on the mask ($widget['args']['0'])
	$display_value = format_display_value($widget['args']['0'], $img, $value, $display);
	
	// Add listings.php link if required.
	if(strcasecmp($widget['args']['1'],'list-link')===0 || 
				strcasecmp($widget['args']['2'],'list-link')===0) //arg[2] if orientation argument provided.
	{
		$display_value = format_listing_link($value, $display_value, $prompt, $s_attribute_type, $widget['type']=='category'?'category':'word');
	}
				
	if(strlen($field) == 0)
	{
		$field = $display_value;
	}
	else
	{
		// Any display_type widgets that rely on the s_attribute_type_lookup display column
		// to format the field, are here.  Currently only category is supported.
		if($widget['type'] == 'category')
			$field .= ' / ' . $display_value;
		else // support orientation
		{
			if(strcasecmp($widget['args']['1'],'HORIZONTAL')===0)
				$field .= ' ' . $display_value;
			else
				$field .= '<br>' . $display_value;
		}
	}
	
	return $field;
}

/**
	Will return an array of links.

	@value Can be an array or a single value.
*/
function format_listing_links($value, $prompt, $s_attribute_type, $attr_match)
{
	if(is_array($value))
		$tokens = $value;
	else
		$tokens[] = $value;
		
	while(list(,$token) = @each($tokens))
	{
		$token = trim($token);
		$lines[] = format_listing_link($token, $token, $prompt, $s_attribute_type, $attr_match);
	}
	
	// If no array passed in, then pass back normal string!
	if(is_array($value))
		return $lines;
	else
		return $lines[0];
}

/**
	$attr_match
		word		A '= $value match' OR 'LIKE % $value% ' OR 'LIKE '%$value ' OR 'LIKE '% $value%'
		exact		A '= "$value match"'
		partial		A 'LIKE %$value%' match
		category	listings will handle this special type, by linking against item.category instead
		of the item_attribute.attribute_val...
*/
function format_listing_link($value, $display, $prompt, $s_attribute_type, $attr_match)
{
	global $LANG_VARS;

	// The % cannot exist in a database column, whereas the '_' can.  This is
	// why we only need to escape the _.  We escape it by specifying it twice!
	$value = trim(str_replace("_", "\\_", $value));
	
	// If any whitespace, then enclose with quotes, otherwise will be treated by boolean parser as 
	// separate words, which is not desirable.
	if(strpos($value, " ")!==FALSE)
		$value = urlencode("\"".$value."\"");
	else if($attr_match == 'exact')
		$value = urlencode("\"".$value."\"");
	else
		$value = urlencode($value);
	
	
	return "<a href=\"listings.php?attribute_list=y&attr_match=$attr_match&attribute_type=$s_attribute_type&s_status_type=ALL&linked_items=include&attribute_val=".$value."&order_by=title&sortorder=ASC\" title=\"".replace_lang_var("prompt", $prompt, $LANG_VARS['list_items_with_same_prompt'])."\" class=\"listlink\">$display</a>";
}

/**
	Reconstitute line, including list-link's if required.

	@param $attr_match
			Specified $attr_match type for list-link.  If FALSE, then no list-link
			to be added.
*/
function format_csv_line($tokens, $delimiter, $prompt=NULL, $s_attribute_type=NULL, $attr_match=FALSE)
{
	// Add listing links if required.
	if($attr_match!==FALSE)
		$tokens = format_listing_links($tokens, $prompt, $s_attribute_type, $attr_match);

	$value = "";
	while(list(, $token) = @each($tokens))
	{
		if(strlen($value)>0)
			$value .= "$delimiter ";

		$value .= $token;
	}
	return $value;
}

/**
	Will generate a formatted list of items, based on $value with newline
	delimiters.  For instance ticks_list(color)

	@param $value	If $value is not already an array, it will be converted to an array
					of one value.

	@param $attr_match
					Specified $attr_match type for list-link.  If FALSE, then no list-link
					to be added.
*/
function format_list_from_array($value, $list_type, $prompt, $s_attribute_type, $attr_match=FALSE)
{
	if(is_array($value))
		$lines = $value;
	else
		$lines[] = $value;

	// Add listing links if required.
	if($attr_match!==FALSE)
		$lines = format_listing_links($lines, $prompt, $s_attribute_type, $attr_match);

	// Must be more than single line to consider processing it.
	if(count($lines)>1)
	{	
		$order=0;
		$value = "";
		while(list(, $line) = each($lines))
		{
			if($list_type == "ordered")
			{
				if(strlen($line)>0)
				{
					$order++;
					$value .= "$order.";
					$value .= "&nbsp;".$line."<br>\n";
				}
				else
					$value .= "<br>";
			}
			else if($list_type == "ticks")
			{
				if(strlen($line)>0)
				{
					$value .= _theme_image("tick.gif", NULL, NULL, "absmiddle");
					$value .= "&nbsp;".$line."<br>\n";
				}
				else
					$value .= "<br>";
			}
			else if($list_type == "nl2br")
			{
				// We do not have to do anything for nl2br!
				$value .= $line."<br>\n";
			}
			else// if($list_type == "unordered"){ == default
			{
				if(strlen($line)>0)
				{
					$value .= _theme_image("dot.gif", NULL, NULL, "absmiddle");
					$value .= "&nbsp;".$line."<br>\n";
				}
				else
					$value .= "<br>";
			}
		}
		return $value;
	}
	else
		return $lines[0];//only one line
}

/**
	Based on the $display_type mask, will format the display text for 
    the combination of $img, $value and $display columns.
    
    This can also be used for input fields where a display component
    is required to be custom.  for fields such as single_select, do
    not pass the $img, and it will work as expected. 
    
    If $img === "none" then we will replace %img% with empty string.  This is
	in contrast to the default functionality of replacing the %img% with %value%
	if a particular record from s_attribute_type_lookup does not specify an
	img value.
	
	If a img value is specified, but it cannot be resolved to a theme_image,
	the display will be used.  This is ac

	NOTE:
		From 0.51-dev6 onwards.  The $alt parameter is no longer $value in
		call to _theme_image.
		
	From 0.55 onwards, if $theme_image_type defined, the $alt parameter will
	be passed in, $display will be used for $alt in this case.
*/
function format_display_value($mask, $img, $value, $display, $theme_image_type=NULL)
{
   	// The default.
   	if(strlen($mask)==0)
       	$mask = "%display%"; 

	// Note: We are only modifying local copy of $mask for return.
	if(strlen(trim($img))>0 && $img!=="none")
	{
		$image = _theme_image($img, strlen($theme_image_type)>0?$display:NULL, $display, "absmiddle", $theme_image_type);
		if(strlen($image)>0)
			$mask = str_replace("%img%", $image, $mask);
		else if(strlen($display)>0)
			$mask = str_replace("%img%", $display, $mask);
		else
			$mask = str_replace("%img%", $value, $mask);
	}
	else if($img === "none") // A image value with "none" indicates we should replace image with empty string.
	{
		$mask = str_replace("%img%", "", $mask);
	}
	else
	{
		// If no %display% mask variable, replace missing image with display field instead.
		if(strpos($mask, '%display%') === FALSE)
		{
			$mask = str_replace("%img%", $display, $mask);
		}
		else if(strpos($mask, '%value%') === FALSE && strcmp($value, $display) !== 0) // but only if display is NOT the same as value
		{
			$mask = str_replace("%img%", $value, $mask);
		}
		else
		{
			$mask = str_replace("%img%", "", $mask);
		}
	}

	$mask = str_replace("%display%", $display, $mask);
	$mask = str_replace("%value%", $value, $mask);
    
    return $mask;       
}

/*
* Stub.
*/
function get_op_confirm_form($PHP_SELF, $confirm_message, $HTTP_VARS)
{
	return format_confirm_form($PHP_SELF, 
							$confirm_message, 
							NULL, 
							$HTTP_VARS);
}

/*
* @parm  $PHP_SELF - Form action
* @param $prompt_message
* @param $form_field_rs Array of form fields, of following format:
* 	$form_field_rs[] = array('prompt'=>'Prompt', 'field'=>'Field contents')
*/
function format_confirm_form($PHP_SELF, $confirm_message, $form_field_rs, $HTTP_VARS)
{
	global $LANG_VARS;
	
	$formContents = 
		$formContents .= "\n<table border=0 frameborder=0 cellspacing=1>".
		"<form action=\"$PHP_SELF\" method=\"POST\">".
		get_url_fields($HTTP_VARS, NULL, array('confirmed')); // Pass all http variables
	
	$align="left";
	
	if(is_not_empty_array($form_field_rs))
	{
		$align="center";
		while(list(,$field) = each($form_field_rs))
		{
			$formContents .= 
				format_field($field['prompt'],
							NULL, // prompt_mask
							$field['field']);
		}
		
		// Spacer
		$formContents .= "<tr><td colspan=2>&nbsp;</td></tr>";
	}
	
	// Add submit button.
	$formContents .= 
		"\n<tr><td colspan=2 align=\"$align\"><input type=\"hidden\" name=\"confirmed\" value=\"false\">".
		"\n<div class=\"colortext\">".$confirm_message."</div>".
		"\n<input type=\"button\" value=\" ".$LANG_VARS['yes']." \" onclick=\"this.form['confirmed'].value='true'; this.form.submit();\">&nbsp;".
		"\n<input type=\"button\" value=\" ".$LANG_VARS['no']." \" onclick=\"this.form['confirmed'].value='false'; this.form.submit();\">".
	"</td></tr></form>".
	"</table>";
	
	return $formContents;
}

/**
	Displays the footer links on a page.
*/
function format_footer_links($footer_links_rs)
{
	$field = "<br><p class=\"footer\">";
	while(list(,$footer_links_r) = @each($footer_links_rs))
	{
		if($footer_links_r['target'] == 'popup')
			$field .= "[<a href=\"javascript:popup('".urlencode($footer_links_r['url'])."','800','600');\">".$footer_links_r['text']."</a>]";
		else
			$field .= "[<a href=\"".$footer_links_r['url']."\"".(strlen($footer_links_r['target'])>0?"target=\"".$footer_links_r['target']."\"":"").">".$footer_links_r['text']."</a>]";
		$field .= "&nbsp;";
	}
	$field .= "</p>";

	return $field;
}

/*
* 	Format of a help entry:
* 
* 		array('Text', 'Text', 'Text'
* 	Or:
* 		array('img'=>image, 'text'=>'Text')
* 	Or:
* 		array('text'=>array('text', 'text', 'text'))
*	Or:
*		Text
* 
*	Expects the 'img' element to be an unexpanded theme image.
*/
function _format_help_entry($help_entry_r)
{
	if(is_array($help_entry_r))
	{
		if(isset($help_entry_r['img']))
			$entry .= _theme_image($help_entry_r['img'], NULL, $help_entry_r['text'])."&nbsp;";
		$entry .= $help_entry_r['text'];
		
		return $entry;
	}
	else
	{
		return $help_entry_r;
	}
}

function _format_help_list($help_entries_rs, $topLevel=FALSE)
{
	if(is_array($help_entries_rs))
	{
		if(!isset($help_entries_rs['text']))
		{
			if(!$topLevel)// remove indenting for top level UL
				$field = "\n<ul>";
				
			while(list($key,$entry) = each($help_entries_rs))
			{
				if(is_array($entry) && !isset($entry['text']))
				{
					$field .= _format_help_list($entry);
					
					if(!$topLevel)
						$field .= "<br>";
				}
				else
				{
					$field .= "\n<li>"._format_help_entry($entry)."</li>";
				}
			}
			
			if(!$topLevel)
				$field .= "\n</ul>";
		}
		else
		{
			return _format_help_entry($help_entries_rs);	
		}
			
		return $field;
	}
	else
	{
		return $help_entries_rs;
	}
}

function format_help_block($help_entries_rs)
{
	if(!is_array($help_entries_rs) && strlen($help_entries_rs)>0)
		$entries[] = $help_entries_rs;
	else
		$entries[] =& $help_entries_rs;
	
	return "\n<div class=\"help\">".
		_format_help_list($entries, TRUE).
		"</div>\n";
}

/*
* If _theme_image returns NULL / FALSE, this indicates that the default
* Action links functionality should be used.  Otherwise a valid Image link
* will be returned for each action.  The situation where some images
* are handled and others are not, will be handled as well, although
* it is not an ideal situation.
* 
* @param wrap_mask - If not null, the action links will be inserted into
* the $wrap_mask, before being returned.  If a %nowrap% variable is present,
* and all action links were handled with _theme_image, then the word 'nowrap'
* will replace %nowrap%, otherwise an empty string will be used.
* 
* @param ifempty - If FALSE, then return NULL.  This also means that the $wrap_mask
* will be ignored in this case as well.  However if $ifempty has a non-FALSE value,
* (even NULL), the $wrap_mask will still be expanded. 
*/
function format_action_links($action_links_rs, $wrap_mask=NULL, $ifempty=FALSE)
{
	$handled_all_images=TRUE;
	$theme_image_handled=TRUE;
	
	// Generate Action block here!
	$field = "";
	while(list(,$action_link_r) = @each($action_links_rs))
	{
		if(strlen($field)>0)
			$field .= "&nbsp;";
			
		// Only if an action image is defined!
		if(strlen($action_link_r['img'])>0)// The action images must have a 'action_' prefix.
			$action_image = _theme_image('action_'.$action_link_r['img'], NULL, $action_link_r['text'], NULL, "action");
		else
			$action_image = FALSE;
		
		// Either strlen($action_link_r['img'])==0 or theme specific theme_image does not want images for actions, and
		// returned NULL as a result.
		if($action_image!==FALSE && strlen($action_image)>0)
		{
			if(!$theme_image_handled)
			{
				if(strlen($field)>0)
					$field .= "/ ";
			}
			$field .= "<a href=\"".$action_link_r['url']."\">$action_image</a>";
			$theme_image_handled=TRUE;
		}
		else
		{
			if(strlen($field)>0)
				$field .= "/ ";
				
			$field .= "<a href=\"".$action_link_r['url']."\">".str_replace(" ","&nbsp;",$action_link_r['text'])."</a>";

			$theme_image_handled=FALSE;
			// At least one default action handled.
			$handled_all_images=FALSE;
		}
	}
	
	if((strlen($field)>0 || $ifempty!==FALSE) && strlen($wrap_mask)>0)
	{
		if($handled_all_images && strlen($field)>0)
			$wrap_mask = str_replace("%nowrap%", "nowrap", $wrap_mask);
		else
			$wrap_mask = str_replace("%nowrap%", "", $wrap_mask);
		
		if(strlen($field)>0)
			$wrap_mask = str_replace("%field%", $field, $wrap_mask);
		else // if we are in here, then $ifempty has to have a value.
			$wrap_mask = str_replace("%field%", $ifempty, $wrap_mask);
			
		return $wrap_mask;
	}
	else if(strlen($field)>0)
		return $field;
	else
		return NULL;
}

/*
* @param $errors an array of errors.  The format of
* each error entry is:
* 	error=>'Main error'
* 	detail=>'Details of error, mysql_error, etc'
* @param $msg_type Indicates the type of error, which might be one
* 		of:
* 			error
* 			warning
* 			information
*/
function format_error_block($errors, $err_type = 'error')
{
	switch($err_type)
	{
		case 'error':
			$class = 'error';
			$smclass = 'smerror';
			break;
		
		case 'smerror':
			$class = 'smerror';
			$smclass = 'smerror';
			break;
			
		case 'warning': // If it becomes necessary, new CSS style classes will be introduced.
		case 'information':
		default:
			$class = 'smsuccess';
			$smclass = 'footer';
	}
	
	if(!is_array($errors))
	{
		if(strlen(trim($errors))==0)
			return NULL;
		else				
			$error_rs[] = array('error'=>$errors,'detail'=>'');
	}
	else if(isset($errors['error']))
		$error_rs[] = $errors;
	else
		$error_rs = $errors;
	
	$error_entries = NULL;
	while(list(,$error) = each($error_rs))
	{
		//print_r($error);
		if(is_not_empty_array($error))
		{
			//print_r($error);
			$error_entry = $error['error'];
		
			if(!is_array($error['detail']) && strlen($error['detail'])>0)
				$detail_rs[] = $error['detail'];
			else if(is_array($error['detail']))
				$detail_rs = $error['detail'];
		
			if(is_not_empty_array($detail_rs))
			{
				$details = "";
				while(list(,$detail) = each($detail_rs))
				{
					$details .= "\n<li class=\"$smclass\">".$detail."</li>";
				}
			
				if(strlen($details)>0)
					$error_entry .= "\n<ul>".$details."</ul>";
			}
		}
		else
			$error_entry = $error;
		
		$error_entries[] = $error_entry;
	}
	
	if(count($error_entries)>1)
	{
		$error_block = "\n<ul>";
		while(list(,$error_entry) = each($error_entries))
		{
			$error_block .= "\n<li class=\"$class\">$error_entry</li>";
		}
		$error_block .= "</ul>";
		
		return $error_block;
	}
	else if(count($error_entries)==1)
	{
		return "\n<p class=\"$class\">".$error_entries[0]."</p>";
	}
	else
		return NULL;
}

/**
	Will fetch the image block for a particular s_item_type.  Will
	use $s_item_type if specified, otherwise will get s_item_type
	for $item_id first.

	@param $islinked is ignored at the moment.  It may be used to
	display a slightly smaller item type image for linked items,
	but we can not provide this logic at level.
	
	NOTE: Assumes that include_once("./include/theme.php"); has been performed.
*/
function get_item_image($s_item_type, $item_id = NULL, $islinked = FALSE)
{
	global $LANG_VARS;
	
	if(strlen($s_item_type)>0 || ($s_item_type = fetch_item_type($item_id)))
	{
		$item_type_r = fetch_item_type_r($s_item_type);
		if(is_array($item_type_r))
		{
			// default
			$imagetext = $s_item_type;

			// Get image block.
			if(strlen($item_type_r['image'])>0)
			{
				if(strlen($item_type_r['description'])>0)
					$title_text = htmlspecialchars($item_type_r['description']);
				else
					$title_text = NULL;

				$imagetext = _theme_image($item_type_r['image'], $s_item_type, $title_text, 'absmiddle', "s_item_type");
			}
	
			if($islinked)
			{
				$imagetext .= _theme_image("linked.gif", NULL, $LANG_VARS['linked_item'], 'top');
			}
				
			return $imagetext;
			
		}
	}
	
	//else
	return FALSE;
}
?>
