/*
 *  Copyright (c) 2007,2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Type.h"
#include "Type_p.h"

#include "Debug.h"
#include "Macros_p.h"

#include <iostream>

#include <llvm/DerivedTypes.h>

#include "wrappers/StructWrap.h"
#include "SharedPointer.h"

using namespace GTLCore;

//-------- Type::StructDataMember -------//

struct Type::StructDataMember::Private : public SharedPointerData
{
  Private() {}
  Private(const Private& _rhs) : name(_rhs.name), type(_rhs.type), initialSizes(_rhs.initialSizes) {}
  GTLCore::String name;
  const Type* type;
  std::list<int> initialSizes;
};

GTL_SHARED_DATA_CLASS(Type::StructDataMember, StructDataMember);

Type::StructDataMember::StructDataMember(const GTLCore::String& _name, const Type* _type) : d(new Private)
{
  d->ref();
  d->name = _name;
  d->type = _type;
  GTL_ASSERT( _type->dataType() != Type::ARRAY);
}

Type::StructDataMember::StructDataMember(const GTLCore::String& _name, const Type* _type, const std::list<int>& _initialSizes) : d(new Private)
{
  d->ref();
  d->name = _name;
  d->type = _type;
  d->initialSizes = _initialSizes;
  GTL_ASSERT( d->initialSizes.empty() or ( not d->initialSizes.empty() and _type->dataType() == Type::ARRAY) );
}

GTLCore::String Type::StructDataMember::name() const
{
  return d->name;
}

const Type* Type::StructDataMember::type() const
{
  return d->type;
}

const std::list<int>& Type::StructDataMember::initialSizes() const
{
  return d->initialSizes;
}

//-------- Type -------//

const Type* Type::Undefined = new Type(Type::UNDEFINED);
const Type* Type::Boolean = new Type(Type::BOOLEAN);
const Type* Type::Integer8 = new Type(Type::INTEGER8);
const Type* Type::UnsignedInteger8 = new Type(Type::UNSIGNED_INTEGER8);
const Type* Type::Integer16 = new Type(Type::INTEGER16);
const Type* Type::UnsignedInteger16 = new Type(Type::UNSIGNED_INTEGER16);
const Type* Type::Integer32 = new Type(Type::INTEGER32);
const Type* Type::UnsignedInteger32 = new Type(Type::UNSIGNED_INTEGER32);
const Type* Type::Integer64 = new Type(Type::INTEGER64);
const Type* Type::UnsignedInteger64 = new Type(Type::UNSIGNED_INTEGER64);
const Type* Type::Float64 = new Type(Type::FLOAT64);
const Type* Type::Float32 = new Type(Type::FLOAT32);
const Type* Type::Float16 = new Type(Type::FLOAT16);
const Type* Type::Void = new Type(Type::VOID);
const Type* Type::Pointer = new Type(Type::POINTER);
const Type* Type::Color = new Type("Color", std::vector<Type::StructDataMember>());

STATIC_DELETER(DeleteTypes)
{
  Type::Private::deleteTypes();
}

Type::Type(DataType _dataType) : d(new Private(_dataType))
{
  GTL_DEBUG("Create type for " << _dataType);
  GTL_ASSERT(_dataType != STRUCTURE and _dataType != ARRAY and _dataType != VECTOR );
  if(_dataType == STRUCTURE or _dataType == ARRAY or _dataType == VECTOR)
  {
    d->dataType = UNDEFINED;
  }
  init(_dataType);
}

Type::Type(const GTLCore::String& _structName, const std::vector<StructDataMember>& _members) : d(new Private)
{
  init(STRUCTURE);
  d->structName = _structName;
  d->structDataMembers = new std::vector<StructDataMember>( );
  // Add the Garbage Collector field
  d->structDataMembers->push_back( StructDataMember( "", GTLCore::Type::Integer32 ) );
  // Add members
  d->structDataMembers->insert( d->structDataMembers->end(), _members.begin(), _members.end() );
  d->m_symbolName = "struct_" + _structName;
}

Type::Type(const Type* _arrayType) : d(new Private)
{
  init(ARRAY);
  d->arrayType = _arrayType;
  d->m_symbolName = "array_" + _arrayType->d->symbolName();
}

Type::Type(int _size, const Type* _vectorType) : d(new Private)
{
  init(VECTOR);
  d->arrayType = _vectorType;
  d->vectorSize = _size;
  d->m_symbolName = "vector" + String::number(_size) + "_" + _vectorType->d->symbolName();
}

void Type::init(DataType _dataType)
{
  d->dataType = _dataType;
  d->arrayType = 0;
  d->structDataMembers = 0;
  d->structFunctionMembers = 0;
  d->structPrivateFunctionMembers = 0;
}

Type::~Type()
{
  delete d;
}

Type::DataType Type::dataType() const
{
  return d->dataType;
}

const Type* Type::embeddedType() const
{
  return d->arrayType;
}

GTLCore::String Type::structName() const
{
  return d->structName;
}

Type::StructDataMember Type::structDataMember(std::size_t index) const
{
  GTL_ASSERT(d->dataType == Type::STRUCTURE );
  GTL_ASSERT(d->structDataMembers);
  GTL_ASSERT( index < countStructDataMembers( ) );
  return (*d->structDataMembers)[index + STRUCT_FIRST_ELEMENT];
}

std::size_t Type::countStructDataMembers( ) const
{
  if( d->structDataMembers )
  {
    return d->structDataMembers->size() - STRUCT_FIRST_ELEMENT;
  } else {
    return 0;
  }
}

int Type::bitsSize() const
{
  switch( dataType() )
  {
    case Type::BOOLEAN:
      return 1;
    case Type::INTEGER8:
    case Type::UNSIGNED_INTEGER8:
      return 8;
    case Type::INTEGER16:
    case Type::UNSIGNED_INTEGER16:
      return 16;
    case Type::INTEGER32:
    case Type::UNSIGNED_INTEGER32:
      return 32;
    case Type::FLOAT16:
      return 16;
    case Type::FLOAT32:
      return 32;
    case Type::VOID:
      return 0;
    case Type::VECTOR:
      return d->vectorSize * embeddedType()->bitsSize();
    default:
      return -1;
      break;
  }

}

bool Type::isSigned() const
{
  return not isUnsigned();
}

bool Type::isUnsigned() const
{
  switch( dataType() )
  {
    case Type::UNSIGNED_INTEGER8:
    case Type::UNSIGNED_INTEGER16:
    case Type::UNSIGNED_INTEGER32:
      return true;
    case Type::VECTOR:
      return embeddedType()->isUnsigned();
    default:
      return false;
  }
}

unsigned int Type::vectorSize() const
{
  GTL_ASSERT( dataType() == Type::VECTOR );
  return d->vectorSize;
}

struct GTLCore::MatrixSize {
};

MatrixSize Type::matrixSize() const
{
  return MatrixSize();
}

bool Type::isNumber() const
{
  return isInteger() or isFloatingPoint();
}

bool Type::isInteger() const
{
  return d->dataType == BOOLEAN or d->dataType == INTEGER8 or d->dataType == UNSIGNED_INTEGER8 or d->dataType == INTEGER16 or d->dataType == UNSIGNED_INTEGER16 or d->dataType == INTEGER32 or d->dataType == UNSIGNED_INTEGER32;
}

bool Type::isFloatingPoint() const
{
  return d->dataType == FLOAT16 or d->dataType == FLOAT32 or d->dataType == FLOAT64;
}

bool Type::isStructure() const
{
  return d->dataType == STRUCTURE;
}
