#!/bin/sh
# path trick for tclsh variants \
# exec tclsh $0 ${1+"$@"}
# We don't actually need the above lines since Mash adds its own header.

# vcc3serv.tcl --
#
#       Core server code sets up a TclDP RPC server to provide remote access
#       to vcc3.  Also parses input file and sets up camera.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

source vcc3-api.tcl
source ../../lib/tcl/dp-lib.tcl

#
# default port is 6905
# default dev is /dev/cuac01
#

global g_port g_serverHandle g_dev g_cf g_label

proc showUsage {} {
    puts stdout "usage:  vcc3d \[-port <port>\] \[-dev <device>\] \[-cf <configuration file>\] \[-label <camera label>\]"
}

proc getFlags {argv} {
    global g_port g_dev g_cf g_label
    
    set g_dev "/dev/cuac01"
    set g_cf ""
    set g_port 6905

    set state "flag"
    foreach arg $argv {
	switch -exact -- $state {
	    flag {
		switch -exact -- $arg {
		    -port {
			set state "port"
		    }
		    -dev {
			set state "dev"
		    }
		    -cf {
			set state "cf"
		    }
		    -label {
			set state "label"
		    }
		    -help {
			showUsage
			exit
		    }
		    default {
			showUsage
			exit
		    }
		}
	    }
	    port {
		set g_port $arg
		set state "flag"
	    }
	    dev {
		set g_dev $arg
		set state "flag"
	    }
	    cf {
		set g_cf $arg
		set state "flag"
	    }
	    label {
		set g_label $arg
		set state "flag"
	    }
	}
    }
}

proc processConfigFile {} {
    global g_cf

    puts stdout "processing file: $g_cf"

    if [catch {open $g_cf r} fd] {
	puts stdout "Cannot open file $g_cf"
	return 0
    }

    while {[gets $fd line] >= 0} {
	if {$line != ""} {
	    set lLine [split $line =]
	    set param [string trim [lindex $lLine 0]]
	    set value [string trim [lindex $lLine 1]]
	    #	puts stdout "param=$param, value=$value"
	    if {[string index $line 0] != "#"} {
		processSetting $param $value
	    }
	}
    }
    close $fd
    return 1
}

proc processSetting {param value} {
    set errMsg ""
    switch -exact -- $param {
	PAN_SPEED {
	    if [catch {vcc3_setPanSpeed $value} err] { set errMsg $err }
	}
	TILT_SPEED {
	    if [catch {vcc3_setTiltSpeed $value} err] { set errMsg $err }
	}
	ZOOM_SPEED {
	    if [catch {vcc3_setZoomSpeed $value} err] { set errMsg $err }
	}
	FOCUS_SPEED {
	    if [catch {vcc3_setFocusSpeed $value} err] { set errMsg $err }
	}
	PAN_POSITION {
	    if [catch {vcc3_setPanPosition $value} err] { set errMsg $err }
	    if {$err == 0} {
		puts stdout "set pan pos failed!"
	    }
	}
	TILT_POSITION {
	    if [catch {vcc3_setTiltPosition $value} err] { set errMsg $err }
	}
	ZOOM_POSITION {
	    if [catch {vcc3_setZoomPosition $value} err] { set errMsg $err }
	}
	PRESET {
	    set temp [split $value " "]
	    set num [string trim [lindex $temp 0]]
	    set info [lrange $temp 1 4]
	    if [catch {vcc3_setPreset $num $info} err] { set errMsg $err }
	}
	default {
	    # do nothing
	    puts stdout "ignoring $param,$value"
	}
    }
    if {$errMsg != ""} {
	puts stdout "Error in configuration file: $err"
    }
}


proc vcc3_killServer {} {
    global g_serverHandle

    dp_CloseRPC $g_serverHandle

    vcc3_closeControl

    # this after is needed to give time for any app remotely invoking
    # amxd_killServer to close the connection
    after 500 { exit }
}

proc vcc3_getLabel {} {
    global g_label

    return $g_label
}
    
#
# main function
#

set g_label "???"

# get the command line arguments
getFlags $argv

# FIXME - printfs for debugging
#puts stdout "g_cf = $g_cf"

puts stdout "port=$g_port  device=$g_dev"

set result [vcc3_init $g_dev]
if {!$result} {
    puts stdout "Error: couldn't init camera; result is $result"
    exit
} else {
    puts stdout "Camera initialization successful"
}

if {$g_cf != ""} {
    set result [processConfigFile]
    if {!$result} {
	puts stdout "Error: couldn't process configuration file $g_cf"
	vcc3_killServer
    }
}

puts stdout "creating server"
set g_serverHandle [dp_MakeRPCServer $g_port]

puts stdout "vcc3d running"

# FIXME - at this point, we should put the camera into regular IR mode
#
# then, when a client attaches to the server, put it in pc control mode
#
# when the last client disconnects, we should put it back in IR mode.  that
#  way, if vcc3d dies for some reason, it will be in the proper state and
#  you won't have to cycle the vcc3 power

vwait forever

