/*
 * whirlpool.cc --
 *
 *      A whirpool-like effect
 *
 * Copyright (c) 1993-2001 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* @(#) $Header: /usr/mash/src/repository/mash/mash-1/fx/effects/whirlpool.cc,v 1.1 2002/02/07 04:18:19 chema Exp $ */


#include <dvmbasic.h>
#include <dvmbytegeom.h>

#include "whirlpool.h"



WhirlpoolEffect::WhirlpoolEffect(int decimation) 
	: EffectModule(1, decimation), old_theta_(0.0)
{
	theta_.from = 0.0;
	//theta_.to = 10.0;
	theta_.to = 6.3; // ~ 2 * pi
	theta_.value = 0.0;
	theta_.incr = 1;
	old_theta_ = 0.0;

	return;
}



int WhirlpoolEffect::command(int argc, const char*const* argv)
{
	return (EffectModule::command(argc, argv));
}



void WhirlpoolEffect::trigger_effect (Uncompressed* input)
{
	float angle;

	//printf ("w(%dx%d) ", input->w_, input->h_); fflush (stdout); // XXX

	// recalculate theta parameter
	if (theta_.incr == 1) {
		theta_.value = theta_.value + 0.1;
	} else {
		theta_.value = theta_.value - 0.1;
	}
	if (theta_.value > theta_.to) {
		theta_.value = theta_.to;
		theta_.incr = 0;
	} else if (theta_.value < theta_.from) {
		theta_.value = theta_.from;
		theta_.incr = 1;
	}
	angle = theta_.value;


	// step 1: initialize the output frame to 0,0x80,0x80 (black)
	ByteImage *out_l = output_->lum_;
	ByteImage *out_cr = output_->cr_;
	ByteImage *out_cb = output_->cb_;
	ByteSet (out_l, 0);
	ByteSet (out_cr, 128);
	ByteSet (out_cb, 128);


	// step 2: carry out the affine whirlpool transformation
	int w = input->w_;
	int h = input->h_;

	float cos_angle = cos(angle);
	float sin_angle = sin(angle);

	float in_css_h = (float) input->css_h_;
	float in_css_v = (float) input->css_v_;
	float out_css_h = (float) output_->css_h_;
	float out_css_v = (float) output_->css_v_;

	float sf = 1.0 - (angle/10.0);
	float la = sf * cos_angle;
	float lb = -(sf * sin_angle);
	float lc = (w - (sf * w * cos_angle) + (sf * h * sin_angle)) / 2.0;
	float ld = sf * sin_angle;
	float le = sf * cos_angle;
	float lf = (h - (h * sf * cos_angle) - (sf * w * sin_angle)) / 2.0;

	float ca = la * in_css_h / out_css_h;
	float cb = lb * in_css_v / out_css_h;
	float cc = lc / out_css_h;
	float cd = ld * in_css_h / out_css_v;
	float ce = le * in_css_v / out_css_v;
	float cf = lf / out_css_v;

	ByteImage *in_l = input->lum_;
	ByteImage *in_cr = input->cr_;
	ByteImage *in_cb = input->cb_;

	ByteAffine (in_l, out_l, la, lb, lc, ld, le, lf);
	ByteAffine (in_cr, out_cr, ca, cb, cc, cd, ce, cf);
	ByteAffine (in_cb, out_cb, ca, cb, cc, cd, ce, cf);

	// deliver the output frame
	deliver_frame();

	return;
}

