/*
 * mb-file.h --
 *
 *      routines for fragmenting and assembling a file
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Header: /usr/mash/src/repository/mash/mash-1/mb/mb-file.h,v 1.7 2002/02/03 03:16:30 lim Exp $
 */

#ifndef MB_FILE_H
#define MB_FILE_H

#include "mb.h"

class MBFrag;

class MBFile {
public:
    static u_long nextTmpNo;

    MBFile() : channel_(0) {}
    virtual ~MBFile() { if (channel_) Tcl_Close(MB_Interp,channel_);
}
    // takes either -file <filename> or <filename>
    // returns FALSE if it encounters any error
    Bool Init(int argc, const char*const* argv);
    static const char* Assemble(int nFrag, MBFrag** ppFragArg,
                                const char* szFileName=NULL);

    /* called when interpreter exited, to delete a temp file */
    static void deleteTemp(ClientData cd, Tcl_Interp *interp);

    // customize this if you want to have more intelligent way of
    // customizing the division of pages.
    //
    // fragmentize the file into blocks of max size
    // returns FALSE if no more fragments after this one
    virtual Bool getNextFrag(int maxFragLen, MBFrag* pFrag);

private:
    Tcl_Channel channel_;
};

//
// This is a fragment that is read from a channel which is usu a file
//
class MBFrag {
public:
    MBFrag():len_(0),pData_(NULL) {}
    virtual ~MBFrag() { delete[] pData_; }

    // returns FALSE if this is the last fragment
    // in case of async the read could be called multiple times
    // if during the first time it is empty
    virtual Bool Read(Tcl_Channel channel, int maxsize) {
        if (!pData_) pData_ = new Byte[maxsize]; // native partitioning
        int len = Tcl_Read(channel, (char*)pData_, maxsize);
        if (len==-1) {
            SignalError(("POSIX err:%d when reading",
                        Tcl_GetErrno()));
            len_=0;
            return FALSE;
        }
        len_ = len;
        if (len_ < maxsize) {
            return !Tcl_Eof(channel); // returns FALSE if last fragment
        } else return TRUE;
    }
    // read from a pointer
    virtual void Read(Byte *pb, u_short len) {
        len_ = len;
        delete[] pData_;
        pData_ = new Byte[len_];
        memcpy(pData_, pb, len_);
    }
    virtual int Write(Tcl_Channel channel) {
        return(Tcl_Write(channel, (char*)pData_, len_));
    }
    virtual u_short Write(Byte *pb) {
        memcpy(pb, pData_, len_);
        return len_;
    }
    Bool IsEmpty() { return (len_==0); }

protected:
    /* DATA
     * ---- */
    u_short len_;
    Byte* pData_;
};


#endif /* #ifdef MB_FILE_H */
