/*
 * mbv2-page.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] = "@(#) $Header: /usr/mash/src/repository/mash/mash-1/mbv2/mbv2-page.cc,v 1.11 2002/02/03 03:17:08 lim Exp $";
#endif


#include "mbv2-obj.h"
#include "mbv2-cmd.h"
#include "mbv2-listsort.h"
#include "mtrace.h"
#include <assert.h>

void srm_recover1(srm_source_t source, unsigned int cid,
		  unsigned int sseq, unsigned int eseq)
{
	MTrace(trcMB, ("Calling srm_recover(%p, %u, %u, %u)", source, cid,
		       sseq, eseq));
	srm_recover(source, cid, sseq, eseq);
}


MBv2Page::MBv2Page(MBv2Source *src, u_int32_t cid, const MBv2PageId &pageid)
	: cmds_(128), nextCmdId_(0), firstHole_(0), src_(src), cid_(cid),
	  id_(pageid), canvas_(NULL), tagCnt_(0)
{
	Tcl_InitHashTable(&htCmd2CanvId_, TCL_ONE_WORD_KEYS);
	Tcl_InitHashTable(&htCanvId2Cmd_, TCL_ONE_WORD_KEYS);
	Tcl_InitHashTable(&htTag2Cmd_,    TCL_ONE_WORD_KEYS);
}


MBv2Page::~MBv2Page()
{
	for (MBv2CmdId i=0; i < nextCmdId_; i++)
		if (cmds_[i]) delete cmds_[i];
	Tcl_DeleteHashTable(&htCmd2CanvId_);
	Tcl_DeleteHashTable(&htCanvId2Cmd_);
	Tcl_DeleteHashTable(&htTag2Cmd_);
}


void
MBv2Page::tag(MBv2Cmd *cmd, MBv2CmdId tagid)
{
	if (tagid==MBv2InvalidCmdId) {
		tagid = tagCnt_++;
	}

	cmd->tag(tagid);

	int created;
	Tcl_HashEntry *entry=Tcl_CreateHashEntry(&htTag2Cmd_, (char*)tagid,
						 &created);
	Tcl_SetHashValue(entry, (ClientData)cmd->id());
}


MBv2CmdId
MBv2Page::tag2cmd(MBv2CmdId tagid)
{
	Tcl_HashEntry *entry=Tcl_FindHashEntry(&htTag2Cmd_, (char*)tagid);
	if (entry) return (MBv2CmdId)Tcl_GetHashValue(entry);
	else return MBv2InvalidCmdId;
}


MBv2CanvId
MBv2Page::get_canvas_item(MBv2CmdId cmdid)
{
	Tcl_HashEntry *entry=Tcl_FindHashEntry(&htCmd2CanvId_, (char*)cmdid);
	if (entry) return (MBv2CanvId)Tcl_GetHashValue(entry);
	else return MBv2InvalidCanvId;
}


MBv2CmdId
MBv2Page::get_cmdid(MBv2CanvId canvid)
{
	Tcl_HashEntry *entry=Tcl_FindHashEntry(&htCanvId2Cmd_, (char*)canvid);
	if (entry) return (MBv2CmdId)Tcl_GetHashValue(entry);
	else return MBv2InvalidCmdId;
}


void
MBv2Page::forget_canvas_item(MBv2CmdId cmdid)
{
	Tcl_HashEntry *entry=Tcl_FindHashEntry(&htCmd2CanvId_, (char*)cmdid);
	if (entry) {
		MBv2CanvId canvid = (MBv2CanvId) Tcl_GetHashValue(entry);
		Tcl_DeleteHashEntry(entry);
		entry = Tcl_FindHashEntry(&htCanvId2Cmd_, (char*)canvid);
		if (entry && cmdid==(MBv2CmdId)(Tcl_GetHashValue(entry)))
			Tcl_DeleteHashEntry(entry);
	}
}


void
MBv2Page::associate_canvas_item(MBv2CmdId cmdid, MBv2CanvId canvId)
{
	int created;
	Tcl_HashEntry *entry=Tcl_CreateHashEntry(&htCmd2CanvId_, (char*)cmdid,
						 &created);
	Tcl_SetHashValue(entry, (ClientData)canvId);

	entry=Tcl_CreateHashEntry(&htCanvId2Cmd_, (char*)canvId,
				  &created);
	Tcl_SetHashValue(entry, (ClientData)cmdid);
}


void
MBv2Page::insert(MBv2Cmd *cmd)
{
	MBv2CmdId cmdId = cmd->id();
	cmds_[cmdId] = cmd;
	if (nextCmdId_ <= cmdId)
		nextCmdId_ = cmdId+1;
	execute(cmd);
}


// returns true if execution succeeded
// returns false if execution has been deferred
Bool
MBv2Page::execute(MBv2Cmd *cmd, Bool inFlushDeferred)
{
	MBv2Dependency dep;
	if (cmd->type()==MBv2Group) {
		MTrace(trcMB, ("DDD executing group command %u", cmd->id()));
	}
	if (!cmd->incomplete(this, &dep)) {
		MTrace(trcMB|trcVerbose, ("cmd %u is complete; executing it",
					  cmd->id()));
		// this command is complete
		// execute it right now

		// first update the firstHole_ parameter
		if (cmd->id()==firstHole_) {
			MBv2CmdId i;
			for (i=firstHole_+1; i<nextCmdId_; i++) {
				if (!cmds_[i]) break;
				if (cmds_[i]->incomplete(this, NULL))
					break;
			}
			firstHole_ = i;
		}

		// now execute the cmd
		MBv2Time startOfEternity;
		startOfEternity.lower = startOfEternity.upper = 0;
		if (!cmd->execute(this, startOfEternity, cmd->timestamp())) {
			MTrace(trcMB|trcVerbose, ("execution of cmd %u failed",
						  cmd->id()));
			return FALSE;
		}

		// rearrange the raise orders on the canvas
		src_->session()->rearrange(this, cmd);

		// notify the tcl code of activity on the canvas
		src_->session()->activity(this, cmd);

		// flush any deferred cmds that may be dependent on this one
		if (!inFlushDeferred) flush_deferred(cmd->id());

		return TRUE;
	} else {
		MTrace(trcMB|trcVerbose, ("cmd %u is incomplete; deferring it",
					  cmd->id()));
		// add this command to the deferred list
		if (dep.cid==cid_) {
			// this command is waiting for another command on
			// the same page
			if (cmd->type()==MBv2Group) {
				MTrace(trcMB,("DDD deferring group command %u",
					      cmd->id()));
			}
			defer_cmd(this, cmd, &dep);
		} else {
			// this command is waiting for another command on
			// a different page

			// FIXME:we may need to schedule a repair request here
			if (is_visible()) {
				srm_recover1(src_->srm_source(), dep.cid,
					    dep.cmd, dep.cmd);
			}
			src_->defer_cmd(this, cmd, &dep);
		}

		// move all the cmds that depend on this cmd to
		// the deferred list for the cmd that this cmd itself
		// depends on
		update_deferred(cmd, &dep);
	}
	return FALSE;
}


void
MBv2Page::update_deferred(MBv2Cmd *cmd, MBv2Dependency *dep)
{
	MBv2Page *dependPage = src_->find_page(dep->cid);
	MBv2CmdId cmdid = cmd->id();
	MBv2SaveDependency *d1;

	// do the easy cases first
	if (deferredCmds_.IsEmpty() ||
	    deferredCmds_.PeekAtTail()->dep.cmd < cmdid) {
		// there is no match here
		return;
	}

	// go thru the entire list looking for cmd->id()
	ListIndex idx, tmp;
	idx=deferredCmds_.getFirst();
	while (!deferredCmds_.IsDone(idx)) {
		d1 = deferredCmds_.getData(idx);
		if (d1->dep.cmd > cmdid) return;
		if (d1->dep.cmd== cmdid) {
			// move this dependency to the appropriate list
			tmp = idx;
			idx=deferredCmds_.getNext(idx);

			deferredCmds_.Remove(tmp);
			d1->dep.cid = dep->cid;
			d1->dep.cmd = dep->cmd;
			d1->dep.degree += dep->degree;
			if (dependPage) {
				// although this may touch the linked list we
				// are working with, since the list is
				// in sorted order, it won't mess with
				// our current pointers
				dependPage->defer_cmd(d1->page, d1->cmd,
						      &d1->dep);
			} else {
				src_->defer_cmd(d1->page, d1->cmd, &d1->dep);
			}
			delete d1;
		} else {
			idx=deferredCmds_.getNext(idx);
		}
	}
}


void
MBv2Page::defer_cmd(MBv2Page *page, MBv2Cmd *cmd, MBv2Dependency *dep)
{
	assert(cid_ == dep->cid);
	MBv2SaveDependency *d = new MBv2SaveDependency;
	d->page = page;
	d->cmd  = cmd;
	d->dep  = *dep;

	// add in increasing order of dep->cmd (followed by dep->degree)
	::insert_sorted(&deferredCmds_, d);
#if 0
	if (deferredCmds_.IsEmpty()) {
		// add at the head of the list
		deferredCmds_.InsertAtHead(d);
		return;
	}

	d1 = deferredCmds_.PeekAtTail();
	if (d1->dep.cmd < dep->cmd || (d1->dep.cmd==dep->cmd &&
				       d1->dep.degree <= dep->degree)) {
		// add at the tail of the list
		deferredCmds_.InsertAtTail(d);
		return;
	}

	// insert somewhere in the middle of the list
	ListIndex idx;
	for (idx=deferredCmds_.getFirst();
	     !deferredCmds_.IsDone(idx);
	     idx=deferredCmds_.getNext(idx))
	{
		d1 = deferredCmds_.getData(idx);
		if (d1->dep.cmd > dep->cmd || (d1->dep.cmd==dep->cmd &&
					       d1->dep.degree > dep->degree)) {
			deferredCmds_.InsertBefore(idx, d);
			return;
		}
	}
#endif
}


// call flush_deferred only if "just_rcvd" is complete
void
MBv2Page::flush_deferred(MBv2CmdId just_rcvd)
{
	ListIndex idx, next;
	MBv2SaveDependency *d;

	idx = deferredCmds_.getFirst();
	while (!deferredCmds_.IsDone(idx)) {
		next = deferredCmds_.getNext(idx);
		d = deferredCmds_.getData(idx);
		MTrace(trcMB,("DDD got %u; flushing deferred cmd %u, depends "
			      "on %u",just_rcvd, d->cmd->id(), d->dep.cmd));
		if (d->dep.cmd > just_rcvd) break;
		if (d->dep.cmd == just_rcvd) {
			MTrace(trcMB,("DDD flushing deferred command %u",
				      d->cmd->id()));
			deferredCmds_.Remove(idx);
			// MBv2Page::execute may insert the command back into
			// the deferred list; however out pointers oughtn't
			// get garbled even though we are still parsing thru
			// deferred list
			d->page->execute(d->cmd, 1);
			delete d;
		}
		idx = next;
	}
}


//  We scan thru the commands from the most recent, until we find the
//  item that is older than timestamp, and refer to and item.
//
//  REVIEW: Can do better? current: O(nM) on average, but const if append
//
MBv2CanvId MBv2Page::find_most_recent(MBv2Time& mostRecent,
				      const MBv2Time &timestamp)
{
	MBv2Time rts;
	MBv2Cmd *cmd=NULL;
	MBv2CmdId cmdId;
	MBv2CanvId canvId, retval=MBv2InvalidCanvId;
	if (nextCmdId_==0) return MBv2InvalidCmdId;
	cmdId = nextCmdId_ - 1;

	/* loop from most recent:
	 *       - skip commands that are null or are older than timestamp
	 *       - exit if we cannot find an item that fits citeria
	 *         (see above)
	 *       - skip if the command does not refer to an item or
	 *         is not 'active' */
	while (cmdId != (MBv2CmdId)-1) {
		cmd = cmds_[cmdId--];
		if (cmd==NULL) continue;
		rts = cmd->raise_timestamp(this);
		if (rts.lower==0 && rts.upper==0) continue;
		if (rts >= timestamp) continue;
		if (mostRecent >= cmd->timestamp()) break;
		if ( (canvId = get_canvas_item(cmd->id()))==MBv2InvalidCanvId)
			continue;

		if (! (mostRecent >= rts) ) {
			mostRecent = rts;
			retval = canvId;
		}
	}

	return retval;
}


// this function gets called whenever we switch pages.
// it schedules repair requests for objects on this page (or objects
// that stuff on this page depends on)
void
MBv2Page::schedule_recovery()
{
	// look for holes in the cmd space and schedule repair requests
	// if necessary

	MBv2Dependency dep;
	MBv2CmdId i, j;

	for (i=firstHole_; i<nextCmdId_; i++) {
		// look for a sequence of holes
		for (j=i; j<nextCmdId_; j++) {
			if (cmds_[j]) break;
		}

		if (i < j) {
			// we have a sequence of holes
			srm_recover1(src_->srm_source(), cid_, i, j-1);
			i = j-1;
		} else {
			// check if there are dependencies on other pages
			// that are missing
			if (cmds_[i]->incomplete(this, &dep) &&
			    dep.cid != cid_){
				srm_recover1(src_->srm_source(),
					    dep.cid, dep.cmd, dep.cmd);
			}
		}
	}
}
