/*
 * hash.h --
 *
 *      Adaptation of Tcl's hash functions for non-Tcl-based applications
 */

/*
 * tcl.h --
 *
 *	This header file describes the externally-visible facilities
 *	of the Tcl interpreter.
 *
 * Copyright (c) 1987-1994 The Regents of the University of California.
 * Copyright (c) 1994-1997 Sun Microsystems, Inc.
 * Copyright (c) 1993-1996 Lucent Technologies.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tcl.h 1.324 97/08/07 10:26:49
 */


#ifndef MASH_HASH_H
#define MASH_HASH_H


/*
 * Forward declaration of HashTable.  Needed by some C++ compilers
 * to prevent errors when the forward reference to HashTable is
 * encountered in the HashEntry structure.
 */

#ifdef __cplusplus
struct HashTable;
#endif

#ifndef _HASH_CLIENTDATA
#   if defined(__STDC__) || defined(__cplusplus)
typedef void *HashClientData;
#   else
typedef int *HashClientData;
#   endif /* __STDC__ */
#define _HASH_CLIENTDATA
#endif

/*
 * Structure definition for an entry in a hash table.  No-one outside
 * Tcl should access any of these fields directly;  use the macros
 * defined below.
 */

typedef struct HashEntry {
	struct HashEntry *nextPtr;	/* Pointer to next entry in this
					 * hash bucket, or NULL for end of
					 * chain. */
	struct HashTable *tablePtr;	/* Pointer to table containing entry.*/
	struct HashEntry **bucketPtr;	/* Pointer to bucket that points to
					 * first entry in this entry's chain:
					 * used for deleting the entry. */
	HashClientData clientData;	/* Application stores something here
					 * with SetHashValue. */
	union {				/* Key has one of these forms: */
		char *oneWordValue;	/* One-word value for key. */
		int words[1];		/* Multiple integer words for key.
					 * The actual size will be as large
					 * as necessary for this table's
					 * keys. */
		char string[4];		/* String for key.  The actual size
					 * will be as large as needed to hold
					 * the key. */
	} key;				/* MUST BE LAST FIELD IN RECORD!! */
} HashEntry;

/*
 * Structure definition for a hash table.  Must be in tcl.h so clients
 * can allocate space for these structures, but clients should never
 * access any fields in this structure.
 */

#define HASH_SMALL_HASH_TABLE 4
typedef struct HashTable {
	HashEntry **buckets;		/* Pointer to bucket array.  Each
					 * element points to first entry in
					 * bucket's hash chain, or NULL. */
	HashEntry *staticBuckets[HASH_SMALL_HASH_TABLE];
	/* Bucket array used for small tables
	 * (to avoid mallocs and frees). */
	int numBuckets;			/* Total number of buckets allocated
					 * at **bucketPtr. */
	int numEntries;			/* Total number of entries present
					 * in table. */
	int rebuildSize;		/* Enlarge table when numEntries gets
					 * to be this large. */
	int downShift;			/* Shift count used in hashing
					 * function.  Designed to use high-
					 * order bits of randomized keys. */
	int mask;			/* Mask value used in hashing
					 * function. */
	int keyType;			/* Type of keys used in this table.
					 * It's either HASH_STRING_KEYS,
					 * HASH_ONE_WORD_KEYS, or an integer
					 * giving the number of ints that
                                         * is the size of the key.
					 */
	HashEntry *(*findProc)(struct HashTable *tablePtr,
			       const char *key);
	HashEntry *(*createProc)(struct HashTable *tablePtr,
				 const char *key, int *newPtr);
} HashTable;

/*
 * Structure definition for information used to keep track of searches
 * through hash tables:
 */

typedef struct HashSearch {
	HashTable *tablePtr;		/* Table being searched. */
	int nextIndex;			/* Index of next bucket to be
					 * enumerated after present one. */
	HashEntry *nextEntryPtr;	/* Next entry to be enumerated in the
					 * the current bucket. */
} HashSearch;

/*
 * Acceptable key types for hash tables:
 */

#define HASH_STRING_KEYS	0
#define HASH_ONE_WORD_KEYS	1

/*
 * Macros for clients to use to access fields of hash entries:
 */

#define GetHashValue(h) ((h)->clientData)
#define SetHashValue(h, value) ((h)->clientData = (HashClientData) (value))
#define GetHashKey(tablePtr, h) \
           ((char *) (((tablePtr)->keyType == HASH_ONE_WORD_KEYS) ? \
		      (h)->key.oneWordValue : (h)->key.string))

/*
 * Macros to use for clients to use to invoke find and create procedures
 * for hash tables:
 */

#define FindHashEntry(tablePtr, key) \
	(*((tablePtr)->findProc))(tablePtr, key)
#define CreateHashEntry(tablePtr, key, newPtr) \
	(*((tablePtr)->createProc))(tablePtr, key, newPtr)


#ifdef __cplusplus
extern "C" {
#endif
	void      DeleteHashEntry(HashEntry *entryPtr);
	void      DeleteHashTable(HashTable *tablePtr);
	HashEntry *FirstHashEntry(HashTable *tablePtr, HashSearch *searchPtr);
	char      *HashStats     (HashTable *tablePtr);
	void      InitHashTable  (HashTable *tablePtr, int keyType);
	HashEntry *NextHashEntry (HashSearch *searchPtr);

#ifdef __cplusplus
}
#endif


#endif /* MASH_HASH_H */
