# ui-canvas.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#  @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/applications/mbv2/ui-canvas.tcl,v 1.19 2002/02/03 04:21:59 lim Exp $


import WidgetClass DropDown Observer ScrolledWidget ScrolledListbox PaneManager


WidgetClass DropDown/CanvasList -superclass {DropDown Observer}


DropDown/CanvasList public build_widget { path } {
	$self next $path
	$self insert end none
}


DropDown/CanvasList public have_canvases { } {
	$self instvar list_
	if { [info exists list_] && [llength $list_] > 0 } { return 1 } \
			else { return 0 }
}


DropDown/CanvasList public insert_item { index value } {
	$self instvar list_ text_ index_

	set pageid [lindex $value 0]
	if { $pageid == "none" } {
		set string none
	} else {
		set string [lindex $value 1]
		if ![info exists list_] {
			# we must delete the "none" entry
			$self delete end
			$self configure -state normal
			unset text_(none)
		}
		lappend list_ $pageid
	}

	set text_($pageid) $string
	$self subwidget menu insert $index command -label $string \
			-command "[list $self] set_var [list $pageid]"

	# FIXME: assuming that we always insert at the end into this list
	# and never delete
	if { $pageid != "none" } {
		if [info exists index_(cnt)] {
			set index_($pageid) $index_(cnt)
			incr index_(cnt)
		} else {
			set index_($pageid) 0
			set index_(cnt) 1
		}
	}
}


DropDown/CanvasList public var_trace { args } {
	upvar #0 [$self set var_] global_var
	if { $global_var=="" } return

	$self instvar text_
	$self subwidget button configure -text $text_($global_var)
}


DropDown/CanvasList public prev_canvas { pageid } {
	$self instvar list_
	if { ![info exists list_] || $pageid == "none" } return
	upvar #0 [$self set var_] global_var

	set idx [lsearch $list_ $pageid]
	if { $idx==-1 } return
	if { $idx==0 } { set idx end } else { incr idx -1 }
	set global_var [lindex $list_ $idx]
}


DropDown/CanvasList public next_canvas { pageid } {
	$self instvar list_
	if { ![info exists list_] || $pageid == "none" } return
	upvar #0 [$self set var_] global_var

	set idx [lsearch $list_ $pageid]
	if { $idx==-1 } return
	incr idx
	if { $idx >= [llength $list_] } { set idx 0 }
	set global_var [lindex $list_ $idx]
}


DropDown/CanvasList public update_pageid { pageid text } {
	$self instvar text_ index_
	if [info exists index_($pageid)] {
		$self subwidget menu entryconfigure $index_($pageid) \
				-label $text

		set text_($pageid) $text
		upvar #0 [$self set var_] global_var
		if { $global_var==$pageid } {
			$self subwidget button configure -text $text
		}
	}
}


WidgetClass ScrolledListbox/CanvasList -superclass {ScrolledListbox Observer} \
		-default {
	{ .scrollbar both }
	{ .Scrollbar.width 10 }
	{ .Scrollbar.borderWidth 1 }
	{ .Scrollbar.highlightThickness 1 }
	{ *Label.borderWidth 1 }
	{ *Canvas.width  200 }
	{ *Canvas.height 125 }
}


ScrolledListbox/CanvasList public update_pageid { pageid text } {
	set w [$self info widget -id $pageid]
	if { $w != {} } {
		$w configure -value $text
	}
}


WidgetClass ScrolledListbox/MemberList -superclass ScrolledListbox \
		-default {
	{ .scrollbar both }
	{ .Scrollbar.width 10 }
	{ .Scrollbar.borderWidth 1 }
	{ .Scrollbar.highlightThickness 1 }
	{ *Radiobutton.borderWidth 1 }
	{ *Canvas.width  200 }
	{ *Canvas.height 125 }
	{ .itemClass MemberListItem }
}


WidgetClass MemberListItem -superclass ListLabelItem

MemberListItem instproc create_root_widget { path } {
	radiobutton $path -anchor w
	if { [option get $path padX Label]=="" } {
		$path configure -padx 1
	}
	if { [option get $path padY Label]=="" } {
		$path configure -pady 0
	}
}


WidgetClass MBv2CanvasMgr -superclass {ScrolledWidget Observable Observer} \
		-default {
	{ *Scrollbar.borderWidth 1 }
	{ *Scrollbar.width 10 }
	{ *Canvas.background white }
	{ *ScrolledListbox/CanvasList*Canvas*background WidgetDefault }
	{ *ScrolledListbox/MemberList*Canvas*background WidgetDefault }
	{ *Canvas.borderWidth 1 }
	{ *Canvas.highlightThickness 0 }
	{ *Canvas.relief sunken }
	{ *Canvas*Label*background  LemonChiffon }
	{ *Canvas*Label*foreground  Black }
	{ *Canvas*Label*relief      raised }
	{ *Canvas*Label*borderWidth 1 }
	{ *Canvas*Label*justify     center }
	{ *Canvas*Label*font        {Helvetica 10} }
}


MBv2CanvasMgr public destroy { } {
	$self instvar activity_
	foreach t [array names activity_ timers,*] {
		after cancel $activity_($t)
	}
	$self next
}


MBv2CanvasMgr public build_widget { path } {
	$self next $path

	frame $path.hscroll_frame
	frame $path.vscroll_frame

	lower $path.hscroll_frame
	lower $path.vscroll_frame

	button $path.left  -image MbIcons(ff-left) -bd 0 \
			-relief raised -highlightthickness 0 -activebackground\
			[WidgetClass widget_default -background] \
			-command "$self grow_scrollregion -1 0"
	button $path.right -image MbIcons(ff-right) -bd 0 \
			-relief raised -highlightthickness 0 -activebackground\
			[WidgetClass widget_default -background] \
			-command "$self grow_scrollregion 1 0"
	button $path.up    -image MbIcons(ff-up) -bd 0 \
			-relief raised -highlightthickness 0 -activebackground\
			[WidgetClass widget_default -background] \
			-command "$self grow_scrollregion 0 -1"
	button $path.down  -image MbIcons(ff-down) -bd 0 \
			-relief raised -highlightthickness 0 -activebackground\
			[WidgetClass widget_default -background] \
			-command "$self grow_scrollregion 0 1"

	pack $path.left    -in $path.hscroll_frame -side left -fill y
	pack $path.hscroll -in $path.hscroll_frame -side left -fill both \
			-expand 1
	pack $path.right   -in $path.hscroll_frame -side left -fill y

	pack $path.up      -in $path.vscroll_frame -side top -fill x
	pack $path.vscroll -in $path.vscroll_frame -side top -fill both \
			-expand 1
	pack $path.down    -in $path.vscroll_frame -side top -fill x

	pack $path.hscroll_frame -side bottom -fill x -before $path.dummy
	pack $path.vscroll_frame -side right  -fill y -in $path.dummy

	$self show_lists 0
}


# override the base class' config_scroll
MBv2CanvasMgr public config_scroll { option scroll } {
}


MBv2CanvasMgr public set_show_canvas_list { f } {
	$self set show_canvas_list_ $f
}


MBv2CanvasMgr public show_lists { {f {}} } {
	$self instvar canvas_list_ member_list_ list_pane_mgr_

	if { $f == {} } {
		set path [$self info path]
		if { [winfo exists $path.lists] && \
				[winfo ismapped $path.lists] } {
			return 1
		} else {
			return 0
		}
	}

	$self set_show_canvas_list $f
	$self set_show_member_list $f
	if $f {
		set path [$self info path]
		if ![winfo exists $path.lists] {
			set t [toplevel $path.lists]
			wm title $t "[wm title [winfo toplevel $path]]"
			wm geometry $t 230x370

			frame $t.frame
			$self create_member_list $t.memberlist
			$self create_canvas_list $t.canvaslist

			set list_pane_mgr_ [new PaneManager $t.memberlist \
					$t.canvaslist -orient vertical \
					-in $t.frame -percent 0.6]
			button $t.dismiss -text "Dismiss" -bd 1 -pady 1 \
					-font {Helvetica 10 bold} \
					-command "$self show_lists 0"
			pack $t.frame -fill both -expand 1
			pack $t.dismiss -anchor e
		}
		wm deiconify $path.lists
	} elseif [info exists canvas_list_] {
		wm withdraw [$self info path].lists
	}
}


MBv2CanvasMgr public create_canvas_list { widget } {
	$self instvar canvas_list_ canvas_
	frame $widget
	label $widget.label -text "Page list:" -anchor w
	set canvas_list_ [ScrolledListbox/CanvasList $widget.list]
	pack $widget.label -fill x
	pack $widget.list  -fill both -expand 1

	# add all existing canvases to the list
	foreach pageid [array names canvas_] {
		if { $pageid != "none" } {
			$canvas_list_ insert end [concat -id $pageid \
					[$self pageid2text $pageid]]
		}
	}

	# highlight the current canvas
	$self tkvar currentPageId_
	if { [info exists currentPageId_] && $currentPageId_ != "none" } {
		$canvas_list_ selection set -id $currentPageId_
	}

	# set up the bindings
	$canvas_list_ configure -browsecmd "$self select_canvas_list_item"
	$self attach_observer $canvas_list_

	bind $canvas_list_ <Destroy> "$self destroy_canvas_list"
}


MBv2CanvasMgr private destroy_canvas_list { } {
	$self instvar canvas_list_ list_pane_mgr_ activity_
	$self detach_observer $canvas_list_
	unset canvas_list_
	if [info exists list_pane_mgr_] {
		delete $list_pane_mgr_
	}

	foreach t [array names activity_ timers,pageid*] {
		after cancel $activity_($t)
		unset activity_($t)
	}
}


MBv2CanvasMgr private select_canvas_list_item { pageid } {
	$self instvar canvas_list_
	if { [llength [$canvas_list_ selection get]] == 0 } {
		# we are selecting the same guy again
		# i.e. deselecting it; don't allow that
		$canvas_list_ selection set -id $pageid
	} else {
		$self switch_canvas $pageid
	}
}


MBv2CanvasMgr public set_show_member_list { f } {
	$self set show_member_list_ $f
}


MBv2CanvasMgr public create_member_list { widget } {
	$self instvar member_list_ canvas_ srcnames_
	frame $widget
	label $widget.label -text "Member list:" -anchor w -pady 0
	frame $widget.title
	label $widget.title.f -text "Follow" -anchor w -font {Helvetica 10} \
			-pady 0
	label $widget.title.n -text "      Name" -anchor w  -pady 0 \
			-font {Helvetica 10}
	set member_list_ [ScrolledListbox/MemberList $widget.list \
			-browsecmd "$self select_member_list_item"]
	radiobutton $widget.follow_any -value any -variable [$self tkvarname \
				source_to_follow_] -anchor w -bd 1 \
				-text "Follow active source" \
				-font {Helvetica 10}
	radiobutton $widget.follow_none -value {} -variable [$self tkvarname \
				source_to_follow_] -anchor w -bd 1 \
				-text "Don't follow anyone" \
				-font {Helvetica 10}

	pack $widget.label -fill x
	pack $widget.follow_any $widget.follow_none -fill x
	pack $widget.title.f -side left -fill y
	pack $widget.title.n -side left -fill both -expand 1
	pack $widget.title -fill x
	pack $widget.list  -fill both -expand 1

	# add all existing sources to the list
	foreach srcid [array names srcnames_] {
		$member_list_ insert end [concat -id $srcid $srcnames_($srcid)]
		set w [$member_list_ info widget -id $srcid]
		$w widget_proc configure -variable [$self tkvarname \
				source_to_follow_] -value $srcid
	}

	bind $member_list_ <Destroy> "$self destroy_member_list"
}


MBv2CanvasMgr private destroy_member_list { } {
	$self instvar member_list_ activity_
	unset member_list_

	foreach t [array names activity_ timers,srcid*] {
		after cancel $activity_($t)
		unset activity_($t)
	}
}


MBv2CanvasMgr private select_member_list_item { srcid } {
	$self instvar member_list_
	$member_list_ selection clear
}


MBv2CanvasMgr public sender { s } {
	$self set sender_ $s
}


MBv2CanvasMgr public recv_only { f } {
	$self instvar recv_only_ tb_path_
	set recv_only_ $f
	if $f { set state normal } else { set state disabled }
	$tb_path_.new configure -state $state
}


MBv2CanvasMgr public create_main_widget { path } {
	$self instvar canv_cnt_ recv_only_ canvas_
	set recv_only_ 0
	set canv_cnt_ 0
	$self source_to_follow any
	#$self fix_to_view 0

	$self show_owner_when_drawn 1

	set c [$self create_widget $path.canvas$canv_cnt_]
	set canvas_(none) [new MBv2TkCanvas $c]
	return $c
}


MBv2CanvasMgr public get_canvas { pageid } {
	$self instvar canv_cnt_ canvas_
	if ![info exists canvas_($pageid)] {
		# we need to create an MBv2Canvas object

		set path [$self info path]
		if [info exists canvas_(none)] {
			# we already have the default canvas
			set canvas_($pageid) $canvas_(none)
			unset canvas_(none)

			# this is the first canvas
			# we should notify everyone of its creation and
			# switch to it
			$self new_canvas $pageid
			$self switch_canvas $pageid
		} else {
			incr canv_cnt_
			set canvpath [$self create_widget \
					$path.canvas$canv_cnt_]
			set canvas_($pageid) [new MBv2TkCanvas $canvpath]
			$self new_canvas $pageid

			# if we are following this source, we ought to
			# switch the canvas now
			set follow [$self source_to_follow]
			set srcid [lindex [split $pageid :] 0]
			if { $follow == "any" || $follow == "$srcid" } {
				$self switch_canvas $pageid
			}
		}
	}

	return $canvas_($pageid)
}


MBv2CanvasMgr private create_widget { path } {
	canvas $path
	bind $path <Enter> "focus $path"
	bind $path <Tab>   "break"

	bind $path <Control-v> "$self notify_observers control_v"
	bind $path <Control-V> "$self notify_observers control_v"

	bind $path <Control-z> "$self notify_observers undo"
	bind $path <Control-Z> "$self notify_observers redo"

	bind $path <ButtonPress-2> \
			"$self notify_observers buttonpress_2 $path %x %y"
	bind $path <ButtonPress-3> "$self notify_observers buttonpress_3"

	#bind $path <Configure> "$self canvas_size_changed %W %w %h"
	return $path
}


MBv2CanvasMgr private new_canvas { pageid } {
	$self instvar canvas_

	# notify any observers
	$self add_to_canvaslist $pageid
	$self notify_observers new_canvas $canvas_($pageid) $pageid
}


MBv2CanvasMgr public switch_canvas { pageid } {
	$self instvar canvas_ tb_path_ sender_ canvas_list_
	$self tkvar scale_

	if [info exists canvas_($pageid)] {
		# change the scale factor on this canvas if necessary
		set scale [string trimright $scale_ %]
		$canvas_($pageid) rescale [expr $scale / 100.0]

		$self replace_main_widget [$canvas_($pageid) path]

		if { $pageid != "none" } {
			# notify the sender object
			$sender_ switch_page -page $pageid

			# update the toolbar
			$self tkvar currentPageId_
			set currentPageId_ $pageid

			# if a canvas list exists, update it
			if [info exists canvas_list_] {
				$canvas_list_ selection set -id $pageid
			}
		}

		# notify any observers
		$self notify_observers switch_canvas $canvas_($pageid) \
				$pageid
	}
}


MBv2CanvasMgr private try_to_switch_canvas { args } {
	$self tkvar currentPageId_
	if { $currentPageId_ == "none" } return

	$self switch_canvas $currentPageId_
}


MBv2CanvasMgr private create_new_canvas { } {
	$self instvar sender_ recv_only_
	if $recv_only_ return

	# this will end up calling get_canvas which'll create the
	# actual canvas
	set pageid [$sender_ new_page]
	$self tkvar currentPageId_
	if { $currentPageId_ != $pageid } {
		$self switch_canvas $pageid
	}
}


MBv2CanvasMgr private add_to_canvaslist { pageid } {
	$self instvar tb_path_ canv_cnt_ canvas_list_
	$tb_path_.canvaslist insert end [list $pageid \
			[$self pageid2text $pageid]]
	if { [info exists canvas_list_] && [winfo exists $canvas_list_] } {
		$canvas_list_ insert end [concat -id $pageid \
				[$self pageid2text $pageid]]
	}

	# if we have more than 1 canvas
	# we should enable the prev/next buttons
	if { $canv_cnt_==1 } {
		# i.e. we have at least 2 canvases
		$tb_path_.prev configure -state normal
		$tb_path_.next configure -state normal
	}
}


MBv2CanvasMgr public source_name { srcid } {
	$self instvar srcnames_
	if [info exists srcnames_($srcid)] {
		set name $srcnames_($srcid)
	} else {
		set name unknown
	}
	return $name
}


MBv2CanvasMgr public pageid2text { pageid } {
	$self instvar srcnames_
	set split [split $pageid :]
	set srcid [lindex $split 0]
	if [info exists srcnames_($srcid)] {
		set name $srcnames_($srcid)
	} else {
		set name unknown
	}

	return "$name: [lindex $split 1]"
}


MBv2CanvasMgr public source_update { src name cname } {
	$self instvar srcnames_ canvas_ member_list_

	if { $name=={} } {
		if { $cname=={} } { set name unknown } else { set name $cname }
	}
	set srcid [$src source_id]
	set srcnames_($srcid) $name

	# update the member list if necessary
	if [info exists member_list_] {
		if [catch {set w [$member_list_ info widget -id $srcid]}] {
			$member_list_ insert end [concat -id $srcid $name]
			set w [$member_list_ info widget -id $srcid]
			$w widget_proc configure -variable [$self tkvarname \
					source_to_follow_] -value $srcid
		} else {
			$w configure -value $name
		}
	}

	# loop thru all the page id's and update the necessary ones
	foreach pageid [array names canvas_] {
		set split [split $pageid :]
		set sid [lindex $split 0]
		if { $sid == $srcid } {
			$self notify_observers update_pageid $pageid \
					"$name: [lindex $split 1]"
		}
	}
}


MBv2CanvasMgr public activity { srcid pageid cmdid canvid islocal } {
	$self instvar activity_ canvas_ srcnames_ show_owner_ canvas_list_ \
			show_canvas_list_ member_list_ show_member_list_ \
			list_pane_mgr_

	if [info exists canvas_($pageid)] {
		$self resize_scrollregion_later $canvas_($pageid)
	}

	# display the owner of the item that was just drawn
	$self tkvar currentPageId_
	if { $show_owner_ && !$islocal && \
			$currentPageId_ == $pageid && \
			[info exists canvas_($pageid)] } {
		if [info exists srcnames_($srcid)] {
			set name $srcnames_($srcid)
		} else { set name unknown }
		$canvas_($pageid) show_owner $srcid $name $canvid
	}

	# highlight the member list
	if { [info exists member_list_] && $show_member_list_ && \
			(![info exists list_pane_mgr_] || \
			[$list_pane_mgr_ percent] > 0.0) } {
		set w [$member_list_ info widget -id $srcid]
		if ![info exists activity_(timers,srcid:$srcid)] {
			$w configure -normalbackground white \
					-selectbackground white
		} else {
			after cancel $activity_(timers,srcid:$srcid)
		}

		set activity_(timers,srcid:$srcid) [after 500 \
				"$self unhilit_member_list_item $srcid $w"]
	}

	# highlight the canvas
	if { [info exists canvas_list_] && $show_canvas_list_ && \
			(![info exists list_pane_mgr_] || \
			[$list_pane_mgr_ percent] < 1.0) } {
		set w [$canvas_list_ info widget -id $pageid]
		if ![info exists activity_(timers,pageid:$pageid)] {
			$w configure -normalbackground white \
					-selectbackground white
		} else {
			after cancel $activity_(timers,pageid:$pageid)
		}

		set activity_(timers,pageid:$pageid) [after 500 \
				"$self unhilit_canvas_list_item $pageid $w"]
	}

	# if we are in followActive/followSrc mode, try switching to the
	# appropriate page
	set follow [$self source_to_follow]
	if { $pageid != {} && $pageid != $currentPageId_ && \
			($follow == "any" || $follow == $srcid) } {
		$self switch_canvas $pageid
	}
}


MBv2CanvasMgr private unhilit_canvas_list_item { pageid w } {
	$self instvar activity_
	unset activity_(timers,pageid:$pageid)
	$w configure -normalbackground [WidgetClass widget_default \
			-background] -selectbackground [WidgetClass \
			widget_default -selectbackground]
}


MBv2CanvasMgr private unhilit_member_list_item { srcid w } {
	$self instvar activity_
	unset activity_(timers,srcid:$srcid)
	$w configure -normalbackground [WidgetClass widget_default \
			-background] -selectbackground [WidgetClass \
			widget_default -selectbackground]
}


MBv2CanvasMgr public source_to_follow { args } {
	$self tkvar source_to_follow_
	if { [llength $args] == 0 } { return $source_to_follow_ }
	set source_to_follow_ [lindex $args 0]
}


MBv2CanvasMgr public resize_scrollregion_later { canv } {
	$self instvar activity_
	if [info exists activity_(timers,resize:$canv)] {
		after cancel $activity_(timers,resize:$canv)
	}

	set activity_(timers,resize:$canv) \
			[after 300 "$canv resize_scrollregion"]
}


MBv2CanvasMgr private grow_scrollregion { x y } {
	$self tkvar currentPageId_
	$self instvar canvas_
	if [info exists canvas_($currentPageId_)] {
		$canvas_($currentPageId_) grow_scrollregion $x $y
	}
}


MBv2CanvasMgr public build_toolbar { tb_path } {
	$self instvar tb_path_
	set tb_path_ $tb_path

	label $tb_path.separator -text "    "
	label $tb_path.canvaslist_label -text "Pages:"
	DropDown/CanvasList $tb_path.canvaslist -state disabled -variable \
			[$self tkvarname currentPageId_]
	TipManager tip $tb_path.canvaslist "Page list"
	$self attach_observer $tb_path.canvaslist
	$tb_path.canvaslist subwidget menu configure -tearoff 0
	$self tkvar currentPageId_
	trace variable currentPageId_ w "$self try_to_switch_canvas"

	button $tb_path.prev -image MbIcons(prev) -padx 1 -state disabled \
			-command "$tb_path.canvaslist prev_canvas \
			\[set [$self tkvarname currentPageId_]\]"
	TipManager tip $tb_path.prev "Previous\npage"
	button $tb_path.next -image MbIcons(next) -padx 1 -state disabled \
			-command "$tb_path.canvaslist next_canvas \
			\[set [$self tkvarname currentPageId_]\]"
	TipManager tip $tb_path.next "Next\npage"
	button $tb_path.new  -image MbIcons(new) \
			-command "$self create_new_canvas"
	TipManager tip $tb_path.new "New\npage"
	DropDown/Text $tb_path.scale -options { {entry.width 5} } \
			-variable [$self tkvarname scale_]
	TipManager tip $tb_path.scale "Change\nzoom"
	$self tkvar scale_
	set scale_ 100%
	trace variable scale_ w "$self switch_scale"

	$tb_path.scale insert end 50% 75% 100% 125% 150% 175% 200% \
			"fit width" "fit height" \
			"fit all"
	pack $tb_path.scale $tb_path.new $tb_path.next $tb_path.prev \
			$tb_path.canvaslist $tb_path.canvaslist_label \
			$tb_path.separator -side right -padx 1
}


# this isn't ready for prime time yet
# I'm not sure I wan't to bother...
MBv2CanvasMgr public fix_to_view { s } {
	$self set fix_to_view_ $s
	set t [winfo toplevel [$self info path]]
	if $s {
		if [winfo ismapped $t] {
			set tw [winfo width  $t]
			set th [winfo height $t]
			# FIXME
			if { [wm frame $t] != [winfo id $t] } {
				# there is an enclosing decorative window
				# FIXME:let's assume it adds 10 pixels to the
				# height
				incr th 10
			}
			wm aspect $t $tw $th $tw $th
		}
	} else {
		wm aspect $t {} {} {} {}
	}
}


MBv2CanvasMgr public toggle_fix_to_view { } {
	$self instvar fix_to_view_
	$self fix_to_view [expr !$fix_to_view_]
}


MBv2CanvasMgr public canvas_size_changed { path w h } {
	$self instvar fix_to_view_ last_widget_ last_widget_w_ last_widget_h_
	if { $fix_to_view_ && [info exists last_widget_] && \
			$last_widget_ == $path } {
		$self tkvar scale_
		if { $last_widget_w_ <= 0 } { set last_widget_w_ 1 }
		if { $last_widget_h_ <= 0 } { set last_widget_h_ 1 }
		if { $w <= 0 } { set w 1 }
		if { $h <= 0 } { set h 1 }

		set sw [expr double($w)/double($last_widget_w_)]
		set sh [expr double($h)/double($last_widget_h_)]
		set s  [string trimright $scale_ %]
		if { $sw < $sh } {
			set scale_ [expr int((double($s) * $sw) + 0.5)]
		} else {
			set scale_ [expr int((double($s) * $sh) + 0.5)]
		}
	} elseif $fix_to_view_ {
		# we must ensure that the aspect ratio remains fixed
		set t  [winfo toplevel $path]
		set tw [winfo width  $t]
		set th [winfo height $t]
		# FIXME
		if { [wm frame $t] != [winfo id $t] } {
			# there is an enclosing decorative window
			# FIXME:let's assume it adds 10 pixels to the height
			incr th 10
		}
		wm aspect $t $tw $th $tw $th
	}

	set last_widget_   $path
	set last_widget_w_ $w
	set last_widget_h_ $h
}


MBv2CanvasMgr public switch_scale { args } {
	$self instvar canvas_
	$self tkvar   currentPageId_ scale_

	if ![info exists canvas_($currentPageId_)] return
	set canv $canvas_($currentPageId_)

	set scale [string trim $scale_]
	set scale [string trimright $scale %]
	set scale [string trimright $scale]

	switch -exact -- $scale {
		"fit width" {
			set path [$canv path]
			set bbox [$path bbox all]
			if {$bbox == {}} {
				$canv rescale 1.0
				set scale 100
			} else {
				set width [expr [lindex $bbox 2] - \
						[lindex $bbox 0]]
				if {$width > 0} {
					# leave room for 10 pix of margin
					set scale [$canv rescale_to_fit \
							$width 0 \
							[expr [winfo width \
							$path]-10] 0]
				} else {
					$canv rescale 1.0
					set scale 100
				}
			}
		}

		"fit height" {
			set path [$canv path]
			set bbox [$path bbox all]
			if {$bbox == {}} {
				$canv rescale 1.0
				set scale 100
			} else {
				set height [expr [lindex $bbox 3] - \
						[lindex $bbox 1]]
				if {$height > 0} {
					# leave room for 10 pix of margin
					set scale [$canv rescale_to_fit \
							0 $height \
							0 [expr [winfo height \
							$path]-10]]
				} else {
					$canv rescale 1.0
					set scale 100
				}
			}
		}

		"fit all" {
			set path [$canv path]
			set bbox [$path bbox all]
			if {$bbox == {}} {
				$canv rescale 1.0
				set scale 100
			} else {
				set width [expr [lindex $bbox 2] - \
						[lindex $bbox 0]]
				set height [expr [lindex $bbox 3] - \
						[lindex $bbox 1]]
				if {$width > 0 || $height > 0} {
					# leave room for 10 pix of margin
					set scale [$canv rescale_to_fit \
							$width $height \
							[expr [winfo width \
							$path]-10] \
							[expr [winfo height \
							$path]-10]]
				} else {
					$canv rescale 1.0
					set scale 100
				}
			}
		}

		default {
			if { ![regexp {^[0-9]+$} $scale] } { set scale 100 }
			$canv rescale [expr $scale / 100.0]
		}
	}

	set scale_ "${scale}%"
	$self notify_observers switch_scale $scale
}


MBv2CanvasMgr public enable_session { f {s {}} } {
	$self instvar  canv_cnt_ tb_path_ recv_only_ session_

	set session_ $s
	set path [$self info path]
	if $f {
		set bg [option get $path.canvas0 background Canvas]
		set state normal
	} else {
		set bg gray50
		set state disabled
	}

	for { set i 0 } { $i <= $canv_cnt_ } { incr i } {
		$path.canvas$i configure -bg $bg
	}

	$tb_path_.scale configure -state $state
	if { $f && $recv_only_ } { set state disabled }
	$tb_path_.new configure -state $state

	if { $f && ![$tb_path_.canvaslist have_canvases] } {
		set state disabled
	}

	$tb_path_.prev configure -state $state
	$tb_path_.next configure -state $state
	$tb_path_.canvaslist configure -state $state
}


MBv2CanvasMgr public show_owner_when_drawn { {flag {}} } {
	$self instvar show_owner_
	if { $flag == {} } {return $show_owner_} else {set show_owner_ $flag}
}


# in order to allow apps (such as collaborator) that import MBv2 to work
# even if mbv2 is not compiled into the mash binary, we explicitly define
# a dummy class MBv2TkCanvas here, in case one doesn't already exist
if { [Class info instances MBv2TkCanvas] == "" } {
	Class MBv2TkCanvas
}


MBv2TkCanvas public init { path } {
	$self set path_ $path
	$self next $path
}


MBv2TkCanvas public destroy { } {
	$self instvar tips_
	foreach t [array names tips_ timer,*] {
		after cancel $tips_($t)
	}
	$self next
}


MBv2TkCanvas public path { } {
	return [$self set path_]
}


MBv2TkCanvas public add_bindtag { tag } {
	$self instvar path_
	set tags [bindtags $path_]
	if { [lsearch $tags $tag]==-1 } {
		bindtags $path_ [concat [list $tag] $tags]
	}
}


MBv2TkCanvas public remove_bindtag { tag } {
	$self instvar path_
	set tags [bindtags $path_]
	set idx [lsearch $tags $tag]
	if { $idx != -1 } {
		bindtags $path_ [lreplace $tags $idx $idx]
	}
}


MBv2TkCanvas public rescale_objects { abs_scale rel_scale {to_fit 0} } {
	$self instvar path_
	$path_ scale all 0.0 0.0 $rel_scale $rel_scale

	MBv2TkCanvas rescale_fonts $abs_scale

	# FIXME: must rescale images as well
	foreach item [$path_ find withtag all] {
		if { [$path_ type $item] == "image" } {
			$self rescale_image $abs_scale $item
		}
	}

	# we may need to reset the origin and resize the scrollregion
	if $to_fit {
		$self resize_scrollregion {} 1
	} else {
		set x [$path_ canvasx 0]
		set y [$path_ canvasy 0]
		if { $x != "0.0" } { set x [expr $x * $rel_scale] }
		if { $y != "0.0" } { set y [expr $y * $rel_scale] }

		set scr [$path_ cget -scrollregion]
		if { $scr != {} } {
			# extract the individual values from the $scr list
			foreach {sx1 sy1 sx2 sy2} $scr  { }
			set sx1 [expr $sx1 * $rel_scale]
			set sy1 [expr $sy1 * $rel_scale]
			set sx2 [expr $sx2 * $rel_scale]
			set sy2 [expr $sy2 * $rel_scale]
			$path_ configure -scrollregion \
					[list $sx1 $sy1 $sx2 $sy2]
			if { $sx2 > $sx1 } {
				$path_ xview moveto [expr ($x - $sx1) / \
						($sx2 - $sx1)]
			}
			if { $sy2 > $sy1 } {
				$path_ yview moveto [expr ($y - $sy1) / \
						($sy2 - $sy1)]
			}
		}

		$self resize_scrollregion
	}
}


MBv2TkCanvas public rescale_image { abs_scale item } {
	$self instvar path_ images_
	set image [$path_ itemcget $item -image]
	if ![info exists images_($item)] { set images_($item) $image }
	if { $images_($item) != $image } { image delete $image }
	if { $abs_scale == 1.0 } {
		$path_ itemconfigure $item -image $images_($item)
	} else {
		set image [image create photo]
		$self scale_image $images_($item) $image
		$path_ itemconfigure $item -image $image
	}
}


MBv2TkCanvas public resize_scrollregion { {margin {}} {to_fit 0} } {
	if { $margin == {} } { set margin 5 }
	$self instvar path_

	set w [winfo width $path_]
	set h [winfo height $path_]
	set x [$path_ canvasx 0]
	set y [$path_ canvasy 0]

	set bbox [$path_ bbox all]
	set scr  [$path_ cget -scrollregion]

	if { $scr == {} } {
		set sx1 [$path_ canvasx 0]
		set sy1 [$path_ canvasy 0]
		set sx2 [expr $sx1 + [winfo width  $path_]]
		set sy2 [expr $sy1 + [winfo height $path_]]
	} else {
		# extract the individual values from the $scr list
		foreach {sx1 sy1 sx2 sy2} $scr  { }
	}

	if { $bbox != {} } {
		foreach {bx1 by1 bx2 by2} $bbox { }
		set bx1 [expr $bx1 - $margin]
		set by1 [expr $by1 - $margin]
		set bx2 [expr $bx2 + $margin]
		set by2 [expr $by2 + $margin]

		if $to_fit {
			$path_ configure -scrollregion \
					[list $bx1 $by1 $bx2 $by2]
			$path_ xview moveto 0.0
			$path_ yview moveto 0.0
		} else {
			if {$bx1 < $sx1} { set sx1 $bx1 }
			if {$by1 < $sy1} { set sy1 $by1 }
			if {$bx2 > $sx2} { set sx2 $bx2 }
			if {$by2 > $sy2} { set sy2 $by2 }
			$path_ configure -scrollregion \
					[list $sx1 $sy1 $sx2 $sy2]
		}
	}
}


MBv2TkCanvas public grow_scrollregion { x y } {
	$self instvar path_

	set scr [$path_ cget -scrollregion]
	if { $scr == {} } {
		set sx1 [$path_ canvasx 0]
		set sy1 [$path_ canvasy 0]
		set sx2 [expr $sx1 + [winfo width  $path_]]
		set sy2 [expr $sy1 + [winfo height $path_]]
	} else {
		# extract the individual values from the $scr list
		foreach {sx1 sy1 sx2 sy2} $scr  { }
	}

	set xstep_size [expr [winfo width  $path_] / 4]
	set ystep_size [expr [winfo height $path_] / 4]
	if { $xstep_size < 25 } { set xstep_size 25 }
	if { $ystep_size < 25 } { set ystep_size 25 }

	if { $x < 0 } {
		set sx1 [expr $sx1 - $xstep_size]
		set xmove 0
	} elseif { $x > 0 } {
		set sx2 [expr $sx2 + $xstep_size]
		set xmove 1
	}

	if { $y < 0 } {
		set sy1 [expr $sy1 - $ystep_size]
		set ymove 0
	} elseif { $y > 0 } {
		set sy2 [expr $sy2 + $ystep_size]
		set ymove 1
	}

	$path_ configure -scrollregion [list $sx1 $sy1 $sx2 $sy2]
	if { $x != 0 } { $path_ xview moveto $xmove }
	if { $y != 0 } { $path_ yview moveto $ymove }
}


MBv2TkCanvas public show_owner { srcid srcname canvid {under_cursor 0} } {
	$self instvar tips_ path_
	if { $canvid == {} || [info exists tips_(label,$canvid)] } return

	set now [clock seconds]
	if { [info exists tips_(srcid,$srcid)] && $under_cursor == 0 } {
		# check how long it's been since the last time we
		# displayed a tip for this canvid

		set then [lindex $tips_(srcid,$srcid) 0]
		if { [expr $now - $then] < 1 } return
		$self hide_owner $srcid [lindex $tips_(srcid,$srcid) 1]
	}

	set tips_(srcid,$srcid) [list $now $canvid]
	set tips_(label,$canvid) [label $path_.owner$canvid -text $srcname]

	if $under_cursor {
		set rx [winfo rootx $path_]
		set wx [winfo pointerx $path_]
		set ry [winfo rooty $path_]
		set wy [winfo pointery $path_]
		set px [$path_ canvasx [expr {$wx - $rx + 5}]]
		set py [$path_ canvasy [expr {$wy - $ry + 10}]]
		set anchor nw
	} else {
		set bbox [$path_ bbox $canvid]
		set px [expr [lindex $bbox 0] - 5]
		set py [expr [lindex $bbox 1] - 15]

		set x1 [$path_ canvasx 0]
		set y1 [$path_ canvasy 0]
		set x2 [expr $x1 + [winfo width  $path_]]
		set y2 [expr $y1 + [winfo height $path_]]

		set anchor {}
		if { $px < $x1 } {
			set px $x1
			append anchor w
		} elseif { $px > $x2 } {
			set px $x2
			append anchor e
		}

		if { $py < $y1 } {
			set py $y1
			append anchor n
		} elseif { $py > $y2 } {
			set py $y2
			append anchor s
		}

		if { $anchor=={} } { set anchor nw }
	}

	set tips_(id,$canvid) [$path_ create window $px $py \
			-window $tips_(label,$canvid) -anchor $anchor]
	set tips_(timer,$canvid) [after 5000 "$self hide_owner $srcid $canvid"]
	$path_ bind $canvid <Leave> "$self hide_owner $srcid $canvid"
}


MBv2TkCanvas public hide_owner { srcid canvid } {
	$self instvar tips_ path_
	if ![info exists tips_(label,$canvid)] return

	if [info exists tips_(srcid,$srcid)] { unset tips_(srcid,$srcid) }
	catch {after cancel $tips_(timer,$canvid)}
	$path_ delete $tips_(id,$canvid)
	destroy $tips_(label,$canvid)
	catch {$path_ bind $canvid <Leave> {}}

	unset tips_(timer,$canvid)
	unset tips_(id,$canvid)
	unset tips_(label,$canvid)
}


MBv2TkCanvas public interpret_font { font scale } {
	MBv2TkCanvas instvar fonts_
	if [info exists fonts_(font,$font)] {
		return $fonts_(font,$font)
	}

	set family [lindex $font 0]
	set size   [lindex $font 1]

	# always use size in pixels i.e. set the size to a negative value
	if { ![regexp {^[-0-9][0-9]*$} $size] } { set size 12 } \
			elseif { $size >= 0 } {
		if { $size < 4 } { set size 4 }
		set size -$size
	} else {
		if { $size > -4 } { set size -4 }
	}

	set weight normal
	set slant  roman
	set underline  0
	set overstrike 0

	foreach style [lrange $font 2 end] {
		switch -exact -- $style {
			normal { set weight normal }
			bold   { set weight bold   }
			roman  { set slant  roman  }
			italic { set slant  italic }
			underline  { set underline  1 }
			overstrike { set overstrike 1 }
		}
	}

	set fonts_(size,$font) $size
	set size [expr int(($size * $scale) + 0.5)]
	set fonts_(font,$font) [font create -family $family -size $size \
			-weight $weight -slant $slant \
			-underline $underline -overstrike $overstrike]
	return $fonts_(font,$font)
}


MBv2TkCanvas proc.public rescale_fonts { scale } {
	$self instvar fonts_
	foreach f [array names fonts_ font,*] {
		set font [string range $f 5 end]
		set size [expr int(($fonts_(size,$font) * $scale) + 0.5)]
		font configure $fonts_(font,$font) -size $size
	}
}



# change the contents of the canvas to postscript
# <ul>
# <li><u>content</u>: if this is <tt>full</tt> the whole canvas is output
#     to postscript. Otherwise only the visible region is converted to
#     postscript.
# <li><u>orient</u>: if this is <tt>portrait</tt>, prints to portrait,
#     else landscape is assumed.
# <li><u>header</u>: additional page header to be included
# <li><u>args</u>: additional arguments to be passed down to the tkcanvas's
#     postscript command.
# <li> empty pages are ignored.
# </ul>
MBv2TkCanvas public to_postscript {file page_num header paper_geom orient \
		colormode eps} {
	$self instvar path_

	# overwrite default behavior
	# so that it prints whole canvas

	set args "-colormode $colormode"
	set hdrStart "%%Page: $page_num $page_num\ngsave\n"
	append hdrStart "%helv font size 10\n/Helvetica-Bold findfont 10 "
	append hdrStart "scalefont ISOEncode setfont\n"
	append hdrStart "%black\n0.000 0.000 0.000 setrgbcolor AdjustColor\n"
	set hdrEnd ") show\ngrestore\n"

        set paper_geom [split $paper_geom +]
        set geom [split [lindex $paper_geom 0] x]

	# pw, ph are in inches; hx, hy are in 1/72 of an inch
	set pw [lindex $geom 0]
	set ph [lindex $geom 1]
	set hx [lindex $paper_geom 1]
	set hy [lindex $paper_geom 2]
	set pw [expr $pw - (2 * $hx)]
	set ph [expr $ph - (2 * $hy)]
	set hx [expr int($hx * 72 + 0.5)]
	set hy [expr int($hy * 72 + 0.5)]

	if {$orient == "landscape"} {
		set tmp $pw
		set pw $ph
		set ph $tmp
		set hy -$hy
		lappend args -rotate 1
		append hdrStart "90 rotate\n"
		append hdrStart "$hx [expr $hy - int($ph * 72 + 0.5) - 4] \
				moveto\n("
	} else {
		append hdrStart "$hx [expr $hy - 4] moveto\n("
	}
	set header "${hdrStart}${header}${hdrEnd}"

	set size [$path_ bbox all]
	if {$size == {}} {
		# empty pages are ignored
		return
	}
	set x [lindex $size 0]
	set y [lindex $size 1]
	set w [expr {[lindex $size 2] - [lindex $size 0]}]
	set h [expr {[lindex $size 3] - [lindex $size 1]}]

	# use the more limiting dimension to scale, so that
	# we can always see the whole picture
	if {$w>0 && ($h/double($w) > $ph/double($pw))} {
		lappend args -pageheight [append ph i]
	} else {
		lappend args -pagewidth [append pw i]
	}
	lappend args -x $x -y $y -width $w -height $h

	if $eps { lappend args -channel $file }
	set postscript [eval $path_ postscript $args]

	if !$eps {
		# get rid of the prolog and the trailer
		set endprolog [string first "%%BeginSetup" $postscript]
		if { $endprolog == -1 } {
			set endprolog [string first "%%BeginPageSetup" \
					$postscript]
		}
		set starttrailer [string last  "%%Trailer" $postscript]

		set pagehdr    [string first "%%Page:" $postscript]
		set endpagehdr [string first "\n" [string range $postscript \
				$pagehdr end]]
		set endpagehdr [expr $pagehdr + $endpagehdr]

		puts "$endprolog $pagehdr $endpagehdr $starttrailer"
		puts -nonewline $file [string range $postscript $endprolog \
				[expr $pagehdr - 1]]$header[string range \
				$postscript $endpagehdr [expr $starttrailer-1]]
	}
}


# compares 2 mb page ids, src name first, then srcid, then page number
MBv2CanvasMgr public sort_cmp {p1 p2} {
	$self instvar srcnames_
	set t  [split $p1 :]
	set s1 [lindex $t 0]
	set n1 [lindex $t 1]

	set t  [split $p2 :]
	set s2 [lindex $t 0]
	set n2 [lindex $t 1]

	puts "$s1 $s2 $n1 $n2"
	if { [info exists srcnames_($s1)] && \
			[info exists srcnames_($s2)] } {
		set r [string compare $srcnames_($s1) $srcnames_($s2)]
	} else { set r 0 }

	if { $r == 0 } { set r [string compare $s1 $s2] }

	if { $r == 0 } { set r [expr $n1 - $n2] }
	return $r
}


# sorts page ids by host first, then page number
MBv2CanvasMgr public sorted_pagelist { range } {
	$self instvar canvas_
	$self tkvar currentPageId_

	if [info exists canvas_(none)]  { return "" }
	if { $range == "current" } {
		return [list $currentPageId_]
	} else {
		return [lsort -command "$self sort_cmp" [array names canvas_]]
	}
}


MBv2CanvasMgr private verify_file { filename } {
	if [file exists $filename] {
		if [file isdirectory $filename] {
			Dialog transient MessageBox -type ok -text \
					"Cannot open \"$filename\":\
					\nis a directory" -image Icons(warning)
			return 0
		}

		set result [Dialog transient MessageBox -type yesno -text \
				"\"$filename\" already exists.\
				\nDo you want to overwrite it?" \
				-image Icons(warning)]
		if { $result == "yes" } { return 1 } else { return 0 }
	}
	return 1
}


MBv2CanvasMgr public save_as_postscript { } {
	$self instvar save_as_ps_
	if ![info exists save_as_ps_] {
		set cnt 0
		while [winfo exists .psdlg$cnt] { incr cnt }
		set save_as_ps_ [PostscriptDialog .psdlg$cnt]
	}

	set filename [$save_as_ps_ invoke]
	if { $filename == "" } return

	$save_as_ps_ tkvar range_ geometry_ orient_ colormode_ eps_
	$self to_postscript $filename $range_ $geometry_ $orient_ $colormode_ \
			$eps_
}


MBv2CanvasMgr public to_postscript {filename range paper_geom orient colormode\
		eps} {
	$self instvar canvas_
	set time [gettimeofday ascii]
	set pagelist [$self sorted_pagelist $range]
	set numpages [llength $pagelist]
	if { $numpages == 0 } {
		Dialog transient MessageBox -type ok -text \
				"There are no pages to print!" \
				-image Icons(warning)
		return
	}

	# create the prolog file
	global tk_library tcl_platform
	set save_tk_library $tk_library
	if { $tcl_platform(platform) == "windows" } {
		set tk_library "C:/"
	} else {
		set tk_library /tmp
	}

	set prolog_name [file join $tk_library prolog.ps]
	if [catch {set prolog [open $prolog_name w 0666]} error] {
		Dialog transient MessageBox -type ok -text \
				"Cannot create file \"$prolog_name\".\
				\n$error" -image Icons(warning)
		set tk_library $save_tk_library
		return
	}

	if $eps {
		if [catch {puts $prolog [postscript_prolog]} error] {
			Dialog transient MessageBox -type ok -text \
					"Error writing to \"$prolog_name\".\
					\n$error" -image Icons(warning)
			set tk_library $save_tk_library
			catch {close $prolog}
			catch {file delete $prolog_name}
			return
		}
	}

	close $prolog

	if { $eps && ($numpages > 1) } {
		set root [file rootname  $filename]
		set ext  [file extension $filename]
		set name ${root}001$ext
	} else {
		set name $filename
	}

	# make sure there isn't already a file called "name"
	if ![$self verify_file $name] {
		set tk_library $save_tk_library
		catch {close $prolog}
		catch {file delete $prolog_name}
		return
	}

        if [catch {set f [open $name "w"]} error] {
		Dialog transient MessageBox -type ok -text \
				"Cannot open file \"$name\".\
				\n$error" -image Icons(warning)
		set tk_library $save_tk_library
		catch {close $prolog}
		catch {file delete $prolog_name}
		return
	}

	set p [lindex $pagelist 0]
        set i 1
        set confName [$self get_option conferenceName]
	if { $confName == "" } { set confName "MediaBoard v2" }
        set hdr "Page $i ---  \[[$self canvas_name \
			$p]\]     $confName     $time"

	if !$eps {
		# dump prolog and document header
		TkCanvPsHeader $f "$confName ($time)"
	}

        $canvas_($p) to_postscript $f $i $hdr $paper_geom $orient $colormode \
			$eps

        foreach p [lrange $pagelist 1 end] {
                incr i
		if $eps {
			# open a new file for each eps
			close $f
			set name [format "${root}%03d$ext" $i]

			# make sure there isn't already a file called "name"
			if ![$self verify_file $name] {
				set tk_library $save_tk_library
				catch {close $prolog}
				catch {file delete $prolog_name}
				return
			}

			if [catch {set f [open $name "w"]} error] {
				Dialog transient MessageBox -type ok -text \
						"Cannot open file \"$name\".\
						\n$error" -image Icons(warning)
				set tk_library $save_tk_library
				catch {close $prolog}
				catch {file delete $prolog_name}
				return
			}
		}

		set hdr "Page $i ---  \[[$self canvas_name \
				$p]\]     $confName     $time"
		$canvas_($p) to_postscript $f $i $hdr $paper_geom $orient \
				$colormode $eps
                if !$eps { puts $f "%%PageTrailer\n" }
	}

        if !$eps { puts $f "%%Trailer\n%%Pages:$i\nend\n%%EOF\n" }
	close $f

	if { $eps && ($numpages > 1) } {
		Dialog transient MessageBox -type ok -text \
				"Saved EPS for all pages to \
				[format "${root}001-%03d$ext" $numpages]"
	}

	set tk_library $save_tk_library
	catch {close $prolog}
	catch {file delete $prolog_name}

	if !$eps { exec ghostview $filename & }
}


MBv2CanvasMgr public canvas_name { pageid } {
	$self instvar srcnames_
	set pageid [split $pageid :]
	set srcid [lindex $pageid 0]
	if [info exists srcnames_($srcid)] { set name $srcnames_($srcid) } \
			else { set name "unknown ($srcid)" }
	return "$name: [lindex $pageid 1]"
}


WidgetClass PostscriptDialog -superclass Dialog -default {
	{ *Radiobutton*font WidgetDefault }
	{ *Radiobutton*borderWidth 1 }
	{ *ImageTextButton.borderWidth 1 }
	{ *DropDown*borderWidth 1 }
	{ *DropDown*font WidgetDefault }
	{ *Entry*borderWidth 1 }
	{ *LabeledWidget*font WidgetDefault }
	{ .title "Save as Postscript..." }
}


PostscriptDialog public build_widget { path } {
	frame $path.f1
	frame $path.f2

	frame $path.geometry -bd 2 -relief groove
	label $path.geometry.label -text "Paper size" -anchor w
	DropDown $path.geometry.size -variable [$self tkvarname paper_size_] \
			-options { {button.width 15} {button.anchor w} }
	$path.geometry.size insert end Letter
	$path.geometry.size insert end A4
	entry $path.geometry.xmargin -textvariable [$self tkvarname xmargin_]
	LabeledWidget $path.geometry.xmargin_label -label "Left/right margin"\
			-widget $path.geometry.xmargin
	entry $path.geometry.ymargin -textvariable [$self tkvarname ymargin_]
	LabeledWidget $path.geometry.ymargin_label -label "Top/bottom margin"\
			-widget $path.geometry.ymargin
	pack $path.geometry.label -side top -anchor nw -fill x
	pack $path.geometry.size -side left -anchor nw -padx 4
	pack $path.geometry.xmargin_label $path.geometry.ymargin_label \
			-side top -fill x -anchor nw -padx 3

	frame $path.orient -bd 2 -relief groove
	label $path.orient.label -text "Orientation" -anchor w
	radiobutton $path.orient.portrait -text "Portrait" -value portrait \
			-variable [$self tkvarname orient_] -anchor w
	radiobutton $path.orient.landscape -text "Landscape" -value landscape \
			-variable [$self tkvarname orient_] -anchor w
	pack $path.orient.label $path.orient.portrait $path.orient.landscape \
			-fill x -pady 0 -anchor nw

	frame $path.color -bd 2 -relief groove
	label $path.color.label -text "Color options" -anchor w
	radiobutton $path.color.color -text "Color" -value color \
			-variable [$self tkvarname colormode_] -anchor w
	radiobutton $path.color.gray -text "Gray-scale" -value gray \
			-variable [$self tkvarname colormode_] -anchor w
	radiobutton $path.color.mono -text "Black-and-white" -value mono \
			-variable [$self tkvarname colormode_] -anchor w
	pack $path.color.label $path.color.color $path.color.gray \
			$path.color.mono -fill x -pady 0 -anchor nw

	frame $path.range -bd 2 -relief groove
	label $path.range.label -text "Range" -anchor w
	radiobutton $path.range.all -text "All pages" -value all \
			-variable [$self tkvarname range_] -anchor w
	radiobutton $path.range.current -text "Current page only" -anchor w \
			-value current -variable [$self tkvarname range_]
	pack $path.range.label $path.range.all $path.range.current \
			-fill x -pady 0 -anchor nw

	frame $path.format -bd 2 -relief groove
	label $path.format.label -text "Format" -anchor w
	radiobutton $path.format.ps -text "Postscript" -value 0 \
			-variable [$self tkvarname eps_] -anchor w
	radiobutton $path.format.eps -text "Encapsulated Postscript" \
			-value 1 -variable [$self tkvarname eps_] -anchor w
	pack $path.format.label $path.format.ps $path.format.eps \
			-fill x -pady 0 -anchor nw

	$self tkvar orient_ colormode_ range_ eps_ paper_size_ \
			xmargin_ ymargin_
	set orient_ portrait
	set colormode_ color
	set range_ all
	set eps_ 0
	set paper_size_ Letter
	set xmargin_ 0.5
	set ymargin_ 0.5

	trace variable eps_ w "$self switch_eps"

	FileBox $path.filebox -filetypes \
			{ {Postscript .ps} {{Encapsulated Postscript} .eps} \
			{{All files} *} } \
			-command "$self save_ps; $self ignore_args"

	frame $path.buttonbox
	ImageTextButton $path.buttonbox.ok -underline 0 -text "Save" \
			-image Icons(check) -orient horizontal \
			-command "$self save_ps"
	ImageTextButton $path.buttonbox.cancel -image Icons(cross) \
			-orient horizontal -text "Cancel" -underline 0 \
			-command "$self cancel"
	bind $path <KeyPress-Escape> "$self cancel"

	pack $path.buttonbox.ok $path.buttonbox.cancel -side left -anchor e\
			-padx 5 -pady 2
	pack $path.buttonbox -side bottom -anchor e

	pack $path.filebox -side top -fill both -expand 1 -pady 2
	pack $path.orient $path.range -fill both -expand 1 -in $path.f1 \
			-side left -padx 3
	pack $path.color $path.format -fill both -expand 1 -in $path.f2 \
			-side left -padx 3

	pack $path.f1 $path.f2 -side top -fill x -pady 3
	pack $path.geometry -side top -fill x -pady 3 -padx 3
}


PostscriptDialog private is_double { value } {
	if { [regexp {^[0-9]+$} $value] || [regexp {^[0-9]+\.[0-9]*$} $value] \
			|| [regexp {^[0-9]*\.[0-9]+$} $value] } { return 1 } \
			else { return 0 }
}


PostscriptDialog public save_ps { } {
	set filebox [$self subwidget filebox]
	set dir  [$filebox cget -directory]
	set file [$filebox cget -filename]
	if  { $file=="" } return

	$self tkvar paper_size_ xmargin_ ymargin_ geometry_
	if ![$self is_double $xmargin_] { set xmargin_ 0.5 }
	if ![$self is_double $ymargin_] { set ymargin_ 0.5 }
	switch $paper_size_ {
		Letter { set geometry_ 8.5x11 }
		A4 { set geometry_ 8.2x11.7 }
	}

	append geometry_ "+${xmargin_}+${ymargin_}"
	set path [file join $dir $file]
	$self config -result $path
}


PostscriptDialog private switch_eps { args } {
	$self tkvar eps_
	if $eps_ {
		$self subwidget filebox configure -current_filetype \
				{{Encapsulated Postscript} .eps}
	} else {
		$self subwidget filebox configure -current_filetype \
				{Postscript .ps}
	}
}
