# timer.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#  @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/common/timer.tcl,v 1.15 2002/02/03 04:25:43 lim Exp $


# This is the base Timer class.
# An application can schedule a timer using the <i>msched</i> method. When
# the timer expires, the <i>timeout</i> method is invoked. Applications
# must subclass this method.
Class Timer

# This class defines a periodic timer; a new timer is scheduled as soon
# as the current one expires. Use the <i>start</i> method to initiate the
# periodic timer, and the <i>cancel</i> method to stop it.
Class Timer/Periodic -superclass Timer

# This is an abstract base class for periodic adaptive timers. Before
# scheduling the next timeout, it invokes the <i>adapt</i> method to
# recompute the timeout. Applications must redefine this method in the
# subclass
Class Timer/Adaptive -superclass Timer

# This class can be used to create timers for sending out network packets
# while maintaining a constant bandwidth utilization
Class Timer/Adaptive/ConstBW -superclass Timer/Adaptive


#
Timer public init {} {
	$self next
	$self randomize 0
	$self set randwt_ 1.0
}

#
Timer public destroy {} {
	$self cancel
	$self next
}

# Use this method to set/unset automatic randomization of the
# timer.
# <ul>
# <li><i>yesno</i> can take one of the following values: <i>yes</i>,
# <i>no</i>, 1, or 0. The default value is 1.
# <li><i>randwt</i> is the weight factor associated with the
# randomization component
Timer public randomize { {yesno 1} {randwt {}} } {
	if { $randwt!={} } {
		$self set randwt_ $randwt
	}
	if {$yesno=="yes"} {set yesno 1} elseif {$yesno=="no"} {set yesno 0}
	$self set randomize_ $yesno
}


# FIXME backward compat
Timer private sched { t } {
	$self msched $t
}


# This method schedules a timeout for <i>t</i> milliseconds. The timeout
# value may be randomized if the <i>randomize</i> method has been invoked
# on this object
Timer public msched { t } {
	$self instvar id_ randomize_ randwt_

	if [info exists id_] {
		puts stderr "warning: $self ([$self info class]):\
				overlapping timers"
	}

	if $randomize_ {
		# Random number in U[-0.5,0.5]
		set r [expr [random]/double(0x7fffffff)-0.5]
		set t [expr $t+$t*$r*$randwt_]
	}

	set t [expr int($t+0.5)]
	set id_ [after $t "$self do_timeout"]
}


# private method invoked every time a timeout occurs. This method invokes
# the <i>timeout</i> method which applications should redefine
Timer private do_timeout {} {
	$self instvar id_
	if ![info exists id_] {
		puts stderr "warning: $self ($class) no timer id_"
	} else {
		unset id_
	}
	$self timeout
}


# Returns a boolean flag that indicates whether a timer has currently
# been scheduled
Timer public is_sched { } {
	$self instvar id_
	return [info exists id_]
}


# Cancels the currently scheduled timer. If there is no timer scheduled,
# this method simply NOPs
Timer public cancel {} {
	$self instvar id_
	if [info exists id_] {
		after cancel $id_
		unset id_
	}
}


#
Timer/Periodic public init { {period 5000} } {
	$self next
	$self set period_ $period
}


# Use this method to start the periodic timer. The timer can be stopped
# using the <i>cancel</i> method. If the <i>period</i> argument is
# specified, the object sets its period to that value, otherwise it
# uses its last set value (possibly set by the init proc). You can invoke
# this method multiple times without invoking <i>cancel</i> in between.
# It will implicitly call <i>cancel</i> if required.
Timer/Periodic public start { {period {}} } {
	$self instvar period_
	if { $period!={} } { set period_ $period }

	if [$self is_sched] { $self cancel }
	$self msched $period_
}


#
Timer/Periodic instproc do_timeout {} {
	$self instvar period_
	$self next
	if { [info commands $self]=="" } return
	$self msched $period_
}


#
Timer/Adaptive public init { {interval 5000} } {
	$self next
	$self set interval_ $interval
}


# Use this method to start the periodic timer. The timer can be stopped
# using the <i>cancel</i> method. You can invoke
# this method multiple times without invoking <i>cancel</i> in between.
# It will implicitly call <i>cancel</i> if required.
Timer/Adaptive public start {} {
	$self instvar interval_

	if [$self is_sched] { $self cancel }
	set interval_ [$self adapt $interval_]
	$self msched [expr int($interval_+0.5)]
}


Timer/Adaptive public do_timeout {} {
	$self next
	if { [info commands $self]=="" } return
	$self instvar interval_
	set interval_ [$self adapt $interval_]
	$self msched [expr int($interval_+0.5)]

}


# applications must subclass this class, and redefine their own
#<i>adapt</i> method in the subclass
Timer/Adaptive private adapt {interval} {
	return $interval
}


#
Timer/Adaptive/ConstBW public init { bw {thresh {}} {size_gain {}} } {
	$self instvar size_gain_ avgsize_ nsrcs_ bw_ thresh_ interval_

	if { $size_gain!={} } {
		set size_gain_ $size_gain
	} else {
		set size_gain_ 0.125
	}

	set avgsize_ 28
	set nsrcs_ 0
	set bw_ $bw

	if { $thresh=={} } {
		# 500ms default threshold
		set thresh_ 500
	} else {
		set thresh_ $thresh
	}
	$self next $thresh_
}

# Call this method to set/retrieve the timer threshold
Timer/Adaptive/ConstBW public threshold { {thresh {}} } {
    $self instvar thresh_
    if {$thresh=={}} {
	return $thresh_
    } else {
	set thresh_ $thresh
    }
}

# Call this method to set/retrieve the timer bandwidth
Timer/Adaptive/ConstBW public bandwidth { {bw {}} } {
    $self instvar bw_
    if {$bw=={}} {
	return $bw_
    } else {
	set bw_ $bw
    }
}


# Call this method to provide a sample of the size of data that is being
# sent/received over the network
Timer/Adaptive/ConstBW public sample_size { size } {
	$self instvar avgsize_ size_gain_
	# FIXME Assume IPv4 header of 28 bytes.
	set avgsize_ [expr $avgsize_ + $size_gain_ * ($size + 28 - $avgsize_)]
}


# Call this method to update the number of sources in the session
Timer/Adaptive/ConstBW public update_nsrcs { nsrcs } {
	$self set nsrcs_ $nsrcs
}


# Call this method to retrieve the number of sources in the session
Timer/Adaptive/ConstBW public nsrcs { nsrcs } {
	return [$self set nsrcs_]
}


# Call this method to increment the number of sources in the session
Timer/Adaptive/ConstBW public incr_nsrcs { {incr 1} } {
        $self instvar nsrcs_
        incr nsrcs_ $incr
}


# This method recomputes the new timeout interval
Timer/Adaptive/ConstBW private adapt {interval} {
	$self instvar avgsize_ bw_ nsrcs_ thresh_

	set t [expr 1000 * ($nsrcs_ * $avgsize_ * 8) / $bw_]
	if { $t < $thresh_ } {
		return $thresh_
	} else {
		return $t
	}
}
