# send-cues.tcl --
#
#       An object for sending cues in a collaborative meeting.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#
# An object for sending cues in a collaborative meeting.
# The cues currently available are "raise hand", "speak
# louder", "agree", and "disagree". The idea is for a
# person to express these cues in a non-verbal manner.
#
# This class makes a gui for the cues; there is just
# one format now. It is a list of checkbuttons with the
# cues images next to them.
#
# It also sends the cues messages on the global
# coordination bus. The client of this class needs to
# pass in this bus object.
#
# See also: CuesReceiver
#
# Examples of usage can be found in vic.
#

import CheckButton

Class CuesSender

#
# Initialize the global coordination bus so the client
# can use this class. The client should use CoordBus_
# and the associated CBChannel/Site object. See the
# creation of buses in application-{vic,vat}.tcl for
# an example
#
CuesSender proc set_cb { cb } {
        # initialize global coordination bus
	CuesSender set glob_chan_ $cb
}

#
# Create a CuesSender object and the gui
# path: the gui is constructed using this path, the
#       client should pass in the name of a frame
# cname: the cname that represents the user
#
CuesSender instproc init { path cname } {
        # initialize some variables
	$self instvar cname_ cues_
        set cname_ $cname
	# FIXME change so client can specify
        set cues_ "hand ear yes no"

        # create the cues buttons
        $self instvar top_
        set top_ $path
        foreach c $cues_ {
	        $self set ${c}_b_ [$self create_cue $c]
	        pack $top_.${c} -side left
	}
}

#
# Create a cue checkbutton
#
CuesSender instproc create_cue { cue } {
        $self instvar top_
        set cb [new CheckButton $top_.${cue} \
		 -bitmap sm_${cue} -command "$self send $cue" \
		 -onvalue ${cue}_on -offvalue ${cue}_off \
 	         -width 20 -height 20 -anchor w ]
	$cb set-val ${cue}_off
	return $cb
}

CuesSender instproc destroy { } {
	$self next
}

#
# Send cue message whenever the user click on a
# cue button. If the checkbutton is clicked down,
# send a message that says this particular cue
# is one, otherwise, send a message that cancels
# this cue.
#
# (punt): because reliable multicast is probably
# too heavyweight for this, we just send the
# three times, hoping most participants can
# get it.
#
CuesSender instproc send { cue } {
        CuesSender instvar glob_chan_
        $self instvar cname_ ${cue}_b_ ${cue}_after_ ${cue}_ids_

        # do nothing is global coordination bus does not exist
        # or there is no cname
        if { ![info exists glob_chan_] || "$cname_" == "" } {
	        return
	}

	# start sending messages
	if { "[[$self set ${cue}_b_] get-val]" == "${cue}_on" } {
		# send the aware message three times
  	        set ${cue}_after_ 1
	        for { set i 0; set t 0 } \
		    { $i < 1 } \
		    { incr i; incr t 500 } {
		        set ${cue}_ids_($i) \
				[after $t "$glob_chan_ send AWARE_$cue $cname_" ]
                }

        } else {
	        # stop sending, if aware messages are queued up
	        # then cancel them
	        if [info exists ${cue}_after_] {
		        unset ${cue}_after_
		        foreach i [array names ${cue}_ids_] {
			        after cancel [set ${cue}_ids_($i)]
			}
	        }

		# send the cancel message three times
	        for { set i 0; set t 0 } \
		    { $i < 1 } \
		    { incr i; incr t 500 } {
			after $t "$glob_chan_ send UNAWARE_$cue $cname_"
	        }
        }
}
