# ui-dcvideo.tcl --
#
#       Stuff for providing the service-specific UI that accompanies
#       video sources when in the preview or broadcast pane.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import DcVideoWidget
import DragNDrop

Class CDcUIVideo

CDcUIVideo public init { uiMain winFrame source service} {
    # member variables
    $self instvar m_uiMain
    $self instvar m_winFrame

    $self instvar m_source
    $self instvar m_service
    $self instvar m_font

    # store away input
    set m_uiMain $uiMain
    set m_winFrame $winFrame

    set m_source $source
    set m_service $service

    set m_font [$self get_option smallfont]
}

CDcUIVideo public destroy { } {
    $self instvar m_source
    $self instvar m_service
    $self instvar m_vwVideo
    $self instvar m_winFrame

    # unmap the ui update message
    if { $m_service != 0 } {
	# unmap the old get_ui_control message
	$m_service UnmapMessage "CONTROL_UI" $self "ControlUI"
	$m_service UnmapMessage "UPDATE_CONTROL_UI" $self "UpdateControlUI"
	$m_service UnmapMessage "UPDATE_INFO" $self "UpdateInfo"
    }

    # detach the video
    $m_vwVideo detach-decoder $m_source

    # destroy the window
    destroy $m_winFrame.frame.video
    destroy $m_winFrame.frame.control
    destroy $m_winFrame.frame.info
    destroy $m_winFrame.frame

    $self next
}

CDcUIVideo private BuildUI {} {
    $self instvar m_uiMain
    $self instvar m_winFrame
    $self instvar m_source
    $self instvar m_font

    # set up the frame for the video and the controls
    frame $m_winFrame.frame -relief raised -borderwidth 3
    pack $m_winFrame.frame -side top

    frame $m_winFrame.frame.video
    pack $m_winFrame.frame.video -side top

    frame $m_winFrame.frame.control -relief ridge -borderwidth 2
    pack $m_winFrame.frame.control -side top -fill x

    # This needs to be created after the video has started since
    # it uses m_vwVideo.

    frame $m_winFrame.frame.info -relief ridge -borderwidth 2
    pack $m_winFrame.frame.info -side bottom -fill x

    # it will be filled in later by calling GET_INFO
    label $m_winFrame.frame.info.label -text "" -font $m_font -relief ridge

    checkbutton $m_winFrame.frame.info.slow$self -text "low refresh rate" \
	-font $m_font -indicatoron true -command "$self ToggleSlow"

    pack $m_winFrame.frame.info.label $m_winFrame.frame.info.slow$self -side left

    # start the video and controls
    $self StartVideo
    $self StartControl


}


CDcUIVideo public ToggleSlow { } {
    $self instvar m_vwVideo
    if {![info exists m_vwVideo]} {
	return
    }
    if {[$m_vwVideo is-slow]} {
	$m_vwVideo set_normal
    } else {
	$m_vwVideo set_slow
    }
}


CDcUIVideo public SwitchVideo { source } {
    $self instvar m_uiMain
    $self instvar m_source
    $self instvar m_vwVideo
    $self instvar m_winFrame

    # detach the video
    # $m_vwVideo detach-decoder $m_source
    $m_vwVideo detach-decoder $m_source

    # now attach the new video
    set m_source $source
    $m_vwVideo attach-decoder $m_source [$m_uiMain set m_ColorModel] 0

    #switch frame color
    set srcidtemp [$m_source srcid]
    set r [expr $srcidtemp % 4096]
    set srcidtemp [expr $srcidtemp / 7]
    set g [expr $srcidtemp % 4096]
    set srcidtemp [expr $srcidtemp / 11]
    set b [expr $srcidtemp % 4096]
    set color [format "#%03x%03x%03x" $r $g $b]
    $m_winFrame.frame configure -background $color

    #change the label
    set hostaddr [$m_source addr]
    set hostname [lookup_host_name $hostaddr]
    set hostname [string tolower $hostname]
    $m_winFrame.frame.info.label configure -text "$hostname"
}


CDcUIVideo public StartVideo { } {
    $self instvar m_winFrame
    $self instvar m_uiMain
    $self instvar m_source

    # set the color for the frame
    # FIXME - this is just weird and doesn't work all that great
    #  it would be better to have several distinct colors and keep
    #  track of which ones are available
    expr srand([$m_source srcid])
    set r [expr int(rand() * 4096)]
    set g [expr int(rand() * 4096)]
    set b [expr int(rand() * 4096)]
    set color [format "#%03x%03x%03x" $r $g $b]
    $m_winFrame.frame configure -background $color

    # create the new preview window
    $self instvar m_vwVideo

    set m_vwVideo [new DcVideoWidget $m_winFrame.frame.video.video 352 288]
    pack $m_winFrame.frame.video.video -side top -fill both -expand 1

    # attach it to the source
    $m_vwVideo attach-decoder $m_source [$m_uiMain set m_ColorModel] 0
}

CDcUIVideo public StartControl { } {
    $self instvar m_service
    $self instvar m_winFrame

    # FIXME - this doesn't seem to be supported anymore so comment it out
    #     it seems like a good idea though....

    # ok put in the main menu stuff, like remote window
    #menubutton $m_winFrame.frame.control.menubutton -text "Win" -menu $m_winFrame.frame.control.menubutton.menu -relief raised
    #pack $m_winFrame.frame.control.menubutton -side right -fill y

    #set winMenu [menu $m_winFrame.frame.control.menubutton.menu]
    #$winMenu add command -label "Remove" -command "$self RemoveSelf"

    # if there's no service then just return
    if { $m_service == 0 } {
	return
    }

    # otherwise try to get the service control ui

    # add the control window with nothing so update ui can destroy it
    frame $m_winFrame.frame.control.serviceFrame
    pack $m_winFrame.frame.control.serviceFrame -side right -fill y

    $self ControlUI $m_service 0
    $self InfoUI $m_service

    # add the final frame to fill up the rest of the space
    frame $m_winFrame.frame.control.filler -relief raised -borderwidth 2
    pack $m_winFrame.frame.control.filler -side left -expand 1 -fill both


    # register this object to get update messages
    $m_service MapMessage "CONTROL_UI" $self "ControlUI"
    $m_service MapMessage "UPDATE_CONTROL_UI" $self "UpdateControlUI"
    $m_service MapMessage "UPDATE_INFO" $self "UpdateInfo"
}

CDcUIVideo public SwitchService { service } {
    $self instvar m_service

    # clear out the control panel
    $self ControlUI $m_service ""

    # unmap the ui update message
    if { $m_service != 0 } {
	# unmap the old get_ui_control message
	$m_service UnmapMessage "CONTROL_UI" $self "ControlUI"
	$m_service UnmapMessage "UPDATE_CONTROL_UI" $self "UpdateControlUI"
	$m_service UnmapMessage "UPDATE_INFO" $self "UpdateInfo"
    }

    # map the new service
    set m_service $service

    # map the new service and get the latest ui
    if { $m_service != 0 } {
	$m_service MapMessage "CONTROL_UI" $self "ControlUI"
	$m_service MapMessage "UPDATE_CONTROL_UI" $self "UpdateControlUI"
	$m_service MapMessage "UPDATE_INFO" $self "UpdateInfo"
	$self ControlUI $m_service 0
	$self InfoUI $m_service
	$m_service Send "FIR"
    }
}

CDcUIVideo public InfoUI {service} {
    $self instvar m_service

    $m_service Send "GET_INFO"
}

CDcUIVideo public ControlUI { service uiCommands } {
    $self instvar m_winFrame
    $self instvar m_service

    # now check if the uiCommands is good
    # this function can be called either by the remote service
    # or local precedure call.  If it's local then the uiCommand
    # will be set to 0 otherwise it's assumed to be the correct
    # commands
    if { $uiCommands == 0 } {
	set uiCommands [$m_service Send "GET_UI_CONTROL" ""]
	return
    }

    # first destroy what we had before
    destroy $m_winFrame.frame.control.serviceFrame

    # the name winFrame is really important here
    # the commands that you get will assume that there
    # is a variable called winFrame to add control into
    # also it will assume that service has the service
    # object to send the control information to

    set service $m_service
    set winFrame [frame $m_winFrame.frame.control.serviceFrame]
    pack $m_winFrame.frame.control.serviceFrame -side right

    eval $uiCommands
}


##############################################################################
#
# CReplayProxyApplication private UpdateControlUI { service arguments }
#
# Input:
# service - the service that called this function
# arguments - the ui commands that will be evaluated to update the ui
#
# Output:
# none
#
# Description:
# it is called whenever we need to update the ui of the controls.
# the differnce from this func and Control UI is that, this doesn't
# destroy the window before the function is called.
#
##############################################################################
CDcUIVideo public UpdateControlUI { service uiCommands } {
    $self instvar m_winFrame
    $self instvar m_service

    # now check if the uiCommands is good
    # this function can be called either by the remote service
    # or local precedure call.  If it's local then the uiCommand
    # will be set to 0 otherwise it's assumed to be the correct
    # commands
    if { $uiCommands == 0 } {
	set uiCommands [$m_service Send "UPDATE_CONTROL_UI" ""]
	return
    }

    # the name winFrame is really important here
    # the commands that you get will assume that there
    # is a variable called winFrame to add control into
    # also it will assume that service has the service
    # object to send the control information to
    set service $m_service
    set winFrame $m_winFrame.frame.control.serviceFrame

    eval $uiCommands
}


CDcUIVideo public UpdateInfo { service newInfo } {
    $self instvar m_winFrame
    $self instvar m_service
    $self instvar m_source

    # FIXME - can also do [$m_source getid], [$m_source ssrc],
    #   [$m_source srcid] to get info
    set hostaddr [$m_source addr]
    set hostname [lookup_host_name $hostaddr]
    set hostname [string tolower $hostname]
    # FIXME - limit the length
    $m_winFrame.frame.info.label configure -text "$hostname: $newInfo"
}


#-----------------------------------------------------------------------------
#
# CDcUIVideo GetSource
# CDcUIVideo GetVideoWidget
#
# Description:
#   Return the source, and the videowidget attached to this object respectively.
#
#-----------------------------------------------------------------------------
CDcUIVideo public GetSource { } {
    $self instvar m_source
    return $m_source
}


CDcUIVideo public GetVideoWidget { } {
    $self instvar m_vwVideo
    return $m_vwVideo 
}
