# application-room405.tcl --
#
#       SDS service to provide control of room 405 lights
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import RTPApplication
import CSdsService
import CServiceManager
import CService
import DpClient

#
# Code for CRnSwitcherApplication
#

Class CRoom405Application -superclass RTPApplication

CRoom405Application instproc InitArgs { options } {
    # for the service discovery service
    $options register_option -sa optServiceAddress
    $options register_option -sp optServicePort
    $options register_option -st optServiceTTL

    # for the service discovery service
    $options register_option -rp optComPort

    # for the amx info
    $options register_option -amxaddr optAMXAddress
    $options register_option -amxport optAMXPort
}

CRoom405Application instproc InitResources { options } {
    # for the service discovery service
    $options add_default optServiceAddress "224.4.6.8"
    $options add_default optServicePort "12344"
    $options add_default optServiceTTL "16"

    # for the service discovery service
    $options add_default optComPort "11410"

    # for the amx
    $options add_default optAMXAddress "htsr.cs.berkeley.edu"
    $options add_default optAMXPort "6901"
}

CRoom405Application instproc InitAMX { } {
    $self instvar m_dpAMX
    $self instvar m_szAMXState
    set options [$self options]

    set inetAMXAddr [$options get_option optAMXAddress]
    set iAMXPort [$options get_option optAMXPort]

    set m_dpAMX [new DpClient $inetAMXAddr $iAMXPort]
}

CRoom405Application instproc init { argv } {
    $self next Room405

    # set some member variables

    # Initiailization of variables and resources.
    set options [$self options]
    $self InitArgs $options
    $self InitResources $options
    $options load_preferences "rtp rvc"
    set argv [$options parse_args $argv]

    # setup the amx stuff if we have to
    $self InitAMX

    # create the sds session object
    set inetServiceAddr [$options get_option optServiceAddress]
    set iServicePort [$options get_option optServicePort]
    set iServiceTTL [$options get_option optServiceTTL]

    $self instvar m_sdsService
    set m_sdsService [new CSdsService $self $inetServiceAddr \
	    $iServicePort $iServicePort $iServiceTTL]

    # setup the service objects
    $self instvar m_serviceManager
    $self instvar m_service

    set iPort [$options get_option optComPort]
    set m_serviceManager [new CServiceManager "ServiceApp" "Service" $iPort]
    $m_serviceManager Attach $self

    set m_service ""
}

##############################################################################
#
# CRoom405Application instproc NewConnection { service } {
#
# Input:
# service - the new service that got connected
#
# Output:
# 1 - if handling this service
# 0 - otherwise
#
# Description:
# A call back function for the service manager.  This function will be called
# when a new connection has just been noticed by the service manager
#
##############################################################################
CRoom405Application instproc NewConnection { service } {
    $self instvar m_service

    # check that we don't already have a connection that we're using
    if { $m_service != "" } {
	$service CloseLink
	return 1
    }

    set m_service $service

    # Add the service calls
    $m_service MapMessage "SYN_SERVICE_IMAGE" $self SynServiceImage
    $m_service MapMessage "GET_UI_WINDOW" $self GetUIWindow
    $m_service MapMessage "AMX_SWITCH" $self AMXSwitch
    $m_service MapMessage "CLOSE_LINK" $self CloseService

    return 1
}

##############################################################################
#
# CRoom405Application instproc CloseService { service arguments }
#
# Input:
# service - the service object that called this function
# arguments - the arguments that are passed to this funct from remote caller
#
# Output:
# none
#
# Description:
# Need to close the service and allow other connections to be made
#
##############################################################################
CRoom405Application instproc CloseService { service arguments } {
    $self instvar m_service
    set m_service ""

    exit
}


##############################################################################
#
# CRoom405Application public SynServiceImage { service arguments } {
#
# Input:
# service - the service that called this function
# arguments - the arguments associated with this call
#
# Output:
# none
#
# Description:
# This is called by the remote service client when it wants the image or text
# to display the service.  The simple protocol goes as such:
# type type_dependent_arguments.  The type could be text and it's argument
# the ascii of title.  It could also be image and be a bitmap of the image
#
##############################################################################
CRoom405Application public SynServiceImage { service arguments } {
    $service Send "SYN_SERVICE_IMAGE" "text Room405"
}


##############################################################################
#
# CRoom405Application public GetUIWindow { service arguments }
#
# Input:
# service - the service that called this function
# arguments - the arguments associated with this call
#
# Output:
# none
#
# Description:
# This is called by the remote service client when it wants the code
# to make the window.  It should send back the code to build the window.
# There are two things that are assumed to be set by the client.  One is
# winFrame the window to draw into.  Another is service, which is the
# m_service object to send back to yourself.
#
##############################################################################
CRoom405Application public GetUIWindow { service arguments } {
    $self instvar m_szAMXState

    set cmd ""

    append cmd "regsub -all -- {\\\.} \$winFrame {_} __name \n"

    # create the frame for the lights
    append cmd "frame \$winFrame.lightFrame -relief raised \
	    -borderwidth 2 \n"
    append cmd "pack \$winFrame.lightFrame -side top -fill both -expand 1 \n"

    # the label for the light
    append cmd "label \$winFrame.lightFrame.title -text \"Lights\" \n"
    append cmd "pack \$winFrame.lightFrame.title -side top \n"

    append cmd "global iLightState\$__name \n"
    append cmd "set iLightState\$__name \"\" \n"

    # create the radio buttons
    set lszLightItem "{Full On} 9 Overlay 43 Slide 44 {Full Off} 55"
    foreach {szLightName szLightVal} $lszLightItem {
	append cmd "frame \$winFrame.lightFrame.$szLightVal \n"
	append cmd "pack \$winFrame.lightFrame.$szLightVal \
		-side top -fill x -expand 1\n"
	append cmd "radiobutton \$winFrame.lightFrame.$szLightVal.radio \
		-command \"\$service Send AMX_SWITCH \\\"5 $szLightVal\\\"\" \
		-text \"$szLightName\" -value $szLightVal \
		-variable iLightState\$__name \n"
	append cmd "pack \$winFrame.lightFrame.$szLightVal.radio -side left \n"
    }

    # now put the close button in
    append cmd "frame \$winFrame.command \n"
    append cmd "pack \$winFrame.command -side top -fill x \n"

    append cmd "button \$winFrame.command.closeButton -text Close \
	    -command \"destroy \$winFrame\" \n"
    append cmd "pack \$winFrame.command.closeButton -side right \n"

    append cmd "label \$winFrame.command.title -relief raised \
	    -text \"Room405\" \n"
    append cmd "pack \$winFrame.command.title -side left -fill both -expand 1\n"


    $service Send "GET_UI_WINDOW" $cmd
}


CRoom405Application public AMXSwitch { service target } {
    $self instvar m_dpAMX

    set device [lindex $target 0]
    set mode [lindex $target 1]

    switch -exact -- $device {
	5 {
	    # lights device
	    if { [lsearch -exact {9 43 44 45 55} $mode] != -1 } {
		$m_dpAMX do send-AMX-command 1 5 $mode
		after 400
		$m_dpAMX do send-AMX-command 2 5 $mode
	    }
	}
    }
}

##############################################################################
#
# CRoom405Application public GetSdsServiceData { } {
#
# Input:
# none
#
# Output:
# the data that the will go out to the sds system
#
# Description:
# This is a callback function for the service discovery system.  Need to
# return the data that will go out to the sds system.  So far there are
# three fields with their values
#
##############################################################################
CRoom405Application public GetSdsServiceData { } {
    $self instvar m_lConfig

    set options [$self options]
    set iComPort [$options get_option optComPort]
    set hostname [exec hostname]

    set location "Berkeley Soda 405"
    set type "405 Room Control"

    set data [list "UPDATE:" [concat $hostname $iComPort] \
	    [list [list "LOCATION:" $location] \
	    	  [list "TYPE:" $type]]]

    return $data
}

