# ui-canvas.tcl --
#
#       This is the class which MediaBoard uses to draw items on
#
# Copyright (c) 1993-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


# Class MBCanvas
# This is the class which MediaBoard uses to draw items on
# It accepts all commands that tkCanvas understand and more.
#

# creates the underlying canvas canvas <br>
# <u>parent</u>: parent widget path
# <u>ops</u>: additional parameters for the underlying tkcanvas
#
MBCanvas public create_canvas {parent ops} {
	$self instvar path_ hilitC_

        $self add_default canvHighlighColor blue
	$self add_default showOwnerTip 1

	set path_ [eval canvas $parent.$self $ops -closeenough 2 \
			-confine 0 \
			-bd 0 -highlightthickness 0 -relief flat \
			-highlightcolor grey]

	$self setpath $path_
	# sets transparent GIF's to be white
	WidgetClass transparent_gif [$path_ cget -bg]

	mtrace trcVerbose "created canvas $path_"

	set hilitC_ [$self get_option canvHighlightColor]
	if {$hilitC_ == {}} {
		set hilitC_ blue
	}
	$self set ownerTip_ [$self get_option showOwnerTip]
	$self set after_id_ 0
	$self set omittedSrc_ {}
	$self set zoomPolicy_ "fix to view"
	bind $path_ <Configure> [list $self reconfig %w %h]
}

# tells canvas to omit showing owner banners for src
# <i>the current implementation can only omit one (i.e. the last) source
#
MBCanvas public omitShowOwner {src} {
	# can only omit one source for now
	$self set omittedSrc_ $src
}

# if <u>show</u> is 1, turn owner banners on, else off
#
MBCanvas instproc showOwner {show} {
	$self set ownerTip_ $show
}

# retrieves the underlying tkcanvas. <br>
# this function is here for efficiency reason, use of it should be restricted
#
MBCanvas public get_win {} {
	return [$self set path_]
}

# hides the marker for the current item
#
MBCanvas public unhilit {} {
 	$self instvar path_ marker_
	if ![info exists marker_] { return }

	$path_ itemconfig marker -outline {}

	# FIXME: Canvas bug: rectangles with no outline and fill should not
	#      be in the outline calculation!
	#
	# use the bbox another item on the canvas so that the marker
	# will not interfere with the actual bounding box of the canvas
	#
#puts "in unhilit"
	set tags [$path_ gettags current]
	set item [$path_ find withtag current]
	if {$tags == {}} {
		set item [$path_ find closest 0 0]
		if {$item == {}} {
			return
		}
		set tags [$path_ gettags $item]
	}
#puts "current item is '$item', tags is '$tags'"
	while {-1 != [lsearch -exact $tags ignore]} {
		set item [$path_ find above $item]
#uts "nextItem is '$item'"
		if {$item == {}} {
			return
		}
		set tags [$path_ gettags $item]
#puts "tags in $tags"
	}
	# since bbox's usu over estimate,
 	eval $path_ coords marker [$path_ bbox $item]
}

# highlights an item <br>
# if <u>itemId</u> is not null, the current item under the cursor is hilited
#
MBCanvas public hilit { {itemId {}} } {
	$self instvar marker_ path_ hilitC_
	if {$itemId == {}} {
		set tags [$path_ gettags current]
		if {-1 != [lsearch -exact $tags ignore]} {
			return
		}
		set currId [$path_ find withtag current]
		set coords [$path_ bbox $currId]
	} else {
		set coords [$path_ bbox $itemId]
		set currId $itemId
	}
	if {$coords!={}} {
		if ![info exists marker_] {
			set marker_ [$path_ create rect 1 1 1 1 \
					-tags {marker ignore} \
					-width 1 -outline {}]
		}
		$path_ coords $marker_ \
			    [expr {[lindex $coords 0] - 2}] \
			    [expr {[lindex $coords 1] - 2}] \
			    [expr {[lindex $coords 2] + 2}] \
			    [expr {[lindex $coords 3] + 2}]
		$path_ itemconfigure $marker_ -outline $hilitC_
		$path_ raise $marker_
	}
}

# specify whether to display the owner tip under the cursor
# (on if <u>tipping</u> is 1, off if it is zero)
#
MBCanvas public enable_tip {tipping} {
	$self instvar ownerTip_ path_ arAfterId_ tip_id_
	set ownerTip_ $tipping
	if {$ownerTip_} {
		set tip_id_ 0
		$path_ bind all <Enter> "$self sched_tip 0 1"
	} else {
		foreach i [array names arAfterId_] {
			after cancel $arAfterId_($i)
			set arAfterId_($i) 0
		}
		$path_ bind all <Enter> {}
		$path_ bind all <Leave> {}
	}
}

# schedules tip for item <u>id</u> <br>
# if <u>atpoint</u> is 1, displays the tip under the cursor, otherwise
# it is displayed near the corner of the item's bounding box
#
MBCanvas private sched_tip {id atpoint} {
	$self instvar tip_id_
	after cancel $tip_id_
	set tip_id_ [after 1000 "$self show_owner $id $atpoint"]
}

# displays the owner banner for <u>id</u>. <br>
# if <u>atpoint</u> is 1, displays the tip under the cursor, otherwise
# it is displayed near the corner of the item's bounding box
#
MBCanvas private show_owner {id atpoint} {
	$self instvar path_ arLabelw_ arLabel_ arAfterId_ tip_id_ omittedSrc_
	if {$id == 0} {
		set id [$path_ find withtag current]
		if {$id == {}} return
		set tags [$path_ gettags $id]
		# don't add tip for local items
#		if {-1 != [lsearch -exact $tags local]} {
#			return
#		}
#uts "newid is $id"
	}
	set owner [$self owner $id]
	if {$owner == {}} {
#		puts stderr "cannot find owner for $id"
		return
	}
	if {$owner == $omittedSrc_} {
		return
	}
#uts "$owner is $owner"
	if $atpoint {
		set rx [winfo rootx $path_]
		set wx [winfo pointerx $path_]
		set ry [winfo rooty $path_]
		set wy [winfo pointery $path_]
		set px [$path_ canvasx [expr {$wx - $rx + 5}]]
		set py [$path_ canvasy [expr {$wy - $ry + 10}]]
		set anchor nw
	} else {
		set bbox [$path_ bbox $id]
		set result [$self clipxy [expr {[lindex $bbox 0] - 5}] \
				[expr {[lindex $bbox 1] - 5}]]
		set px [lindex $result 0]
		set py [lindex $result 1]
		set anchor [lindex $result 2]
	}
	if {![info exists arLabel_($owner)]} {
		set arLabel_($owner) [label .l$self$owner \
				-font [$self get_option smallfont] \
				-bg beige -relief raised -text [$owner cname]]
	} else {
		$arLabel_($owner) configure -text [$owner cname]
	}
	if {![info exists arLabelw_($owner)]} {
		set arLabelw_($owner) [$path_ create window $px $py \
				-anchor nw]
		set arAfterId_($owner) {}
	}
	$path_ itemconfigure $arLabelw_($owner) -window $arLabel_($owner) \
			-anchor $anchor
	$path_ coord $arLabelw_($owner) $px $py
	after cancel $arAfterId_($owner)
	if {$atpoint} {
		$path_ bind $id <Leave> "$self hide_owner $owner"
		set arAfterId_($owner) [after 5000 "$self hide_owner $owner"]
	} else {
		set arAfterId_($owner) [after 5000 "$self hide_owner $owner"]
	}
}

# hide the owner banner for <u>owner</u>. <u>owner</u> is a source object
#
MBCanvas private hide_owner {owner} {
#	puts "in hide owner $owner"
	$self instvar arLabelw_ arLabel_ arAfterId_ path_
	after cancel $arAfterId_($owner)
	set arAfterId_($owner) 0
	if [info exists arLabel_($owner)] {
		destroy $arLabel_($owner)
		unset arLabel_($owner)
	}
}

# resets the state of the canvas, including standard bindings, focus,
# hiliting of markers etc.
#
MBCanvas public resetBindings {} {
	$self instvar path_

	$path_ bind local <Enter> {}
	bind $path_ <Button-1> {}
	bind $path_ <B1-Motion> {}
	bind $path_ <ButtonRelease-1> {}
	$path_ focus {}
	$self unhilit
	$self enable_tip [$self set ownerTip_]
}

# updates the current marker position, this is called e.g. when an item
# gets deleted, so that the marker won't be over a non-existing item
#
MBCanvas public resetMarker {} {
	$self instvar path_

	set tags [$path_ gettags current]
	if {-1 != [lsearch -exact $tags local]} {
		$self hilit
	} else {
		$self unhilit
	}
}

# turns on showing of marker over the current item
#
MBCanvas public setHilit {} {
	$self instvar path_ marker_

	$path_ bind local <Enter> "$self hilit"
	if [info exists marker_] {
		$path_ bind $marker_ <Enter> {}
	}
}

# returns if <u>itemid</u> overlaps the region {x1 y1 x2 y2}
#
MBCanvas private overlap {itemid x1 y1 x2 y2} {
	# this is a slow algoritshm, we could use the item's bbox to check
	# as well, but that may not be accurate (e.g. wrt filled/unfilled rects)
	set overlap [[$self set path_] find overlapping $x1 $y1 $x2 $y2]
	return [expr {([lsearch -exact $overlap $itemid]==-1) ? 0 : 1}]
}

# change the contents of the canvas to postscript
# <ul>
# <li><u>content</u>: if this is <tt>full</tt> the whole canvas is output
#     to postscript. Otherwise only the visible region is converted to
#     postscript.
# <li><u>orient</u>: if this is <tt>portrait</tt>, prints to portrait,
#     else landscape is assumed.
# <li><u>header</u>: additional page header to be included
# <li><u>args</u>: additional arguments to be passed down to the tkcanvas's
#     postscript command.
# <li> empty pages are ignored.
# </ul>
MBCanvas public to_ps {content orient header args} {
	$self instvar path_

	# overwrite default behavior
	# so that it prints whole canvas

	set hdrStart "gsave
%helv font size 10
/Helvetica-Bold findfont 10 scalefont ISOEncode setfont
%black
0.000 0.000 0.000 setrgbcolor AdjustColor\n"
	set hdrEnd ") show\ngrestore\n"
	if {$orient == "portrait"} {
		# inches
		set ph 10
		set pw 7.5
		# 1/72 of an inch
		set hx 27
		set hy 27
	} else {
		set pw 10
		set ph 7.5
		set hx 27
		set hy -27
		lappend args -rotate 1
		append hdrStart "90 rotate\n"
	}
	append hdrStart "$hx $hy moveto\n("
	set str $hdrStart
	append str $header
	append str $hdrEnd
	lappend args -pageheader $str

	if {$content == "full"} {
		set size [$path_ bbox all]
		if {$size == {}} {
			# empty pages are ignored
			return
		}
		set x [lindex $size 0]
		set y [lindex $size 1]
		set w [expr {[lindex $size 2] - [lindex $size 0]}]
		set h [expr {[lindex $size 3] - [lindex $size 1]}
		]
	} else {
		set x [$path_ canvasx 0]
		set y [$path_ canvasy 0]
		set w [$path_ canvasx [winfo width $path_]]
		set h [$path_ canvasy [winfo height $path_]]
	}
	# use the more limiting dimension to scale, so that
	# we can always see the whole picture
	if {$w>0 && ($h/double($w) > $ph/double($pw))} {
		lappend args -pageheight [append ph i]
	} else {
		lappend args -pagewidth [append pw i]
	}
	lappend args -x $x -y $y -width $w -height $h

	eval $path_ postscript $args
}

# returns the canvas x-y coordinate for the pointer
MBCanvas public pointerxy {} {
	$self instvar path_
	set rx [winfo rootx $path_]
	set wx [winfo pointerx $path_]
	set ry [winfo rooty $path_]
	set wy [winfo pointery $path_]

	if {($wx == -1) || ($wy == -1)} {
		return {}
	}
	return [$self canvasxy [expr {$wx - $rx}] [expr {$wy - $ry}]]
}

# if newRegion is larger than the scrollregion, update it
# <u>inc</u> is the amount of increase for the region (all around)
# $newRegion is a list of coordinates {x0 y0 x1 y1} and any of the
# ordinates can be null to mean "nocare"
#
MBCanvas private expandScrReg {newRegion {inc 0}} {

	$self instvar path_
	set region [$path_ cget -scrollregion]
	set oldRegion $region
	set rsz 0
	#	puts "expandScrReg: $newRegion"
	# new size could be a bounding box, which includes insets
	foreach i {2 3} {
		set newVal [lindex $newRegion $i]
		if {$newVal != {} && $newVal > [lindex $region $i]} {
			set region [lreplace $region $i $i \
					[expr {$newVal + $inc}]]
			set rsz 1
		}
	}
	foreach i {0 1} {
		set newVal [lindex $newRegion $i]
		if {$newVal != {} && $newVal < [lindex $region $i]} {
			set region [lreplace $region $i $i \
					[expr {$newVal - $inc}]]
			set rsz 1
		}
	}
        if $rsz {
		mtrace trcMB "expandScrReg: new $region old:$oldRegion"
                $path_ configure -scrollregion $region
        }
}

# Change zoom policy <p>
# if $policy is "fix to view", scale will adjusted so that roughly
# the same view is displayed as window size changes <br>
# if $policy is a number (possibly followed by '%'), sets scale to $policy
# (in percentage of normal size) <br>
# if $policy is "fit width", the scale is adjusted to fit the <i>current</i>
# width of the bounding box. (Note: automatic update as bounding box changes
# not supported) <br>
# if $policy is "fit height", the scale is adjusted to fit the <i>current</i>
# height of the bounding box. (Note: automatic update as bounding box changes
# not supported) <br>
# if $policy is "fit all", the scale is adjusted to fit the <i>current</i>
# bounding box. (Note: automatic update as bounding box changes
# not supported) <br>
#
MBCanvas public zoom_policy {policy} {
	$self instvar path_
	$self set zoomPolicy_ $policy
	set leftFract 0
	set topFract 0
	set changeSR 1
	switch -exact $policy {
		"fix to view" {
			set changeSR 0
		}
		"fit width" {
			set bbox [$path_ bbox all]
			if {$bbox == {}} { return }
			set width [expr {[lindex $bbox 2] - [lindex $bbox 0]}]
			if {$width > 1} {
				$self fit $width 0
			}
		}
		"fit height" {
			set bbox [$path_ bbox all]
			if {$bbox == {}} { return }
			set height [expr {[lindex $bbox 3] - [lindex $bbox 1]}]
			if {$height > 1} {
				$self fit 0 $height
			}
		}
		"fit all" {
			set bbox [$path_ bbox all]
			if {$bbox == {}} { return }
			set width [expr {[lindex $bbox 2] - [lindex $bbox 0]}]
			set height [expr {[lindex $bbox 3] - [lindex $bbox 1]}]
			$self fit $width $height
		}
		default {
			set topLeft [$self canvasxy 0 0]
			puts "topLeft = $topLeft"
			if [regexp {([0-9]+)%?} $policy {} scale] {
				$self rescale [expr {$scale / 100.0}]
			} else {
				error "invalid zoom policy: $policy"
				return
			}
			set changeSR 0
			# move back so that the top left corner is still
			# at the same location
			$self scrollTopLeft $topLeft
		}
	}
	if $changeSR {
		set bbox [$path_ bbox all]
		if {$bbox == {}} { return }
		$path_ configure -scrollregion $bbox

		# note: since we trap xview and yview commands,
		#       must not call $path_ directly for them
		$self yview moveto 0
		$self xview moveto 0
	}
}

# scrolls the canvas so that xy (in normalized coordinates) is at the
# top left corner
MBCanvas private scrollTopLeft {topLeft} {
	$self instvar path_
	set sr [$path_ cget -scrollregion]
	set scale [$self getscale]
	set x [expr {[lindex $topLeft 0]*$scale}]
	set y [expr {[lindex $topLeft 1]*$scale}]
#	puts "x = $x, y = $y, sr=$sr"
	set leftFract [expr {($x - [lindex $sr 0])/([lindex $sr 2] \
			- [lindex $sr 0])}]
	set topFract [expr {($y - [lindex $sr 1])/([lindex $sr 3] \
			- [lindex $sr 1])}]
#	puts "topFract = $topFract, leftFract = $leftFract"
	$self yview moveto $topFract
	$self xview moveto $leftFract
}

# reconfigure the canvas to {w h}
MBCanvas private reconfig {w h} {
	$self instvar path_ zoomPolicy_
#	puts "in reconfig, zoomPolicy_=$zoomPolicy_"
	if {$zoomPolicy_ == "fix to view"} {
		set topleft [$self canvasxy 0 0]
		set inset [expr {[$path_ cget -borderwidth] \
				+ [$path_ cget -highlightthickness]}]
		$self resize [expr {$w - 2*$inset}] [expr {$w - 2*$inset}]

		# scroll back so that the top left corner is still at
		# the same location
		$self scrollTopLeft $topleft
	}
	$self updScrReg 5
	return
}

# sets the scroll region to be the union of the displayed area and
# the bounding box, inc gives the extra boundary
MBCanvas private updScrReg { {inc 0} } {
	#	puts "in updScrReg, inc=$inc"
	$self instvar path_
	set w [winfo width $path_]
	set h [winfo height $path_]
	set x [$path_ canvasx 0]
	set y [$path_ canvasy 0]
	set bbox [$path_ bbox all]
	set x1 $x
	set x2 [expr {$x + $w}]
	set y1 $y
	set y2 [expr {$y + $h}]
	if {$bbox == {}} {
		$self expandScrReg [list $x1 $y1 $x2 $y2]
		return
	}
	foreach {bx1 by1 bx2 by2} $bbox {
		if {$x1 > $bx1} { set x1 $bx1 }
		if {$y1 > $by1} { set y1 $by1 }
		if {$x2 < $bx2} { set x2 $bx2 }
		if {$y2 < $by2} { set y2 $by2 }
	}
	$self expandScrReg [list $x1 $y1 $x2 $y2]
	return
}

# shows the busy cursor in the canvas <br>
# if <u>isBusy</u> is 1, return the displaced cursor, it is then the caller's
# reponsibily to restore the old cursor by call show_busy 0 oldcursorname
#
MBCanvas public show_busy {isBusy {prevCursor {}}} {
	$self instvar path_
	if {$isBusy} {
		set oldCursor [$path_ cget -cursor]
		$path_ configure -cursor watch
		return $oldCursor
	} elseif {$prevCursor != {}} {
		$path_ configure -cursor $prevCursor
	}
}

# attaches a pair of scrollbars to the canvas
MBCanvas public attach_scrollbar {xscroll yscroll} {
	$path_ configure -xscrollcommand [list $xscroll set] \
			-yscrollcommand [list $yscroll set]
}

# transfers configurations from one canvas to another, useful when
# we hide one canvas and displays another but want the displayed canvas to
# have the same configuration as the about to be hidden one
#
MBCanvas public transfer_state_from {canvas} {
	$self enable_tip [$canvas set ownerTip_]

	# transfer zoom scale. If view is fixed don't rescale
	# (otherwise view will not be fixed. Otherwise, change the zoom
	# value.
	set zoomPolicy [$canvas set zoomPolicy_]
	if ![regexp  {fix*} $zoomPolicy] {
#		puts "rescaling to [$canvas getscale]"
		$self rescale [$canvas getscale]
	}
	if ![regexp "fit*" $zoomPolicy] {
		$self zoom_policy $zoomPolicy
	} else {
		$self set zoomPolicy_ $zoomPolicy
	}
}

# resizes the canvas to fit the new bounding box
#
MBCanvas public refreshScrReg {} {
	$self instvar path_
	#	puts "in refreshScrReg"
	$self updScrReg 5
}

# tells the canvas to pack itself
MBCanvas public pack {args} {
	$self instvar path_
	$self refreshScrReg
	bind $path_ <Enter> "focus $path_"
	bind $path_ <Leave> "focus ."
	eval pack $path_ $args
}

# tells the canvas to unpack itself
MBCanvas public unpack {} {
	$self resetBindings
	pack forget [$self set path_]
}
