# ui-activesource.tcl --
#
#       An ActiveSource window is created to encapsulate a slow-update
#       postage-stamp-sized VideoWidget for displaying the video of a source
#       and some of its statistics once the source is "activated".
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#  @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/vic/ui-activesource.tcl,v 1.16 2002/02/13 22:04:51 lim Exp $


import InfoWindow RtpStatWindow ErrorWindow ScubaInfoWindow VideoWidget MtraceWindow Configuration VideoBox

# to get update_rate proc
import VicUI

#
# An ActiveSource window is created to encapsulate a slow-update
# postage-stamp-sized VideoWidget for displaying the video of a source
# and some of its statistics once the source is "activated".  The VicUI
# lays out these thumbnail-sized windows in grid.  Selecting an
# ActiveSource's VideoWidget will toggle the mapping of a UserWindow for the
# represented source.
#
Class ActiveSource -superclass TkWindow

#
# Every second, update the elements of the global rate and source
# description arrays indexed by the source (instvar src_) of this
# ActiveSource.
#
ActiveSource instproc update {} {
	$self instvar src_ ui_
	global ftext
	if ![info exists ftext($src_)] {
		return
	}
	#FIXME
	update_rate $src_
	#FIXME
	$ui_ trigger_sdes $src_

	after 1000 "$self update"
}


ActiveSource instproc name {} {
	global src_nickname
	$self instvar src_
	return $src_nickname($src_)
}

#
# For the represented source, delete the VideoWidget and any of the following windows if they exist: <br>
#    <dd> InfoWindow
#    <dd> RtpStatWindow displaying rtp statistics
#    <dd> RtpStatWindow displaying decoder statistics
#    <dd> ScubaInfoWindow
#
ActiveSource instproc destroy {} {
	$self instvar vw_ info_win_ rtp_win_ decoder_win_ scuba_win_
	$vw_ destroy
	if [info exists info_win_] {
		delete $info_win_
	}
	if [info exists rtp_win_] {
		delete $rtp_win_
	}
	if [info exists decoder_win_] {
		delete $decoder_win_
	}
	if [info exists scuba_win_] {
		delete $scuba_win_
	}
	$self next
}

#
# If an InfoWindow exists for this source, delete it.  Else, create one.
#
ActiveSource instproc create-info-window {} {
	$self instvar src_ info_win_
	if [info exists info_win_] {
		$self delete-info-window
	} else {
		set info_win_ [new InfoWindow .info$src_ $src_ $self]
	}
}

#
# Delete the InfoWindow for this source.
#
ActiveSource instproc delete-info-window {} {
	$self instvar info_win_
	delete $info_win_
	unset info_win_
}

#
# For the represented source, return a string
# describing its network statistics.
#
ActiveSource instproc stats {} {
	$self instvar src_
	return "Kilobits [expr [$src_ layer-stat nb_] >> (10-3)] \
		Frames [$src_ layer-stat nf_] \
		Packets [$src_ layer-stat np_] \
		Missing [$src_ missing] \
		Misordered [$src_ layer-stat nm_] \
		Runts [$src_ layer-stat nrunt_] \
		Dups [$src_ layer-stat ndup_] \
		Bad-S-Len [$src_ set badsesslen_] \
		Bad-S-Ver [$src_ set badsessver_] \
		Bad-S-Opt [$src_ set badsessopt_] \
		Bad-Sdes [$src_ set badsdes_] \
		Bad-Bye [$src_ set badbye_]"
}

#
# For the represented source, return the decoder statistics.
#
ActiveSource instproc decoder-stats {} {
	$self instvar src_
	set d [$src_ handler]
	#FIXME
	return [$d stats]
}

#
# If a window exists for displaying the RTP Statistics of this source, delete it.  Else, create one.
#
ActiveSource instproc create-rtp-window {} {
	$self instvar src_ rtp_win_
	if [info exists rtp_win_] {
		$self delete-rtp-window
	} else {
		set rtp_win_ [new RtpStatWindow .rtp$src_ $src_ \
					"RTP Statistics" \
					"$self stats" \
					"$self delete-rtp-window"]
	}
}

#
# Delete the RTP Statisctics window for this source.
#
ActiveSource instproc delete-rtp-window {} {
	$self instvar rtp_win_
	delete $rtp_win_
	unset rtp_win_
}

#
# If a window exists for displaying the Decoder Statistics of this source, delete it.  Else, create one.
#
ActiveSource instproc create-decoder-window {} {
	$self instvar src_ decoder_win_
	if [info exists decoder_win_] {
		$self delete-decoder-window
	} else {
		if { "[$src_ handler]" == "" } {
			new ErrorWindow "no decoder stats yet"
			return
		}
		set decoder_win_ [new RtpStatWindow .decoder$src_ $src_  \
				"Decoder Statistics" \
				"$self decoder-stats" \
				"$self delete-decoder-window"]
	}
}

#
# Delete the Decoder Statisctics window for this source.
#
ActiveSource instproc delete-decoder-window {} {
	$self instvar decoder_win_
	if [info exists decoder_win_] {
		delete $decoder_win_
		unset decoder_win_
	}
}

#
# Create a menu for accessing information and data associated with the represented source: <br>
#    <dd> "Site Info"
#    <dd> "RTP Stats"
#    <dd> "Decoder Stats"
#    <dd> "Mtrace from"
#    <dd> "Mtrace to"
#    <dd> and possibly "Scuba Info"
#
ActiveSource instproc build_info_menu {src m} {
	menu $m
	set f [$self get_option smallfont]
	$m add command -label "Site Info" \
		-command "$self create-info-window" -font $f
	$m add command -label "RTP Stats"\
		-command "$self create-rtp-window" -font $f
	$m add command -label "Decoder Stats" \
		-command "$self create-decoder-window" -font $f
	$self instvar ui_
	if [in_multicast [[$ui_ set videoAgent_] session-addr]] {
		$m add command -label "Mtrace from" \
			-command "$self create-mtrace-window from" -font $f
		$m add command -label "Mtrace to" \
			-command "$self create-mtrace-window to" -font $f
	}
	$ui_ instvar scuba_sess_
	if [info exists scuba_sess_] {
		$m add command -label "Scuba Info" -font $f \
			-command "$self create-scuba-window"
	}
}

#
# If a window exists for displaying the Scuba Votes for this source, delete it.  Else, create one.
#
ActiveSource instproc create-scuba-window {} {
	$self instvar scuba_win_ ui_ src_
	$ui_ instvar scuba_sess_
	if [info exists scuba_win_] {
		$self delete-scuba-window
	} else {
		set scuba_win_ [new ScubaInfoWindow .scubainfo$self \
				$src_ $self $scuba_sess_]
		[$scuba_sess_ source-manager] attach $scuba_win_
		$scuba_win_ timeout
	}
}

#
# Delete the Scuba Votes window for this source.
#
ActiveSource instproc delete-scuba-window {} {
	$self instvar scuba_win_ ui_
	$ui_ instvar scuba_sess_
	[$scuba_sess_ source-manager] detach $scuba_win_
	delete $scuba_win_
	unset scuba_win_
}

#
# Create a UserWindow for this ActiveSource. Returns the UserWindow.
#
ActiveSource private create_user_window {{x false} {y false} {showMenus true} {scale 1}} {
	$self instvar asm_ localChannel_

    # If scale is 0, don't create this window.
    if {!$scale} {
        return ""
    }

	set uw [new UserWindow $asm_ $self [$self yesno useCues] $localChannel_ {} $showMenus]
	if {$x != "false" && $y != "false"} {
		wm geometry [$uw path] "+$x+$y"
	}
	$uw resize-zoom $scale
	return $uw
}

#
# For implementing drag-and-drop feature so that thumbnail can be
# dragged onto a VideoBox and cause the Source displayed to switch to
# that displayed by the thumbnail.
#
ActiveSource private select-thumbnail-to-drag {} {
	$self instvar draggableToplevel_
	raise $draggableToplevel_
	puts stderr "selecting ActiveSource = $self"
}

#
# For implementing drag-and-drop feature so that thumbnail can be
# dragged onto a VideoBox and cause the Source displayed to switch to
# that displayed by the thumbnail.
#
ActiveSource private move-thumbnail {} {
	$self instvar draggableToplevel_
	set x [winfo pointerx $draggableToplevel_]
	set y [winfo pointery $draggableToplevel_]
	puts "moving +$x+$y"
	raise $draggableToplevel_
	wm geometry $draggableToplevel_ +$x+$y
	wm deiconify $draggableToplevel_
	update idletasks
}

#
# For implementing drag-and-drop feature so that thumbnail can be
# dragged onto a VideoBox and cause the Source displayed to switch to
# that displayed by the thumbnail.
#
ActiveSource private release-thumbnail {} {
	$self instvar draggableToplevel_
	set x [expr [expr [winfo width $draggableToplevel_] / 2] + [winfo rootx $draggableToplevel_]]
	set y [expr [expr [winfo height $draggableToplevel_] / 2] + [winfo rooty $draggableToplevel_]]
	wm withdraw $draggableToplevel_
	puts "dropped off at [winfo containing $x $y]"
	puts "Have vbm go thru vb's and find the one w/ the matching widgetpath & put the selected ActiveSource in there"
# once find vb
# $vidbox unset_manager_switched
# $vidbox switch $src
}

#
# Given widget, <i>w</i>, a frame ($w.stamp) is created and packed.
# Within this frame, an "is_slow" postage-stamp sized (80x60)
# VideoWidget (instvar vw_) is created.  A decoder is attached to this
# VideoWidget for the specified <i>src</i>.
# This ActiveSource is added to the provided ActiveSourceManager, <i>asm</i>.
# <p>
# The <i>ui</i> must implement the <i>trigger_sdes</i> method
# and must have instvars <i>videoAgent_</i> and <i>vframe_</i> and optionally
# <i>scuba_sess_</i>.
#
ActiveSource instproc init { asm ui w src localChannel } {
	$self next $w

	# must add src to active_ list before actually making the new ActiveSource
	# because the videobox created in the ActiveSource init method needs the
	# src/as mapping.
	$asm add_active $self $src

	frame $w -relief groove -borderwidth 0 \
		-visual [Application set visual_] \
		-colormap [Application set colormap_]
	$self instvar src_ ui_ thumbnail_ userwindows_ videoboxes_ asm_ localChannel_
	set src_ $src
	set ui_ $ui
	set userwindows_ ""
	set videoboxes_ ""
	set asm_ $asm
	set localChannel_ $localChannel
	after 1000 "$self update"

	set stamp $w.stamp
	frame $stamp -relief ridge -borderwidth 2
	bind $stamp <Enter> "%W configure -background gray90"
	bind $stamp <Leave> "%W configure -background \
		[$self get_option background]"

	# FIXME this is a temporary hack while cindy works on the VideoBox class... trying it out in mui before using in vic
	if {[$ui_ info class] != "MuiUI"} {
		set video_widget_path $stamp.video
		set thumbnail_ [new VideoWidget $video_widget_path 80 60]
		$thumbnail_ set is_slow_ 1
		$self attach-thumbnail
		pack $stamp.video -side left -anchor c -padx 2
		pack $stamp -side left -fill y
		bind $video_widget_path <ButtonPress-1> "$self select-thumbnail"

		frame $w.r
		$self display_cw $w.r
		$self display_ctrl $w.r
		pack $w.r -side left -expand 1 -fill x
	} else {
		set vb [VideoBox $stamp.video -size thumbnail -border false -nametag true -ui $ui_ -updateSpeed slow -src $src_]
		pack $stamp.video -side left -anchor c -padx 0
		pack $stamp -side top -fill y
		set thumbnail_ [$vb get_video_widget]
		set video_widget_path [$vb get_video_widget_path]
		bind $video_widget_path <ButtonPress-2> "$self select-thumbnail"

		# Setting up drag-and-drop feature for selecting which Source is displayed in the large VideoBoxes
		$self instvar draggableToplevel_
		set draggableToplevel_ .$self
		toplevel $draggableToplevel_
		#  prevent title bar and border from being drawn around this window
		wm overrideredirect $draggableToplevel_ 1
		VideoBox $draggableToplevel_.vb -size thumbnail -border false -nametag true -ui $ui_ -updateSpeed slow -src $src_
		pack $draggableToplevel_.vb
		wm withdraw $draggableToplevel_

		bind $video_widget_path <ButtonPress-1> "$self select-thumbnail-to-drag"
		bind $video_widget_path <B1-Motion> "$self move-thumbnail"
		bind $video_widget_path <ButtonRelease-1> "$self release-thumbnail"
	}

	bind $video_widget_path <Enter> { focus %W }
	bind $video_widget_path <d> "$src deactivate"

	update idletasks
}

#
# Display a small panel of network statistics for this ActiveSource
# within the widget <i>w</i>.
#
ActiveSource public display_cw { w } {
	$self instvar src_

	set f [$self get_option smallfont]

	frame $w.cw -relief groove -borderwidth 2

	pack $w.cw -side left -expand 1 -fill both -anchor w -padx 0

	label $w.cw.name -textvariable src_nickname($src_) -font $f \
		-pady 1 -borderwidth 0 -anchor w
	label $w.cw.addr -textvariable src_info($src_) -font $f \
		-pady 1 -borderwidth 0 -anchor w

	global ftext btext ltext
	set ftext($src_) "0.0 f/s"
	set btext($src_) "0.0 kb/s"
	set ltext($src_) "(0%)"
	frame $w.cw.rateinfo
	label $w.cw.rateinfo.fps -textvariable ftext($src_) -width 6 \
		-font $f -pady 0 -borderwidth 0
	label $w.cw.rateinfo.bps -textvariable btext($src_) -width 8 \
		-font $f -pady 0 -borderwidth 0
	label $w.cw.rateinfo.loss -textvariable ltext($src_) -width 6 \
		-font $f -pady 0 -borderwidth 0


	pack $w.cw.rateinfo.fps $w.cw.rateinfo.bps $w.cw.rateinfo.loss \
		-side left -anchor w
	pack $w.cw.name $w.cw.addr $w.cw.rateinfo -anchor w -fill x

	pack $w.cw -fill x -side top
}

#
# Display a mute button, a color button, and an info menu for this
# ActiveSource within the widget <i>w</i>.
#
ActiveSource public display_ctrl { w } {
	$self instvar src_ ui_

	set f [$self get_option smallfont]

	frame $w.ctrl -borderwidth 0

	global mutebutton
	set mutebutton($src_) [$ui_ mute_new_sources]
	$src_ mute $mutebutton($src_)
	checkbutton $w.ctrl.mute -text mute -borderwidth 2 \
		-highlightthickness 1 \
		-relief groove -font $f -width 4 \
		-command "$src_ mute \$mutebutton($src_)" \
		-variable mutebutton($src_)

	checkbutton $w.ctrl.color -text color -borderwidth 2 \
		-highlightthickness 1 \
		-relief groove -font $f -width 4 \
		-command "\[$src_ handler\] color \$colorbutton($src_)" \
		-variable colorbutton($src_)

	set m $w.ctrl.info.menu$src_
	menubutton $w.ctrl.info -text info... -borderwidth 2 \
		-highlightthickness 1 \
		-relief groove -font $f -width 5 \
		-menu $m
	$self build_info_menu $src_ $m

	pack $w.ctrl.mute -side left -fill x -expand 1
	pack $w.ctrl.color -side left -fill x -expand 1
	pack $w.ctrl.info -side left -fill both -expand 1
#	pack $w.ctrl.options -side left -fill x -expand 1

	global colorbutton
	set colorbutton($src_) 1

	pack $w.ctrl -fill x -side top
}

#
# Returns true if the represented source is h261 format.
#
ActiveSource instproc isCIF {} {
	$self instvar src_
	return [expr [string compare [$src_ format_name] h261] == 0]
}

#
# Bind a source to a window so that the video stream from the
# represented source appears in UserWindow <i>uw</i>.
#
ActiveSource instproc attach-window uw {
	$self instvar src_ ui_
	[$uw video-widget] attach-decoder $src_ [[$ui_ set vframe_] set colorModel_] [$ui_ use_hw_decode] [$uw use_heuristics]
	$self instvar userwindows_
	lappend userwindows_ $uw
	$uw set-name [$src_ getid]

	$ui_ instvar scuba_sess_
	if [info exists scuba_sess_] {
		$scuba_sess_ scuba_focus $src_
	}
}

#
# Bind a source to a window so that the video stream from the
# represented source appears in VideoBox <i>vb</i>.
#
ActiveSource instproc attach_videobox { vb } {
	$self instvar src_ ui_ vb_
#vbx	puts "$self attach_videobox $vb"
	set vb_ $vb

	[$vb get_video_widget] attach-decoder $src_ [[$ui_ set vframe_] set colorModel_] [$ui_ use_hw_decode]

	$self instvar videoboxes_
	lappend videoboxes_ $vb

#vbx	$uw set-name [$src_ getid]

	$ui_ instvar scuba_sess_
	if [info exists scuba_sess_] {
		$scuba_sess_ scuba_focus $src_
	}
}

#
# Return a list of UserWindows to which this ActiveSource is attached.
#
ActiveSource instproc user-windows {} {
	return [$self set userwindows_]
}

#
# Return a list of VideoBoxes to which this ActiveSource is attached.
#
ActiveSource instproc video_boxes {} {
	return [$self set videoboxes_]
}

#
# Return the Source attached to this ActiveSource.
#
ActiveSource instproc source {} {
	return [$self set src_]
}

#
# Discontinue the representation of this ActiveSource in UserWindow <i>uw</i>.
#
ActiveSource instproc detach-window uw {
	$self instvar userwindows_ src_ ui_
	$ui_ instvar scuba_sess_
	if [info exists scuba_sess_] {
		$scuba_sess_ scuba_unfocus $src_
	}
	[$uw video-widget] detach-decoder $src_
	# there must be an easier way to do this
	set k [lsearch -exact $userwindows_ $uw]
	if { $k < 0 } {
		puts "[$self get_option appname]: detach-window: FIXME"
		exit 1
	}
	set userwindows_ [lreplace $userwindows_ $k $k]
}

#
# Discontinue the representation of this ActiveSource in VideoBox <i>vb</i>.
#
ActiveSource instproc detach_videobox vb {
	$self instvar videoboxes_ src_ ui_
	$ui_ instvar scuba_sess_
	if [info exists scuba_sess_] {
		$scuba_sess_ scuba_unfocus $src_
	}
	[$vb get_video_widget] detach-decoder $src_
	# there must be an easier way to do this
	set k [lsearch -exact $videoboxes_ $vb]
	if { $k < 0 } {
		puts "[$self get_option appname]: detach-window: FIXME"
		exit 1
	}
	set videoboxes_ [lreplace $videoboxes_ $k $k]
}

#
# Discontinue the representation of this ActiveSource in all of the
# UserWindows to which it is currently attached.
#
ActiveSource instproc detach-windows {} {
	$self instvar userwindows_
	foreach uw $userwindows_ {
		$self detach-window $uw
	}
	$self detach-thumbnail
}

#
# Discontinue the representation of this ActiveSource in all of the
# VideoBoxes to which it is currently attached.
#
ActiveSource instproc detach_videoboxes {} {
	$self instvar videoboxes_
	foreach vb $videoboxes_ {
		$self detach_videobox $vb
	}
}

#
ActiveSource instproc attach-thumbnail {} {
        #mjh
        $self instvar src_ ui_ thumbnail_
        $thumbnail_ attach-decoder $src_ [[$ui_ set vframe_] set colorModel_] [$ui_ use_hw_decode]
}

#
ActiveSource instproc detach-thumbnail {} {
        #mjh
        $self instvar src_ thumbnail_
        $thumbnail_ detach-decoder $src_
}

#
# If a window does not exist for displaying the MTrace info for this source, create one.
#
ActiveSource instproc create-mtrace-window {dir} {
        $self instvar mtrace_win_ src_
        set w .mtrace$src_
        if ![winfo exists $w] {
                set mtrace_win_ [new MtraceWindow $w $src_ $self]
        }
        $mtrace_win_ do_mtrace
}

#
# Called when user clicks on thumbnail video window.
# Create a new window only if the window already
# isn't being displayed (in locked mode).  In this
# case, delete the existing window (i.e., clicking
# on the thumbnail gives a toggle action, but not
# for voice-switched or browse-mode windows).
# Return the new UserWindow.
#
ActiveSource public select-thumbnail {} {
	foreach uw [$self user-windows] {
		if { [$uw attached-source] == "$self" && ![$uw is-switched] } {
			$uw destroy
			return
		}
	}
	#FIXME a little weird
	$self instvar ui_
	return [$self create_user_window]
}

#
# Called to place a new window only if the window
# isn't already being displayed. If it is, don't
# do anything. Return the new UserWindow.
#
ActiveSource public place {x y {scale 1}} {
    foreach uw [$self user-windows] {
	if {[$uw attached-source] == "$self"} {
	    return
	}
    }
    #FIXME a little weird: apparently ui_ needs to exist otherwise
    # you can't destroy the UserWindow by clicking on the thumbnail.
    $self instvar ui_
    return [$self create_user_window $x $y false $scale]
}
