/*
 * tracker.cc --
 *
 *      Implementation of Tracker class.  The Tracker looks at the blocks
 *      replenished by the H261 algorithm to find motion and do camera tracking
 *
 * Copyright (c) 2000-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include "tclcl.h"
#include "tracker.h"

/* SCALE_SIZE is the pixels displayed per block */

#define VAR_SIZE

#ifdef VAR_SIZE
int SCALE_SIZE = 8;
#else
#define SCALE_SIZE 8
#endif

int numBlocksPerFrame = 0;
// int colorIndex = 0xffff;

//#define DEBUG
#define PRINT_OK 1

// t3 = t2 - t1
// FIXME - this modifies t1!
void subTimes2(struct timeval* t1, struct timeval* t2, struct timeval* t3)
{
  if(t1->tv_usec < t2->tv_usec)
  {
    (t1->tv_sec) -= 1;
    (t1->tv_usec) += 1000000;
  }
  t3->tv_usec = t1->tv_usec - t2->tv_usec;
  t3->tv_sec = t1->tv_sec - t2->tv_sec;
}


#define LEFT_THRESHOLD 10
#define RIGHT_THRESHOLD 10

#define TOTAL_THRESHOLD 250
#define CRAZY_THRESHOLD 20
#define MOVE_THRESHOLD NUM_HISTS - 5

#define MAX_DIFFS 8

#define NUM_DELAYS 0
int disp_counter = 0;

// FIXME - this needs to be local to the tracker or something like that
Tcl& tcl = Tcl::instance();

Tracker::~Tracker()
{
  printf("destructor being called!\n");
}

Tracker::Tracker()
{
  printf("default being called!\n");
}


Tracker::Tracker(int argc, const char*const* argv) : num_left(0), num_right(0), num_center(5)
{
  int x;

#ifdef DEBUG
  printf("initing the Tracker class\n");
#endif

#ifdef DEBUG
  printf("argc is %d\n", argc);
  for(x = 0; x < argc; x++)
  {
    printf("argv[%d] is %s\n", x, argv[x]);
  }
#endif

  if(argc > 4)
  {
    sprintf(frame_name, "%s", argv[4]);
  }
#ifdef VAR_SIZE
  if(argc > 5)
  {
    SCALE_SIZE = atoi(argv[5]);
  }
#endif

  tcl = Tcl::instance();

  /* bind object variable to a Tcl instvar */
  bind("tracking_on_", &tracking_on);

  /*
    this is how you can call a Tcl-defined function on this particular object
    instance without knowing the name

    but it doesn't work here because this is the constructor and there is no
    name yet!
  */
  /*
    temp = Invokef("fireball");
  */

  initGobsArray();

  // init the slices array, which keeps track of how many blocks changed in that region
  for(x = 0; x < NUM_AREAS; ++x)
  {
    slices[x] = 0;
  }
  woojblockoff = 0;


  hist_index = 0;
  for(x = 0; x < NUM_HISTS; ++x)
  {
    hist_buf[x] = 0;
  }
  hist_val = 0;
  tracking_on = 0;
  total_pkts = 0;
  total_lost = 0;
  num_moves = 0;
  pause_counter = 0;

#ifdef DEBUG
  printf("finished initing, hist_val = %d, tracking_on = %d\n", hist_val, tracking_on);
#endif
}

void Tracker::initGobsArray()
{
  int x;
  int y;

  // init the Group Of Blocks array, recording their pixel location
  // do this once at the beginning and just read it later
  for(x = 0; x < NUM_GOBS; ++x)
  {
    gobs[x].blocks = (t_block*)malloc(sizeof(t_block) * BLOCKS_PER_GOB);
    for(y = 0; y < BLOCKS_PER_GOB; ++y)
    {
      gobs[x].blocks[y].color = 0;
      gobs[x].blocks[y].changed = 0;
      gobs[x].blocks[y].x_tl = ((x % 2) * SCALE_SIZE * 11) + ((y % 11) * SCALE_SIZE);
      gobs[x].blocks[y].y_tl = ((x / 2) * SCALE_SIZE * 3) + ((y / 11) * SCALE_SIZE);
      gobs[x].blocks[y].x_br = ((x % 2) * SCALE_SIZE * 11) + ((y % 11) * SCALE_SIZE) + SCALE_SIZE - 1;
      gobs[x].blocks[y].y_br = ((x / 2) * SCALE_SIZE * 3) + ((y / 11) * SCALE_SIZE) + SCALE_SIZE - 1;

      gobs[x].blocks[y].x_pos = ((x % 2) * 11) + ((y % 11));
      gobs[x].blocks[y].y_pos = ((x / 2) * 3) + ((y / 11));
    }
  }
}

void Tracker::rescale(int newSize)
{
#ifdef VAR_SIZE
  SCALE_SIZE = newSize;
  initGobsArray();
#endif
}

void Tracker::refresh()
{
  tcl.evalc("update idletasks");
}

/*
  counts the number of packets received and number lost
  should be called every time a packet is received
*/
void Tracker::lostPackets(int num_lost)
{
#ifdef DEBUG
  printf("entering lostPackets\n");
#endif
  total_pkts += num_lost + 1;
  total_lost += num_lost;
}


int Tracker::endFrame()
{
  int x;
  int y;
  int index;
  static int init = 0;
  static int cnt = 0;
  struct timeval curTime;
  static struct timeval lastTime;
  struct timeval diffTime;
  float msec;

#ifdef DEBUG

#endif
  gettimeofday(&curTime, NULL);
  if(!init)
  {
    init = 1;
    lastTime = curTime;
  }
  else
  {
    int res = 15;
    cnt++;
    cnt = cnt % res;
    if(!cnt)
    {
      subTimes2(&curTime, &lastTime, &diffTime);
      msec = diffTime.tv_sec * 1000;
      msec += diffTime.tv_usec / 1000;
      lastTime = curTime;
      if(PRINT_OK)
      {
	printf("getting end o`f frame, 30 frameTime = %f msec => %f fps\n",  msec, 1000 / msec * res);
	printf("avg numBlocksPerFrame = %d\n", numBlocksPerFrame / res);
      }
      numBlocksPerFrame = 0;
    }
  }

//  colorIndex++;
//  colorIndex = colorIndex % (256 * 256);

  // blank everything out by drawing a solid blue rectangle
  tcl.evalf("%s.t_canvas create rect 0 0 %d %d -fill blue", frame_name, SCALE_SIZE * 22, SCALE_SIZE * 18);

  // if a block changed, reset it; keep statistics as we go
  for(x = 0; x < NUM_GOBS; ++x)
  {
    for(y = 0; y < BLOCKS_PER_GOB; ++y)
    {
      if(gobs[x].blocks[y].changed == 1)
      {
	gobs[x].blocks[y].changed = 0;
	//printf("x, y = %d, %d\n",gobs[x].blocks[y].x_pos, gobs[x].blocks[y].y_pos);
	index = gobs[x].blocks[y].x_pos / AREA_SIZE;
	if(index >= NUM_AREAS)
	{
	  index = NUM_AREAS - 1;
	}
	else
	{
	  slices[index]++;
	}
      }
    }
  }
// FIXME - I commented out the drawing of the bars
  /*
  if(woojblockoff)
  {
    tcl.evalf("%s.t_output create rect 0 0 100 %d -fill blue", frame_name, NUM_AREAS * 10);

    for(x = 0; x < NUM_AREAS; ++x)
    {
      tcl.evalf("%s.t_output create rect 0 %d %d %d -fill yellow", frame_name, x * 10, slices[x] * 4, ((x + 1) * 10) - 1);
    }
  }
  */

  moveCamera();

  for(x = 0; x < NUM_AREAS; ++x)
  {
    slices[x] = 0;
  }

  // usleep(100);

  num_right = 0;
  num_left = 0;
  num_center = 0;
  return(1);
}

/*
  this function is called by t_p64.cc when a block is decoded as changed
  it draws a red block and marks it as changed
*/
void Tracker::changeBlock(int gob, int block_num)
{
#ifdef DEBUG
  printf("entering changeBlock\n");
#endif

  if (woojblockoff==1)
  {
    if(disp_counter++ > NUM_DELAYS)
    {
//      tcl.evalf("%s.t_canvas create rect %d %d %d %d -fill #%.4x%.4x%.4x", frame_name, gobs[gob].blocks[block_num].x_tl, gobs[gob].blocks[block_num].y_tl, gobs[gob].blocks[block_num].x_br, gobs[gob].blocks[block_num].y_br, colorIndex, colorIndex, colorIndex);
      tcl.evalf("%s.t_canvas create rect %d %d %d %d -fill %s", frame_name, gobs[gob].blocks[block_num].x_tl, gobs[gob].blocks[block_num].y_tl, gobs[gob].blocks[block_num].x_br, gobs[gob].blocks[block_num].y_br, "red");
      disp_counter = 0;
    }
  }

  numBlocksPerFrame++;

  gobs[gob].blocks[block_num].changed = 1;
}


void Tracker::print_test()
{
  printf("I printed successfully  num_center is %d\n", num_center);
}

/*
  this handles commands called in tcl code
*/
int Tracker::command(int argc, const char* const* argv)
{
  int x;
  const char* cmd = argv[1];
  if(argc >= 2)
  {
    if(!strcmp(cmd, "toggleblockonoff"))
    {
      woojblockoff = !woojblockoff;
      if (woojblockoff ==1)
	tcl.evalf("%s.togs1.bicanvas create rect 5 5 16 16 -fill red", frame_name);
      else
	tcl.evalf("%s.togs1.bicanvas create rect 5 5 16 16 -fill green", frame_name);
      return(TCL_OK);
    }
    if(!strcmp(cmd, "disable_tracking"))
    {
      //	  printf("Tracker::Command: disable_tracking called\n");
      tracking_on = 0;
      hist_val = 0;
      return(TCL_OK);
    }
    if(!strcmp(cmd, "enable_tracking"))
    {
      //	  printf("Tracker::Command: enable_tracking called\n");
      hist_val = 0;
      hist_index = 0;
      //	  if(hist_buf == NULL)
      //{
      //	      hist_buf = (int*)malloc(sizeof(int) * NUM_HISTS);
      //}
      for(x = 0; x < NUM_HISTS; ++x)
      {
	hist_buf[x] = 0;
      }
      tracking_on = 1;
      return(TCL_OK);
    }
    if(!strcmp(cmd, "get_stats"))
    {
      getStats();
      return(TCL_OK);
    }
    if(!strcmp(cmd, "rescale"))
    {
      if(argc > 2)
      {
	rescale(atoi(argv[2]));
	return(TCL_OK);
      }
    }
  }
  return(TclObject::command(argc, argv));
}

int Tracker::getStats()
{
  double pct_lost;

  printf("total pkts = %d, total lost = %d, percent lost = ", total_pkts, total_lost);
  if(total_lost != 0)
  {
    pct_lost = (double)total_lost/(double)total_pkts;
    printf("%f\n", pct_lost);
  }
  else
  {
    printf("0\n");
  }
  return(total_lost);
}

int Tracker::moveCamera()
{
  int x;
  int max = 0;
  int max2 = 0;
  int index = NUM_AREAS/2;
  int total = 0;
  int last_index;
  int result;

#ifdef DEBUG
  printf("entering moveCamera(), tracking_on is %d\n", tracking_on);
#endif

  if(pause_counter > 0)
  {
    pause_counter--;
    return(0);
  }

  if(hist_buf == NULL)
  {
    printf("major problem, hist_buf is NULL\n");
  }

  for(x = 0; x < NUM_AREAS; ++x)
  {
    total += slices[x];
    if(slices[x] > max)
    {
      max2 = max;
      max = slices[x];
      index = x;
    }
  }

  // name() gets the OTcl object name
  // printf("name is %s\n", name());

  if(tracking_on != 0)
  {
    last_index = hist_index + 1;
    last_index = last_index % NUM_HISTS;
    num_moves -= abs(hist_buf[last_index]);

    if(total > TOTAL_THRESHOLD)
    {
      //   printf("above threshold for max\n");
      return(0);
    }
    //  if(max > LEFT_THRESHOLD)
    if(max - max2 > MAX_DIFFS)
    {
      //   printf("index is %d, hist_buf is %d\n", hist_index, hist_buf);
      switch(index)
      {
      case 0:
	hist_val += -1;
	hist_buf[hist_index] = -1;
	num_moves++;
	pause_counter = PAUSE_COUNT;
	// we should use invokef because invoke may mess with our string...I think
	result = Invokef("move_camera left 4");
	//      printf("Left 10 hist_val = %d\n", hist_val);
	break;
      case 1:
	hist_val += -1;
	hist_buf[hist_index] = -1;
	num_moves++;
	pause_counter = PAUSE_COUNT;
	result = Invokef("move_camera left 3");
	//printf("Left 5 hist_val = %d\n", hist_val);
	break;
	/*
	  case 2:
	*/
      case 3:
	hist_val += 1;
	hist_buf[hist_index] = 1;
	num_moves++;
	pause_counter = PAUSE_COUNT;
	result = Invokef("move_camera right 3");
	//printf("RIGHT 5 hist_val = %d\n", hist_val);
	break;
      case 4:
	hist_val += 1;
	hist_buf[hist_index] = 1;
	num_moves++;
	pause_counter = PAUSE_COUNT;
	result = Invokef("move_camera right 4");
	//printf("RIGHT 5 hist_val = %d\n", hist_val);
	break;
      default:
	hist_buf[hist_index] = 0;
      }

      /*
	filter

	last_index = hist_index - 1;
	last_index = last_index % NUM_HISTS;

	if(hist_buf[last_index] == hist_buf[hist_index] && hist_buf[hist_index] == 1)
	{
	tcl.evalf("%s move_camera right 3", agent_name);
	hist_val++;
	num_moves++;
	}
	if(hist_buf[last_index] == hist_buf[hist_index] && hist_buf[hist_index] == -1)
	{
	tcl.evalf("%s move_camera left 3", agent_name);
	hist_val--;
	num_moves++;
	}
      */

      /* hist_buf is a circular buffer */
      hist_index += 1;
      hist_index = hist_index % NUM_HISTS;
      if((hist_val > CRAZY_THRESHOLD) || (-hist_val > CRAZY_THRESHOLD))
      {
	//    printf("signaling crazy becaue of %d tracking_on is %d\n", hist_val, tracking_on);
	Invokef("signal_crazy");
      }
    }
    //  printf("num_moves is %d\n", num_moves);
    if(num_moves > MOVE_THRESHOLD)
    {
      num_moves = 0;
      Invokef("move_camera out 1");
    }
  }
  return(1);
}

