/*
 * assistor-mme.cc --
 *
 *      Support for generating an analogue output from the J300 and associated
 *      boards using the MME interface.
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* @(#) assistor-mme.cc 1.3@(#)
**
** Support for generating an analogue output from the J300 and associated
** boards using the MME interface.
**
** An environment variable VIC_OUT is used to select between pal, ntsc and
** secam. Another environment variable VIC_MONOCHROME is used to select
** a monochrome output.
**
** Atanu Ghosh
** <atanu@socs.uts.edu.au>
** 7th May 1997
*/

#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>

extern "C" {
#include <mme/mme_api.h>
}

#include "jvs.h"
#include "inet.h"
#include "tclcl.h"
#include "rtp.h"
#include "decoder.h"
#include "renderer.h"
#include "renderer-window.h"
#include "vw.h"
#include "iohandler.h"
#include "device-output.h"

class MmeDecoder : public IOHandler {
	HVIDEO video;
	LPBITMAPINFOHEADER   lpbi;
	struct extbi {
		EXBMINFOHEADER exbi;
		JPEGINFOHEADER jpegbi;
	} *ep;

	VIDEOHDR *vhdr;
	unsigned char *vbuf;
	int frame_size;
	int monochrome;

protected:
	MmeDecoder(int ft);
	void analogueOut(JpegFrame *jf);
	void analogueOut(YuvFrame *jf, int ft);
	int checkSize(int frame_size, int width, int height);
public:
	int active;
	~MmeDecoder();
	char *errorString(int type);
	void warning(char *fmt, ...);
};

void
MmeDecoder::warning(char *fmt, ...)
{
	va_list ap;
	char buf[1024];

	va_start(ap, fmt);
	vsprintf(buf, fmt, ap);

	fprintf(stderr, "MmeDecoder::%s\n", buf);

	va_end(ap);
}

MmeDecoder::MmeDecoder(int ft) : active(0), ep(0), vhdr(0), vbuf(0)
{
	MMRESULT res;

	/*
	** Find the number of devices
	*/
	int num = videoGetNumDevs();
	if(0 == num) {
		if(-1 == mmeServerFileDescriptor())
			warning("MmeDecoder: Cannot contact mmeserver");
		else {
			warning("MmeDecoder: No output devices");
		}

		return;
	}

	monochrome = 0;
	if(getenv("VIC_MONOCHROME"))
		monochrome = 1;
	/*
	** Open the analogue output port
	*/
	if(DV_ERR_OK != (res = videoOpen(&video, 0, VIDEO_OUT))) {
		warning("MmeDecoder: videoOpen failed %s", errorString(res));
		return;
	}

	/*
	** Allow the video output type to be set from an environment variable.
	*/
	char *s = getenv("VIC_OUT");
	int standard = VIDEO_STANDARD_NTSC;
	if(s) {
		if(0 == strcmp(s, "pal"))
			standard = VIDEO_STANDARD_PAL;
		else if(0 == strcmp(s, "ntsc"))
			standard = VIDEO_STANDARD_NTSC;
		else if(0 == strcmp(s, "secam"))
			standard = VIDEO_STANDARD_SECAM;
		else
			warning("MmeDecoder: VID_OUT=pal/ntsc/secam not %s",s);
	}

	if(videoSetStandard(video, standard) != DV_ERR_OK) {
		warning("MmeDecoder: Unable to select video standard %s",s);
		return;
	}

	EXBMINFOHEADER *exbi;
	JPEGINFOHEADER *jpegbi;
	int size;
	int bitcount;
	int compression;

	switch(ft) {
	case FT_JPEG:
		size = sizeof(EXBMINFOHEADER) + sizeof(JPEGINFOHEADER);
		bitcount = 24;
		compression = MJPG_DIB;
		break;
	case FT_YUV_411:
	case FT_YUV_422:
		size = sizeof(BITMAPINFOHEADER);
		bitcount = 16;
		compression = BICOMP_DECYUVDIB;
		break;
	default:
		warning("MmeDecoder: unknown type %d", ft);
		return;
	}

	if(0 == (ep = (extbi *)mmeAllocMem(size))) {
		warning("MmeDecoder: Could not allocate header");
		return;
	}

	lpbi = (LPBITMAPINFOHEADER) ep;
	lpbi->biSize = size;

	/*
	** Nail in CIF for the time being. It doesn't
	** matter if the size is wrong. It will be changed
	** in the output routine.
	*/
	lpbi->biWidth = 352;
	lpbi->biHeight = 288;
	lpbi->biPlanes = 1;
	lpbi->biBitCount = bitcount;
	lpbi->biCompression = compression;
	lpbi->biSizeImage = 0;
	lpbi->biXPelsPerMeter = 0;
	lpbi->biYPelsPerMeter = 0;
	lpbi->biClrUsed = 0;
	/*bi.biClrImportant Not set */

	switch(ft) {
	case FT_JPEG:
		exbi = (EXBMINFOHEADER *)lpbi;
		exbi->biExtDataOffset   = sizeof(EXBMINFOHEADER);

		jpegbi = (JPEGINFOHEADER *)
			((unsigned long)exbi + exbi->biExtDataOffset);
		jpegbi->JPEGSize = sizeof(JPEGINFOHEADER);
		jpegbi->JPEGProcess = JPEG_PROCESS_BASELINE;
		jpegbi->JPEGColorSpaceID = JPEG_YCbCr;
		jpegbi->JPEGBitsPerSample = 8;
		jpegbi->JPEGHSubSampling = JPEG_SUBSAMPLING_2;
		jpegbi->JPEGVSubSampling = JPEG_SUBSAMPLING_1;
		break;
	case FT_YUV_411:
	case FT_YUV_422:
		break;
	default:
		warning("MmeDecoder: unknown type %d", ft);
		return;
	}

	res = videoConfigure(video,
			     DVM_FORMAT,
			     VIDEO_CONFIGURE_SET,
			     0,
			     lpbi,
			     lpbi->biSize,
			     0,0);

	if(DV_ERR_OK != res) {
		warning("videoConfigure failed %d\n", errorString(res));
		return;
	}

	/*
	** Allocate the video Header
	*/
	vhdr = (VIDEOHDR *)mmeAllocMem(sizeof(VIDEOHDR));
	memset(vhdr, 0, sizeof(VIDEOHDR));

	/*
	** Force the memory allocation of the video buffer in analogueOut().
	*/
	vbuf = 0;

	active = 1;
}

int
MmeDecoder::checkSize(int fsize, int width, int height)
{
	if(fsize > frame_size) {
		if(vbuf && !mmeFreeBuffer(vbuf)) {
			warning("checkSize: mmeFreeBuffer failed");
		}
		vbuf = 0;
		frame_size = 0;
	}
	if(0 == vbuf) {
		vbuf = (unsigned char *)mmeAllocBuffer(fsize);
		if(0 == vbuf) {
			warning("checkSize: mmeAllocBuffer failed");
			return 0;
		}
		frame_size = fsize;
		if(monochrome)
			memset(vbuf, 128, frame_size);
	}
	if(lpbi->biWidth != width || lpbi->biHeight != height) {
		printf("MmeDecoder::checkSize: size change(%d,%d)\n",
		       width, height);
		lpbi->biWidth = width;
		lpbi->biHeight = height;

		MMRESULT res = videoConfigure(video,
					      DVM_FORMAT,
					      VIDEO_CONFIGURE_SET,
					      0,
					      lpbi,
					      lpbi->biSize,
					      0,0);

		if(DV_ERR_OK != res) {
			warning("checkSize: videoConfigure failed %s",
				   errorString(res));
			return 0;
		}
	}

	return 1;
}

/*
** Input: Motion Jpeg Frame
*/
void
MmeDecoder::analogueOut(JpegFrame *jf)
{
	if(!checkSize(jf->len_, jf->width_, jf->height_))
		return;

	memcpy(vbuf, jf->bp_, jf->len_);

	vhdr->lpData = (char *)vbuf;
	vhdr->dwBufferLength = jf->len_;
	vhdr->dwBytesUsed = jf->len_;

	MMRESULT res = videoFrame(video, vhdr);

	if(DV_ERR_OK != res) {
		warning("analogueOut: videoFrame failed %s",
			   errorString(res));
	}
}

/*
** Input: YUV 411 or YUV 422
** Convert to DEC style YUV 422.
*/
void
MmeDecoder::analogueOut(YuvFrame *jf, int ft)
{
	int width = jf->width_;
	int image_size = jf->width_ * jf->height_;
	int len = image_size * 2;

	if(!checkSize(len, jf->width_, jf->height_))
		return;

	unsigned char *y_inptr;
	unsigned char *u_inptr;
	unsigned char *v_inptr;

	unsigned char *y_outptr;
	unsigned char *uv_outptr;
	unsigned char *uv_outptr_advance;

	int stride = width * 2;

	y_inptr = jf->bp_;
	y_outptr = vbuf;
	uv_outptr = vbuf;
	uv_outptr_advance = vbuf + stride;

	u_inptr = jf->bp_ + image_size;

	/*
	** Set up the luminance first
	*/
	for(int i = 0; i < image_size / 2; i++) {
		*(y_outptr + 0) = *(y_inptr + 0);
		*(y_outptr + 2) = *(y_inptr + 1);

		y_outptr += 4;
		y_inptr += 2;
	}

	if(!monochrome) {
		switch(ft) {
		case FT_YUV_411:
			v_inptr = u_inptr + image_size / 4;
			for(int i = 0; i < jf->height_ / 2; i++) {
				for(int j = 0; j < width / 2; j++) {
					*(uv_outptr + 1) = *u_inptr;
					*(uv_outptr + 3) = *v_inptr;

					*(uv_outptr_advance + 1) = *u_inptr;
					*(uv_outptr_advance + 3) = *v_inptr;

					uv_outptr += 4;
					uv_outptr_advance += 4;

					u_inptr += 1;
					v_inptr += 1;
				}
				uv_outptr += stride;
				uv_outptr_advance += stride;
			}
			break;
		case FT_YUV_422:
			v_inptr = u_inptr + image_size / 2;
			for(int i = 0; i < image_size / 2; i++) {
				*(uv_outptr + 1) = *u_inptr;
				*(uv_outptr + 3) = *v_inptr;

				uv_outptr += 4;

				u_inptr += 1;
				v_inptr += 1;
			}
			break;
		}
	}

	vhdr->lpData = (char *)vbuf;
	vhdr->dwBufferLength = len;
	vhdr->dwBytesUsed = len;

	MMRESULT res = videoFrame(video, vhdr);

	if(DV_ERR_OK != res) {
		warning("analogueOut: videoFrame failed %s",
			   errorString(res));
	}
}

MmeDecoder::~MmeDecoder()
{
	MMRESULT res = videoClose(video);
	if(DV_ERR_OK != res) {
		warning("~MmeDecoder: videoClose failed %s",
			errorString(res));
	}

	if(ep && !mmeFreeMem(lpbi))
		warning("~MmeDecode: mmeFreeMem failed");

	if(vhdr && !mmeFreeMem(vhdr))
		warning("~MmeDecode: mmeFreeMem failed");

	if(vbuf && !mmeFreeBuffer(vbuf))
		warning("~MmeDecode: mmeFreeBuffer failed");

	active = 0;
}

char *
MmeDecoder::errorString(int error)
{
	static char *buf = 0;

	if(0 == buf)
		if(0 == (buf = (char *)mmeAllocMem(MAXERRORLENGTH)))
			return "MmeDecoder::errorString: mmeAlocMem failed";

	if(videoGetErrorText(video, error, buf, MAXERRORLENGTH) == DV_ERR_OK)
		return buf;

	switch(error) {
	case DV_ERR_NONSPECIFIC:
		sprintf(buf, "DV_ERR_NONSPECIFIC");
		break;
	case DV_ERR_BADFORMAT:
		sprintf(buf, "DV_ERR_BADFORMAT");
		break;
	case DV_ERR_CREATEPALETTE:
		sprintf(buf, "DV_ERR_CREATEPALETTE");
		break;
	case DV_ERR_INVALHANDLE:
		sprintf(buf, "DV_ERR_INVALHANDLE");
		break;
	case DV_ERR_NOTSUPPORTED:
		sprintf(buf, "DV_ERR_NOTSUPPORTED");
		break;
	case DV_ERR_ALLOCATED:
		sprintf(buf, "device is allocated");
		break;
	default:
		sprintf(buf, "unknown error %d", error);
		break;
	}

	return buf;
}

class MmeOutboardAssistor : public Renderer, public MmeDecoder {
public:
	inline MmeOutboardAssistor(int ft) : Renderer(ft), MmeDecoder(ft) {}
protected:
	virtual int command(int argc, const char*const* argv);
	virtual void dispatch(int);
 	virtual int consume(const VideoFrame* vf) {
		if(!active)
			return 0;
		switch(ft_) {
		case FT_JPEG:
			analogueOut((JpegFrame *)vf);
			break;
		case FT_YUV_411:
			analogueOut((YuvFrame *)vf, ft_);
			break;
		case FT_YUV_422:
			analogueOut((YuvFrame *)vf, ft_);
			break;
		}
		return 0;
 	}
};

int MmeOutboardAssistor::command(int argc, const char*const* argv)
{
	if(argc == 3) {
		if(strcmp(argv[1], "scale") == 0) {
			/* ignore */
			return (TCL_OK);
		}
	}
	return Renderer::command(argc, argv);
}

void MmeOutboardAssistor::dispatch(int)
{
	printf("MmeOutboardAssistor::dispatch()\n");
}

static class MmeOutputDevice : public OutputDevice {
 public:
	MmeOutputDevice() : OutputDevice("MME") {}
	virtual int command(int argc, const char*const* argv);
} jv_output;


int MmeOutputDevice::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	if(argc == 3) {
		if(strcmp(argv[1], "assistor") == 0 &&
		   strcmp(argv[2], "jpeg/422") == 0) {
 			MmeOutboardAssistor* p = new
				MmeOutboardAssistor(FT_JPEG);
			tcl.result(p->name());
			return TCL_OK;
		}
		if(strcmp(argv[1], "renderer") == 0 &&
		   strcmp(argv[2], "422") == 0) {
 			MmeOutboardAssistor* p = new
				MmeOutboardAssistor(FT_YUV_422);
			tcl.result(p->name());
			return TCL_OK;
		}
		if(strcmp(argv[1], "renderer") == 0 &&
		   strcmp(argv[2], "411") == 0) {
 			MmeOutboardAssistor* p = new
				MmeOutboardAssistor(FT_YUV_411);
			tcl.result(p->name());
			return TCL_OK;
		}
	}
	return OutputDevice::command(argc, argv);
}
