/*
 * video-device.h --
 *
 *      Device template for the video capture hardware.
 *
 * Copyright (c) 1993-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef mash_video_device_h
#define mash_video_device_h

#include "config.h"
#include "timer.h"
#include "media-timer.h"
#include "tclcl.h"
#include "module.h"
#include "crdef.h"

/*
 * Device template for the video capture hardware.
 * Device::instance returns the detected device.  When multiple
 * devices are present, they are distinguished by the device
 * specific type name.  Video capture objects, assistors, and
 * output channels are allocated via methods invoked on the
 * device and the attribute abstractions tells us which of
 * the above are supported.
 */
class VideoDevice : public TclClass {
 public:
	VideoDevice(const char* clsname, const char* nickname);
 protected:
	virtual void bind();
	const char* nickname_;
	const char* attributes_;
};

/*
 * Number of horizontal lines to pad at the top and bottom of the
 * framer buffers that are handed to the Encoder encode method.
 * Some encoders can operate more efficiently with this scratch
 * space (i.e., the bvc coder).
 */
#define VIDCAP_VPAD 1

class VideoCapture : public SourceModule, public Timer, public MediaTimer,
	public ConditionalReplenisher {
 protected:
	VideoCapture();
 public:
	virtual ~VideoCapture();
	virtual int command(int argc, const char*const* argv);
	virtual void fps(int);
	inline int fps() const { return (fps_); }
	virtual void bps(int);
	virtual void start();
	virtual void stop();
	inline int status() const { return (status_); }

	virtual void grab();
	virtual void timeout();
	int background() const;
 protected:
	double tick();
	/* hooks for conditional replenishment algorithm */
	void suppress(const u_char* devbuf);
	void saveblks(u_char* lum);
	virtual void adjust_frameclock(int bps);
	double advance();

	void set_size_422(int w, int h);
	void set_size_411(int w, int h);
	void set_size_cif(int w, int h);
	void allocref();

	int vstart_;
	int vstop_;
	int hstart_;
	int hstop_;

	u_int framesize_;
	u_char* framebase_;
	u_char* frame_;
	u_char* ref_;/*FIXME*/
	int inw_;
	int inh_;
	int outw_;
	int outh_;
	int send_all_;

	EncoderModule* encoder_;

	double frametime_;	/* 1 / fps */
	double frameclock_;	/* logical clock for pumping out frames */
	double lastfc_;		/* last frameclock_ value */

	int fps_;		/* desired frames per second */
	int bps_;		/* desired bits per second */
	int running_;		/* true when capture object active */
	int status_;		/* true if capture object okay */

	double delta_;		/* most recent inter-frame time */

	/* brightness/contrast controls */
	void contrast(double c);
	u_int ymin_;
	u_int ymax_;
	double contrast_;
	u_char ynorm_[256];
};

#define REPLENISH(devbuf, refbuf, ds, bpp, hstart, hstop, vstart, vstop) \
{ \
	/* \
	 * First age the blocks from the previous frame. \
	 */ \
	int mark = age_blocks() | CR_MOTION_BIT | CR_LQ; \
\
	register int _ds = ds; \
	register int _rs = outw_; \
	const u_char* rb = &(refbuf)[scan_ * _rs]; \
	const u_char* db = &(devbuf)[scan_ * _ds]; \
	int w = blkw_; \
	u_char* crv = crvec_; \
 \
	crv += (vstart) * w; \
	for (int y = vstart; y < vstop; ++y) { \
		const u_char* ndb = db; \
		const u_char* nrb = rb; \
		u_char* ncrv = crv; \
		crv += hstart; \
		for (int x = hstart; x < hstop; x++) { \
			int left = 0; \
			int right = 0; \
			int top = 0; \
			int bottom = 0; \
			DIFFLINE(db, rb, left, top, right); \
			db += _ds << 3; \
			rb += _rs << 3; \
			DIFFLINE(db, rb, left, bottom, right); \
			db -= _ds << 3; \
			rb -= _rs << 3; \
 \
			int center = 0; \
			if (left >= 48 && x > 0) { \
				crv[-1] = mark; \
				center = 1; \
			} \
			if (right >= 48 && x < w - 1) { \
				crv[1] = mark; \
				center = 1; \
			} \
			if (bottom >= 48 && y < blkh_ - 1) { \
				crv[w] = mark; \
				center = 1; \
			} \
			if (top >= 48 && y > 0) { \
				crv[-w] = mark; \
				center = 1; \
			} \
			if (center) \
				crv[0] = mark; \
 \
			db += 16 * (bpp); \
			rb += 16; \
			++crv; \
		} \
		db = ndb + (_ds << 4); \
		rb = nrb + (_rs << 4); \
		crv = ncrv + w; \
	} \
}

#endif
