/*
 * Copyright (c) 1997 Carnegie Mellon University. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation is hereby granted (including for commercial or
 * for-profit use), provided that both the copyright notice and this
 * permission notice appear in all copies of the software, derivative
 * works, or modified versions, and any portions thereof, and that
 * both notices appear in supporting documentation, and that credit
 * is given to Carnegie Mellon University in all publications reporting
 * on direct or indirect use of this code or its derivatives.
 *
 * THIS SOFTWARE IS EXPERIMENTAL AND IS KNOWN TO HAVE BUGS, SOME OF
 * WHICH MAY HAVE SERIOUS CONSEQUENCES.  CARNEGIE MELLON PROVIDES THIS
 * SOFTWARE IN ITS ``AS IS'' CONDITION, AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 *
 * Carnegie Mellon encourages (but does not require) users of this
 * software to return any improvements or extensions that they make,
 * and to grant Carnegie Mellon the rights to redistribute these
 * changes without encumbrance.
 */

#define LOOP_DEBUG

#ifdef BSD_KERNEL
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/malloc.h>
#else
#include <stdlib.h>
#endif

#include "hfsc_err.h"
#include "hfsc_def.h"
#include "hfsc_decl.h"
#include "hfsc_fun.h"


/****************************************************************
*
*  NAME: allocCalHeapStruct
*
*  DESCRIPTION:
*    Allocate memory and initialize a 'CalHeapStructure'
*    data structure. 
*
*  PARAMETERS:
*    levels      - the number of levels of the multi-level
*                  calendar queue that stores the eligible
*                  guaranteed requests 
*    relLogUnits - vector containing the relative ratios between 
*                  the size of the time units at two consecutive
*                  levels
*
****************************************************************/
 
CalHeapStruct *allocCalHeapStruct(levels, relLogUnits)
  int    levels;
  u_int *relLogUnits;
{
  CalHeapStruct *pQueue;

  if (!levels)
    return NULL;

  /* allocate memory */
#ifdef BSD_KERNEL
  if (!(pQueue = 
    (CalHeapStruct *)malloc(sizeof(CalHeapStruct), M_TEMP, M_NOWAIT)))
    hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);
  bzero(pQueue, sizeof(CalHeapStruct));
#else
  if (!(pQueue = (CalHeapStruct *)calloc(1, sizeof(CalHeapStruct))))
    hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);
#endif

#ifdef HFSC_CALENDAR
hfsc_panic("Shouldn't execute this!!!!!\n");
  pQueue->pHeap = allocCalPriorQueue(levels, relLogUnits);
#else
  pQueue->pHeap = newHeap();
#endif

  return pQueue;
}  




/****************************************************************
*
*  NAME: rehashCal
*
*  DESCRIPTION:
*    Rehash calendar, i.e., move all the eligible requests into the
*    heap.
*
*  PARAMETERS:
*    pQueue - the input queue
*    ctime  - current time
*
*  NOTE: (yhchu)
*    Replaced all (a op b) comparison operation to (a - b op 0).
*    This is to accomondate clock wrap around.
*    
****************************************************************/

void rehashCal(pQueue, ctime)
  CalHeapStruct *pQueue;
  u_int          ctime;
{
  register int begin = pQueue->ctime & CAL_MASK, i;
  register int end   = ctime & CAL_MASK;
  ReqItem  *pReq, *pTemp;

  if (!pQueue->num) {
    /* nothing to rehash */
    pQueue->ctime = ctime;
    return;
  }

  if (ctime - pQueue->ctime >= CAL_SIZE) {
    /* clock wraped arround ; we need to vist all the beens */
    for (i = 0; i < CAL_SIZE; i++) {
      for (pReq = pQueue->v[i]; pReq; ) {
	/* yc: CAL_SIZE = 1024
	 * CAL_MASK == 1023 
	 * & is like mod 1024
	 */
        if ((pReq->e & CAL_MASK) == i && (int)pReq->e - (int)ctime<=0) { /* yc: */
          pTemp = pReq->next;
          REMOVE_REQ(pQueue->v[i], pReq);
          enqueueReq(pQueue->pHeap, pReq);
          pQueue->num--;
          pReq = pTemp;
	} else {
          pReq = pReq->next;
        }
      }
    }
  } else {
    if (begin - end < 0)  /* yc: */
      for (i = begin; (i - end <= 0); i++) {   /* yc: */
	for (pReq = pQueue->v[i]; pReq; ) {
          if ((pReq->e & CAL_MASK) == i && (int)pReq->e-(int)ctime<=0) { /* yc: */
	    pTemp = pReq->next;
	    REMOVE_REQ(pQueue->v[i], pReq);
	    enqueueReq(pQueue->pHeap, pReq);
	    pQueue->num--;
	    pReq = pTemp;
	  } else {
	    pReq = pReq->next;
	  }
	}
      }
    else {
      for (i = begin; (i - CAL_SIZE) < 0; i++) {
	for (pReq = pQueue->v[i]; pReq; ) {
          if ((pReq->e & CAL_MASK) == i && (int)pReq->e-(int)ctime<=0) { /* yc: */
	    pTemp = pReq->next;
	    REMOVE_REQ(pQueue->v[i], pReq);
	    enqueueReq(pQueue->pHeap, pReq);
	    pQueue->num--;
	    pReq = pTemp;
	  } else
	    pReq = pReq->next;
	}
      }
      for (i = 0; (i - end <= 0); i++) {   /* yc: */
	for (pReq = pQueue->v[i]; pReq; ) {
          if ((pReq->e & CAL_MASK) == i && (int)(pReq->e)-(int)ctime<=0) { /* yc: */
	    pTemp = pReq->next;
	    REMOVE_REQ(pQueue->v[i], pReq);
	    enqueueReq(pQueue->pHeap, pReq);
	    pQueue->num--;
	    pReq = pTemp;
	  } else
	    pReq = pReq->next;
	}
      }
    }
  }
  pQueue->ctime = ctime;
}
    


/****************************************************************
*
*  NAME: enqueueReq2
*
*  DESCRIPTION:
*    Enqueue request. As long as the eligible key (e) of 
*    'pReq' is larger than 'ctime' the request is inserted in
*    the calendar 'v'. Otherwise the request is inserted in the
*    'heap'. 
*
*  PARAMETERS:
*    pQueue - 
*    pReq   - request to be enqueued
*    ctime  - current time
*    
*  NOTE: (yhchu)
*    Replaced all (a op b) comparison operation to (a - b op 0).
*    This is to accomondate clock wrap around.
*    
****************************************************************/
 
void enqueueReq2(pQueue, pReq, ctime) 
  CalHeapStruct *pQueue;
  ReqItem       *pReq;
  u_int          ctime;
{
  u_int idx;

#ifdef REQ_DEBUG
  if (pReq->status != INACTIVE)
    hfsc_panic("error in enqueueReq2\n");
  pReq->status = IN_CALENDAR;
#endif
  if ((int)ctime - (int)pQueue->ctime > 0)  /* yc: */
    rehashCal(pQueue, ctime);

  if ((int)pReq->e - (int)ctime <= 0)   /* yc: */
    enqueueReq(pQueue->pHeap, pReq);
  else {
    (pQueue->num)++;
    INSERT_REQ(pQueue->v[idx = (pReq->e & CAL_MASK)], pReq);
  }
}



/****************************************************************
*
*  NAME: dequeueFirstReq2
*
*  DESCRIPTION:
*    Dequeue the eligible request with the smallest deadline
*
*  PARAMETERS:
*    pQueue - 
*    ctime  - current time
*    
*  NOTE: (yhchu)
*    Replaced all (a op b) comparison operation to (a - b op 0).
*    This is to accomondate clock wrap around.
*    
****************************************************************/
 
ReqItem *dequeueFirstReq2(pQueue, ctime) 
  CalHeapStruct *pQueue;
  u_int          ctime;
{
  /* was if ((int)ctime > (int)pQueue->ctime) */
  if (((int)ctime - (int)pQueue->ctime > 0) ||  /* yc: */
      /* hack, to deal with the case where ctime < 0 initially */
      (pQueue->ctime == 0))    
    rehashCal(pQueue, ctime);
  return dequeueReq(pQueue->pHeap);
}


/****************************************************************
*
*  NAME: dequeueReq2
*
*  DESCRIPTION:
*    Dequeue the specified request
*
*  PARAMETERS:
*    pQueue - 
*    pReq   - request to be dequeued
*    ctime  - current time
*    
*  NOTE: (yhchu)
*    Replaced all (a op b) comparison operation to (a - b op 0).
*    This is to accomondate clock wrap around.
*    
****************************************************************/
 
void dequeueReq2(pQueue, pReq, ctime) 
  CalHeapStruct *pQueue;
  ReqItem       *pReq;
  u_int          ctime;
{
  u_int idx;

  if ((int)ctime - (int)pQueue->ctime > 0) /* yc: */
    rehashCal(pQueue, ctime);

  if ((int)pReq->e - (int)ctime <= 0) {    /* yc: */
    /* request is in heap */
    removeReq(pQueue->pHeap, pReq->idx);
    /* dequeueReq(pQueue->pHeap, pReq); */
  }
  else {
    /* request is in multi-level calendar queue; remove it */
    pQueue->num--;
#ifdef REQ_DEBUG    
    pReq->status = INACTIVE;
#endif
    REMOVE_REQ(pQueue->v[idx = (pReq->e & CAL_MASK)], pReq);
  }
}


