/*
 * Copyright (c) 1998 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *       This product includes software developed by the Computer Systems
 *       Engineering Group at Lawrence Berkeley Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * ---------------------------
 *
 * Filename: ns-srmv2.cc
 *   -- Author: Suchitra Raman <suchi@cs.berkeley.edu>
 *
 * @(#) $Header: /usr/mash/src/repository/srmv2/srmv2/ns-srmv2.cc,v 1.23 2000/12/10 02:37:24 lim Exp $
 * 
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>

#include "common/ntp-time.h"
#include "srmv2/srmv2.h"
#include "srmv2/srmv2-api.h"
#include "srmv2/ns-srmv2.h"
#include "srmv2/source-srmv2.h"
#include "common/random.h"


SRMv2_NameSpace::SRMv2_NameSpace(u_int32_t /*rootID*/)
{
        dirty_sig_ = 1;
	/*
	 * FIXME make this randomized, 
	 * but let's worry about that later... 
	 */
	SrmInitHashTable(&nodes_, HASH_ONE_WORD_KEYS);
	/*nodes_ = new (NS_Node*) [MAX_NODES];
	for (int i=0; i < MAX_NODES; i++) 
		nodes_[i] = 0;*/
	init_prio();
	cid_  = 0;
	root_ = new NS_Node(0, cid_, 0, 0);
	unsigned char rootname[5];
	strcpy((char*)rootname, "root");
	root_->name(rootname, 5);
	insert_node(cid_, root_);
	
}


SRMv2_NameSpace::~SRMv2_NameSpace()
{
	HashSearch search;
	HashEntry  *entry;
	NS_Node    *node;

	/* Delete all nodes here
	 * Since the root has also been inserted into the hash table,
	 * we don't need to explicitly delete root_
	 */
	while ( (entry = FirstHashEntry(&nodes_, &search))!=NULL ) {
		node = (NS_Node*) GetHashValue(entry);
		delete node;
	}

	DeleteHashTable(&nodes_);
}


unsigned int
SRMv2_NameSpace::calloc(unsigned int parent, unsigned int off, 
			int *created, unsigned int cid)
{
	NS_Node *pnode = getnode(parent);
	NS_Node *cnode = 0;
	
	if (cid == SRMv2_ALLOC) {
		cid = ++cid_;
	}
	if (pnode != 0) {
		cnode = getnode(cid);
		if (cnode==0) {
			cnode = pnode->calloc(cid, off, created);
			insert_node(cid, cnode);
		}
	}

	if (*created) {
	  dirty_signature();
	}

	/* Recompute signatures. */
	this->compute_signature();
	if (cnode != 0)
		return cnode->getcid();
	return 0;
}

/*
 * Recursively compute the signature of nodes in the 
 * tree, starting from 'node'. Since the 128-bit MD-5 
 * hash function is too long, we XOR the individual 
 * random 32-bit quantities to get a 32-bit quantity. 
 * Since MD-5 guarantees that the 128-bits generated 
 * are random, so is their XOR.
 */
void
SRMv2_NameSpace::compute_signature()
{
  if (dirty_sig_) {
    root()->compute_signature();
  }
  dirty_sig_ = 0;
}

/*
 * Return an NS_Node* from a cid.
 */
NS_Node *
SRMv2_NameSpace::getnode(unsigned int id)
{
	HashEntry *entry;
	entry = FindHashEntry(&nodes_, (const char*)id);
	if (entry) {
		return (NS_Node*) GetHashValue(entry);
	} else return NULL;
	
	//return nodes_[id];
}

void
SRMv2_NameSpace::insert_node(unsigned int id, NS_Node* node)
{
	int isNew;
	HashEntry *entry;
	entry = CreateHashEntry(&nodes_, (const char*)id, &isNew);
	if (isNew) {
		SetHashValue(entry, node);
	}
}

/* 
 * This is necessary only for sources for which we send SRMv2_ANNOUNCE
 * messages. Should be invoked explicity, for other SRMv2_Source
 * objects, prio_ is unused.  
 */
void
SRMv2_NameSpace::init_prio()
{
	prio_ = new activity[MAX_NODES];
	for (int i=0; i<MAX_NODES; i++) {
		prio_[i].tix = SRMv2_INIT_PRIO;
		prio_[i].hot = SRMv2_UNUSED;
		prio_[i].selected = 0;
	}
}
void
SRMv2_NameSpace::update_prio(unsigned int cid)
{
	if (prio_[cid].tix < 2 * SRMv2_INIT_PRIO) 
		prio_[cid].tix = 2 * SRMv2_INIT_PRIO;
	prio_[cid].hot = SRMv2_HOT;
	prio_[cid].selected = 0;
}

/* 
 * Generate the k (= 'count') highest "lottery winners". 
 * Do this probabilistically, so that there's no starvation. 
 * Coupon collecting analysis shows that we need on the average about
 * count*log(count) random numbers to get 'count' distinct winners. 
 */
void
SRMv2_NameSpace::hiprio(naddr **comps, int *count, int lim)
{
	int i, j, k, total = 0;
	int success = 0;
	NS_Node *node = 0;

	/* No prio matirx! Don't know what happened. */
	if (!prio_) {
		*count = 0; 
		*comps = 0;
		return;
	}

	/* If there's room to send all containers do so */
	if (1 + cid_ <= (unsigned) lim) {
		*count = cid_+ 1; // cid_  + 1
		*comps = new naddr[*count];
		for (j = cid_; j >= 0; j--) {
			node = getnode(j);
			if (node && prio_[j].tix) {
				build_record(& (*comps)[success++], j);
				prio_[j].tix = SRMv2_INIT_PRIO;
			}
		}
		return;
	}
	float* winner = new float[lim * lim];
	success = 0;
	*count = 0;
	*comps = new naddr[lim];
	while (lim) {
		total = 0;
		for (i=0; (i<MAX_NODES) && (prio_[i].selected == 0); i++)
		        total += prio_[i].tix;
		for (i=0; i < lim * lim; i++)
			winner[i] = total * (double)random()/double(INT_MAX);
		sort_prio(winner, lim * lim);
		i = 0, j = 0, total = 0;
		while (j < MAX_NODES && i < lim * lim) {
		        /* winner should catch up to total */
		        while (winner[i] < total) {
				i++;
				continue;
			}
			if (getnode(j) != 0) {
			  if (!prio_[j].selected) {
			    if (winner[i] <
				total + prio_[j].tix) {
			      prio_[j].selected = 1;
			      success++;
			      build_record(& (*comps)[(*count)++], j);
			      /* 
			       * Re-initialize priority, now that we've 
			       * selected this container 
			       */
			      prio_[j].tix = SRMv2_INIT_PRIO;
			      total += prio_[j++].tix;
			      if (success == lim) {
				for (k = 0; k < MAX_NODES; k++) 
				  prio_[k].selected = 0;
				delete [] winner;
				return;
			      }
			    } else {
			      /* total should catch up to winner */
			      total += prio_[j].tix;
			      prio_[j++].tix ++; /* additively increase prio
						    if we pass this container*/
			    }
			  } else {
			    j++;
			  }
			} else {
			  j++;
			}
		}
		lim -= success;
		success = 0;
	}
	return;
}

void
SRMv2_NameSpace::sort_prio(float* a, int sz)
{
	int i, j;
	float temp;

	for (i = 1; i < sz; i++) {
		j = i;
		temp = a[j];
		while (j > 0 && a[j-1] > temp) {
			a[j] = a[j-1];
			j--;
		}
		a[j] = temp;
	}
}

void
SRMv2_NameSpace::build_record(naddr *ar, unsigned int cid)
{
	NS_Node* node = getnode(cid);
	const SRMv2_Signature &sign = node->signature();
	ar->cid       = cid;
	ar->pcid      = node->getpcid();
	ar->childsig  = sign.childsig;
	ar->adu       = sign.edge.adu;
	ar->byte      = sign.edge.byte;
	ar->offset    = 0;

	if (node->parent()) {
		int offset = node->parent()->offset(cid);
		if (offset > 0) {
			ar->offset = offset;
		}
	}
	srm_trace(srmSRM, ("[%d, %d, 0x%x, %d, %d, %d]\n", ar->cid, ar->pcid, 
			   ar->childsig, ar->adu, ar->byte, ar->offset));
	return;
}

void
SRMv2_NameSpace::update_edge(unsigned int cid, unsigned int seqno, 
			     unsigned int ebytes)
{
	NS_Node* node = getnode(cid);
	node->regadu(seqno, ebytes);
}

void
SRMv2_NameSpace::ancestors(unsigned int cid, naddr** path, unsigned int *npar)
{
	NS_Node* np = getnode(cid);
	SRMv2_Signature sign;
	unsigned int count = np->depth() + 1;
	*path = new naddr [count];
	*npar = count;
	naddr* comps = *path;
	
	while (np) {
		--count;
		sign = np->signature();
		comps[count].offset    = np->offset();
		comps[count].cid       = np->getcid();
		comps[count].pcid      = np->getpcid();
		comps[count].childsig  = sign.childsig;
		comps[count].adu       = sign.edge.adu;
		comps[count].byte      = sign.edge.byte;
		np = np->parent();
	}
}

/* ********************* NS_Node ************************* */

NS_Node::NS_Node(NS_Node* parent, unsigned int cid, unsigned int off, 
		 unsigned int depth, int max_degree, int max_dbufs) :
	degree_(0), max_degree_(max_degree), 
	ndbufs_(0), max_dbufs_(max_dbufs), 
	cid_(cid), off_(off), depth_(depth), 
	child_(0), parent_(parent),
	name_(NULL), name_len_(0)
{
	if (max_degree > 0) 
		child_ = new (NS_Node *)[max_degree];
	for (int i=0; i<max_degree_; i++) {
		child_[i] = 0;
	}

	if (max_dbufs_ > 0) 
	        dbufs_ = new adubuf[max_dbufs_];
}

NS_Node::~NS_Node() 
{
	for (int i = 0; i < degree_; i ++)
		delete [] child_[i];
	delete [] child_;

	if (name_ && name_len_) delete [] name_;
}

/* 
 * writef() may generate a repair request if necessary. 
 */
void
NS_Node::writef(SRMv2_pktbuf *pb, SRMv2_Source* src, SRMv2_Session* session, 
		u_int32_t seqno, Bool eoa, u_short *ttl)
{
	/* FIXME else, need to grow the array */
	if (seqno >= (unsigned) max_dbufs_) {
		srm_trace(srmSRM,
			  ("Slots in node exceeded (have %d, got seqno %u). "
			   "Calling realloc...", max_dbufs_, seqno));
		expand_dbufs(seqno);
		srm_trace(srmSRM,
			  ("Slots in node expanded to %d", max_dbufs_));
	}
	SRMv2_Edge left_edge(seqno, pb->offset);
	
	/* Compute the next byte past the last one we received. */
	SRMv2_Edge right_edge;
	if (eoa) 
		right_edge.set_edge(seqno + 1, 0);
	else
		right_edge.set_edge(seqno, pb->offset + pb->len);

	/* 
	 * We do not explicitly query the application through the 
	 * session object to before requesting a repair. FIXME We need 
	 * a way to garbage collect unassembled fragments. 
	 */
	dbufs_[seqno].insert(pb);
	requests_.cancel(cid_, left_edge, right_edge, ttl);
	replies_.cancel(cid_, left_edge, right_edge);
	
	/* 
	 * Check for missing pieces only if there is receiver interest. 
	 * If this returns 1, a repair request is scheduled for the 
	 * entire range. Both start and end are included in the range. 
	 */

	check_holes(src, session, left_edge);
	
	/* Update the right edge if necessary. */
	if (signature_.edge < right_edge)
		signature_.edge.set_edge(right_edge.adu, right_edge.byte);
}

/* check_holes -- 
 * Use left to detect a gap. If one is found, schedule a request.
 */
void
NS_Node::check_holes(SRMv2_Source* src, SRMv2_Session* s,
		     const SRMv2_Edge& left) 
{
	SRMv2_Edge &expected = signature_.edge;
	if (!(expected < left)) {
		/* we've already looked for holes in that region */
		return;
	}

	unsigned int i, last;
	if (left.byte == 0) 
		last = left.adu - 1;
        else last = left.adu;
	
	/* Check first if the application wants this repaired. */
	if (s->srm_should_recover(src, cid_, expected.adu, last) != 0) {
		for (i=expected.adu; i < last; i++) {
			dbufs_[i].ignore = 1;
		}
		SRMv2_Request *req = new SRMv2_Request(cid_, expected, left);
		req->session(s);
		req->source(src);
		requests_.schedule(req);
	}
}

void
NS_Node::expand_dbufs(u_int32_t seqno) {
	/* Grow exponentially */
	adubuf *new_dbufs;
	int old_max;

	old_max = max_dbufs_;

	while (seqno >= (unsigned) max_dbufs_) {
		max_dbufs_  *= 2;
	}

	new_dbufs = new adubuf[max_dbufs_];

	int i;
	for(i=0; i< old_max; i++) {
		new_dbufs[i] = dbufs_[i];
		dbufs_[i].data = 0;
	}
	delete [] dbufs_;

	dbufs_ = new_dbufs;

	if (dbufs_ == 0) {
		printf("Realloc failed.\n");
		abort();
	}
}

/*void
NS_Node::set_edge(unsigned int a, unsigned int b) {
	expected_.set_edge(a, b);
}*/

NS_Node*
NS_Node::is_child(unsigned int id) 
{
	NS_Node *child;

        for (int i = 0; i < degree_; i++) {
		child = child_[i];
		if (id == child->getcid()) 
			return child;
	}
	return NULL;
}

int
NS_Node::offset(unsigned int id) 
{
	NS_Node *child;

        for (int i = 0; i < degree_; i++) {
		child = child_[i];
		if (id == child->getcid()) 
			return i;
	}
	return -1;
}

NS_Node*
NS_Node::child(unsigned int id) 
{
	if (id < (unsigned) degree_) {
		return child_[id];
	} else return 0;
}

NS_Node*
NS_Node::calloc(unsigned int cid, unsigned int off, int *created)
{
	*created = 0;
	/* Special case: use the next available offset. */
	if (off == SRMv2_ALLOC) {
		off = degree_;
		degree_++;
	} else if ((int) off >= degree_) { /* Update degree_ to be the
					    * right-most offset seen. */
		degree_ = off + 1;
		srm_trace(srmSRM, ("Incrementing degree(%d) to %d "
				   "(points to %d)", cid_, degree_, cid));
	}

	if ((int) off < max_degree_ && child_[off] == 0) {
		*created = 1;
		child_[off] = new NS_Node(this, cid, off, depth_+1);
	} else {
		// FIXME: must grow the node array
		srm_trace(srmSRM, ("exceeded max_degree (%d) of node %d,\n"
				   "or child already exists at offset %d\n",
				   max_degree_, cid_, off));
	}

	return child_[off];
}

adubuf*
NS_Node::reassemble(int seqno)
{
	if (dbufs_[seqno].concatenate() > 0) 
		return &dbufs_[seqno];
	else return 0;
}

void
NS_Node::regadu(unsigned int seqno, unsigned int ebytes)
{
        if (seqno >= (unsigned) max_dbufs_) {
		srm_trace(srmSRM,
			  ("Slots in node exceeded (have %d, got seqno %u). "
			   "Calling realloc...", max_dbufs_, seqno));
		expand_dbufs(seqno);
		srm_trace(srmSRM,
			  ("Slots in node expanded to %d", max_dbufs_));
	}

	if ((unsigned) dbufs_[seqno].len < ebytes + 1)
		dbufs_[seqno].len = ebytes + 1;
	/* 
	 * Since this is only meaningful on the sender's side,
	 * we need not schedule a RREQ. We must push ndbufs_ 
	 * to the right, and never to the left. 
	 */
	if ((unsigned) ndbufs_ < seqno)
		ndbufs_ = seqno;
	signature_.edge.set_edge(seqno + 1, 0);
}

#ifdef YYY
adubuf*
NS_Node::get_nodeinfo() 
{
	int i; 
	adubuf* buffer = new adubuf; 
	buffer->data = (unsigned char *) new srmv2_announcehdr[degree_ + 1];
	srmv2_announcehdr* ahdr = (srmv2_announcehdr*) buffer->data;
	/* First fill in data about this node : ndbufs_ and ebytes */
	ahdr[0].cid    = cid_;
	if (degree_) {
		ahdr[0].sign   = ndbufs_;
	} else {
		ahdr[0].sign = expected_.byte;
	}
	for (i=0; i<degree_; i++) {
		ahdr[i+1].cid    = child_[i]->getcid();
		ahdr[i+1].sign   = child_[i]->signature();
	}
	buffer->len = (degree_ + 1) * sizeof(srmv2_announcehdr);
	return buffer;
}
#endif


/*Bool
NS_Node::match_signature(SRMv2_Edge *sign) 
{	
	return (*sign == signature_);
}*/

void
#ifndef NDEBUG
NS_Node::display(int indent)
#else
NS_Node::display(int /* indent */)
#endif
{
#ifndef NDEBUG
	int i;
	srm_trace_start(srmSRM);
	for (i = 0; i < indent; i++)
		srm_trace_out(srmSRM, ("   "));
	srm_trace_out(srmSRM, ("%4d | 0x%x %u %u", cid_, signature_.childsig,
			       signature_.edge.adu, signature_.edge.byte));
	srm_trace_end(srmSRM);

	for (i = 0; i < degree_; i++) {
		if (child_[i]) {
			child_[i]->display(indent+1);
		} else {
			srm_trace(srmSRM, ("null child %d encountered\n", i));
		}
	}
#endif
}

adubuf::adubuf() : len(0), unused(0), 
	seqno(SRMv2_ERROR), data(0),
	dp(0), fraglist(0), lastseen(0), ignore(0)
{ }

adubuf::adubuf(unsigned int length) : len(length), fraglist(0) 
{ 
	data = new unsigned char[length];
	dp = data;
}

adubuf::~adubuf() 
{
	if (data) delete [] data;
}

void 
adubuf::copy(unsigned char *buffer, int buflen) 
{
	len = buflen;
	data = new unsigned char [len+1];
	memcpy(data, buffer, buflen);
	dp = data;
}

/*
 * Caller must ascertain that there is enough space.
 * The only way to do this now is to allocate enough 
 * at the beginning. 
 */

void
adubuf::append(unsigned *buffer, int buflen)
{
	memcpy(data + unused, buffer, buflen);
	unused += buflen;
	dp += buflen;
}


void
NS_Node::request_missing(SRMv2_Source *src, SRMv2_Session *session,
			 unsigned int seqno)
{
	SRMv2_pktbuf *curr = dbufs_[seqno].fraglist;
	SRMv2_pktbuf *prev=0;
	SRMv2_Edge left, right;
	SRMv2_Request *req;
	
	left.set_edge(seqno, 0);
 	while (curr) {
		if ((int) left.byte < curr->offset) {
			right.set_edge(seqno, curr->offset);
			req = new SRMv2_Request(cid_, left, right);
			req->session(session);
			req->source(src);
			requests_.schedule(req);
			left.set_edge(seqno, curr->offset + curr->len);
		}
		prev = curr;
		curr = curr->next;
	}

	/* This ADU is missing a tail piece. Request it. */
	if (dbufs_[seqno].lastseen == 0) {
		right.set_edge(seqno + 1, 0);
		req = new SRMv2_Request(cid_, left, right);
		req->session(session);
		req->source(src);
		requests_.schedule(req);
	}
}


void
adubuf::insert(SRMv2_pktbuf *pb) 
{
	SRMv2_pktbuf *prev = 0;
	SRMv2_pktbuf *curr;
	int i;

	if (pb->offset + pb->len > len) {
		len = pb->offset + pb->len;
	}
	
	/* 
	 * Do nothing if the current ADU was already passed 
	 * up to the application.
	 */
	if (fraglist == 0 && lastseen == 1) {
		pb->release();
		return;
	}		

        /*
         * Find a segment which begins after this one does.
         */
	for (curr = fraglist; curr != 0; prev = curr, curr = curr->next)
		if (curr->offset > pb->offset) 
			break;

	/*
         * If there is a preceding segment, it may provide some of
         * our data already.  If so, drop the data from the incoming
         * segment.  If it provides all of our data, drop us.
         */
	if (prev != 0) {
		i = prev->offset + prev->len - pb->offset;
		if (i > 0) {
			if (i >= pb->len)
				pb->release();
			else {
				pb->offset += i;
				pb->dp  += i;
				pb->len -= i;
				prev->next = pb;
				pb->next = curr;
				prev = pb;
			}
		} else {
			prev->next = pb;
			pb->next = curr;
			prev = pb;
		}
	} else {
		pb->next = curr;
		fraglist = pb;
	}

	/*
         * While we overlap succeeding segments trim them or,
         * if they are completely covered, dequeue them.
         */
	while (curr != 0 && pb->offset + pb->len > curr->offset) {
		i = (pb->offset + pb->len) - curr->offset;
		if (i < curr->len) {
			curr->len -= i;
			curr->offset += i;
			curr->dp += i;
			break;
		}
		prev->next = curr->next;
		curr->release();
		curr = prev->next;
	}
}

/* 
 * Retransmissions should have the "more" bit set to 0
 * This will trigger the following test. 
 */
int
adubuf::contiguous() 
{
	SRMv2_pktbuf *p;
	unsigned int bytes = 0;
	for (p = fraglist; p != 0; p = p->next) {
		if ((unsigned) p->offset != bytes) 
			return (-1);
		bytes += p->len;
	}
	return (bytes);
}

/*
 * Concatenate fragments.
 */
int
adubuf::concatenate() 
{
	SRMv2_pktbuf *prev = 0;
	SRMv2_pktbuf *curr = fraglist;

	/* 
	 * This returns the length of the ADU.
	 */
	int adulen = contiguous();
	if (adulen > 0) {
		len = adulen;
		data = new unsigned char[adulen+1];
		unsigned char *dp = data;
		while (curr) {
			memcpy(dp, curr->dp, curr->len);
			dp += curr->len;
			prev = curr;
			curr = curr->next;
			if (prev != 0) 
				prev->release();
		}
		fraglist = 0;
	}
	return adulen;
}

/* 
 * The signature function is computed as 
 * s(n) = md5_digest(ndbufs_, s(n1), ..., s(nk)), for internal nodes.
 *              = ndbufs_, for leaf nodes. 
 * Since MD-5 only takes strings, we feed it with the concatenation 
 * of signatures of nodes n1, n2, ..., nk in hex, prepended by the 
 * ndbufs_.
 */

void
NS_Node::compute_signature(int top_down=1) {
	int sig_words[4];
	int i = 0;
	/* Leaf node */

	//signature_.edge = expected_;
	
	if (degree_ == 0) {
		signature_.childsig = 0;
		return;
	}

	/* Internal node */
	char* inpstr = new char[degree_*30 + 10];
	inpstr[0] = 0;

	char tmpstr[31];
	for (i = 0; i < degree_; i++) { 
		if (child_[i] != 0) {
			if (top_down)
				child_[i]->compute_signature();
			sprintf(tmpstr, "%x%x%x",
				child_[i]->signature_.childsig,
				child_[i]->signature_.edge.adu,
				child_[i]->signature_.edge.byte);
			strcat(inpstr, tmpstr);
		} 
	}
	md5_digest(inpstr, (unsigned char *) &sig_words[0]);
	signature_.childsig = sig_words[0];
	for (i = 1; i < 4; i++)
		signature_.childsig ^= sig_words[i];

	delete [] inpstr;
	return;
}

