/*
 * Copyright (c) 1998 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *       This product includes software developed by the Computer Systems
 *       Engineering Group at Lawrence Berkeley Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * ---------------------------
 *
 * Filename: pktbuf-srmv2.cc
 *   -- Author: Suchitra Raman <suchi@cs.berkeley.edu>
 *
 * @(#) $Header: /usr/mash/src/repository/srmv2/srmv2/pktbuf-srmv2.cc,v 1.20 2000/12/10 02:37:24 lim Exp $
 * 
 */


#include "pktbuf-srmv2.h"
#include "ntp-time.h"
#include "pktbuf.h"
#include "srmv2.h"
#include "session-srmv2.h"
#include "ns-srmv2.h"
#include <math.h>

SRMv2_pktbuf*
SRMv2_BufferPool::alloc(unsigned char tcid, unsigned char type=0, unsigned int addr=0)
{
	SRMv2_pktbuf* pb = SRMv2_BufferPool_::alloc();
	pb->mode = SRMv2_MULTICAST; /* Default mode is multicast. */

#ifdef SRMv2_LOC_REC
	if (type == SRMv2_REXMIT) 
		pb->mode = SRMv2_UNICAST;
#endif
	pb->addr    = addr;
	pb->manager = this;
	pb->tcid    = tcid;
	build_cmnhdr(pb, tcid, type);
	return (pb);
}

/*
 *     (8)      (8)         (8)          (8)
 * +--------+---------+--------------+-----------+
 * | flags  | pkttype |  tcid (HFSC) |  reserved |
 * +--------+---------+--------------+-----------+
 *
 *
 * SRMv2 Flags.
 * 0   1   2   3   4   5   6   7   
 * +---+---+---+---+---+---+---+---+
 * |   V   | P | U |   Reserved    |
 * +---+---+---+---+---+---+---+---+
 * 
 * V: 2-bit SRM version number
 * 
 * P: 1-bit padding, set when padded to closest
 *    word boundary in encrypted packets. 
 *
 * U: Unicast bit set when packet is unicast.
 *    Unicast address appears in the IP header.
 *    This is used in 2-step recovery when the initial
 *    response is unicast to the requesting host. 
 *
 * Reserved: Unused bits
 * 
 * pkttype: SRMv2 protocol message type.
 *          One of SRMv2_DATA, SRMv2_REXMIT, SRMv2_RREQ,
 *          SRMv2_NSREQUEST, SRMv2_NSREPAIR,
 *          SRMv2_ANNOUNCE, SRMv2_DELAYS,
 * Note: It is not necessary that original and retransmitted data
 * have seperate codes. This difference must go away eventually. 
 * 
 * tcid: See hfsc-api.h for default traffic class IDs.
 */


void
SRMv2_BufferPool::build_cmnhdr(SRMv2_pktbuf *pb, unsigned char tcid, unsigned char type)
{
	srmv2_cmnhdr *sh = (srmv2_cmnhdr *)pb->data;
	sh->tcid  = tcid;
	sh->type  = type;
	unsigned char mode;
#ifdef SRMv2_LOC_REC
	mode = (type == SRMv2_REXMIT) ? 1 : 0;
#else
	mode = 0;
#endif
	sh->unused = 0;
	sh->ipaddr = htonl(LookupLocalAddr());
	sh->flags = (SRMv2_VERSION << 6) | (mode << 4);
	pb->dp = pb->data + sizeof(srmv2_cmnhdr);
	pb->len = sizeof(srmv2_cmnhdr);
	pb->ttl = sttl_;
	return;
}


/* Container header: To identify the container of the
 *                   data carried in this packet. 
 * 
 *                    (32)
 * +--------+---------+---------------------------+
 * |               source ID (0-31)               | 
 * +--------+---------+---------------------------+
 * |               source ID (32-63)              |
 * +--------+---------+---------------------------+
 * |               container ID                   |
 * +--------+---------+---------------------------+
 *
 * source ID: Obtained by the exclusive OR of the two halves
 * of the 16 B quantity from md5_digest(). See source-srmv2.cc
 * for more details on the digest function and its return value. 
 * 
 * conatiner ID: ID allocated to corresponding container by
 *               the original source (also called owner).                 
 *
 * This is the {source, container ID} w.r.t the original source. 
 * Hence, repair responses do not contain the IDs of the responding
 * host. Data is identified, not the host currently transmitting it. 
 */

void
SRMv2_BufferPool::build_conhdr(SRMv2_pktbuf *pb, int *srcid, u_int32_t cid)
{
	srmv2_conhdr *sh   = (srmv2_conhdr *)pb->dp;
	for (int i = 0; i < 4; i ++) 
		sh->srcid[i]  = htonl(srcid[i]);
	sh->cid       = htonl(cid);
	
	pb->dp += sizeof(srmv2_conhdr);
	pb->len += sizeof(srmv2_conhdr);
	return;
}


/*
 *                    (32)
 * +--------+---------+---------------------------+
 * |               source ID (0-31)               | 
 * +--------+---------+---------------------------+
 * |               source ID (32-63)              |
 * +--------+---------+---------------------------+
 *
 * The Source header needs to be used in the two-step
 * recovery process, where the first response is transmitted
 * via unicast to the requesting host. In this case, the
 * requesting host includes its source ID to the packet header. 
 * Again, source ID is as explained under "container header"
 * build_conhdr above.
 */
void
SRMv2_BufferPool::build_srchdr(SRMv2_pktbuf *pb, int *srcid)
{
	int* sid   = (int *)pb->dp;
	for (int i = 0; i < 4; i ++) 
		sid[i]  = htonl(srcid[i]);
	pb->dp += 4 * sizeof(int);
	pb->len += 4 * sizeof(int);
	return;
}


/*
 * An ADU header appears ahead of each ADU/fragment.
 *
 * F: Fragment flag indicates whether this is a whole ADU or
 *    a fragment. 
 * L: LastADU flag indicates whether this is the last ADU in a
 *    stack of ADUs within a packet or not
 * resv: Unused bits
 * adutype: ADU data type set by the application. This may be used,
 *          for example, in mb for the different media and drawing types
 *          in webcast, for HTML, GIF, JPEG, etc. 
 * adulen/fraglen: Length of ADU is F is not set, or
 *                 fragment length otherwise. 	   
 * seq#: Last level in the SNAP hierarchy permits data
 *       to be identified using this sequence number.
 *        
 * adu timestamp: 64-bit timestamp that is supplied by the application.
 *                May be used as the creation time of the ADU at the
 *                application. Note: This may be (much) earlier than the
 *                instant of transmission. 
 *
 *     (8)      (8)           (16)
 * +-+------+---------+---------------------------+
 * |F|L|resv| adutype |      adulen/fraglen       |
 * +-+------+---------+---------------------------+
 * |              seq# (or adu#)                  |
 * +--------+---------+---------------------------+
 * |               adu timestamp (0-31)           | 
 * +--------+---------+---------------------------+
 * |               adu timestamp (32-63)          | 
 * +--------+---------+---------------------------+
 *
 */

void
SRMv2_BufferPool::build_aduhdr(SRMv2_pktbuf *pb, u_int8_t atype,u_int16_t alen,
			       u_int32_t seqno, ntp64 timestamp, u_int8_t frag)
{
	srmv2_aduhdr *sh   = (srmv2_aduhdr *)pb->dp;
	sh->flags        = frag << 7;
	sh->flags       |= 1 << 6;    /* always set the L bit to 1;
				       * set it to 0 explicitly in
				       * SRMv2_Session::send_adu */
	sh->atype        = atype;
	sh->alen         = htons(alen);
	sh->seqno        = htonl(seqno);

	sh->ts[0]  = htonl(timestamp.lower);
	sh->ts[1]  = htonl(timestamp.upper);
	
	pb->dp  += sizeof(srmv2_aduhdr);
	pb->len += sizeof(srmv2_aduhdr);
	return;
}

/* A Fragment header appears at the head of each fragment.
 *
 * M: "More" bit that indicates that this is not the last fragment
 *     of the ADU. 
 * [Note: The sender knows the total length of the ADU at the time it's
 *        transmission starts at the sender. Hence, an alternative to the
 *        "stream" approach with the "more" bit is to include the total
 *         ADU length in all fragments. This was not done in order to save
 *         32-bit per packet overhead. IP performs the same optimization.
 *         We cannot simply include the ADU length in the first fragment alone,
 *         since this would be sensitive to the loss of the first fragment.] 
 * 
 * offset:  Is the byte offset of the first byte in the fragment
 *          relative to the first byte in the adu 
 * 
 *  (1)                (31)
 * +---+---------+---------------------------+
 * | M |        offset                       |
 * +---+---------+---------------------------+
 *
 */
void
SRMv2_BufferPool::build_fraghdr(SRMv2_pktbuf *pb, u_int8_t more, u_int32_t offset)
{
	srmv2_fraghdr *sh = (srmv2_fraghdr *)pb->dp;
	if (offset >> 31) {
		printf("oversized fragment! This can't happen.\n");
		abort();
	}
	*sh = offset;
	if (more == 1) 
		*sh  |= 0x80000000;
	*sh = htonl(*sh);
	pb->dp  += sizeof(srmv2_fraghdr);
	pb->len += sizeof(srmv2_fraghdr);
	return;
}




/* RREQ header: Appears before each repair request. 
 *
 * Note: Multiple repair reqeusts may eb stacked in one
 * packet. Our current implementation (timer-per-request)
 * makes this slightly more involved. 
 *
 * Each RREQ is a fixed length (128B) record that contains
 * the left and right boundaries of the missing ADUs.
 * Each boundary of the two-dimensional ADU space is
 * a tuple {seqno, byte offset}. Note that a single 
 * RREQ record cannot contain a gap that spans
 * multiple containers. The lost range includes the
 * left, but not the right boundary. 
 *
 *                    (32)
 * +--------+---------+---------------------------+
 * |               start adu seqno                | 
 * +--------+---------+---------------------------+
 * |               start offset (Bytes)           | 
 * +--------+---------+---------------------------+
 * |                 end adu seqno                | 
 * +--------+---------+---------------------------+
 * |                 end offset (Bytes)           | 
 * +--------+---------+---------------------------+
 *
 */

void 
SRMv2_BufferPool::build_rreqhdr(SRMv2_pktbuf *pb, unsigned int ss,
				unsigned int sbytes, unsigned int es,
				unsigned int ebytes)
{
	srmv2_rreqhdr *rh = (srmv2_rreqhdr *)pb->dp;
	rh->ss          = htonl(ss);
	rh->sbytes      = htonl(sbytes);
	rh->es          = htonl(es);
	rh->ebytes      = htonl(ebytes);
	pb->dp  += sizeof(srmv2_rreqhdr);
	pb->len += sizeof(srmv2_rreqhdr);
}


/* +--------+---------+-------------------------+
 * |length of src's app_info_ -- 2 bytes        |
 * | (0xFFFF indicates no app_info in this pkt) |
 * +--------+---------+-------------------------+
 * |                 app info                   |
 * +--------+---------+-------------------------+
 * |         (padding to multiple of 4)         | 
 * +--------+---------+-------------------------+
 */
void
SRMv2_BufferPool::build_srcinfohdr(SRMv2_pktbuf *pb, SRMv2_Source *source,
				   int include_srcinfo)
{
	const unsigned char *app_info;
	int info_len;
	u_int16_t len;
	if (include_srcinfo) {
		source->get_app_info(&app_info, &info_len);
		if (!app_info) info_len = 0;
		len = (u_int16_t)info_len;
		*((u_int16_t*)pb->dp) = htons(len);
		if (app_info)
			memcpy((char*) pb->dp + sizeof(u_int16_t),
			       app_info,len);
	} else {
		*((u_int16_t*)pb->dp) = 0xFFFF;
		len = 0;
	}
	
	if ((len+2)%4) len += (2 - len%4);
	pb->dp  += sizeof(u_int16_t) + len;
	pb->len += sizeof(u_int16_t) + len;
}


/*
 * SRMv2_ANNOUNCE: Each announcement record contains
 * summary information for one container (internal or
 * external node in the SNAP hierarchy). 
 * Note: We may have to revisit this when we allow
 * internal SNAP nodes to contain ADUs in addition to
 * other SNAP nodes. 
 * 
 *                      (32)
 * +--------+---------+---------------------------+
 * |               container ID                   | 
 * +--------+---------+---------------------------+
 * |                signature (last adu#)         | 
 * +--------+---------+---------------------------+
 */
void
SRMv2_BufferPool::build_announcehdr(SRMv2_pktbuf *pb,
				    srmv2_announcehdr *salist, int count)
{
	srmv2_announcehdr *al = (srmv2_announcehdr *)pb->dp;
	int i;
	/*
	 * SRMv2_ERROR indicates that  the corresponding
	 * record must not be included for transmission.
	 */
	for (i = 0; i < count && salist[i].cid != SRMv2_ERROR; i++) {
		al[i].cid = htonl(salist[i].cid);
		al[i].sign = htonl(salist[i].sign);
	}
	pb->dp  += i * sizeof(srmv2_announcehdr);
	pb->len += i * sizeof(srmv2_announcehdr);
}


/*
 * SRMv2_NSHEADER : When a new container is created, updates to the 
 * namespace are transmitted to the entire group using this packet. 
 *                      (32)
 * +--------+---------+---------------------------+
 * |               parent ID                      | 
 * +--------+---------+---------------------------+
 * |               container ID                   | 
 * +--------+---------+---------------------------+
 * |             offset within parent             | 
 * +--------+---------+---------------------------+
 * |             signature                        | 
 * +--------+---------+---------------------------+
 * |      length of container name (2 bytes)      | 
 * +--------+---------+---------------------------+
 * |              container name                  |
 * +--------+---------+---------------------------+
 * |          (padding to multiple of 4)          | 
 * +========+=========+===========================+
 * |               parent ID                      | 
 * +--------+---------+---------------------------+
 * |               container ID                   | 
 * +--------+---------+---------------------------+
 * |             offset within parent             | 
 * +--------+---------+---------------------------+
 * |             signature                        | 
 * +--------+---------+---------------------------+
 * |      length of container name (2 bytes)      | 
 * +--------+---------+---------------------------+
 * |              container name                  |
 * +--------+---------+---------------------------+
 * |          (padding to multiple of 4)          | 
 * +========+=========+===========================+
 *                 ...
 * +--------+---------+---------------------------+
 * |               parent ID                      | 
 * +--------+---------+---------------------------+
 * |               container ID                   | 
 * +--------+---------+---------------------------+
 * |             offset within parent             | 
 * +--------+---------+---------------------------+
 * |             signature                        | 
 * +--------+---------+---------------------------+
 * |      length of container name (2 bytes)      | 
 * +--------+---------+---------------------------+
 * |              container name                  |
 * +--------+---------+---------------------------+
 * |          (padding to multiple of 4)          | 
 * +--------+---------+---------------------------+
 */

void 
SRMv2_BufferPool::build_nshdr(SRMv2_pktbuf *pb, SRMv2_Source *source,
			      naddr *comps, int count)
{
	naddr_plus_name *addrmap = (naddr_plus_name *)pb->dp;
	int name_len;
	const unsigned char *name;
	int i;
	NS_Node *node;
	
	/* FIXME: how do I know if the entire header will fit in the pktbuf
	 */
	for (i = 0; i < count; i++) {
		addrmap = (naddr_plus_name *)pb->dp;
		addrmap->addr.offset  = htonl(comps[i].offset);
		addrmap->addr.cid     = htonl(comps[i].cid);
		addrmap->addr.pcid    = htonl(comps[i].pcid);
		addrmap->addr.childsig= htonl(comps[i].childsig);
		addrmap->addr.adu     = htonl(comps[i].adu);
		addrmap->addr.byte    = htonl(comps[i].byte);

		node = source->name_space()->getnode(comps[i].cid);
		if (node) {
			name = node->name();
			name_len = node->name_len();
		} else {
			name = NULL;
			name_len = 0;
		}

		if (!name) name_len = 0;
		addrmap->name_len = htonl((u_int32_t)name_len);
		if (name && name_len > 0)
			memcpy(addrmap->name, name, name_len);

		if (name_len%4) name_len += (4 - name_len%4);
		pb->dp  += sizeof(naddr_plus_name) + name_len;
		pb->len += sizeof(naddr_plus_name) + name_len;
	}
} 

/* Overview of SRMv2 DATA and RREQ packets in Backus-Naur style:

(Fragmented ADUs contain fragment per packet.)
SRMv2_DATA : <common header>
                <container header>      
                        <ADU header>
                        <fragment header>
                        <fragment data>

        OR
                                                
(Small ADUs can be stacked in a single packet.)
           <common header>      
                <container header>      
                        <ADU header>
                        <ADU data>
                            |
                            |
                        <ADU header>
                        <ADU data>
                <container header>      
                        <ADU header>
                        <ADU data>
                            |
                            |
                        <ADU header>
                        <ADU data>
                        |
                        |
                        |
                <container header>      
                        <ADU header>
                        <ADU data>
                            |
                            |
                        <ADU header>
                        <ADU data>
                                


SRMv2_REQUEST: <common header>
                   <container header>
                           <basic request record>
                           <basic request record>
                                    |
                                    |
                           <basic request record>
                   <container header> 
                           <basic request record>
                           <basic request record>
                                    |
                                    |
                           <basic request record>
	            
Note: Even though this present implementation does not stack RREQs,
      the formats do support multiple RREQs per packet. 
*/
			  
