;;;-*- Mode: Lisp; Package: SPARC -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   This file is part of OpenMCL.  
;;;
;;;   OpenMCL is licensed under the terms of the Lisp Lesser GNU Public
;;;   License , known as the LLGPL and distributed with OpenMCL as the
;;;   file "LICENSE".  The LLGPL consists of a preamble and the LGPL,
;;;   which is distributed with OpenMCL as the file "LGPL".  Where these
;;;   conflict, the preamble takes precedence.  
;;;
;;;   OpenMCL is referenced in the preamble as the "LIBRARY."
;;;
;;;   The LLGPL is also available online at
;;;   http://opensource.franz.com/preamble.html
(in-package "SPARC")

;;; This doesn't -quite- match what Sun/GNU assemblers accept; 
;;; there's a one-to-one mapping between mnemonics and opcodes, so
;;; "ld" is used to load a GPR, "ldf" to load an FPR, etc.  (Both
;;; as and gas overload "ld" to stand for any 32-bit load; gas actually
;;; hides the other type-specific mnemonics.)
;;; This defines all v8 Sparc instructions (even the privileged ones),
;;; mostly for the disassembler's benefit.
 

  

(defenum ()
  f-delayed           ; a delayed branch
  f-alias           ; alias for a "real" instruction
  f-unbr            ; unconditional branch
  f-condbr          ; conditional branch
  f-jsr           ; subroutine call
  f-float           ; fp instruction (not a branch)
  f-fbr           ; fp branch
  f-branch-computed  ; branch target depends on regs (e.g., JUMPL)
  f-setsCC           ; instruction sets CC
  f-readsCC          ; instruction reads condition codes
  f-store            ; RD is a source, not a dest
  f-pcrel            ; pc-relative branch of some sort
)



(defmacro op (n) `(ash (logand ,n 3) 30))
(defmacro op2 (n) `(ash (logand ,n 7) 22))
(defmacro op3 (n) `(ash (logand ,n #x3f) 19))
(defmacro opf (n) `(ash (logand ,n #x1ff) 5))

(defconstant op-mask (op -1))


(defconstant op2-mask (logior op-mask (op2 -1)))
(defconstant op3-mask (op3 -1))
(defconstant opf-mask (opf -1))



(defmacro f3f (x y z) `(logior (op ,x) (op3 ,y) (opf ,z)))
(defmacro f3i (n) `(ash (logand ,n 1) 13))

(defmacro f1 (x) `(op ,x))
(defconstant f1-mask (f1 -1))

(defmacro f2 (x y &rest rest) `(logior (op ,x) (op2 ,y) ,@rest))
(defconstant f2-mask (f2 -1 -1))

(defmacro f3 (x y z) `(logior (op ,x) (op3 ,y) (f3i ,z)))

(defmacro f3op3 (op3 &rest rest) `(logior (op 3) (op3 ,op3) ,@rest))
(defmacro f3op2 (op3 &rest rest) `(logior (op 2) (op3 ,op3) ,@rest))

(defconstant f3-mask (logior (op -1) (op3 -1)))
(defconstant f3-maski (logior f3-mask (f3i 1)))
(defconstant annul (ash 1 29))

(defmacro xcond (c) `(ash (logand ,c #xf) 25))

(defconstant trap-mask (logior f3-mask annul (xcond -1)))

(defmacro fpop1 (opf) `(f3op2 #x34 (opf ,opf)))
(defmacro fpop2 (opf) `(f3op2 #x35 (opf ,opf)))

(defconstant f3fpopmask (logior f3-mask (opf -1)))

(defmacro disp30 (x) `(ldb (byte 30 0) ,x))
(defmacro asi (n) `(ash (logand ,n #xff) 5))
(defmacro rs2 (n) `(logand ,n #x1f))
(defmacro simm13 (n) `(logand ,n #x1fff))
(defmacro rd (n) `(ash (logand ,n #x1f) 25))
(defmacro rs1 (n) `(ash (logand ,n #x1f) 14))
(defmacro asi-rs2 (n) `(simm13 ,n))
(defconstant immed (f3i 1))
(defconstant rd-g0 (rd (lognot 0)))
(defconstant rs1-g0 (rs1 (lognot 0)))
(defconstant rs2-g0 (rs2 (lognot 0)))
(defconstant op2-g0 (1- (ash 1 14)))

(defmacro branch (cond &optional a)
  `(f2 0 2 (xcond ,cond) ,(if a annul 0)))

(defmacro fbranch (cond &optional a)
  `(f2 0 6 (xcond ,cond) ,(if a annul 0)))

(defmacro cbranch (cond &optional a)
  `(f2 0 7 (xcond ,cond) ,(if a annul 0)))

(defmacro trap (cond)
  `(f3op2 #x3a (xcond ,cond)))
  
(defconstant branch-mask (f2 -1 -1 (xcond -1) annul))

(defenum ()
  sparc-operand-signed
  sparc-operand-gpr
  sparc-operand-fpr
  sparc-operand-relative
  sparc-operand-source
  sparc-operand-dest
  sparc-operand-cpr			;coprocessor register
  sparc-operand-double			;register pair
  sparc-operand-quad			;register quadruple
  sparc-operand-address
  sparc-operand-op2
  sparc-operand-regaddr
  )



(eval-when (:compile-toplevel :execute)
  (defmacro sparc-op (name width offset &optional insert-function extract-function &rest flags)
    `(arch::make-operand :index ',name
      :width ,width 
      :offset ,offset 
      :insert-function ',insert-function
      :extract-function ',extract-function
      :flags (logior ,@(mapcar #'(lambda (f) `(ash 1 ,f)) flags)))))

(eval-when (:compile-toplevel :load-toplevel :execute)
(defparameter *sparc-operands*
  (vector
   (sparc-op 'unused 0 0)
   (sparc-op :address 19 0 insert-address extract-address sparc-operand-address)
   (sparc-op :asi 9 5 nil nil)
   (sparc-op :asr 5 25 nil nil)
   (sparc-op :cd 5 25 insert-rd extract-rd sparc-operand-cpr)
   (sparc-op :cdd 5 25 insert-rd extract-rd sparc-operand-cpr sparc-operand-double)
   (sparc-op :cond 4 25 nil nil)
   (sparc-op :cpop 9 5 nil nil)
   (sparc-op :crs1 5 14 insert-rs1 extract-rs1 sparc-operand-cpr)
   (sparc-op :crs2 5 0 insert-rs2 extract-rs2 sparc-operand-cpr)
   (sparc-op :fcond 4 25 nil nil)
   (sparc-op :frd 5 25 insert-rd extract-rd sparc-operand-fpr)
   (sparc-op :frdd 5 25 insert-rd extract-rd sparc-operand-fpr sparc-operand-double)
   (sparc-op :frdq 5 25 insert-rd extract-rd sparc-operand-fpr sparc-operand-quad)
   (sparc-op :frs1 5 14 insert-rs1 extract-rs1 sparc-operand-fpr)
   (sparc-op :frs1d 5 14 insert-rs1 extract-rs1 sparc-operand-fpr sparc-operand-double)
   (sparc-op :frs1q 5 14 insert-rs1 extract-rs1 sparc-operand-fpr sparc-operand-quad)
   (sparc-op :frs2 5 0 insert-rs2 extract-rs2 sparc-operand-fpr)
   (sparc-op :frs2d 5 0 insert-rs2 extract-rs2 sparc-operand-fpr sparc-operand-double)
   (sparc-op :frs2q 5 0 insert-rs2 extract-rs2 sparc-operand-fpr sparc-operand-quad)
   (sparc-op :imm22 22 0 insert-imm22 extract-imm22 sparc-operand-signed)
   (sparc-op :label22 22 0 insert-label22 extract-label22 sparc-operand-signed sparc-operand-relative)
   (sparc-op :label30 30 0 insert-label30 extract-label30 sparc-operand-signed sparc-operand-relative)
   (sparc-op :op2 14 0 insert-op2 extract-op2 sparc-operand-op2)
   (sparc-op :rd 5 25 insert-rd extract-rd sparc-operand-gpr)
   (sparc-op :rdd 5 25 insert-rd extract-rd sparc-operand-gpr sparc-operand-double)
   (sparc-op :regaddr 19 0 insert-regaddr extract-regaddr sparc-operand-regaddr)
   (sparc-op :rs1 5 14 insert-rs1 extract-rs1 sparc-operand-gpr)
   (sparc-op :rs2 5 0 insert-rs2 extract-rs2 sparc-operand-gpr)
   (sparc-op :shift-op2 14 0 insert-shiftop2 extract-op2 sparc-operand-op2)
   (sparc-op :trap-number 14 0 insert-trap-number extract-trap-number sparc-operand-op2)
   (sparc-op :uuo-minor 6 1 nil nil)
   (sparc-op :uuo-rb 5 7 nil nil sparc-operand-gpr)
   (sparc-op :uuo-ra 5 12 nil nil sparc-operand-gpr)
   (sparc-op :uuo-rt 5 17 nil nil sparc-operand-gpr)
   (sparc-op :uuo-errnum 10 12 nil nil)
   (sparc-op :uuo-small-errnum 5 17 nil nil)
   (sparc-op :subprim 14 0 insert-subprim extract-subprim)
   ))
)
  
   

(defmacro sparc-opcode (&whole w name op mask (&rest flags) &rest operands)
  (let* ((ops (mapcar #'(lambda (op)
			     (let* ((p (find op *sparc-operands*
						 :key #'arch::operand-index)))
			       (or p (warn "Unknown operand: ~s" op))))
			 operands))
	 (len (length ops)))
  
    `(arch::make-opcode :name ,(string name)
      :opcode ,op
      :mask ,mask
      :min-args ,len
      :max-args ,len
      :flags (logior ,@(mapcar #'(lambda (f) `(ash 1 ,f)) flags))
      :operands ',ops)))

(defconstant uuo-mask #xffc0007f)
(defconstant uuorb-mask (logior uuo-mask (ash 31 7)))
(defmacro uuo (n) `(ash ,n 1))

(defparameter *sparc-opcodes* 
  (vector 
   ;; There's exactly one format 1 instruction:

   #.(sparc-opcode call (f1 #x1) f1-mask (f-jsr f-delayed f-pcrel f-unbr) :label30)

   ;; Format-2 instructions: UNIMP, sethi, Bicc, FBfcc, CBccc
   #.(sparc-opcode uuo_box_signed (uuo 1) uuorb-mask () :uuo-rt :uuo-ra)
   #.(sparc-opcode uuo_box_unsigned (uuo 2) uuorb-mask () :uuo-rt :uuo-ra)
   #.(sparc-opcode uuo_interr (uuo 11) uuo-mask () :uuo-errnum :uuo-rb)
   #.(sparc-opcode uuo_intcerr (uuo 12) uuo-mask () :uuo-errnum :uuo-rb)
   #.(sparc-opcode uuo_interr2 (uuo 13) uuo-mask () :uuo-small-errnum :uuo-ra :uuo-rb)
   #.(sparc-opcode uuo_intcerr2 (uuo 14) uuo-mask () :uuo-small-errnum :uuo-ra :uuo-rb)

   #.(sparc-opcode uuo_fixnum_overflow (uuo 21) uuorb-mask () :uuo-rt :uuo-ra)
   #.(sparc-opcode uuo_multiply_fixnums (uuo 23) uuo-mask () :uuo-rt :uuo-ra :uuo-rb)
   #.(sparc-opcode uuo_xalloc (uuo 24) uuo-mask ()  :uuo-rt :uuo-ra :uuo-rb)
   #.(sparc-opcode  unimp (f2 0 0) f2-mask () :imm22)

   ;; NOP is short for sethi 0,%g0
   #.(sparc-opcode nop (f2 0 4) #xffffffff ())
   #.(sparc-opcode  sethi (f2 0 4) f2-mask () :imm22 :rd)

   ;; Integer unit condtional branches, and a general branch-on (computed) icc.
   ;; Then, the same thing with the annul bit set.
   #.(sparc-opcode b (branch conda) branch-mask (f-unbr f-pcrel f-delayed) :label22)
   #.(sparc-opcode ba (branch conda) branch-mask (f-unbr f-pcrel f-delayed) :label22)
   #.(sparc-opcode bn (branch condn) branch-mask (f-delayed) :label22)
   #.(sparc-opcode bne (branch condne) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bnz (branch condne) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode be (branch conde) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bz (branch conde) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bg (branch condg) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode ble (branch condle) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bge (branch condge) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bl (branch condl) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bgu (branch condgu) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bleu (branch condleu) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bcc (branch condcc) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bgeu (branch condcc) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bcs (branch condcs) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode blu (branch condcs) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bpos (branch condpos) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bneg (branch condneg) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bvc (branch condvc) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bvs (branch condvs) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bicc (f2 0 2) (logand branch-mask (lognot (branch -1 t)))
		 (f-condbr f-pcrel f-readsCC f-delayed) :cond :label22)

   #.(sparc-opcode b.a (branch conda t) branch-mask (f-unbr f-delayed) :label22)
   #.(sparc-opcode ba.a (branch conda t) branch-mask (f-unbr f-delayed) :label22)
   #.(sparc-opcode bn.a (branch condn t) branch-mask (f-delayed) :label22)
   #.(sparc-opcode bne.a (branch condne t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bnz.a (branch condne t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode be.a (branch conde t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bz.a (branch conde t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bg.a (branch condg t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode ble.a (branch condle t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bge.a (branch condge t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bl.a (branch condl t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bgu.a (branch condgu t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bleu.a (branch condleu t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bcc.a (branch condcc t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bgeu.a (branch condcc t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bcs.a (branch condcs t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode blu.a (branch condcs t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bpos.a (branch condpos t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bneg.a (branch condneg t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bvc.a (branch condvc t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bvs.a (branch condvs t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode bicc.a (f2 0 2 annul) (logand branch-mask (lognot (branch -1 t)))
		 (f-condbr f-pcrel f-readsCC f-delayed) :cond :label22)

   ;; FP branches.  Pretty much the same as the integer unit branches, only
   ;; slightly different names for/encodings of conditions.

   #.(sparc-opcode fba (fbranch fconda) branch-mask (f-fbr f-unbr f-delayed) :label22)
   #.(sparc-opcode fbn (fbranch fcondn) branch-mask (f-fbr f-delayed) :label22)
   #.(sparc-opcode fbu (fbranch fcondu) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbg (fbranch fcondg) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbug (fbranch fcondug) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbl (fbranch fcondl) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbul (fbranch fcondul) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fblg (fbranch fcondlg) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbne (fbranch fcondne) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbnz (fbranch fcondne) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbe (fbranch fconde) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbz (fbranch fconde) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbue (fbranch fcondue) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbge (fbranch fcondge) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbuge (fbranch fconduge) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fble (fbranch fcondle) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbule (fbranch fcondule) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbo (fbranch fcondo) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbfcc (f2 0 6) (logand branch-mask (lognot (branch -1 t)))
		 (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :fcond :label22)

   #.(sparc-opcode fba.a (fbranch fconda t) branch-mask (f-fbr f-unbr f-delayed) :label22)
   #.(sparc-opcode fbn.a (fbranch fcondn t) branch-mask (f-fbr f-delayed) :label22)
   #.(sparc-opcode fbu.a (fbranch fcondu t) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbg.a (fbranch fcondg t) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbug.a (fbranch fcondug t) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbl.a (fbranch fcondl t) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbul.a (fbranch fcondul t) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fblg.a (fbranch fcondlg t) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbne.a (fbranch fcondne t) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbnz.a (fbranch fcondne t) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbe.a (fbranch fconde t) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbz.a (fbranch fconde t) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbue.a (fbranch fcondue t) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbge.a (fbranch fcondge t) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbuge.a (fbranch fconduge t) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fble.a (fbranch fcondle t) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbule.a (fbranch fcondule t) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbo.a (fbranch fcondo t) branch-mask (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode fbfcc.a (f2 0 6 annul) (logand branch-mask (lognot (branch -1 t)))
		 (f-fbr f-condbr f-pcrel f-readsCC f-delayed) :fcond :label22)

;;; If we don't define coprocessor branch instructions, we'll probably
;;; regret it someday.  Or maybe not.
   #.(sparc-opcode cba (cbranch 8) branch-mask (f-unbr f-delayed) :label22)
   #.(sparc-opcode cbn (cbranch 0) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb3 (cbranch 7) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb2 (cbranch 6) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb23 (cbranch 5) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb1 (cbranch 4) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb13 (cbranch 3) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb12 (cbranch 2) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb123 (cbranch 1) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb0 (cbranch 9) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb03 (cbranch 10) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb02 (cbranch 11) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb023 (cbranch 12) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb01 (cbranch 13) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode c013 (cbranch 14) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode c012 (cbranch 15) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)

   #.(sparc-opcode cba.a (cbranch 8 t) branch-mask (f-unbr f-delayed) :label22)
   #.(sparc-opcode cbn.a (cbranch 0 t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb3.a (cbranch 7 t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb2.a (cbranch 6 t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb23.a (cbranch 5 t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb1.a (cbranch 4 t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb13.a (cbranch 3 t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb12.a (cbranch 2 t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb123.a (cbranch 1 t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb0.a (cbranch 9 t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb03.a (cbranch 10 t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb02.a (cbranch 11 t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb023.a (cbranch 12 t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode cb01.a (cbranch 13 t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode c013.a (cbranch 14 t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)
   #.(sparc-opcode c012.a (cbranch 15 t) branch-mask (f-condbr f-pcrel f-readsCC f-delayed) :label22)

;;; That's really pushing it.  That's also the end of the format-2 opcodes.

;;; Format 3, op 3 instructions are loads & stores.  See above: we want
;;; the mnemonic to uniquely determine the opcode.

   #.(sparc-opcode ld (f3op3 #x00) f3-mask () :address :rd)
   #.(sparc-opcode ldub (f3op3 #x01) f3-mask () :address :rd)
   #.(sparc-opcode lduh (f3op3 #x02) f3-mask () :address :rd)
   #.(sparc-opcode ldd (f3op3 #x03) f3-mask () :address :rdd)
   #.(sparc-opcode st (f3op3 #x04) f3-mask (f-store) :rd :address)
   #.(sparc-opcode stb (f3op3 #x05) f3-mask (f-store) :rd :address)
   #.(sparc-opcode sth (f3op3 #x06) f3-mask (f-store) :rd :address)
   #.(sparc-opcode std (f3op3 #x07) f3-mask (f-store) :rdd :address)
   #.(sparc-opcode ldsb (f3op3 #x09) f3-mask () :address :rd)
   #.(sparc-opcode ldsh (f3op3 #x0a) f3-mask () :address :rd)
   #.(sparc-opcode ldstub (f3op3 #x0d) f3-mask () :address :rd)
   #.(sparc-opcode swap (f3op3 #x0f) f3-mask () :address :rd)

   ;; Alternate-space loads & stores are privileged; the op2 part of the
   ;; address can never be immediate.
   #.(sparc-opcode lda (f3op3 #x10) f3-maski () :regaddr :asi :rd)
   #.(sparc-opcode lduba (f3op3 #x11) f3-maski () :regaddr :asi :rd)
   #.(sparc-opcode lduha (f3op3 #x12) f3-maski () :regaddr :asi :rd)
   #.(sparc-opcode ldda (f3op3 #x13) f3-maski () :regaddr :asi :rdd)
   #.(sparc-opcode sta (f3op3 #x14) f3-maski (f-store) :rd :regaddr :asi)
   #.(sparc-opcode stba (f3op3 #x15) f3-maski (f-store) :rd :regaddr :asi)
   #.(sparc-opcode stha (f3op3 #x16) f3-maski (f-store) :rd :regaddr :asi)
   #.(sparc-opcode stda (f3op3 #x17) f3-maski (f-store) :rdd :regaddr :asi)
   #.(sparc-opcode ldsba (f3op3 #x19) f3-maski () :regaddr :asi :rd)
   #.(sparc-opcode ldsha (f3op3 #x1a) f3-maski () :regaddr :asi :rd)
   #.(sparc-opcode ldstuba (f3op3 #x1d) f3-maski () :regaddr :asi :rd)
   #.(sparc-opcode swapa (f3op3 #x1f) f3-maski () :regaddr :asi :rd)

;;; Float loads & stores
   #.(sparc-opcode ldf (f3op3 #x20) f3-mask () :address :frd)
   #.(sparc-opcode ldfsr (f3op3 #x21) (logior f3-mask rd-g0) () :address)
   #.(sparc-opcode lddf (f3op3 #x23) f3-mask () :address :frdd)
   #.(sparc-opcode stf (f3op3 #x24) f3-mask (f-store) :frd :address)
   #.(sparc-opcode stfsr (f3op3 #x25) (logior f3-mask rd-g0) () :address)
   #.(sparc-opcode stdfq (f3op3 #x26) (logior f3-mask rd-g0) () :address)
   #.(sparc-opcode stdf (f3op3 #x27) f3-mask (f-store) :frdd :address)

;;; Coprocessor loads & stores.
   #.(sparc-opcode ldc (f3op3 #x30) f3-mask () :address :cd)
   #.(sparc-opcode ldcsr (f3op3 #x31) (logior f3-mask rd-g0) () :address)
   #.(sparc-opcode lddc (f3op3 #x33) f3-mask () :address :cdd)
   #.(sparc-opcode stc (f3op3 #x34) f3-mask () :cd :address)
   #.(sparc-opcode stcsr (f3op3 #x35) (logior f3-mask rd-g0) () :address)
   #.(sparc-opcode stdcq (f3op3 #x36) (logior f3-mask rd-g0) () :address)
   #.(sparc-opcode stdc (f3op3 #x37) f3-mask () :cdd :address)

;;; For all of these alu instructions, we'll insist on rs1 and op2 both
;;; being specified (and in that order).  Hopefull, lapmacros will handle
;;; cases where this would be tedious.

   #.(sparc-opcode add (f3op2 #x00) f3-mask () :rs1 :op2 :rd)
   #.(sparc-opcode and (f3op2 #x01) f3-mask () :rs1 :op2 :rd)
   #.(sparc-opcode mov (f3op2 #x02) (logior f3-mask rs1-g0) () :op2 :rd)
   #.(sparc-opcode or (f3op2 #x02) f3-mask () :rs1 :op2 :rd)
   #.(sparc-opcode xor (f3op2 #x03) f3-mask () :rs1 :op2 :rd)
   #.(sparc-opcode sub (f3op2 #x04) f3-mask () :rs1 :op2 :rd)
   #.(sparc-opcode andn (f3op2 #x05) f3-mask () :rs1 :op2 :rd)
   #.(sparc-opcode orn (f3op2 #x06) f3-mask () :rs1 :op2 :rd)
   #.(sparc-opcode xnor (f3op2 #x07) f3-mask () :rs1 :op2 :rd)
   #.(sparc-opcode addx (f3op2 #x08) f3-mask () :rs1 :op2 :rd)
   #.(sparc-opcode umul (f3op2 #x0a) f3-mask () :rs1 :op2 :rd)
   #.(sparc-opcode smul (f3op2 #x0b) f3-mask () :rs1 :op2 :rd)
   #.(sparc-opcode subx (f3op2 #x0c) f3-mask () :rs1 :op2 :rd)
   #.(sparc-opcode udiv (f3op2 #x0e) f3-mask () :rs1 :op2 :rd)
   #.(sparc-opcode sdiv (f3op2 #x0f) f3-mask () :rs1 :op2 :rd)

   #.(sparc-opcode addcc (f3op2 #x10) f3-mask (f-setsCC) :rs1 :op2 :rd)
   #.(sparc-opcode andcc (f3op2 #x11) f3-mask (f-setsCC) :rs1 :op2 :rd)
   #.(sparc-opcode tst (f3op2 #x12) (logior f3-mask op2-g0 rd-g0) (f-setsCC) :rs1)
   #.(sparc-opcode orcc (f3op2 #x12) f3-mask (f-setsCC) :rs1 :op2 :rd)
   #.(sparc-opcode xorcc (f3op2 #x13) f3-mask (f-setsCC) :rs1 :op2 :rd)

   ;; CMP is a handy (and simple) synthetic instruction
   #.(sparc-opcode cmp (f3op2 #x14) (logior f3-mask rd-g0) (f-setsCC) :rs1 :op2)
   #.(sparc-opcode subcc (f3op2 #x14) f3-mask (f-setsCC) :rs1 :op2 :rd)

   #.(sparc-opcode andncc (f3op2 #x15) f3-mask (f-setsCC) :rs1 :op2 :rd)
   #.(sparc-opcode orncc (f3op2 #x16) f3-mask (f-setsCC) :rs1 :op2 :rd)
   #.(sparc-opcode xnorcc (f3op2 #x17) f3-mask (f-setsCC) :rs1 :op2 :rd)
   #.(sparc-opcode addxcc (f3op2 #x18) f3-mask (f-setsCC) :rs1 :op2 :rd)
   #.(sparc-opcode umulcc (f3op2 #x1a) f3-mask (f-setsCC) :rs1 :op2 :rd)
   #.(sparc-opcode smulcc (f3op2 #x1b) f3-mask (f-setsCC) :rs1 :op2 :rd)
   #.(sparc-opcode subxcc (f3op2 #x1c) f3-mask (f-setsCC) :rs1 :op2 :rd)
   #.(sparc-opcode udivcc (f3op2 #x1e) f3-mask (f-setsCC) :rs1 :op2 :rd)
   #.(sparc-opcode sdivcc (f3op2 #x1f) f3-mask (f-setsCC) :rs1 :op2 :rd)

   #.(sparc-opcode taddcc (f3op2 #x20) f3-mask (f-setsCC) :rs1 :op2 :rd)
   #.(sparc-opcode tsubcc (f3op2 #x21) f3-mask (f-setsCC) :rs1 :op2 :rd)
   #.(sparc-opcode taddcctv (f3op2 #x22) f3-mask (f-setsCC) :rs1 :op2 :rd)
   #.(sparc-opcode tsubcctv (f3op2 #x23) f3-mask (f-setsCC) :rs1 :op2 :rd)
   #.(sparc-opcode mulscc (f3op2 #x24) f3-mask (f-setsCC) :rs1 :op2 :rd)

   #.(sparc-opcode sll (f3op2 #x25) f3-mask () :rs1 :shift-op2 :rd)
   #.(sparc-opcode srl (f3op2 #x26) f3-mask () :rs1 :shift-op2 :rd)
   #.(sparc-opcode sra (f3op2 #x27) f3-mask () :rs1 :shift-op2 :rd)

					; RDASR, RDY, and STBAR are distinguished from each other by the values
					; of rs1/rd.  The entire OP2 field of the instruction is always 0.
   #.(sparc-opcode stbar (f3op2 #x28 (rs1 15)) #xffffffff ())
   #.(sparc-opcode rdy (f3op2 #x28) (logand #xffffffff (lognot rd-g0)) () :rd)
   #.(sparc-opcode rdasr (f3op2 #x28) (logand #xffffffff (lognot rd-g0) (lognot rs1-g0)) () :asr :rd)

   #.(sparc-opcode rdpsr (f3op2 #x29) (logior f3-mask (lognot rd-g0)) () :rd)
   #.(sparc-opcode rdwim (f3op2 #x2a) (logior f3-mask (lognot rd-g0)) () :rd)
   #.(sparc-opcode rdtbr (f3op2 #x2b) (logior f3-mask (lognot rd-g0)) () :rd)

   #.(sparc-opcode wry (f3op2 #x30) (logand #xffffffff (lognot rd-g0)) () :rs1 :op2)
   #.(sparc-opcode wrasr (f3op2 #x30) f3-mask () :rs1 :op2 :asr)

   #.(sparc-opcode wrpsr (f3op2 #x31) (logior f3-mask (lognot rd-g0)) () :rs1 :op2)
   #.(sparc-opcode wrwim (f3op2 #x32) (logior f3-mask (lognot rd-g0)) () :rs1 :op2)
   #.(sparc-opcode wrtbr (f3op2 #x33) (logior f3-mask (lognot rd-g0)) () :rs1 :op2)

   #.(sparc-opcode fmovs (fpop1 #x01) (logior f3fpopmask rs1-g0) () :frs2 :frd)
   #.(sparc-opcode fnegs (fpop1 #x05) (logior f3fpopmask rs1-g0) () :frs2 :frd)
   #.(sparc-opcode fabss (fpop1 #x09) (logior f3fpopmask rs1-g0) () :frs2 :frd)
   #.(sparc-opcode fqrts (fpop1 #x29) (logior f3fpopmask rs1-g0) () :frs2 :frd)
   #.(sparc-opcode fqrtd (fpop1 #x2a) (logior f3fpopmask rs1-g0) () :frs2d :frdd)
   #.(sparc-opcode fqrtq (fpop1 #x2b) (logior f3fpopmask rs1-g0) () :frs2q :frdq)
   #.(sparc-opcode fadds (fpop1 #x41) f3fpopmask () :frs1 :frs2 :frd)
   #.(sparc-opcode faddd (fpop1 #x42) f3fpopmask () :frs1d :frs2d :frdd)
   #.(sparc-opcode faddq (fpop1 #x43) f3fpopmask () :frs1q :frs2q :frdq)
   #.(sparc-opcode fsubs (fpop1 #x45) f3fpopmask () :frs1 :frs2 :frd)
   #.(sparc-opcode fsubd (fpop1 #x46) f3fpopmask () :frs1d :frs2d :frdd)
   #.(sparc-opcode fsubq (fpop1 #x47) f3fpopmask () :frs1q :frs2q :frdq)
   #.(sparc-opcode fmuls (fpop1 #x49) f3fpopmask () :frs1 :frs2 :frd)
   #.(sparc-opcode fmuld (fpop1 #x4a) f3fpopmask () :frs1d :frs2d :frdd)
   #.(sparc-opcode fmulq (fpop1 #x4b) f3fpopmask () :frs1q :frs2q :frdq)
   #.(sparc-opcode fdivs (fpop1 #x4d) f3fpopmask () :frs1 :frs2 :frd)
   #.(sparc-opcode fdivd (fpop1 #x4e) f3fpopmask () :frs1d :frs2d :frdd)
   #.(sparc-opcode fdivq (fpop1 #x4f) f3fpopmask () :frs1q :frs2q :frdq)
   #.(sparc-opcode fsmuld (fpop1 #x69) f3fpopmask () :frs1 :frs2 :frdd)
   #.(sparc-opcode fdmulq (fpop1 #x6e) f3fpopmask () :frs1d :frs2d :frdq)

   #.(sparc-opcode fitos (fpop1 #xc4) (logior f3fpopmask rs1-g0) () :frs2 :frd)
   #.(sparc-opcode fdtos (fpop1 #xc6) (logior f3fpopmask rs1-g0) () :frs2d :frd)
   #.(sparc-opcode fqtos (fpop1 #xc7) (logior f3fpopmask rs1-g0) () :frs2q :frd)
   #.(sparc-opcode fitod (fpop1 #xc8) (logior f3fpopmask rs1-g0) () :frs2 :frdd)
   #.(sparc-opcode fstod (fpop1 #xc9) (logior f3fpopmask rs1-g0) () :frs2 :frdd)
   #.(sparc-opcode fqtod (fpop1 #xcb) (logior f3fpopmask rs1-g0) () :frs2q :frdd)
   #.(sparc-opcode fitoq (fpop1 #xcc) (logior f3fpopmask rs1-g0) () :frs2 :frdq)
   #.(sparc-opcode fstoq (fpop1 #xcd) (logior f3fpopmask rs1-g0) () :frs2 :frdq)
   #.(sparc-opcode fdtoq (fpop1 #xce) (logior f3fpopmask rs1-g0) () :frs2d :frdq)
   #.(sparc-opcode fstoi (fpop1 #xd1) (logior f3fpopmask rs1-g0) () :frs2 :frd)
   #.(sparc-opcode fdtoi (fpop1 #xd2) (logior f3fpopmask rs1-g0) () :frs2 :frd)
   #.(sparc-opcode fqtoi (fpop1 #xd3) (logior f3fpopmask rs1-g0) () :frs2 :frd)

   #.(sparc-opcode fcmps (fpop2 #x51) (logior f3fpopmask rd-g0) () :frs1 :frs2)
   #.(sparc-opcode fcmpd (fpop2 #x52) (logior f3fpopmask rd-g0) () :frs1d :frs2d)
   #.(sparc-opcode fcmpq (fpop2 #x53) (logior f3fpopmask rd-g0) () :frs1q :frs2q)
   #.(sparc-opcode fcmpes (fpop2 #x55) (logior f3fpopmask rd-g0) () :frs1 :frs2)
   #.(sparc-opcode fcmped (fpop2 #x56) (logior f3fpopmask rd-g0) () :frs1d :frs2d)
   #.(sparc-opcode fcmpeq (fpop2 #x57) (logior f3fpopmask rd-g0) () :frs1q :frs2q)

   ;; Whew.  That's the last of the floating-point.
   ;;; Generic coprocessor operations.
   #.(sparc-opcode cpop1 (f3op2 #x36) f3-mask () :cpop :crs1 :crs2 :cd)
   #.(sparc-opcode cpop2 (f3op2 #x37) f3-mask () :cpop :crs1 :crs2 :cd)

;;; A few handy forms of JMPL.
   ;; Return to [%o7+8], as when returning from a Lisp function or C leaf function.
   #.(sparc-opcode retl (f3op2 #x38 (rs1 sparc::%ra0) immed (simm13 8)) #xffffffff (f-unbr f-delayed f-branch-computed))
   ;; C code might use this to return before restoring the register window.
   #.(sparc-opcode ret (f3op2 #x38 (rs1 sparc::%ra1) immed (simm13 8)) #xffffffff (f-unbr f-delayed f-branch-computed))
   ;; Recognize subprim calls and jumps.  They're "constant branches" as far as
   ;; branch delay optimization goes.
   #.(sparc-opcode jump-subprim (f3op2 #x38 (rs1 %rnil)) (logior f3-mask rd-g0 rs1-g0) (f-unbr f-delayed) :subprim)
   #.(sparc-opcode call-subprim (f3op2 #x38 (rs1 %rnil) (rd %ra0)) (logior f3-mask rd-g0 rs1-g0) (f-unbr f-delayed f-jsr) :subprim)
   #.(sparc-opcode call-subprim* (f3op2 #x38 (rs1 %rnil) (rd %ra1)) (logior f3-mask rd-g0 rs1-g0) (f-unbr f-delayed f-jsr) :subprim)
   
   ;; For tailcalls, etc.
   #.(sparc-opcode jmp (f3op2 #x38) (logior f3-mask rd-g0) (f-unbr f-delayed f-branch-computed)  :rs1 :op2)
   ;; In general, make 'em specify rs1 and op2, in that order, explicitly.
   #.(sparc-opcode jmpl (f3op2 #x38) f3-mask (f-unbr f-delayed f-branch-computed f-jsr) :rs1 :op2 :rd)

   #.(sparc-opcode rett (f3op2 #x39) (logior f3-mask rd-g0) () :rs1 :op2)

					;Conditional traps.

   #.(sparc-opcode t (trap conda) trap-mask () :trap-number)
   #.(sparc-opcode ta (trap conda) trap-mask () :trap-number)
   #.(sparc-opcode tn (trap condn) trap-mask () :trap-number)
   #.(sparc-opcode tne (trap condne) trap-mask (f-readsCC) :trap-number)
   #.(sparc-opcode tnz (trap condne) trap-mask (f-readsCC) :trap-number)
   #.(sparc-opcode te (trap conde) trap-mask (f-readsCC) :trap-number)
   #.(sparc-opcode tz (trap conde) trap-mask (f-readsCC) :trap-number)
   #.(sparc-opcode tg (trap condg) trap-mask (f-readsCC) :trap-number)
   #.(sparc-opcode tle (trap condle) trap-mask (f-readsCC) :trap-number)
   #.(sparc-opcode tge (trap condge) trap-mask (f-readsCC) :trap-number)
   #.(sparc-opcode tl (trap condl) trap-mask (f-readsCC) :trap-number)
   #.(sparc-opcode tgu (trap condgu) trap-mask (f-readsCC) :trap-number)
   #.(sparc-opcode tleu (trap condleu) trap-mask (f-readsCC) :trap-number)
   #.(sparc-opcode tcc (trap condcc) trap-mask (f-readsCC) :trap-number)
   #.(sparc-opcode tgeu (trap condcc) trap-mask (f-readsCC) :trap-number)
   #.(sparc-opcode tcs (trap condcs) trap-mask (f-readsCC) :trap-number)
   #.(sparc-opcode tlu (trap condcs) trap-mask (f-readsCC) :trap-number)
   #.(sparc-opcode tpos (trap condpos) trap-mask (f-readsCC) :trap-number)
   #.(sparc-opcode tneg (trap condneg) trap-mask (f-readsCC) :trap-number)
   #.(sparc-opcode tvc (trap condvc) trap-mask (f-readsCC) :trap-number)
   #.(sparc-opcode tvs (trap condvs) trap-mask (f-readsCC) :trap-number)

   #.(sparc-opcode flush (f3op2 #x3b) (logior f3-mask rd-g0) () :rs1 :op2)

   #.(sparc-opcode save (f3op2 #x3c) f3-mask () :rs1 :op2 :rd)
   #.(sparc-opcode restore (f3op2 #x3d) f3-mask () :rs1 :op2 :rd)
   ))

(defun insert-default (operand high low val)
  (let* ((width (arch::operand-width operand))
         (offset (arch::operand-offset operand))
         (msbit (1- (+ width offset))))
    (declare (fixnum width offset msbit))
    (if (>= offset 16)
      (values (dpb val (byte width (- offset 16)) high) low)
      (if (< msbit 16)
        (values high (dpb val (byte width offset) low))
        (let* ((lowbits (- 16 offset)))
          (values
           (dpb (the fixnum (ash val (the fixnum (- lowbits))))
                (byte  (the fixnum (- width lowbits)) 0) 
                high)
           (dpb val (byte lowbits offset) low)))))))

(defun extract-default (operand instr)
  (let* ((width (arch::operand-width operand))
           (op (ldb (byte width (arch::operand-offset operand)) instr)))
    (if (and (logbitp sparc-operand-signed (arch::operand-flags operand))
                (logbitp (1- width) op))
         (- op (ash 1 width))
       op)))

(defun insert-label30 (high low val)
  (setq val (ash val -2))
  (values (dpb (ldb (byte 14 16) val) (byte 14 0) high)
	  (dpb (ldb (byte 16 0) val) (byte 16 0) low)))

(defun extract-label30 (instr)
  (let* ((val (ldb (byte 30 0) instr)))
    (ash (if (logbitp 20 val)
	     (- val (ash 1 30))
	     val)
	 2)))

(defun insert-rd (high low val)
  (values (dpb val (byte 5 9) high) low))

(defun extract-rd (instr)
  (ldb (byte 5 25) instr))

(defun insert-rs1 (high low val)
  (values (dpb (ldb (byte 3 2) val) (byte 3 0) high)
          (dpb (ldb (byte 2 0) val) (byte 2 14) low)))

(defun extract-rs1 (instr)
  (ldb (byte 5 14) instr))

(defun insert-rs2 (high low val)
  (values high (dpb (ldb (byte 5 0) val) (byte 5 0) low)))

(defun extract-rs2 (instr)
  (ldb (byte 5 0) instr))

(defun insert-op2 (high low val)
  (values high (dpb val (byte 14 0) low)))

(defun insert-shiftop2 (high low val)
  (insert-op2 high low (logior (logand val (ash 1 13))
			       (logand val (1- (ash 1 5))))))
(defun extract-shiftop2 (instr)
  (extract-op2 instr))

(defun insert-address (high low val)
  (values (dpb (ldb (byte 3 16) val) (byte 3 0) high)
	  (dpb (ldb (byte 16 0) val) (byte 16 0) low)))

(defun extract-address (instr)
  (ldb (byte 22 0) instr))

(defun extract-op2 (instr)
  (ldb (byte 14 0) instr))

(defun extract-rs2 (instr)
  (ldb (byte 5 0) instr))





(defun insert-label22 (high low val)
  (insert-imm22 high low (ash val -2))
)

(defun insert-imm22 (high low val)
  (values (dpb (ldb (byte 6 16) val) (byte 6 0) high)
	  (dpb (ldb (byte 16 0) val) (byte 16 0) low)))

(defun extract-label22 (instr)
  (let* ((val (ldb (byte 22 0) instr)))
    (ash (if (logbitp 21 val)
      (- val (ash 1 22))
      val)
	 2)))

(defun insert-trap-number (high low val)
  (values high (dpb (ldb (byte 14 0) val) (byte 14 0) low)))

(defun extract-trap-number (instr)
  (ldb (byte 14 0) instr))

(defun extract-imm22 (instr)
  (ldb (byte 22 0) instr))

(defun insert-subprim (high low val)
  (insert-op2 high low (logior (ash 1 13)
			       (ldb (byte 13 0) (sparc-subprim-nil-offset val)))))

(defun extract-subprim (instr)
  (+ (- (ldb (byte 13 0) instr) (ash 1 13))
     (+ arch::fulltag-nil 2048)))
    
  
(defvar *sparc-opcode-numbers* (make-hash-table :test #'equalp))

;; break operands of type :ADDRESS into :RS1, :OP2
(defun sparc-vinsn-operands (operands)
  (if (not (position :address operands :key #'arch::operand-index))
    operands
    (let* ((vinsn-ops ())
	   (rs1 (find :rs1 sparc::*sparc-operands* :key #'arch::operand-index))
	   (op2 (find :op2 sparc::*sparc-operands* :key #'arch::operand-index)))
      (dolist (op operands (nreverse vinsn-ops))
	(case (arch::operand-index op)
	  (:address
	   (push rs1 vinsn-ops)
	   (push op2 vinsn-ops))
	  (t
	   (push op vinsn-ops)))))))

(defun initialize-sparc-opcode-numbers ()
  (clrhash *sparc-opcode-numbers*)
  (dotimes (i (length *sparc-opcodes*))
      (let* ((code (svref *sparc-opcodes* i))
             (opcode (arch::opcode-opcode code))
             (mask (arch::opcode-mask code))
	     (operands (arch::opcode-operands code))
	     (vinsn-operands (sparc-vinsn-operands operands))
	     (vo-len (length vinsn-operands)))
	    
	
        (setf (gethash (string (arch::opcode-name code))  *sparc-opcode-numbers*) i)
        (setf (arch::opcode-op-high code) (ldb (byte 16 16) opcode)
              (arch::opcode-op-low code) (ldb (byte 16 0) opcode)
              (arch::opcode-mask-high code) (ldb (byte 16 16) mask)
              (arch::opcode-mask-low code) (ldb (byte 16 0) mask)
	      (arch::opcode-vinsn-operands code) vinsn-operands
	      (arch::opcode-max-vinsn-args code) vo-len
	      (arch::opcode-min-vinsn-args code) vo-len)
	
        ))
  (when (fboundp 'ccl::fixup-vinsn-templates)   ; not defined yet at bootstrap time
    (locally (declare (special ccl::*sparc-vinsn-templates*))
      (when (boundp 'ccl::*sparc-vinsn-templates*)
	(funcall 'ccl::fixup-vinsn-templates ccl::*sparc-vinsn-templates* *sparc-opcode-numbers*)))))

(initialize-sparc-opcode-numbers)

(defparameter *sparc-nop-opcode* (svref sparc::*sparc-opcodes*
                                        (gethash "NOP" sparc::*sparc-opcode-numbers*)))

(defparameter *sparc-bicc.a-opcode* (svref sparc::*sparc-opcodes*
                                        (gethash "BICC.A" sparc::*sparc-opcode-numbers*)))


(ccl::provide "SPARC-ASM")
