;;;-*- Mode: Lisp; Package: CCL -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   This file is part of OpenMCL.  
;;;
;;;   OpenMCL is licensed under the terms of the Lisp Lesser GNU Public
;;;   License , known as the LLGPL and distributed with OpenMCL as the
;;;   file "LICENSE".  The LLGPL consists of a preamble and the LGPL,
;;;   which is distributed with OpenMCL as the file "LGPL".  Where these
;;;   conflict, the preamble takes precedence.  
;;;
;;;   OpenMCL is referenced in the preamble as the "LIBRARY."
;;;
;;;   The LLGPL is also available online at
;;;   http://opensource.franz.com/preamble.html


(in-package "CCL")

(eval-when (:compile-toplevel :load-toplevel :execute)
  (require "VINSN")
  (require "SPARC-BACKEND"))

(eval-when (:compile-toplevel :execute)
  (require "SPARCENV"))

(defmacro define-sparc-vinsn (vinsn-name (results args &optional temps) &body body)
;  (print vinsn-name)
  (%define-vinsn *sparc-backend* vinsn-name results args temps body))

(eval-when (:compile-toplevel :load-toplevel :execute)
  (defun simm13 (val)
    (logior (ash 1 13) (ldb (byte 13 0) val))))


; Index "scaling" and constant-offset misc-ref vinsns.

(define-sparc-vinsn scale-32bit-misc-index (((dest :u32))
					    ((idx :imm)      ; A fixnum
					     )
					    ())
  (add idx #.(simm13 arch::misc-data-offset) dest))

(define-sparc-vinsn scale-16bit-misc-index (((dest :u32))
					    ((idx :imm)      ; A fixnum
					     )
					    ())
  (srl idx #.(simm13 1) dest)
  (add dest #.(simm13 arch::misc-data-offset) dest))



(define-sparc-vinsn scale-8bit-misc-index (((dest :u32))
					   ((idx :imm)      ; A fixnum
					    )
					   ())
  (srl idx #.(simm13 2) dest)
  (add dest #.(simm13 arch::misc-data-offset) dest))

(define-sparc-vinsn scale-64bit-misc-index (((dest :u32))
					    ((idx :imm)      ; A fixnum
					     )
					    ())
  (sll idx #.(simm13 1) dest)
  (add dest #.(simm13 arch::misc-dfloat-offset) dest))

(define-sparc-vinsn scale-1bit-misc-index (((word-index :u32)
					    (bitnum :u8))     ; (unsigned-byte 5)
					   ((idx :imm)      ; A fixnum
					    )
					   )
  ; Logically, we want to:
  ; 1) Unbox the index by shifting it right 2 bits.
  ; 2) Shift (1) right 5 bits
  ; 3) Scale (2) by shifting it left 2 bits.
  (srl idx #.(simm13 7) word-index)
  (sll word-index #.(simm13 2) word-index)
  (add word-index  #.(simm13 arch::misc-data-offset) word-index)
  (srl idx #.(simm13 2) bitnum)
  (and bitnum #.(simm13 31) bitnum))

(define-sparc-vinsn misc-ref-u32  (((dest :u32))
				   ((v :lisp)
				    (scaled-idx :u32))
				   ())
  (ld v scaled-idx dest))

(define-sparc-vinsn misc-ref-c-u32  (((dest :u32))
				     ((v :lisp)
				      (idx :u32const))
				     ())
  (ld v (:apply simm13 (:apply + arch::misc-data-offset (:apply ash idx 2))) dest))

(define-sparc-vinsn misc-ref-s32 (((dest :s32))
                             ((v :lisp)
                              (scaled-idx :u32))
                             ())
   (ld v scaled-idx dest))

(define-sparc-vinsn misc-ref-c-s32  (((dest :s32))
                               ((v :lisp)
                                (idx :u32const))
                               ())
  (ld v (:apply simm13 (:apply + arch::misc-data-offset (:apply ash idx 2))) dest))


(define-sparc-vinsn misc-set-c-u32 (()
				    ((val :u32)
				     (v :lisp)
				     (idx :u32const)))
  (st val v (:apply simm13 (:apply + arch::misc-data-offset (:apply ash idx 2)))))

(define-sparc-vinsn misc-set-u32 (()
				  ((val :u32)
				   (v :lisp)
				   (scaled-idx :u32)))
  (st val v scaled-idx))

                              
(define-sparc-vinsn misc-ref-single-float  (((dest :single-float))
                                     ((v :lisp)
                                      (scaled-idx :u32))
                                     ())
  (ldf v scaled-idx dest))

(define-sparc-vinsn misc-ref-c-single-float  (((dest :single-float))
					      ((v :lisp)
					       (idx :u32const))
					      ())
  (ldf v (:apply simm13 (:apply + arch::misc-data-offset (:apply ash idx 2))) dest))

(define-sparc-vinsn misc-ref-double-float  (((dest :double-float))
					    ((v :lisp)
					     (scaled-idx :u32))
					    ())
  (lddf v scaled-idx dest))


(define-sparc-vinsn misc-ref-c-double-float  (((dest :double-float))
					      ((v :lisp)
					       (idx :u32const))
					      ())
  (lddf v (:apply simm13 (:apply + arch::misc-dfloat-offset (:apply ash idx 3))) dest))

(define-sparc-vinsn misc-set-c-double-float (((val :double-float))
					     ((v :lisp)
					      (idx :u32const)))
  (stdf val v (:apply simm13 (:apply + arch::misc-dfloat-offset (:apply ash idx 3)))))

(define-sparc-vinsn misc-set-double-float (()
					   ((val :double-float)
					    (v :lisp)
					    (scaled-idx :u32)))
  (stdf val v scaled-idx))


(define-sparc-vinsn misc-ref-u16  (((dest :u16))
				   ((v :lisp)
				    (scaled-idx :u32))
				   ())
  (lduh v scaled-idx dest))

(define-sparc-vinsn misc-ref-c-u16  (((dest :u16))
				     ((v :lisp)
				      (idx :u32const))
				     ())
  (lduh v (:apply simm13 (:apply + arch::misc-data-offset (:apply ash idx 1))) dest))

(define-sparc-vinsn misc-set-c-u16  (((val :u16))
				     ((v :lisp)
				      (idx :u32const))
				     ())
  (sth val v (:apply simm13 (:apply + arch::misc-data-offset (:apply ash idx 1)))))

(define-sparc-vinsn misc-set-u16 (((val :u16))
				  ((v :lisp)
				   (scaled-idx :s32)))
  (sth val v scaled-idx))

(define-sparc-vinsn misc-ref-s16  (((dest :s16))
                             ((v :lisp)
                              (scaled-idx :u32))
                             ())
  (ldsh v scaled-idx dest))

(define-sparc-vinsn misc-ref-c-s16  (((dest :s16))
                               ((v :lisp)
                                (idx :u32const))
                               ())
  (ldsh v (:apply simm13 (:apply + arch::misc-data-offset (:apply ash idx 1))) dest))

(define-sparc-vinsn misc-ref-u8  (((dest :u8))
				  ((v :lisp)
				   (scaled-idx :u32))
				  ())
  (ldub v scaled-idx dest))

(define-sparc-vinsn misc-ref-c-u8  (((dest :u8))
				    ((v :lisp)
				     (idx :u32const))
				    ())
  (ldub v (:apply simm13 (:apply + arch::misc-data-offset idx)) dest))

(define-sparc-vinsn misc-set-c-u8  (((val :u8))
				    ((v :lisp)
				     (idx :u32const))
				    ())
  (stb val v (:apply simm13 (:apply + arch::misc-data-offset idx))))

(define-sparc-vinsn misc-set-u8  (((val :u8))
				  ((v :lisp)
				   (scaled-idx :u32))
				  ())
  (stb val v scaled-idx))

(define-sparc-vinsn misc-ref-s8  (((dest :s8))
				  ((v :lisp)
				   (scaled-idx :u32))
				  ())
  (ldsb v scaled-idx dest))

(define-sparc-vinsn misc-ref-c-s8  (((dest :s8))
				    ((v :lisp)
				     (idx :u32const))
				    ())
  (ldsb v (:apply simm13 (:apply + arch::misc-data-offset idx)) dest))


(define-sparc-vinsn misc-ref-c-bit (((dest :u8))
				    ((v :lisp)
				     (idx :u32const))
				    ())
  (ld v (:apply simm13 (:apply + arch::misc-data-offset (:apply ash idx -5))) dest)
  (srl dest (:apply simm13 (:apply - 31 (:apply logand idx #x1f))) dest)
  (and dest #.(simm13 1) dest))


(define-sparc-vinsn misc-ref-c-bit[fixnum] (((dest :imm))
					    ((v :lisp)
					     (idx :u32const))
					    ((temp :u32)))
  (ld v (:apply simm13 (:apply + arch::misc-data-offset (:apply ash idx -5))) temp)
  (srl temp (:apply simm13 (:apply - 31 (:apply logand idx #x1f))) temp)
  (and temp #.(simm13 1) temp)
  (sll temp #.(simm13 arch::fixnumshift) dest))


(define-sparc-vinsn misc-ref-node  (((dest :lisp))
				    ((v :lisp)
				     (scaled-idx :s32))
				    ())
  (ld v scaled-idx dest))

(define-sparc-vinsn misc-set-node (()
				   ((val :lisp)
				    (v :lisp)
				    (scaled-idx :s32))
				   ())
  (st val v scaled-idx))

(define-sparc-vinsn misc-set-node& (()
				    ((val :lisp)
				     (v :lisp)
				     (scaled-idx :s32))
				    ())
  (add v scaled-idx sparc::%loc-g)
  (st sparc::%loc-g sparc::%memo #.(simm13 -4))
  (add sparc::%memo #.(simm13 -4) sparc::%memo)
  (st val sparc::%loc-g sparc::%rzero))


(define-sparc-vinsn misc-ref-c-node (((dest :lisp))
				     ((v :lisp)
				      (idx :s16const))
				     ())
  (ld v (:apply simm13 (:apply + arch::misc-data-offset (:apply ash idx 2))) dest))

(define-sparc-vinsn misc-set-c-node (()
				     ((val :lisp)
				      (v :lisp)
				      (idx :s16const))
				     ())
  (st val v (:apply simm13 (:apply + arch::misc-data-offset (:apply ash idx 2)))))

(define-sparc-vinsn misc-set-c-node& (()
				      ((val :lisp)
				       (v :lisp)
				       (idx :s16const))
				      ())
  (add v (:apply simm13 (:apply + arch::misc-data-offset (:apply ash idx 2))) sparc::%loc-g)
  (st sparc::%loc-g sparc::%memo #.(simm13 -4))
  (add sparc::%memo #.(simm13 -4) sparc::%memo)
  (st val sparc::%loc-g sparc::%rzero))

; We should rarely need to do this; more often interested in
; the element-count or subtag than in the "whole" header.
(define-sparc-vinsn misc-ref-header (((dest :u32))
				     ((v :lisp))
				     ())
  (ld v #.(simm13 arch::misc-header-offset) dest))

(define-sparc-vinsn misc-element-count[fixnum] (((dest :imm))
						((v :lisp))
						((temp :u32)))
  (ld v #.(simm13 arch::misc-header-offset) temp)
  (srl temp #.(simm13 arch::num-subtag-bits) temp)
  (sll temp #.(simm13 arch::fixnumshift) dest))

(define-sparc-vinsn check-misc-bound (()
				      ((idx :imm)
				       (v :lisp))
				      ((temp :u32)))
  (ld v #.(simm13 arch::misc-header-offset) temp)
  (srl temp #.(simm13 arch::num-subtag-bits) temp)
  (sll temp #.(simm13 arch::fixnumshift) temp)
  (cmp idx temp)
  (tcc #.(simm13 sparc::trap-bounds-check)))

(define-sparc-vinsn 2d-df-scaled-index (((dest :u32))
					((array :lisp)
					 (i :imm)
					 (j :imm)
					 (dim1 :u32)))
  (umul i dim1 dest)
  (add dest j dest)
  (add dest dest dest)
  (add dest #.(simm13 arch::misc-dfloat-offset) dest))

(define-sparc-vinsn 2d-dim1 (((dest :u32))
			     ((header :lisp)))
  (ld header (simm13 (+ arch::misc-data-offset (* 4 (1+ arch::arrayH.dim0-cell)))) dest)
  (sra dest #.(simm13 arch::fixnumshift) dest))

;; Return dim1 (unboxed)
(define-sparc-vinsn check-2d-bound (((dim :u32))
				    ((i :imm)
				     (j :imm)
				     (header :lisp)))
  (ld header (simm13 (+ arch::misc-data-offset (* 4 arch::arrayH.dim0-cell))) dim)
  (cmp i dim)
  (tcc #.(simm13 sparc::trap-bounds-check))
  (ld header (simm13 (+ arch::misc-data-offset (* 4 (1+ arch::arrayH.dim0-cell)))) dim)
  (cmp j dim)
  (tcc #.(simm13 sparc::trap-bounds-check))  
  (sra dim #.(simm13 arch::fixnumshift) dim))

(define-sparc-vinsn array-data-vector-ref (((dest :lisp))
					   ((header :lisp)))
  (ld header #.(simm13 arch::arrayH.data-vector) dest))
  

(define-sparc-vinsn check-arrayH-rank (()
				       ((header :lisp)
					(expected :u32const))
				       ((rank :imm)))
  (ld header #.(simm13 arch::arrayH.rank) rank)
  (cmp rank (:apply simm13 (:apply ash expected arch::fixnumshift)))
  (tne #.(simm13 sparc::trap-rank-check)))

(define-sparc-vinsn check-arrayH-flags (()
                                 ((header :lisp)
                                  (expected :u16const))
                                 ((flags :imm)
                                  (xreg :u32)))
  (sethi (:apply ldb (byte 22 10) (:apply ash expected arch::fixnumshift)) xreg)
  (or xreg (:apply simm13 (:apply ldb (byte 10 0) (:apply ash expected arch::fixnumshift))) xreg)
  (ld header #.(simm13 arch::arrayH.flags) flags)
  (cmp xreg flags)
  (tne #.(simm13 sparc::trap-array-flags-check)))

(define-sparc-vinsn misc-element-count[u32] (((dest :u32))
					     ((v :lisp))
					     ())
		    
  (ld v #.(simm13 arch::misc-header-offset) dest)
  (srl dest #.(simm13 arch::num-subtag-bits) dest))

(define-sparc-vinsn misc-subtag[fixnum] (((dest :imm))
					 ((v :lisp))
					 ((temp :u32)))
  (ldub v #.(simm13 arch::misc-subtag-offset) temp)
  (sll temp #.(simm13 arch::fixnumshift) dest))

(define-sparc-vinsn misc-subtag[u32] (((dest :u32))
				      ((v :lisp))
				      ())
  (ldub v #.(simm13 arch::misc-subtag-offset) dest))

(define-sparc-vinsn header->subtag[u32] (((dest :u32))
					 ((header :u32))
					 ())
  (and header #.(simm13 (1- (ash 1 arch::num-subtag-bits))) dest))

(define-sparc-vinsn header->subtag[fixnum] (((dest :imm))
					    ((header :u32))
					    ())
  (sll header #.(simm13 (- 32 arch::num-subtag-bits)) dest)
  (srl dest #.(simm13 (- 32 (+ arch::num-subtag-bits arch::fixnumshift))) dest))

(define-sparc-vinsn header->element-count[u32] (((dest :u32))
						((header :u32))
						())
  (srl header #.(simm13 arch::num-subtag-bits) dest))

(define-sparc-vinsn node-slot-ref  (((dest :lisp))
				    ((node :lisp)
				     (cellno :u32const)))
  (ld node (:apply simm13 (:apply + arch::misc-data-offset (:apply ash cellno 2))) dest))

(define-sparc-vinsn node-slot-set (()
				   ((node :lisp)
				    (cellno :u32const)
				    (newval :lisp)))
  (st newval node (:apply simm13 (:apply + arch::misc-data-offset (:apply ash cellno 2)))))


(define-sparc-vinsn node-slot-set& (()
				    ((node :lisp)
				     (cellno :u32const)
				     (newval :lisp)))
  (add node (:apply simm13 (:apply + arch::misc-data-offset (:apply ash cellno 2))) sparc::%loc-g)
  (st sparc::%loc-g sparc::%memo #.(simm13 -4))
  (add sparc::%memo #.(simm13 -4) sparc::%memo)
  (st newval sparc::%loc-g sparc::%rzero))


; Untagged memory reference & assignment.

(define-sparc-vinsn mem-ref-c-fullword (((dest :u32))
					((src :address)
					 (index :s16const)))
  (ld src (:apply simm13 index) dest))

(define-sparc-vinsn mem-ref-fullword (((dest :u32))
				      ((src :address)
				       (index :s16)))
  (ld src index dest))

(define-sparc-vinsn mem-ref-c-u16 (((dest :u16))
				   ((src :address)
				    (index :s16const)))
  (lduh src (:apply simm13 index) dest))

(define-sparc-vinsn mem-ref-u16 (((dest :u16))
				 ((src :address)
				  (index :s16)))
  (lduh src index dest))


(define-sparc-vinsn mem-ref-c-s16 (((dest :s16))
				   ((src :address)
				    (index :s16const)))
  (ldsh src (:apply simm13 index) dest))

(define-sparc-vinsn mem-ref-s16 (((dest :s16))
				 ((src :address)
				  (index :s16)))
  (ldsh src index dest))

(define-sparc-vinsn mem-ref-c-u8 (((dest :u8))
				  ((src :address)
				   (index :s16const)))
  (ldub src (:apply simm13 index) dest))

(define-sparc-vinsn mem-ref-u8 (((dest :u8))
				((src :address)
				 (index :s16)))
  (ldub src index dest))

(define-sparc-vinsn mem-ref-c-s8 (((dest :s8))
				  ((src :address)
				   (index :s16const)))
  (ldsb src (:apply simm13 index) dest))

(define-sparc-vinsn mem-ref-s8 (((dest :s8))
				((src :address)
				 (index :s16)))
  (ldsb src index dest))


(define-sparc-vinsn mem-set-c-fullword (()
					((val :u32)
					 (src :address)
					 (index :s16const)))
  (st val src (:apply simm13 index)))

(define-sparc-vinsn mem-set-fullword (()
				      ((val :u32)
				       (src :address)
				       (index :s32)))
  (st val src index))

(define-sparc-vinsn mem-set-c-halfword (()
					((val :u16)
					 (src :address)
					 (index :s16const)))
  (sth val src (:apply simm13 index)))

(define-sparc-vinsn mem-set-halfword (()
				      ((val :u16)
				       (src :address)
				       (index :s32)))
  (sth val src index))

(define-sparc-vinsn mem-set-c-byte (()
				    ((val :u16)
				     (src :address)
				     (index :s16const)))
  (stb val src (:apply simm13 index)))

(define-sparc-vinsn mem-set-byte (()
				  ((val :u16)
				   (src :address)
				   (index :s32)))
  (stb val src index))


; Tag and subtag extraction, comparison, checking, trapping ...

(define-sparc-vinsn extract-tag (((tag :u8)) 
				 ((object :lisp)) 
				 ())
  (and object #.(simm13 arch::tagmask) tag))

(define-sparc-vinsn extract-tag[fixnum] (((tag :imm))
					 ((object :lisp)))
  (sll object #.(simm13 (- arch::nbits-in-word arch::nlisptagbits)) tag)
  (srl tag #.(simm13 (- arch::nbits-in-word (+ arch::fixnumshift
					       arch::nlisptagbits))) tag))


(define-sparc-vinsn extract-fulltag (((tag :u8))
				     ((object :lisp))
				     ())
  (and object #.(simm13 arch::fulltagmask) tag))


(define-sparc-vinsn extract-fulltag[fixnum] (((tag :imm))
                                       ((object :lisp)))
  (sll object #.(simm13 (- arch::nbits-in-word arch::ntagbits)) tag)
  (srl tag #.(simm13 (- arch::nbits-in-word (+ arch::fixnumshift)
					       arch::ntagbits))
       tag))

(define-sparc-vinsn extract-typecode (((code :u8))
				      ((object :lisp))
				      ())
  (and object #.(simm13 arch::tagmask) code)
  (cmp code #.(simm13 arch::tag-misc))
  (be.a :have-code)
    (ldub object #.(simm13 arch::misc-subtag-offset) code)
  :have-code)

(define-sparc-vinsn extract-typecode[fixnum] (((code :imm))
					      ((object (:lisp (:ne code))))
					      ( (subtag :u8)))

  (and object #.(simm13 arch::tagmask) subtag)
  (cmp subtag #.(simm13 arch::tag-misc))
  (be.a :have-code)
    (ldub object #.(simm13 arch::misc-subtag-offset) subtag)
  :have-code
  (sll subtag #.(simm13 arch::fixnumshift) code))


(define-sparc-vinsn require-fixnum (()
                              ((object :lisp))
                              ((tag :u8)))
  :again
  (andcc object #.(simm13 arch::tagmask) tag)
  (bne.a :again)
    (uuo_intcerr arch::error-object-not-fixnum object))

(define-sparc-vinsn require-integer (()
				     ((object :lisp))
				     ((tag :u8)))
  :again
  (andcc object #.(simm13 arch::tagmask) tag)
  (be :got-it)
    (cmp tag #.(simm13 arch::tag-misc))
  (bne.a :again)
    (uuo_intcerr arch::error-object-not-integer object)
  (ldub object #.(simm13 arch::misc-subtag-offset) tag)
  (cmp tag #.(simm13 arch::subtag-bignum))
  (bne.a :again)
    (uuo_intcerr arch::error-object-not-integer object)
  :got-it)

(define-sparc-vinsn require-simple-vector (()
					   ((object :lisp))
					   ((tag :u8)))
  :again
  (and object #.(simm13 arch::tagmask) tag)
  (cmp object #.(simm13 arch::tag-misc))
  (bne.a :again)
    (uuo_intcerr arch::error-object-not-simple-vector object)
  (ldub object #.(simm13 arch::misc-subtag-offset) tag)
  (cmp tag #.(simm13 arch::subtag-simple-vector))
  (bne.a :again)
    (uuo_intcerr arch::error-object-not-simple-vector object))

(define-sparc-vinsn require-simple-string (()
					   ((object :lisp))
					   ((tag :u8)))
  :again
  (and object #.(simm13 arch::tagmask) tag)
  (cmp tag #.(simm13 arch::tag-misc))
  (bne.a :again)
    (uuo_intcerr arch::error-object-not-simple-string object)
  (ldub object #.(simm13 arch::misc-subtag-offset) tag)
  (cmp tag #.(simm13 arch::subtag-simple-base-string))
  (bne.a :again)
    (uuo_intcerr arch::error-object-not-simple-string object))

  
(define-sparc-vinsn require-real (()
                            ((object :lisp))
                            ((tag :u8)))
  :again
  (andcc object #.(simm13 arch::tagmask) tag)
  (be :got-it)
    (cmp tag #.(simm13 arch::tag-misc))
  (bne.a :again)
    (uuo_intcerr arch::error-object-not-real object)
  (ldub object #.(simm13 arch::misc-subtag-offset) tag)
  (cmp tag #.(simm13 ppc::max-real-subtag))
  (bg.a :again)
    (uuo_intcerr arch::error-object-not-real object)
  :got-it)

(define-sparc-vinsn require-number (()
				    ((object :lisp))
				    ((tag :u8)))
  :again
  (andcc object #.(simm13 arch::tagmask) tag)
  (be :got-it)
    (cmp tag #.(simm13 arch::tag-misc))
  (bne.a :again)
    (uuo_intcerr arch::error-object-not-real object)
  (ldub object #.(simm13 arch::misc-subtag-offset) tag)
  (cmp tag #.(simm13 ppc::max-numeric-subtag))
  (bg.a :again)
    (uuo_intcerr arch::error-object-not-number object)
  :got-it)


(define-sparc-vinsn require-list (()
				  ((object :lisp))
				  ((tag :u8)))
  :again
  (and object #.(simm13 arch::tagmask) tag)
  (cmp tag #.(simm13 arch::tag-list))
  (bne.a :again)
    (uuo_intcerr ppc::error-object-not-list object))

(define-sparc-vinsn require-symbol (()
				    ((object :lisp))
				    ((tag :u8)))
  :again
  (cmp object sparc::%rnil)
  (and object #.(simm13 arch::tagmask) tag)
  (be :got-it)
    (cmp tag #.(simm13 arch::tag-misc))
  (bne.a :again)
    (uuo_intcerr arch::error-object-not-symbol object)  
  (ldub object #.(simm13 arch::misc-subtag-offset) tag)
  (cmp tag #.(simm13 arch::subtag-symbol))
  (bne.a :again)
    (uuo_intcerr arch::error-object-not-symbol object)
  :got-it)

(define-sparc-vinsn require-character (()
				       ((object :lisp))
				       ((tag :u8)))
  :again
  (and object #.(simm13 arch::subtag-mask) tag)
  (cmp tag #.(simm13 arch::subtag-character))
  (bne.a :again)
    (uuo_intcerr ppc::error-object-not-character object))


(define-sparc-vinsn require-u8 (()
				((object :lisp))
				())
  :again
  (andncc object #.(simm13 (ash (1- (ash 1 8)) arch::fixnumshift)) sparc::%rzero)
  (bne.a :again)
    (uuo_intcerr ppc::error-object-not-unsigned-byte-8 object))

(define-sparc-vinsn box-fixnum (((dest :imm))
				((src :s32)))
  (sll src #.(simm13 arch::fixnumshift) dest))

(define-sparc-vinsn fixnum->s32 (((dest :s32))
				 ((src :imm)))
  (sra src #.(simm13 arch::fixnumshift) dest))


(define-sparc-vinsn fixnum->u32 (((dest :u32))
				 ((src :imm)))
  (srl src #.(simm13 arch::fixnumshift) dest))


; An object is of type (UNSIGNED-BYTE 32) iff
;  a) it's of type (UNSIGNED-BYTE 30) (e.g., an unsigned fixnum)
;  b) it's a bignum of length 1 and the 0'th digit is positive
;  c) it's a bignum of length 2 and the sign-digit is 0.

(define-sparc-vinsn unbox-u32 (((dest :u32))
			       ((src :lisp))
			       ((header :u32)))
  :again
  ;; If a fixnum, the bottom arch::fixnumshift bits and the top bit must
  ;; all be zero.
  (sethi (ash #x80000000 -10) header)
  (or header #.(simm13 arch::fixnummask) header)
  (andcc src header header)
  (be :got-it)
    (srl src #.(simm13 arch::fixnumshift) dest)
  (and src #.(simm13 arch::tagmask) dest)
  (cmp dest #.(simm13 arch::tag-misc))
  (bne.a :again)
    (uuo_interr arch::error-object-not-unsigned-byte-32 src)
  (ld src #.(simm13 arch::misc-header-offset) header)
  (ld src #.(simm13 arch::misc-data-offset) dest)
  (cmp header #.(simm13 arch::one-digit-bignum-header))
  (bne :maybe-two)
    (tst dest)
  (bg :got-it)
    (nop)
  (b :again)
    (uuo_interr arch::error-object-not-unsigned-byte-32 src)
  :maybe-two
  (cmp header #.(simm13 arch::two-digit-bignum-header))
  (bne.a :again)
    (uuo_interr arch::error-object-not-unsigned-byte-32 src)
  (ld src #.(simm13 (+ arch::misc-data-offset 4)) header)
  (tst header)
  (bne.a :again)
    (uuo_interr arch::error-object-not-unsigned-byte-32 src)
  :got-it)

; an object is of type (SIGNED-BYTE 32) iff
; a) it's a fixnum
; b) it's a bignum with exactly one digit.

(define-sparc-vinsn unbox-s32 (((dest :s32))
			       ((src :lisp))
			       ((tag :u32)))
  :again
  (andcc src #.(simm13 arch::tagmask) tag)
  (be :got-it)
    (sra src #.(simm13 arch::fixnumshift) dest)
  (cmp tag #.(simm13 arch::tag-misc))
  (bne.a :again)
    (uuo_interr arch::error-object-not-signed-byte-32 src)
  (ld src #.(simm13 arch::misc-header-offset) tag)
  (cmp tag #.(simm13 arch::one-digit-bignum-header))
  (ld src #.(simm13 arch::misc-data-offset) dest)
  (bne.a :again)
    (uuo_interr arch::error-object-not-signed-byte-32 src)
  :got-it)



(define-sparc-vinsn unbox-u16 (((dest :u16))
			       ((src :lisp))
			       ())

  ;; The bottom arch::fixnumshift bits and the top (- 31 (+ arch::fixnumshift 16)) must all be zero.
  (sethi (ash #xfffc0000 -10) dest)
  (or dest #.(simm13 arch::fixnummask) dest)
  (andcc src dest dest)
  (be.a :got-it)
    (srl src #.(simm13 arch::fixnumshift) dest)
  (uuo_interr arch::error-object-not-unsigned-byte-16 src)
  :got-it)

(define-sparc-vinsn unbox-s16 (((dest :s16))
			       ((src :lisp))
			       ())
  :again
  (sll src #.(simm13 (- 16 arch::fixnumshift)) dest)
  (sra dest #.(simm13 (- 16 arch::fixnumshift)) dest)
  (cmp src dest)
  (bne.a :again)
    (uuo_interr ppc::error-object-not-signed-byte-16 src)
  (andcc src #.(simm13 arch::tagmask) 0) 
  (bne.a :again)
    (uuo_interr ppc::error-object-not-signed-byte-16 src))
  
  
(define-sparc-vinsn unbox-u8 (((dest :u8))
			      ((src :lisp))
			      ())
  ; The bottom arch::fixnumshift bits and the top (- 31 (+ arch::fixnumshift 8)) must all be zero.
  :again
  (mov #.(simm13 (ash (1- (ash 1 8)) arch::fixnumshift)) dest)
  (andncc src dest dest)
  (bne.a :again)
   (uuo_interr ppc::error-object-not-unsigned-byte-8 src)
  (srl src #.(simm13 arch::fixnumshift) dest))

(define-sparc-vinsn unbox-s8 (((dest :s8))
			      ((src :lisp))
			      )
  :again
  (sll src #.(simm13 (- ppc::nbits-in-word (+ 8 arch::fixnumshift))) dest)
  (sra dest #.(simm13 (- ppc::nbits-in-word (+ 8 arch::fixnumshift))) dest)
  (cmp dest src)
  (bne.a :again)
    (uuo_interr ppc::error-object-not-signed-byte-16 src)
  (andcc src #.(simm13 arch::tagmask) dest) 
  (bne.a :again)
    (uuo_interr ppc::error-object-not-signed-byte-16 src)
  (sra src #.(simm13 arch::fixnumshift) dest))

;; A bit more fuss than we need, given that EXTENDED-CHARs can't happen.
(define-sparc-vinsn unbox-base-char (((dest :u32))
				     ((src :lisp))
				     ())
  :again
  (sethi (ash #xff000000 -10) dest)
  (or dest #.(simm13 arch::subtag-mask) dest)
  (and src dest dest)
  (cmp dest #.(simm13 arch::subtag-character))
  (srl src #.(simm13 arch::charcode-shift) dest)
  (bne.a :again)
    (uuo_interr arch:error-object-not-base-char src))

(define-sparc-vinsn unbox-character (((dest :u32))
				     ((src :lisp))
				     ())
  :again
  (and src #.(simm13 arch::subtag-mask) dest)
  (cmp dest #.(simm13 arch::subtag-character))
  (srl src #.(simm13 arch::charcode-shift) dest)
  (bne.a :again)
    (uuo_interr arch::error-object-not-character src))

(define-sparc-vinsn unbox-bit (((dest :u32))
			       ((src :lisp))
			       ())
  :again
  (cmp src #.(simm13 (ash 1 arch::fixnumshift)))
  (sra src #.(simm13 arch::fixnumshift) dest)
  (bcc.a :again)
    (uuo_interr arch::error-object-not-bit src))

(define-sparc-vinsn unbox-bit-bit0 (((dest :u32))
				    ((src :lisp))
				    ())
  :again
  (cmp src #.(simm13 (ash 1 arch::fixnumshift)))
  (sll src #.(simm13 (- 32 (1+ arch::fixnumshift))) dest)
  (bcc.a :again)
    (uuo_interr arch::error-object-not-bit src))

(define-sparc-vinsn fixnum->fpr (((dest :double-float))
				 ((src :lisp))
				 ((imm :s32)))
  (add sparc::%sp #.(simm13 -16) sparc::%sp)
  (sra src #.(simm13 arch::fixnumshift) imm)
  (st imm sparc::%sp #.(simm13 64))
  (ldf sparc::%sp #.(simm13 64) dest)
  (add sparc::%sp #.(simm13 16) sparc::%sp)
  (fitod dest dest))


(define-sparc-vinsn shift-right-variable-word (((dest :u32))
					       ((src :u32)
						(sh :u32)))
  (srl src sh dest))

(define-sparc-vinsn u32logandc2 (((dest :u32))
				 ((x :u32)
				  (y :u32)))
  (andn x y dest))

(define-sparc-vinsn u32logior (((dest :u32))
			       ((x :u32)
				(y :u32)))
  (or x y dest))



(define-sparc-vinsn complement-shift-count (((dest :u32))
					    ((src :u32)))
  (mov #.(simm13 32) dest)
  (sub dest src dest))

(define-sparc-vinsn extract-lowbyte (((dest :u32))
				     ((src :lisp)))
  (and src #.(simm13 arch::subtag-mask) dest))



; Set the "EQ" bit in condition-register if object is
; a fixnum.  Leave the object's tag in TAG.

(define-sparc-vinsn eq-if-fixnum (((tag :u8))
				  ((object :lisp))
				  ())
  (andcc object #.(simm13 arch::tagmask) tag))


(define-sparc-vinsn trap-unless-tag= (()
				      ((object :lisp)
				       (tagval :u16const))
				      ((tag :u8)))
  <
  (and object #.(simm13 arch::tagmask) tag)
  (cmp tag (:apply simm13 tagval))
  (tne #.(simm13 sparc::trap-lisptag-check))
  >)

(define-sparc-vinsn trap-unless-fulltag= (()
					  ((object :lisp)
					   (tagval :u16const))
					  ((tag :u8)))
  <
  (and object #.(simm13 arch::fulltagmask) tag)
  (cmp tag (:apply simm13 tagval))
  (tne #.(simm13 sparc::trap-fulltag-check))
  >)

(define-sparc-vinsn trap-unless-lowbyte= (()
					  ((object :lisp)
					   (tagval :u16const))
					  ((tag :u8)))
  <
  (and object #.(simm13 arch::subtag-mask) tag)
  (cmp tag (:apply simm13 tagval))
  (tne #.(simm13 sparc::trap-lowbyte-check))
  >)

(define-sparc-vinsn trap-unless-typecode= (()
					   ((object :lisp)
					    (tagval :u16const))
					   ((tag :u8)
					    ))
  (and object #.(simm13 arch::tagmask) tag)
  (cmp tag #.(simm13 arch::tag-misc))
  (be.a :do-trap)
  <
    (ldub object #.(simm13 arch::misc-subtag-offset) tag)
  :do-trap
  (cmp tag (:apply simm13 tagval))
  (tne #.(simm13 sparc::trap-typecode-check))
  >)
  
(define-sparc-vinsn subtract-constant (((dest :imm))
				       ((src :imm)
					(const :s16const)))
  (sub src (:apply simm13 const) dest))

;; Bit-extraction & boolean operations

(eval-when (:compile-toplevel :execute)
  (assert (= ppc::t-offset #b10001)))         ; PPC-bits 31 and 27 set

;; For some mind-numbing reason, IBM decided to call the most significant
;; bit in a 32-bit word "bit 0" and the least significant bit "bit 31"
;; (this despite the fact that it's essentially a big-endian architecture
;; (it was exclusively big-endian when this decision was made.))
;; We'll probably be least confused if we consistently use this backwards
;; bit ordering (letting things that have a "sane" bit-number worry about
;; it at compile-time or run-time (subtracting the "sane" bit number from
;; 31.))

(define-sparc-vinsn extract-variable-bit[fixnum] (((dest :imm))
						  ((src :u32)
						   (bitnum :u8))
						  ((temp :u32)))
  (sll src bitnum temp)
  (srl temp #.(simm13 31) temp)
  (sll temp #.(simm13 arch::fixnumshift) dest))


;; Sometimes we try to extract a single bit from some source register
;; into a destination bit (typically 31, sometimes fixnum bit 0 = 29).
;; If we copy bit 0 (whoops, I mean "bit 31") to bit 4 (aka 27) in a
;; given register, we get a value that's either 17 (the arithmetic difference
;; between T and NIL) or 0.

(define-sparc-vinsn bit31->truth (((dest :lisp)
				   (bits :u32))
				  ((bits :u32))
				  ((temp :u32)))
  (sll bits #.(simm13 4) temp)
  (or temp bits bits)
  (add sparc::%rnil bits dest))


(define-sparc-vinsn invert-bit31 (((bits :u32))
				  ((bits :u32))
				  ())
  (xor bits #.(simm13 1) bits))

;; Some of the obscure-looking instruction sequences - which map some relation
;; to the LSBit of some register - were found by the GNU SuperOptimizer.

(define-sparc-vinsn eq0->bit31 (((bits :u32))
				((src (t (:ne bits)))))
  (addcc src #.(simm13 -1) bits)
  (subx src bits bits))

(define-sparc-vinsn ne0->bit31 (((bits :u32))
				((src (t (:ne bits)))))
  (addcc src #.(simm13 -1) bits)
  (addx 0 0 bits))

(define-sparc-vinsn lt0->bit31 (((bits :u32))
				((src (t (:ne bits)))))
  (srl src #.(simm13 31) bits))                   ; bits = 0000...000X


(define-sparc-vinsn ge0->bit31 (((bits :u32))
				((src (t (:ne bits)))))
  (srl src #.(simm13 31) bits)
  (xor bits #.(simm13 1) bits))                   ; bits = 0000...000X


(define-sparc-vinsn le0->bit31 (((bits :u32))
				((src (t (:ne bits)))))
  (sll src #.(simm13 1) bits)
  (subcc src bits bits)
  (addx bits src bits)
  (xor bits 1 bits))

(define-sparc-vinsn gt0->bit31 (((bits :u32))
				((src (t (:ne bits)))))
  (sll src #.(simm13 1) bits)
  (subcc src bits bits)
  (addx bits src bits))

(define-sparc-vinsn ne->boolean (((dest t))
				 ((x t)
				  (y t))
				 ())
  (cmp x y)
  (bne.a :done)
    (add sparc::%rnil #.(simm13 arch::t-offset) dest)
  (mov sparc::%rnil dest)
  :done)

(define-sparc-vinsn eq->boolean (((dest t))
				 ((x t)
				  (y t)))
  (cmp x y)
  (be.a :done)
    (add sparc::%rnil #.(simm13 arch::t-offset) dest)
  (mov sparc::%rnil dest)
  :done)

(define-sparc-vinsn lt->boolean (((dest t))
				 ((x t)
				  (y t)))
  (cmp x y)
  (bl.a :done)
    (add sparc::%rnil #.(simm13 arch::t-offset) dest)
  (mov sparc::%rnil dest)
  :done)


(define-sparc-vinsn le->boolean (((dest t))
				 ((x t)
				  (y t)))
  (cmp x y)
  (ble.a :done)
    (add sparc::%rnil #.(simm13 arch::t-offset) dest)
  (mov sparc::%rnil dest)
  :done)



(define-sparc-vinsn gt->boolean (((dest t))
				 ((x t)
				  (y t)))
  (cmp x y)
  (bg.a :done)
    (add sparc::%rnil #.(simm13 arch::t-offset) dest)
  (mov sparc::%rnil dest)
  :done)

(define-sparc-vinsn ge->boolean (((dest t))
				 ((x t)
				  (y t)))
  (cmp x y)
  (bge.a :done)
    (add sparc::%rnil #.(simm13 arch::t-offset) dest)
  (mov sparc::%rnil dest)
  :done)


(define-sparc-vinsn compare (()
			     ((arg0 t)
			      (arg1 t))
			     ())
  (cmp arg0 arg1))




              

(define-sparc-vinsn double-float+-2 (((result :double-float))
				     ((x :double-float)
				      (y :double-float))
				     )
  (faddd x y result)
  (fmovs 0 0))

(define-sparc-vinsn double-float--2 (((result :double-float))
                                ((x :double-float)
                                 (y :double-float))
                                ())
  (fsubd x y result)
  (fmovs 0 0))


(define-sparc-vinsn double-float*-2 (((result :double-float))
				     ((x :double-float)
				      (y :double-float))
				     ())
  (fmuld x y result)
  (fmovs 0 0))

(define-sparc-vinsn double-float/-2 (((result :double-float))
                                ((x :double-float)
                                 (y :double-float))
                                )
  (fdivd x y result)
  (fmovs 0 0))

(define-sparc-vinsn single-float+-2 (((result :single-float))
                                ((x :single-float)
                                 (y :single-float))
                                ())
  (fadds x y result)
  (fmovs 0 0))


(define-sparc-vinsn single-float--2 (((result :single-float))
				     ((x :single-float)
				      (y :single-float))
				     ())
  (fsubs x y result)
  (fmovs 0 0))


(define-sparc-vinsn single-float*-2 (((result :single-float))
				     ((x :single-float)
				      (y :single-float))
				     ())
  (fmuls x y result)
  (fmovs 0 0))


(define-sparc-vinsn single-float/-2 (((result :single-float))
				     ((x :single-float)
				      (y :single-float))
				     ())
  (fdivs x y result)
  (fmovs 0 0))

;; I think that this can be a NOP on the SPARC
(define-sparc-vinsn clear-fpu-exceptions (()
                                    ((freg :double-float)))
)

; Someone should have made sure that the constant is of type (SIGNED-BYTE 13).
(define-sparc-vinsn compare-signed-s13const (()
					     ((arg0 :imm)
					      (imm :s16const))
					     ())
  (cmp arg0 (:apply simm13 imm)))

; and here it needs to be of type (UNSIGNED-BYTE 12)
(define-sparc-vinsn compare-unsigned-u16const (()
					       ((arg0 :imm)
						(imm :u16const))
					       ())
  (cmp arg0 (:apply simm13 imm)))




;; Extract a constant bit (0-31) from src; make it be bit 31 of dest.
;; Bitnum is treated mod 32.
(define-sparc-vinsn extract-constant-ppc-bit (((dest :u32))
					      ((src :imm)
					       (bitnum :u16const))
					      ())
  ((:pred /= bitnum 31)
   (srl src (:apply simm13 (:apply - 31 bitnum)) dest))
  ((:pred /= bitnum 0)
   (and dest #.(simm13 1) dest)))

(define-sparc-vinsn set-constant-ppc-bit-to-variable-value (((dest :u32))
							    ((src :u32)
							     (bitval :u32)      ; 0 or 1
							     (bitnum :u8const))
							    ((mask :u32)))
  ;; It's not OK to use a sign-extended 13-bit constant mask: we want the mask
  ;; to contain exactly one set bit.
  ((:pred < (:apply (- 31 bitnum)) 12)
   (andn src (:apply simm13 (:apply ash 1 (:apply - 31 bitnum))) mask))
  ((:pred >= (:apply (- 31 bitnum)) 12)
   (sethi (:apply ash #x80000000 (:apply - (:apply + 10 bitnum))) mask)
   (andn src mask mask))
  (sll bitval (:apply simm13 bitnum) dest)
  (or dest mask dest))


(define-sparc-vinsn set-constant-ppc-bit-to-1 (((dest :u32))
					       ((src :u32)
						(bitnum :u8const))
					       ((mask :u32)))
  ((:pred < (:apply - 31 bitnum) 12)
   (or src (:apply simm13 (:apply ash 1 (:apply - 31 bitnum))) dest))
  ((:pred >= (:apply - 31 bitnum) 12)
   (sethi (:apply ash #x80000000 (:apply - (:apply + 10 bitnum))) mask)
   (or src mask dest)))

(define-sparc-vinsn set-constant-ppc-bit-to-0 (((dest :u32))
					       ((src :u32)
						(bitnum :u8const))
					       ((mask :u32)))
  ((:pred < (:apply - 31 bitnum) 12)
   (andn src (:apply simm13 (:apply ash 1 (:apply - 31 bitnum))) dest))
  ((:pred >= (:apply - 31 bitnum) 12)
   (sethi (:apply ash #x80000000 (:apply - (:apply + 10 bitnum))) mask)
   (andn src mask dest)))

  
; The bit number is boxed and wants to think of the least-significant bit as 0.
; Imagine that.
(define-sparc-vinsn extract-variable-non-insane-bit (((dest :u32))
						     ((src :imm)
						      (bit :imm))
						     ((temp :u32)))
  (srl bit #.(simm13 arch::fixnumshift) temp)
  (add temp #.(simm13 arch::fixnumshift)  temp)
  (srl src temp dest)
  (and dest #.(simm13 1) dest))



; Operations on lists and cons cells

(define-sparc-vinsn %cdr (((dest :lisp))
                    ((src :lisp)))
  (ld src #.(simm13 arch::cons.cdr) dest))

(define-sparc-vinsn %car (((dest :lisp))
                    ((src :lisp)))
  (ld src #.(simm13 arch::cons.car) dest))


(define-sparc-vinsn %set-car (()
                        ((cell :lisp)
                         (new :lisp)))
  (st new cell #.(simm13 arch::cons.car)))

(define-sparc-vinsn %set-car& (()
                         ((cell :lisp)
                          (new :lisp)))
  (st cell sparc::%memo #.(simm13 -4))
  (add sparc::%memo #.(simm13 -4) sparc::%memo)
  (st new cell #.(simm13 arch::cons.car)))

(define-sparc-vinsn %set-cdr (()
                        ((cell :lisp)
                         (new :lisp)))
  (st new cell #.(simm13 arch::cons.cdr)))

(define-sparc-vinsn %set-cdr& (()
			       ((cell :lisp)
				(new :lisp)))
  (st cell sparc::%memo #.(simm13 -4))
  (add sparc::%memo #.(simm13 -4) sparc::%memo)
  (st new cell #.(simm13 arch::cons.cdr)))

(define-sparc-vinsn set-nargs (()
                         ((n :s16const)))
  (mov (:apply simm13 (:apply ash n ppc::word-shift)) sparc::%nargs))

(define-sparc-vinsn scale-nargs (()
                           ((nfixed :s16const)))
  ((:pred > nfixed 0)
   (add sparc::%nargs (:apply simm13 (:apply - (:apply ash nfixed ppc::word-shift))) sparc::%nargs)))

(define-sparc-vinsn (vpush-register :push :node :vsp)
              (()
               ((reg :lisp)))
  (st reg sparc::%vsp #.(simm13 -4))
  (add sparc::%vsp #.(simm13 -4) sparc::%vsp))

(define-sparc-vinsn (vpush-register-arg :push :node :vsp :outgoing-argument)
              (()
               ((reg :lisp)))
  (st reg sparc::%vsp #.(simm13 -4))
  (add sparc::%vsp #.(simm13 -4) sparc::%vsp))

(define-sparc-vinsn (vpop-register :pop :node :vsp) (((dest :lisp))
						     ())
  (ld sparc::%vsp 0 dest)
  (add sparc::%vsp #.(simm13 4) sparc::%vsp))

;; Caller has ensured that source & dest differ.
;; Worse things could happen ...
(define-sparc-vinsn copy-node-gpr (((dest :lisp))
				   ((src :lisp)))
  (mov src dest))

(define-sparc-vinsn copy-gpr (((dest t))
			      ((src t)))
  (mov src dest))

;; This copies an FPR pair.  Wonder if that's always what we want ...
(define-sparc-vinsn copy-fpr (((dest t))
			      ((src t)))
  (fmovs src dest)
  (fmovs (:apply 1+ src) (:apply 1+ dest)))

(define-sparc-vinsn vcell-ref (((dest :lisp))
			       ((vcell :lisp)))
  (ld vcell #.(simm13 arch::misc-data-offset) dest))

(define-sparc-vinsn vcell-set (()
                         ((vcell :lisp)
                          (value :lisp)))
  (st value vcell #.(simm13 arch::misc-data-offset)))

(define-sparc-vinsn vcell-set& (()
				((vcell :lisp)
				 (value :lisp)))
  (st vcell sparc::%memo #.(simm13 -4))
  (add sparc::%memo #.(simm13 -4) sparc::%memo)
  (st value vcell #.(simm13 arch::misc-data-offset)))


(define-sparc-vinsn make-vcell (((dest :lisp))
				((closed :lisp))
				((header :u32)))
  (st 0 sparc::%freeptr #.(simm13 arch::value-cell.size))
  (add sparc::%freeptr #.(simm13 (+ arch::value-cell.size arch::fulltag-misc)) sparc::%freeptr)
  (mov #.(simm13 arch::value-cell-header) header)
  (st header sparc::%freeptr #.(simm13 (- (+ arch::value-cell.size arch::fulltag-misc))))
  (st closed sparc::%freeptr #.(simm13 (+ 4 (- (+ arch::value-cell.size arch::fulltag-misc)))))
  (sub sparc::%freeptr #.(simm13 arch::value-cell.size) dest)
  (andn sparc::%freeptr #.(simm13 arch::fulltagmask) sparc::%freeptr))

(define-sparc-vinsn make-tsp-vcell (((dest :lisp))
				    ((closed :lisp))
				    ((header :u32)))
  (or sparc::%freeptr #.(simm13 arch::fulltag-nil) sparc::%freeptr) ;; no interrupts
  (st sparc::%tsp sparc::%tsp #.(simm13 -16))
  (add sparc::%tsp #.(simm13 -16) sparc::%tsp)
  (st 0 sparc::%tsp #.(simm13 4))
  (mov #.(simm13 arch::value-cell-header) header)
  (st header sparc::%tsp #.(simm13 8))
  (st closed sparc::%tsp #.(simm13 12))
  (add sparc::%tsp #.(simm13 (+ 8 arch::fulltag-misc)) dest)
  (andn sparc::%freeptr #.(simm13 arch::fulltagmask) sparc::%freeptr))

(define-sparc-vinsn make-tsp-cons (((dest :lisp))
				   ((car :lisp)
				    (cdr :lisp))
				   ())
  (or sparc::%freeptr #.(simm13 arch::fulltag-nil) sparc::%freeptr) ;; no interrupts
  (st sparc::%tsp sparc::%tsp #.(simm13 -16))
  (add sparc::%tsp #.(simm13 -16) sparc::%tsp)
  (st 0 sparc::%tsp #.(simm13 4))
  (st car sparc::%tsp #.(simm13 (+ 8 (+ arch::fulltag-cons arch::cons.car))))
  (st cdr sparc::%tsp #.(simm13 (+ 8 (+ arch::fulltag-cons arch::cons.cdr))))
  (add sparc::%tsp #.(simm13 (+ 8 arch::fulltag-cons)) dest)
  (andn sparc::%freeptr #.(simm13 arch::fulltagmask) sparc::%freeptr))

(define-sparc-vinsn %closure-code% (((dest :lisp))
                             ())
  (ld sparc::%rnil #.(simm13 (+ arch::symbol.vcell (arch::nrs-offset %closure-code%))) dest))



(define-sparc-vinsn (call-subprim :call :subprim-call) (()
                                                  ((spno :s16const)))
  (call-subprim spno)
    (nop))

;; Call subprim, save return address in %ra1
(define-sparc-vinsn (call-subprim* :call :subprim-call) (()
                                                  ((spno :s16const)))
  (call-subprim* spno)
    (nop))


(define-sparc-vinsn (jump-subprim :jumpLR) (()
                            ((spno :s16const)))
  (jump-subprim spno)
    (nop))

; Same as "call-subprim", but gives us a place to 
; track args, results, etc.
(define-sparc-vinsn (call-subprim-0 :call :subprim-call) (((dest t))
                                                     ((spno :s16const)))
  
  (call-subprim spno)
    (nop))

(define-sparc-vinsn (call-subprim-1 :call :subprim-call) (((dest t))
                                                    ((spno :s16const)
                                                     (z t)))
  (call-subprim spno)
    (nop))
  
(define-sparc-vinsn (call-subprim-2 :call :subprim-call) (((dest t))
							  ((spno :s16const)
							   (y t)
							   (z t)))
  (call-subprim spno)
  (nop))

(define-sparc-vinsn (call-subprim-3 :call :subprim-call) (((dest t))
							  ((spno :s16const)
							   (x t)
							   (y t)
							   (z t)))
  (call-subprim spno)
  (nop))

(define-sparc-vinsn event-poll (()
                          ())
  (ld sparc::%rnil #.(simm13 (+ arch::symbol.vcell (arch::nrs-offset *interrupt-level*))) sparc::%nargs)
  (tst sparc::%nargs)
  (tg #.(simm13 sparc::trap-event-poll)))

; Unconditional (pc-relative) branch
(define-sparc-vinsn (jump :jump) (()
				  ((label :label)))
  (b label)
    (nop))

(define-sparc-vinsn (call-label :jump) (()
					((label :label)))
  (call label)
    (nop))

; just like JUMP, only (implicitly) asserts that the following 
; code is somehow reachable.
(define-sparc-vinsn (non-barrier-jump :xref) (()
					      ((label :label)))
  (bn.a label))


(define-sparc-vinsn (build-catch-frame :call :subprim-call :xref) (()
                                                                   ((spno :u16const)
                                                                    (label :label)))
  (call-subprim spno)
  (bn label))

    
; Sometimes, the condition is known or easy to express ...

(define-sparc-vinsn (bne :branch) (()
				   ((label :label)))
  (bne  label)
    (nop))

; Most of the time, it isn't.

(define-sparc-vinsn (branch-true :branch) (()
					   ((cond :u8const)
					    (label :label)))
  (bicc cond label)
     (nop))

(define-sparc-vinsn (branch-false :branch) (()
					    ((cond :u8const)
					     (label :label)))
  (bicc (:apply logxor 8 cond) label)
    (nop))

(define-sparc-vinsn double-float-compare->boolean (((dest :lisp))
						   ((fcond :u8const)
						    (x :double-float)
						    (y :double-float)))
  (fcmped x y)
  (fmovs 0 0)				; for FP exception
  (mov sparc::%rnil dest)
  (fbfcc.a fcond :true)
    (add sparc::%rnil arch::t-offset dest)
  :true)

(define-sparc-vinsn single-float-compare->boolean (((dest :lisp))
						   ((fcond :u8const)
						    (x :single-float)
						    (y :single-float)))
  (fcmpes x y)
  (fmovs 0 0)				; for FP exception
  (fbfcc.a (:apply logand 15 fcond) :true)
    (add sparc::%rnil #.(simm13 arch::t-offset) dest)
  (mov sparc::%rnil dest)
  :true)

(define-sparc-vinsn double-float-compare (()
					    ((x :double-float)
					     (y :double-float)))
  (fcmped x y)
  (fmovs 0 0))



(define-sparc-vinsn single-float-compare (()
					   ((x :single-float)
					    (y :single-float)))
  (fcmpes x y)
  (fmovs 0 0))

(define-sparc-vinsn (float-branch-true :branch) (()
							((fcond :u8const)
							 (label :label)))
  (fbfcc (:apply logand 15 fcond) label)
     (or 0 0 0))

(define-sparc-vinsn (float-branch-false :branch) (()
							((fcond :u8const)
							 (label :label)))
  (fbfcc (:apply logxor 8 (:apply logand 15 fcond)) label)
     (or 0 0 0))




; We don't have much control over section alignment, so we
; can't depend on bits that may differ between T and NIL.
(define-sparc-vinsn invert-boolean (((dest :lisp))
				    ((src :lisp))
				    ((diff :u32)
				     (temp :u32)))
  (mov #.(simm13 arch::t-offset) temp)
  (sub src sparc::%rnil diff)
  (sub temp diff diff)
  (add diff sparc::%rnil dest))
                              
(define-sparc-vinsn lisp-word-ref (((dest :lisp))
				   ((base :imm)
				    (offset :imm)))
  (ld base offset dest))

(define-sparc-vinsn lisp-word-ref-c (((dest :lisp))
				     ((base :imm)
				      (offset :s16const)))
  (ld base (:apply simm13 offset) dest))


(define-sparc-vinsn load-s13 (((dest :imm))
			      ((val :s16const)))
  (mov (:apply simm13 val) dest))


;; Load an unsigned, 32-bit constant into a destination register.
;; It's not worth checking for the case where this could be a simple
;; MOV; a caller should have done that for us.
(define-sparc-vinsn lwi (((dest :imm))
			 ((intval :u32const))
			 ())
  (sethi (:apply ash intval -10) dest)
  ((:not (:pred = (:apply logand (1- (ash 1 10)) intval) 0))
   (or dest (:apply simm13 (:apply logand (1- (ash 1 10)) intval)) dest))
)



(define-sparc-vinsn discard-temp-frame (()
					())
  (ld sparc::%tsp 0 sparc::%tsp))





(define-sparc-vinsn load-nil (((dest :lisp))
			      ())
  (mov sparc::%rnil dest))

(define-sparc-vinsn load-t (((dest :lisp))
			    ())
  (add sparc::%rnil #.(simm13 arch::t-offset) dest))

(define-sparc-vinsn ref-constant (((dest :lisp))
				  ((src :s16const)))
  (ld sparc::%fn (:apply simm13 (:apply + arch::misc-data-offset (:apply ash (:apply 1+ src) 2))) dest))

(define-sparc-vinsn ref-indexed-constant (((dest :lisp))
					  ((idxreg :s32)))
  (ld sparc::%fn idxreg dest))

(define-sparc-vinsn cons (((dest :lisp))
			  ((newcar :lisp)
			   (newcdr :lisp)))
  (st 0 sparc::%freeptr #.(simm13 arch::cons.size))
  (add sparc::%freeptr #.(simm13 (+ arch::cons.size arch::fulltag-cons)) sparc::%freeptr)
  (st newcdr sparc::%freeptr #.(simm13 (- arch::cons.cdr arch::cons.size)))
  (st newcar sparc::%freeptr #.(simm13 (- arch::cons.car arch::cons.size)))
  (sub sparc::%freeptr #.(simm13 arch::cons.size) dest)
  (andn sparc::%freeptr #.(simm13 arch::fulltagmask) sparc::%freeptr))  

;; subtag had better be a PPC-NODE-SUBTAG of some sort!
(define-sparc-vinsn %ppc-gvector (((dest :lisp))
				  ((Rheader :u32) 
				   (nbytes :u32const))
				  ((immtemp0 :u32)                   
				   (nodetemp :lisp)
				   (initptr :u32)))
  (st 0 sparc::%freeptr (:apply simm13 (:apply logand (lognot 7) (:apply + (+ 7 4) nbytes))))
  (add sparc::%freeptr (:apply simm13 (:apply + arch::fulltag-misc (:apply logand (lognot 7) (:apply + (+ 7 4) nbytes)))) sparc::%freeptr)
  (sub sparc::%freeptr (:apply simm13 (:apply logand (lognot 7) (:apply + (+ 7 4) nbytes))) dest)
  (st rheader dest #.(simm13 arch::misc-header-offset))
  (andn sparc::%freeptr #.(simm13 arch::fulltagmask) sparc::%freeptr)
  
  ((:not (:pred = nbytes 0))
   (mov (:apply simm13  (:apply + nbytes arch::misc-data-offset)) initptr)
   (mov (:apply simm13 (:apply ash nbytes (- arch::word-shift))) immtemp0)
   :loop
   (subcc immtemp0 #.(simm13 1) immtemp0)
   (ld sparc::%vsp 0 nodetemp)
   (add sparc::%vsp #.(simm13 4) sparc::%vsp)
   (sub initptr #.(simm13 4) initptr)
   (bne :loop)
     (st nodetemp dest initptr)))


;; allocate a small (phys size <= 32K bytes) misc obj of known size/subtag
;; The 32K limit has to do with how many guard pages are beyond the freeptr;
;; we may have to use a temp reg to indicate the physical size if it won't
;; fit in a signed 13-bit constant
(define-sparc-vinsn %alloc-misc-fixed (((dest :lisp))
				       ((Rheader :u32)
					(nbytes :u32const))
				       ((size :u32)))
  ((:pred < (:apply + arch::fulltag-misc (:apply logand (lognot 7) (:apply + 11 nbytes))) (ash 1 12))
   (st 0 sparc::%freeptr (:apply simm13 (:apply logand (lognot 7) (:apply + (+ 7 4) nbytes))))
   (add sparc::%freeptr (:apply simm13 (:apply + arch::fulltag-misc  (:apply logand (lognot 7) (:apply + (+ 7 4) nbytes)))) sparc::%freeptr)
   (sub sparc::%freeptr (:apply simm13 (:apply logand (lognot 7) (:apply + (+ 7 4) nbytes))) dest)
   (st rheader dest #.(simm13 arch::misc-header-offset))
   (andn sparc::%freeptr #.(simm13 arch::fulltagmask) sparc::%freeptr))
  ((:pred >=
	  (:apply + arch::fulltag-misc
		  (:apply logand (lognot 7) (:apply + 11 nbytes)))
	  (ash 1 12))
   (sethi (:apply ash (:apply logand (lognot 7) (:apply + 11 nbytes)) -10) size)
   ((:pred /= 0 (:apply logand (1- (ash 1 10)) (:apply logand (lognot 7) (:apply + 11 nbytes))))
    (or size (:apply simm13 (:apply logand (1- (ash 1 10)) (:apply logand (lognot 7) (:apply + 11 nbytes)))) size))
   (st 0 sparc::%freeptr size)
   (add sparc::%freeptr size sparc::%freeptr)
   (or sparc::%freeptr #.(simm13 arch::fulltag-misc) sparc::%freeptr)
   (sub sparc::%freeptr size dest)
   (st rheader dest #.(simm13 arch::misc-header-offset))
   (andn sparc::%freeptr #.(simm13 arch::fulltagmask) sparc::%freeptr)))

(define-sparc-vinsn vstack-discard (()
				    ((nwords :u32const)))
  ((:not (:pred = nwords 0))
   (add sparc::%vsp (:apply simm13 (:apply ash nwords ppc::word-shift)) sparc::%vsp)))


(define-sparc-vinsn lcell-load (((dest :lisp))
				((cell :lcell)
				 (top :lcell)))
  (ld sparc::%vsp (:apply simm13
			  (:apply - 
				  (:apply - (:apply calc-lcell-depth top) 4)
				  (:apply calc-lcell-offset cell))) dest))

(define-sparc-vinsn vframe-load (((dest :lisp))
				 ((frame-offset :u16const)
				  (cur-vsp :u16const)))
  (ld sparc::%vsp (:apply simm13 (:apply - (:apply - cur-vsp 4) frame-offset)) dest))

(define-sparc-vinsn lcell-store (()
				 ((src :lisp)
				  (cell :lcell)
				  (top :lcell)))
  (st src sparc::%vsp (:apply simm13
			      (:apply - 
				      (:apply - (:apply calc-lcell-depth top) 4)
				      (:apply calc-lcell-offset cell)))))

(define-sparc-vinsn vframe-store (()
				  ((src :lisp)
				   (frame-offset :u16const)
				   (cur-vsp :u16const)))
  (st src sparc::%vsp (:apply simm13 (:apply - (:apply - cur-vsp 4) frame-offset))))

(define-sparc-vinsn load-vframe-address (((dest :imm))
					 ((offset :s16const)))
  (add sparc::%vsp (:apply simm13 offset) dest))

(define-sparc-vinsn copy-lexpr-argument (()
					 ()
					 ((temp :lisp)))
  (ld sparc::%vsp sparc::%nargs temp)
  (st temp sparc::%vsp #.(simm13 -4))
  (add sparc::%vsp #.(simm13 -4) sparc::%vsp))


; Boxing/unboxing of integers.

; Treat the low 8 bits of VAL as an unsigned integer; set RESULT to the equivalent fixnum.
(define-sparc-vinsn u8->fixnum (((result :imm)) 
                          ((val :u8)) 
                          ())
  (sll val #.(simm13 24) result)
  (srl result #.(simm13 (- 24 arch::fixnumshift)) result))

; Treat the low 8 bits of VAL as a signed integer; set RESULT to the equivalent fixnum.
(define-sparc-vinsn s8->fixnum (((result :imm)) 
				((val :s8)) 
				())
  (sll val #.(simm13 24) result)
  (sra result #.(simm13 (- 24 arch::fixnumshift)) result))


; Treat the low 16 bits of VAL as an unsigned integer; set RESULT to the equivalent fixnum.
(define-sparc-vinsn u16->fixnum (((result :imm)) 
				 ((val :u16)) 
				 ())
  (sll val #.(simm13 16) result)
  (srl result #.(simm13 (- 16 arch::fixnumshift)) result))


; Treat the low 16 bits of VAL as a signed integer; set RESULT to the equivalent fixnum.
(define-sparc-vinsn s16->fixnum (((result :imm)) 
				 ((val :s16)) 
				 ())
  (sll val #.(simm13 16)  result)
  (sra result #.(simm13 (- 16 arch::fixnumshift)) result))

(define-sparc-vinsn fixnum->s16 (((result :s16))
				 ((src :imm)))
  (sra src #.(simm13 arch::fixnumshift) result))

; A signed 32-bit untagged value can be at worst a 1-digit bignum.
; There should be something very much like this that takes a stack-consed
; bignum result ...

(define-sparc-vinsn s32->integer (((result :lisp))
				  ((src :s32))
				  ((temp :s32)))        
  (addcc src src temp)
  (bvc.a :lab1)
    (addcc temp temp result)
  :lab1
  (bvs.a :lab2)
    (uuo_box_signed result src)
  :lab2)

; An unsigned 32-bit untagged value can be either a 1 or a 2-digit bignum.
(define-sparc-vinsn u32->integer (((result :lisp))
				  ((src :u32))
				  ((temp :u32)))
  (sethi (ash #xe0000000 -10) temp)
  (andcc temp src temp)
  (sll src #.(simm13 arch::fixnumshift) result)
  (bne.a :done)
    (uuo_box_unsigned result src)
  :done)

(define-sparc-vinsn u16->u32 (((dest :u32))
			      ((src :u16)))
  (sll src #.(simm13 16) dest)
  (srl dest #.(simm13 16) dest))


(define-sparc-vinsn u8->u32 (((dest :u32))
			     ((src :u8)))
  (and src #.(simm13 #xff) dest))


(define-sparc-vinsn s16->s32 (((dest :s32))
                        ((src :s16)))
  (sll src #.(simm13 16) dest)
  (sra dest #.(simm13 16) dest))

(define-sparc-vinsn s8->s32 (((dest :s32))
			     ((src :s8)))
  (sll src #.(simm13 24) dest)
  (sra dest #.(simm13 24) dest))

; ... of floats ...

; Heap-cons a double-float to store contents of FPREG.  Hope that we don't do
; this blindly.
(define-sparc-vinsn double->heap (((result :lisp))    ; tagged as a double-float
                            ((fpreg :double-float)) 
                            ((header-temp :u32)))
  (mov #.(simm13 arch::double-float-header) header-temp)
  (st 0 sparc::%freeptr #.(simm13 arch::double-float.size))
  (add sparc::%freeptr #.(simm13 (+ arch::double-float.size arch::fulltag-misc)) sparc::%freeptr)
  (sub sparc::%freeptr #.(simm13 arch::double-float.size) result)
  (st header-temp result #.(simm13 arch::misc-header-offset))
  (andn sparc::%freeptr #.(simm13 arch::fulltagmask) sparc::%freeptr)
  (stf fpreg result #.(simm13 arch::double-float.value)))



; This is about as bad as heap-consing a double-float.  (In terms of verbosity).
; Wouldn't kill us to do either/both out-of-line, but need to make visible to
; compiler so unnecessary heap-consing can be elided.
(define-sparc-vinsn single->heap (((result :lisp))    ; tagged as a single-float
				  ((fpreg :single-float))
				  ((header-temp :u32)))
  (mov #.(simm13 arch::single-float-header) header-temp)
  (st 0 sparc::%freeptr #.(simm13 arch::single-float.size))
  (add sparc::%freeptr #.(simm13 (+ arch::single-float.size arch::fulltag-misc)) sparc::%freeptr)
  (sub sparc::%freeptr #.(simm13 arch::single-float.size) result)
  (st header-temp result #.(simm13 arch::misc-header-offset))
  (andn sparc::%freeptr #.(simm13 arch::fulltagmask) sparc::%freeptr)
  (stf fpreg result #.(simm13 arch::single-float.value)))



; "dest" is preallocated, presumably on a stack somewhere.
(define-sparc-vinsn store-double (()
				  ((dest :lisp)
				   (source :double-float))
				  ())
  (stdf source dest #.(simm13 arch::double-float.value)))


(define-sparc-vinsn get-double (((target :double-float))
				((source :lisp))
				())
  (lddf source #.(simm13 arch::double-float.value) target))

;;; Extract a double-float value, typechecking in the process.
;;; IWBNI we could simply call the "trap-unless-typecode=" vinsn here,
;;; instead of replicating it ..

(define-sparc-vinsn get-double? (((target :double-float))
				 ((source :lisp))
				 ((tag :u8)))
  (and source #.(simm13 arch::tagmask) tag)
  (cmp tag #.(simm13 arch::tag-misc))
  (be.a :do-trap)
  <
    (ldub source #.(simm13 arch::misc-subtag-offset) tag)
  :do-trap
  (cmp tag #.(simm13 arch::subtag-double-float))
  (tne #.(simm13 sparc::trap-typecode-check))
  >
  (lddf source #.(simm13 arch::double-float.value) target))
  

(define-sparc-vinsn store-single (()
				  ((dest :lisp)
				   (source :single-float))
				  ())
  (stf source dest #.(simm13 arch::single-float.value)))

(define-sparc-vinsn get-single (((target :single-float))
				((source :lisp))
				())
  (ldf source #.(simm13 arch::single-float.value) target))


; ... of characters ...
(define-sparc-vinsn charcode->u16 (((dest :u16))
				   ((src :imm))
				   ())
  (srl src #.(simm13 arch::charcode-shift) dest))

(define-sparc-vinsn character->fixnum (((dest :lisp))
				      ((src :lisp))
				      ((temp :u32)))
  (srl src #.(simm13 arch::charcode-shift) temp)
  (sll temp #.(simm13 arch::fixnumshift) dest))

(define-sparc-vinsn character->code (((dest :u32))
				     ((src :lisp)))
  (srl src #.(simm13 arch::charcode-shift) dest))


(define-sparc-vinsn fixnum->char (((dest :lisp))
				  ((src :imm))
				  ())
  (sll src #.(simm13 (- arch::charcode-shift arch::fixnumshift)) dest)
  (or dest #.(simm13 arch::subtag-character) dest))  

(define-sparc-vinsn u8->char (((dest :lisp))
			      ((src :u8))
			      ())
  (sll src #.(simm13 arch::charcode-shift) dest)
  (or dest #.(simm13 arch::subtag-character) dest))

;; ... Macptrs ...

(define-sparc-vinsn deref-macptr (((addr :address))
				  ((src :lisp))
				  ())
  (ld src #.(simm13 arch::macptr.address) addr))

(define-sparc-vinsn set-macptr-address (()
					((addr :address)
					 (src :lisp))
					())
  (st addr src #.(simm13 arch::macptr.address)))

(define-sparc-vinsn macptr->heap (((dest :lisp))
				  ((address :address))
				  ((header :u32)))
  (mov #.(simm13 arch::macptr-header) header)
  (st 0 sparc::%freeptr #.(simm13 arch::macptr.size))
  (add sparc::%freeptr #.(simm13 (+ arch::macptr.size arch::fulltag-misc)) sparc::%freeptr)
  (st header sparc::%freeptr #.(simm13 (- (+ arch::macptr.size arch::fulltag-misc))))
  (sub sparc::%freeptr #.(simm13 arch::macptr.size) dest)
  (andn sparc::%freeptr #.(simm13 arch::fulltagmask) sparc::%freeptr)
  (st address dest #.(simm13 arch::macptr.address)))


(define-sparc-vinsn macptr->stack (((dest :lisp))
				   ((address :address))
				   ((header :u32)))
  (mov #.(simm13 arch::macptr-header) header)
  (or sparc::%freeptr #.(simm13 arch::fulltag-nil) sparc::%freeptr)
  (st sparc::%tsp sparc::%tsp  #.(simm13 (- (+ 8 arch::macptr.size))))
  (add sparc::%tsp #.(simm13 (- (+ 8 arch::macptr.size))) sparc::%tsp)
  (st 0 sparc::%tsp #.(simm13 4))
  (st header sparc::%tsp #.(simm13 8))
  (add sparc::%tsp #.(simm13 (+ 8 arch::fulltag-misc)) dest)
  (andn sparc::%freeptr #.(simm13 arch::fulltagmask) sparc::%freeptr)  
  (st address dest #.(simm13 arch::macptr.address)))

  
(define-sparc-vinsn adjust-stack-register (()
                                     ((reg t)
                                      (amount :s16const)))
  (add reg (:apply simm13 amount) reg))


;; Arithmetic on fixnums & unboxed numbers

(define-sparc-vinsn u32-lognot (((dest :u32))
				((src :u32))
				())
  (xnor src 0 dest))

(define-sparc-vinsn fixnum-lognot (((dest :imm))
				   ((src :imm))
				   ((temp :u32)))
  (xnor src 0 temp)
  (andn temp #.(simm13 arch::fixnum-mask) dest))

(define-sparc-vinsn negate-fixnum (((dest :lisp))
				   ((src :imm))
				   ())
  (tsubcctv 0 src dest))

  
(define-sparc-vinsn negate-fixnum-no-ovf (((dest :lisp))
					  ((src :imm)))
  (tsubcc 0 src dest))

; Constant is a (SIGNED-BYTE 13), or the unsigned equivalent thereof
(define-sparc-vinsn logior-low (((dest :imm))
				((src :imm)
				 (low :s16const)))
  (or src (:apply simm13 low) dest))

                           
                           
(define-sparc-vinsn %logior2 (((dest :imm))
                        ((x :imm)
                         (y :imm))
                        ())
  (or x y dest))


(define-sparc-vinsn logand-low (((dest :imm))
                           ((src :imm)
                            (low :s16const))
                           )
  (and src (:apply simm13 low) dest))


(define-sparc-vinsn %logand2 (((dest :imm))
                        ((x :imm)
                         (y :imm))
                        ())
  (and x y dest))


(define-sparc-vinsn logxor-low (((dest :imm))
                           ((src :imm)
                            (low :s16const)))
  (xor src (:apply simm13 low) dest))

                           

(define-sparc-vinsn %logxor2 (((dest :imm))
                        ((x :imm)
                         (y :imm))
                        ())
  (xor x y dest))

(define-sparc-vinsn %ilsl (((dest :imm))
			   ((count :imm)
			    (src :imm))
			   ((temp :u32)))
  (cmp count #.(simm13 (ash 31 arch::fixnumshift)))
  (srl count #.(simm13 arch::fixnumshift) temp)
  (sll src temp dest)
  (bg.a :foo)
    (mov 0 dest)
  :foo)


(define-sparc-vinsn %ilsl-c (((dest :imm))
			     ((count :u8const)
			      (src :imm)))
  (sll src (:apply simm13 count) dest))

(define-sparc-vinsn %ilsr-c (((dest :imm))
			     ((count :u8const)
			      (src :imm))
			     ((temp :u32)))
  (srl src (:apply simm13 count) temp)
  (andn temp #.(simm13 arch::fixnum-mask) dest))

; 68k did the right thing for counts < 64 - fixnumshift but not if greater
; so load-byte fails in 3.0 also

(define-sparc-vinsn %iasr (((dest :imm))
			   ((count :imm)
			    (src :imm))
			   ((temp :s32)))
  (cmp count #.(simm13 (ash 31 arch::fixnumshift)))
  (sra count #.(simm13 arch::fixnumshift) temp)
  (sra src temp temp)
  (bg.a :foo)
    (sra src #.(simm13 31) temp)
  :foo
  (andn temp #.(simm13 arch::fixnum-mask) dest))

(define-sparc-vinsn %iasr-c (((dest :imm))
			     ((count :u8const)
			      (src :imm))
			     ((temp :s32)))
  (sra src (:apply simm13 count) temp)
  (andn temp #.(simm13 arch::fixnum-mask) dest))


(define-sparc-vinsn %ilsr (((dest :imm))
			   ((count :imm)
			    (src :imm))
			   ((temp :s32)))
  (cmp count #.(simm13 (ash 31 arch::fixnumshift)))
  (srl count #.(simm13 arch::fixnumshift) temp)
  (srl src temp temp)
  (andn temp #.(simm13 arch::fixnum-mask) dest)
  (bg.a :foo)
    (mov 0 dest)
  :foo
  )


(define-sparc-vinsn sign-extend-halfword (((dest :imm))
					  ((src :imm)))
  (sll src #.(simm13 (- 16 arch::fixnumshift)) dest)
  (sra dest #.(simm13 (- 16 arch::fixnumshift)) dest))


(define-sparc-vinsn s32-highword (((dest :imm))
				  ((src :s32))
				  ((temp :s32)))
  (sra src #.(simm13 16) temp)
  (sll temp #.(simm13 arch::fixnumshift) dest))

(define-sparc-vinsn fixnum-add (((dest t))
				((x t)
				 (y t)))
  (taddcc x y dest))

(define-sparc-vinsn fixnum-add-constant (((dest t))
					 ((x t)
					  (y :s16const)))
  (taddcc x (:apply simm13 y) dest))

(define-sparc-vinsn fixnum-add-overflow (((dest :imm))
					 ((x :imm)
					  (y :imm))
					 ())
  (taddcctv x y dest))

(define-sparc-vinsn fixnum-add-constant-overflow (((dest t))
						  ((x t)
						   (y :s16const)))
  (taddcctv x (:apply simm13 y) dest))

;  (setq dest (- x y))
(define-sparc-vinsn fixnum-sub (((dest t))
				((x t)
				 (y t)))
  (tsubcc x y dest))

(define-sparc-vinsn fixnum-sub-constant (((dest t))
				   ((x t)
				    (y :s16const)))
  (tsubcc x (:apply simm13 y) dest))

(define-sparc-vinsn fixnum-sub-overflow (((dest :imm))
					 ((x :imm)
					  (y :imm))
					 ())
  (tsubcctv x y dest))

(define-sparc-vinsn fixnum-sub-constant-overflow (((dest :imm))
						  ((x :imm)
						   (y :s16const))
						  ())
  (tsubcctv x (:apply simm13 y) dest))



;This must unbox one reg, but hard to tell which is better.
;(The one with the smaller absolute value might be)
(define-sparc-vinsn multiply-fixnums (((dest :imm))
				      ((a :imm)
				       (b :imm))
				      ((unboxed :s32)))
  (sra b #.(simm13 arch::fixnumshift) unboxed)
  (smul a unboxed dest))

(define-sparc-vinsn multiply-immediate (((dest :imm))
					((boxed :imm)
					 (const :s16const))
					)
  (smul boxed (:apply simm13 const) dest))

; Mask out the code field of a base character; the result
; should be EXACTLY = to subtag-base-char
(define-sparc-vinsn mask-base-char (((dest :u32))
				    ((src :imm)))
  (sll src #.(simm13 arch::charcode-shift) dest)
  (srl dest #.(simm13 arch::charcode-shift) dest))

;; Boundp, fboundp stuff.
(define-sparc-vinsn symbol-value (((val :lisp))
				  ((sym (:lisp (:ne val)))))
  <
  (ld sym #.(simm13 arch::symbol.vcell) val)
  (cmp val #.(simm13 arch::unbound-marker))
  (te #.(simm13 sparc::trap-unbound-variable))
  >)

(define-sparc-vinsn symbol-function (((val :lisp))
				     ((sym (:lisp (:ne val))))
				     ((tag :u32)))
  :again
  (ld sym #.(simm13 arch::symbol.fcell) val)
  (and val #.(simm13 arch::tagmask) tag)
  (cmp tag #.(simm13 arch::tag-misc))
  (be.a :check)
    (ldub val #.(simm13 arch::misc-subtag-offset) tag)
  :check
  (cmp tag #.(simm13 arch::subtag-function))
  (bne.a :again)
    (uuo_interr arch::error-udf sym))

;; Probably cheaper to push/pop unboxed stuff on the C stack

(define-sparc-vinsn (temp-push-unboxed-word :push :word :tsp)
              (()
               ((w :u32)))
  (st sparc::%tsp sparc::%tsp #.(simm13 -16))
  (add sparc::%tsp #.(simm13 -16) sparc::%tsp)
  (st sparc::%tsp sparc::%tsp #.(simm13 4))
  (st w sparc::%tsp #.(simm13 8)))

(define-sparc-vinsn (temp-pop-unboxed-word :pop :word :tsp)
    (((w :u32))
     ())
  (ld sparc::%tsp #.(simm13 8) w)
  (ld sparc::%tsp 0 sparc::%tsp))


(define-sparc-vinsn (temp-push-double-float :push :doubleword :tsp)
              (((d :double-float))
               ())
  (st sparc::%tsp sparc::%tsp #.(simm13 -16))
  (add sparc::%tsp #.(simm13 -16) sparc::%tsp)
  (st sparc::%tsp sparc::%tsp #.(simm13 4))
  (stdf d sparc::%tsp #.(simm13 8)))


(define-sparc-vinsn (temp-pop-double-float :pop :doubleword :tsp)
              (()
               ((d :double-float)))
  (lddf sparc::%tsp #.(simm13 8) d)
  (ld sparc::%tsp 0 sparc::%tsp))


(define-sparc-vinsn (temp-push-single-float :push :word :tsp)
    (((s :single-float))
     ())
  (st sparc::%tsp sparc::%tsp #.(simm13 -16))
  (add sparc::%tsp #.(simm13 -16) sparc::%tsp)
  (st sparc::%tsp sparc::%tsp #.(simm13 4))
  (stf s sparc::%tsp #.(simm13 8)))

(define-sparc-vinsn (temp-pop-single-float :pop :word :tsp)
              (()
               ((s :single-float)))
  (ldf sparc::%tsp #.(simm13 8) s)
  (ld sparc::%tsp 0 sparc::%tsp))

;; The PPC convention was that SAVE0 was always stored at the lowest
;; address and stored last.  Follow part of that convention here (so
;; "first" refers to the NVR furthest from SAVE0), but always save
;; SAVE0 first (at the lowest address.)  Since there's no STM, we
;; have to do sort of a big CASE statement here.
;; Note that this decremements VSP before storing anything; that's
;; the opposite of what "vpush" does.
(define-sparc-vinsn (save-nvrs :push :node :vsp :multiple)
              (()
               ((first :u8const)))
  (sub sparc::%vsp (:apply simm13 (:apply * 4 (:apply 1+ (:apply - first sparc::%save0)))) sparc::%vsp)
  ;;; THIS might trap on a write to a guard page.
  (st sparc::%save0 sparc::%vsp 0)  ; note the implict use of %g0
  ((:pred > first sparc::%save0)
   (st sparc::%save1 sparc::%vsp #.(simm13 4))
   ((:pred > first sparc::%save1)
    (st sparc::%save2 sparc::%vsp #.(simm13 8))
    ((:pred > first sparc::%save2)
     (st sparc::%save3 sparc::%vsp #.(simm13 12))
     ((:pred > first sparc::%save3)
      (st sparc::%save4 sparc::%vsp #.(simm13 16))
      ((:pred > first sparc::%save4)
       (st sparc::%save5 sparc::%vsp #.(simm13 20))))))))


(define-sparc-vinsn (restore-nvrs :pop :node :vsp :multiple)
    (()
     ((firstreg :u8const)
      (basereg :imm)
      (offset :s16const)))
  (ld basereg (:apply simm13 offset) sparc::%save0)
  ((:pred >= firstreg sparc::%save1)
   (ld basereg (:apply simm13 (:apply + 4 offset)) sparc::%save1)   
   ((:pred >= firstreg sparc::%save2)
    (ld basereg (:apply simm13 (:apply + 8 offset)) sparc::%save2)   
    ((:pred >= firstreg sparc::%save3)
     (ld basereg (:apply simm13 (:apply + 12 offset)) sparc::%save3)   
     ((:pred >= firstreg sparc::%save4)
      (ld basereg (:apply simm13 (:apply + 16 offset)) sparc::%save4)   
      ((:pred = firstreg sparc::%save5)
       (ld basereg (:apply simm13 (:apply + 20 offset)) sparc::%save5)))))))

(define-sparc-vinsn (vpush2 :push :node :vsp :multiple) (()
							 ((rlast :lisp)
							  (r0 :lisp)))
  (st rlast sparc::%vsp #.(simm13 -8))
  (add sparc::%vsp #.(simm13 -8) sparc::%vsp)
  (st r0 sparc::%vsp #.(simm13 4)))

(define-sparc-vinsn (vpush3 :push :node :vsp :multiple) (()
							 ((rlast :lisp)
							  (r1 :lisp)
							  (r0 :lisp)))
  (st rlast sparc::%vsp #.(simm13 -12))
  (add sparc::%vsp #.(simm13 -12) sparc::%vsp)
  (st r1 sparc::%vsp #.(simm13 4))
  (st r0 sparc::%vsp #.(simm13 8)))

(define-sparc-vinsn (vpush4 :push :node :vsp :multiple) (()
							 ((rlast :lisp)
							  (r2 :lisp)
							  (r1 :lisp)
							  (r0 :lisp)))
  (st rlast sparc::%vsp #.(simm13 -16))
  (add sparc::%vsp #.(simm13 -16) sparc::%vsp)
  (st r2 sparc::%vsp #.(simm13 4))
  (st r1 sparc::%vsp #.(simm13 8))
  (st r0 sparc::%vsp #.(simm13 12)))

(define-sparc-vinsn (vpush5 :push :node :vsp :multiple) (()
							 ((rlast :lisp)
							  (r3 :lisp)
							  (r2 :lisp)
							  (r1 :lisp)
							  (r0 :lisp)))
  (st rlast sparc::%vsp #.(simm13 -20))
  (add sparc::%vsp #.(simm13 -20) sparc::%vsp)
  (st r3 sparc::%vsp #.(simm13 4))
  (st r2 sparc::%vsp #.(simm13 8))
  (st r1 sparc::%vsp #.(simm13 12))
  (st r0 sparc::%vsp #.(simm13 16)))

(define-sparc-vinsn (vpush6 :push :node :vsp :multiple) (()
							 ((rlast :lisp)
							  (r4 :lisp)
							  (r3 :lisp)
							  (r2 :lisp)
							  (r1 :lisp)
							  (r0 :lisp)))
  (st rlast sparc::%vsp #.(simm13 -24))
  (add sparc::%vsp #.(simm13 -24) sparc::%vsp)
  (st r4 sparc::%vsp #.(simm13 4))
  (st r3 sparc::%vsp #.(simm13 8))
  (st r2 sparc::%vsp #.(simm13 12))
  (st r1 sparc::%vsp #.(simm13 16))
  (st r0 sparc::%vsp #.(simm13 20)))

(define-sparc-vinsn (vpush7 :push :node :vsp :multiple) (()
							 ((rlast :lisp)
							  (r5 :lisp)
							  (r4 :lisp)
							  (r3 :lisp)
							  (r2 :lisp)
							  (r1 :lisp)
							  (r0 :lisp)))
  (st rlast sparc::%vsp #.(simm13 -28))
  (add sparc::%vsp #.(simm13 -28) sparc::%vsp)
  (st r5 sparc::%vsp #.(simm13 4))
  (st r4 sparc::%vsp #.(simm13 8))
  (st r3 sparc::%vsp #.(simm13 12))
  (st r2 sparc::%vsp #.(simm13 16))
  (st r1 sparc::%vsp #.(simm13 20))
  (st r0 sparc::%vsp #.(simm13 24)))

(define-sparc-vinsn (vpush8 :push :node :vsp :multiple) (()
							 ((rlast :lisp)
							  (r6 :lisp)
							  (r5 :lisp)
							  (r4 :lisp)
							  (r3 :lisp)
							  (r2 :lisp)
							  (r1 :lisp)
							  (r0 :lisp)))
  (st rlast sparc::%vsp #.(simm13 -32))
  (add sparc::%vsp #.(simm13 -32) sparc::%vsp)
  (st r6 sparc::%vsp #.(simm13 4))
  (st r5 sparc::%vsp #.(simm13 8))
  (st r4 sparc::%vsp #.(simm13 12))
  (st r3 sparc::%vsp #.(simm13 16))
  (st r2 sparc::%vsp #.(simm13 20))
  (st r1 sparc::%vsp #.(simm13 24))
  (st r0 sparc::%vsp #.(simm13 28)))

(define-sparc-vinsn (vpush9 :push :node :vsp :multiple) (()
							 ((rlast :lisp)
							  (r7 :lisp)
							  (r6 :lisp)
							  (r5 :lisp)
							  (r4 :lisp)
							  (r3 :lisp)
							  (r2 :lisp)
							  (r1 :lisp)
							  (r0 :lisp)))
  (st rlast sparc::%vsp #.(simm13 -36))
  (add sparc::%vsp #.(simm13 -36) sparc::%vsp)
  (st r7 sparc::%vsp #.(simm13 4))
  (st r6 sparc::%vsp #.(simm13 8))
  (st r5 sparc::%vsp #.(simm13 12))
  (st r4 sparc::%vsp #.(simm13 16))
  (st r3 sparc::%vsp #.(simm13 20))
  (st r2 sparc::%vsp #.(simm13 24))
  (st r1 sparc::%vsp #.(simm13 28))
  (st r0 sparc::%vsp #.(simm13 32)))


(define-sparc-vinsn (vpop2 :pop :node :vsp :multiple) (((rlast :lisp)
							(r0 :lisp))
						       ())
  (ld sparc::%vsp #.(simm13 0) rlast)
  (ld sparc::%vsp #.(simm13 4) r0)
  (add sparc::%vsp #.(simm13 8) sparc::%vsp))
    
(define-sparc-vinsn (vpop3 :pop :node :vsp :multiple) (((rlast :lisp)
						  (r1 :lisp)
						  (r0 :lisp))
						 ())
  (ld sparc::%vsp #.(simm13 0) rlast)
  (ld sparc::%vsp #.(simm13 4) r1)
  (ld sparc::%vsp #.(simm13 8) r0)  
  (add sparc::%vsp #.(simm13 12) sparc::%vsp))


(define-sparc-vinsn (vpop4 :pop :node :vsp :multiple) (((rlast :lisp)
						  (r2 :lisp)
						  (r1 :lisp)
						  (r0 :lisp))
						 ())
  (ld sparc::%vsp #.(simm13 0) rlast)
  (ld sparc::%vsp #.(simm13 4) r2)
  (ld sparc::%vsp #.(simm13 8) r1)  
  (ld sparc::%vsp #.(simm13 12) r0)  
  (add sparc::%vsp #.(simm13 16) sparc::%vsp))




(define-sparc-vinsn (dpayback :call :subprim-call) (()
                                              ((n :s16const))
                                              ((temp (:u32 #.sparc::%imm0))))
  ((:pred > n 1)
  (call-subprim .SPunbind-n)
   (mov (:apply simm13 n) temp))
  ((:pred = n 1)
   (call-subprim .SPunbind)   
   (nop)))

(define-sparc-vinsn zero-double-float-register 
              (((dest :double-float))
               ())
  (fmovs sparc::%fp-zero dest)
  (fmovs (1+ sparc::%fp-zero) (:apply 1+ dest)))


(define-sparc-vinsn zero-single-float-register 
              (((dest :single-float))
               ())
  (fmovs sparc::%fp-zero dest))

(define-sparc-vinsn load-double-float-constant
              (((dest :double-float))
               ((high t)
                (low t)))
  (add sparc::%sp #.(simm13 -8) sparc::%sp)
  (st high sparc::%sp #.(simm13 64))
  (st low sparc::%sp #.(simm13 68))
  (lddf sparc::%sp #.(simm13 64) dest)
  (add sparc::%sp #.(simm13 8) sparc::%sp))


(define-sparc-vinsn load-single-float-constant (((dest :single-float))
						((src t)))
  (add sparc::%sp #.(simm13 -8) sparc::%sp)
  (st src sparc::%sp #.(simm13 64))
  (ldf sparc::%sp #.(simm13 64) dest)
  (add sparc::%sp #.(simm13 8) sparc::%sp))

(define-sparc-vinsn load-indexed-node (((node :lisp))
				       ((base :lisp)
					(offset :s16const)))
  (ld base (:apply simm13 offset) node))

(define-sparc-vinsn recover-saved-vsp (((dest :imm))
				       ())
  (ld sparc::%lsp #.(simm13 sparc::lisp-frame.savevsp) dest))

(define-sparc-vinsn check-exact-nargs (()
                                 ((n :u16const)))
  (cmp sparc::%nargs (:apply simm13 (:apply ash n 2)))
  (tne #.(simm13 sparc::trap-nargs)))


(define-sparc-vinsn check-min-nargs (()
                               ((min :u16const)))
  (cmp sparc::%nargs (:apply simm13 (:apply ash min 2)))
  (tlu #.(simm13 sparc::trap-nargs)))


(define-sparc-vinsn check-max-nargs (()
				     ((max :u16const)))
  (cmp sparc::%nargs (:apply simm13 (:apply ash max 2)))
  (tgu #.(simm13 sparc::trap-nargs)))

; Save context and establish FN.  The current VSP is the the
; same as the caller's, e.g., no arguments were vpushed.
(define-sparc-vinsn save-lisp-context-vsp (()
                                     ()
                                     ((imm :u32)))
  (add sparc::%sp #.(simm13 (- sparc::lisp-frame.size)) sparc::%sp)
  (st sparc::%lsp sparc::%sp #.(simm13 64))
  (add sparc::%sp #.(simm13 64) sparc::%lsp)
  (st sparc::%fn sparc::%lsp #.(simm13 sparc::lisp-frame.savefn))
  (st sparc::%ra0 sparc::%lsp #.(simm13 sparc::lisp-frame.savelr))
  (st sparc::%vsp sparc::%lsp #.(simm13 sparc::lisp-frame.savevsp))
  (mov sparc::%nfn sparc::%fn)
  ; Do a stack-probe ...
  (ld sparc::%rnil #.(simm13 (arch::kernel-global cs-overflow-limit)) imm)
  (cmp sparc::%sp imm)
  (tcs #.(simm13 sparc::trap-sp-overflow)))

; Do the same thing via a subprim call.
(define-sparc-vinsn (save-lisp-context-vsp-ool :call :subprim-call) (()
                                                               ()
                                                               ((imm (:u32 #.sparc::%imm0))))
  (call-subprim* .SPsavecontextvsp)
     (nop))

(define-sparc-vinsn save-lisp-context-offset (()
                                        ((nbytes-vpushed :u16const))
                                        ((imm :u32)))
  (add sparc::%vsp (:apply simm13 nbytes-vpushed) imm)
  (add sparc::%sp #.(simm13 (- sparc::lisp-frame.size)) sparc::%sp)
  (st sparc::%lsp sparc::%sp #.(simm13 64))
  (add sparc::%sp #.(simm13 64) sparc::%lsp)
  (st sparc::%fn sparc::%lsp #.(simm13 sparc::lisp-frame.savefn))
  (st sparc::%ra0 sparc::%lsp #.(simm13 sparc::lisp-frame.savelr))
  (st imm sparc::%lsp #.(simm13 sparc::lisp-frame.savevsp))
  (mov sparc::%nfn sparc::%fn)
  ; Do a stack-probe ...
  (ld sparc::%rnil #.(simm13 (arch::kernel-global cs-overflow-limit)) imm)
  (cmp sparc::%sp imm)
  (tcs #.(simm13 sparc::trap-sp-overflow)))


(define-sparc-vinsn save-lisp-context-offset-ool (()
                                            ((nbytes-vpushed :u16const))
                                            ((imm (:u32 #.sparc::%imm0))))
  (call-subprim* .SPsavecontext0)
    (mov (:apply simm13 nbytes-vpushed) imm))


(define-sparc-vinsn save-lisp-context-lexpr (()
                                       ()
                                       ((imm :u32)))
  (add sparc::%sp #.(simm13 (- sparc::lisp-frame.size)) sparc::%sp)
  (st sparc::%lsp sparc::%sp #.(simm13 64))
  (add sparc::%sp #.(simm13 64) sparc::%lsp)
  (st 0 sparc::%lsp #.(simm13 sparc::lisp-frame.savefn))
  (st sparc::%ra0 sparc::%lsp #.(simm13 sparc::lisp-frame.savelr))
  (st sparc::%vsp  sparc::%lsp #.(simm13 sparc::lisp-frame.savevsp))
  (mov sparc::%nfn sparc::%fn)
  ; Do a stack-probe ...
  (ld sparc::%rnil #.(simm13 (arch::kernel-global cs-overflow-limit)) imm)
  (cmp sparc::%sp imm)
  (tcs #.(simm13 sparc::trap-sp-overflow)))
  
(define-sparc-vinsn save-cleanup-context (()
                                    ())
  ; SP was this deep just a second ago, so no need to do a stack-probe.
  (add sparc::%sp #.(simm13 (- sparc::lisp-frame.size)) sparc::%sp)
  (st sparc::%lsp sparc::%sp #.(simm13 64))
  (add sparc::%sp #.(simm13 64) sparc::%lsp)
  (st 0 sparc::%lsp #.(simm13 sparc::lisp-frame.savefn))
  (st sparc::%ra0 sparc::%lsp #.(simm13 sparc::lisp-frame.savelr))
  (st sparc::%vsp sparc::%lsp #.(simm13 sparc::lisp-frame.savevsp)))


;; Vpush the argument registers.  We got at least "min-fixed" args;
;; that knowledge may help us generate better code.

(define-sparc-vinsn (save-lexpr-argregs :call :subprim-call)
    (()
     ((min-fixed :u16const))
     ((entry-vsp (:u32 #.sparc::%imm0))
      (arg-temp :u32)))
  ((:pred >= min-fixed $numsparcargregs)
   (st sparc::%arg_x sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp)
   (st sparc::%arg_y sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp)
   (st sparc::%arg_z sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp))
  ((:pred = min-fixed 2)                ; at least 2 args
   (cmp sparc::%nargs #.(simm13 (ash 2 ppc::word-shift)))
   (be :yz2)                      ; skip arg_x if exactly 2
     (nop)
   (st sparc::%arg_x sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp)
   :yz2
   (st sparc::%arg_y sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp)
   (st sparc::%arg_z sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp))
  ((:pred = min-fixed 1)                ; at least one arg
   (cmp sparc::%nargs #.(simm13 (ash 2 ppc::word-shift)))
   (bl :z1)                       ; branch if exactly one
     (nop)
   (be :yz1)                      ; branch if exactly two
     (nop)
   (st sparc::%arg_x sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp)

   :yz1
   (st sparc::%arg_y sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp)   
   :z1
   (st sparc::%arg_z sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp))
  ((:pred = min-fixed 0)
   (cmp sparc::%nargs #.(simm13 (ash 2 ppc::word-shift)))
   (be :yz0)				; exactly two
     (tst sparc::%nargs)
   (be :none)				; exactly zero
     (cmp sparc::%nargs #.(simm13 (ash 2 ppc::word-shift)))
   (bl :z0)                        ; one
     (nop)
   ; Three or more ...
   (st sparc::%arg_x sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp)
   :yz0
   (st sparc::%arg_y sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp)
   :z0
   (st sparc::%arg_z sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp)
   :none
   )
  ((:pred = min-fixed 0)
   (st sparc::%nargs sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp))
  ((:not (:pred = min-fixed 0))
   (sub sparc::%nargs (:apply simm13 (:apply ash min-fixed ppc::word-shift))
	arg-temp)
   (st arg-temp sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp))
  (add sparc::%vsp sparc::%nargs entry-vsp)
  (call-subprim* .SPlexpr-entry)
    (add entry-vsp #.(simm13 4) entry-vsp))


(define-sparc-vinsn (jump-return-pc :jumpLR)
              (()
               ())
  (retl)
    (nop))

(define-sparc-vinsn (restore-full-lisp-context :lispcontext :pop :csp :lrRestore)
              (()
               ())
  (ld sparc::%lsp #.(simm13 sparc::lisp-frame.savelr) sparc::%ra0)
  (ld sparc::%lsp #.(simm13 sparc::lisp-frame.savefn) sparc::%fn)
  (ld sparc::%lsp #.(simm13 sparc::lisp-frame.savevsp) sparc::%vsp)
  (ld sparc::%lsp #.(simm13 sparc::lisp-frame.backlink) sparc::%lsp)
  (add sparc::%lsp #.(simm13 -64) sparc::%sp))


(define-sparc-vinsn (restore-full-lisp-context-ool :lispcontext :pop :csp :lrRestore)
              (()
               ())
  (call-subprim* .SPrestorefullcontext)
    (nop))

(define-sparc-vinsn (popj :lispcontext :pop :csp :lrRestore :jumpLR)
              (() 
               ())
  (jump-subprim .SPpopj)
    (nop))

; Exiting from an UNWIND-PROTECT cleanup is similar to
; (and a little simpler than) returning from a function.
(define-sparc-vinsn restore-cleanup-context (()
                                       ())
  (ld sparc::%lsp #.(simm13 sparc::lisp-frame.savelr) sparc::%ra0)
  (ld sparc::%lsp #.(simm13 sparc::lisp-frame.backlink) sparc::%lsp)
  (add sparc::%lsp #.(simm13 -64) sparc::%sp))


(define-sparc-vinsn default-1-arg (()
                             ((min :u16const))
                             ())
  (cmp sparc::%nargs (:apply simm13 (:apply ash min 2)))
  (bne :done)
    (nop)
  ((:pred >= min 3)
   (st sparc::%arg_x sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp))
  ((:pred >= min 2)
   (mov sparc::%arg_y sparc::%arg_x))
  ((:pred >= min 1)
   (mov sparc::%arg_z sparc::%arg_y))
  (mov sparc::%rnil sparc::%arg_z)
  :done)


(define-sparc-vinsn default-2-args (()
                             ((min :u16const))
                             ())
  (cmp sparc::%nargs (:apply simm13 (:apply ash (:apply 1+ min) 2)))
  (bg :done)
    (nop)
  (be :one)
    (nop)
  ; We got "min" args; arg_y & arg_z default to nil
  ((:pred >= min 3)
   (st sparc::%arg_x sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp))
  ((:pred >= min 2)
   (st sparc::%arg_y sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp))
  ((:pred >= min 1)
   (mov sparc::%arg_z sparc::%arg_x))
  (b :last)
    (mov sparc::%rnil sparc::%arg_y)  
   :one
  ; We got min+1 args: arg_y was supplied, arg_z defaults to nil.
  ((:pred >= min 2)
   (st sparc::%arg_x sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp))
  ((:pred >= min 1)
   (mov sparc::%arg_y sparc::%arg_x))
  (mov sparc::%arg_z sparc::%arg_y)
  :last
  (mov sparc::%rnil sparc::%arg_z)
  :done)


(define-sparc-vinsn default-3-args (()
				    ((min :u16const))
				    ())
  (cmp sparc::%nargs (:apply simm13 (:apply ash (:apply + 2 min) 2)))
  (bg :done)
    (nop)
  (be :two)
    (cmp sparc::%nargs (:apply simm13 (:apply ash min 2)))
  (be :none)
    (nop)
  ; The first (of three) &optional args was supplied.
  ((:pred >= min 2)
   (st sparc::%arg_x sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp))
  ((:pred >= min 1)
   (st sparc::%arg_y sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp))
  (b :last-2)
    (mov sparc::%arg_z sparc::%arg_x)
  :two
  ; The first two (of three) &optional args were supplied.
  ((:pred >= min 1)
   (st sparc::%arg_x sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp))
  (mov sparc::%arg_y sparc::%arg_x)
  (b :last-1)
    (mov sparc::%arg_z sparc::%arg_y)
  ; None of the three &optional args was provided.
  :none
  ((:pred >= min 3)
   (st sparc::%arg_x sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp))
  ((:pred >= min 2)
   (st sparc::%arg_y sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp))
  ((:pred >= min 1)
   (st sparc::%arg_z sparc::%vsp #.(simm13 -4))
   (add sparc::%vsp #.(simm13 -4) sparc::%vsp))
  (mov sparc::%rnil sparc::%arg_x)
  :last-2
  (mov sparc::%rnil sparc::%arg_y)
  :last-1
  (mov sparc::%rnil sparc::%arg_z)
  :done)



(define-sparc-vinsn save-lr (()
                       ())
)

;; "n" is the sum of the number of required args + 
;; the number of &optionals.
(define-sparc-vinsn (default-optionals :call :subprim-call) (()
                                                       ((n :u16const)))
  (call-subprim* .SPdefault-optional-args)
    (mov (:apply simm13 (:apply ash n 2)) sparc::%imm0))

; fname contains a known symbol
(define-sparc-vinsn (call-known-symbol :call) (()
					       ())
  (ld sparc::%fname #.(simm13 arch::symbol.fcell) sparc::%nfn)
  (ld sparc::%nfn #.(simm13 arch::misc-data-offset) sparc::%temp0)
  (jmpl sparc::%temp0 #.(simm13 arch::misc-data-offset) sparc::%ra0)
    (nop))

(define-sparc-vinsn (jump-known-symbol :jumplr) (()
                                           ())
  (ld sparc::%fname #.(simm13 arch::symbol.fcell) sparc::%nfn)
  (ld sparc::%nfn #.(simm13 arch::misc-data-offset) sparc::%temp0)
  (jmp sparc::%temp0 #.(simm13 arch::misc-data-offset))
    (nop))


(define-sparc-vinsn (call-known-function :call) (()
                                           ())
  (ld sparc::%nfn #.(simm13 arch::misc-data-offset) sparc::%temp0)
  (jmpl sparc::%temp0 #.(simm13 arch::misc-data-offset) sparc::%ra0)
    (nop))


(define-sparc-vinsn (jump-known-function :jumplr) (()
						   ())
  (ld sparc::%nfn #.(simm13 arch::misc-data-offset) sparc::%temp0)
  (jmp sparc::%temp0 #.(simm13 arch::misc-data-offset))
    (nop))

(define-sparc-vinsn %schar (((char :imm))
                      ((str :lisp)
                       (idx :imm))
                      ((imm :u32)))
  (srl idx #.(simm13 arch::fixnumshift) imm)
  (add imm #.(simm13 arch::misc-data-offset) imm)
  (ldub imm str imm)
  (sll imm #.(simm13 arch::charcode-shift) char)
  (or char #.(simm13 arch::subtag-character) char))

(define-sparc-vinsn add-unboxed-constant (((dest t))
					  ((src t)
					   (const :s32const))
					  ((temp :s32)))
  ((:or (:pred < const (ash -1 12))
	(:pred >= const (ash 1 12)))
   (sethi (:apply ash const -10) temp)
   ((:pred /= (:apply logand (1- (ash 1 10)) const))
    (or temp (:apply simm13 (:apply logand (1- (ash 1 10)) const)) temp))
   (add src temp dest))
  ((:and (:pred >= const (ash -1 12))
	 (:pred < const (ash 1 12)))
   (add src (:apply simm13 const) dest)))

(define-sparc-vinsn add-unboxed (((dest t))
			   ((x t)
			    (y t)))
  (add x y dest))

(define-sparc-vinsn %set-schar (()
                          ((str :lisp)
                           (idx :imm)
                           (char :imm))
                          ((imm :u32)
                           (imm1 :u32)))
  (srl idx #.(simm13 arch::fixnumshift) imm)
  (add imm #.(simm13 arch::misc-data-offset) imm)
  (srl char #.(simm13 arch::charcode-shift) imm1)
  (stb imm1 str imm)
  )

(define-sparc-vinsn %set-scharcode (()
				    ((str :lisp)
				     (idx :imm)
				     (code :imm))
				    ((imm :u32)
				     (imm1 :u32)))
  (srl idx #.(simm13 arch::fixnumshift) imm)
  (add imm #.(simm13 arch::misc-data-offset) imm)
  (srl code #.(simm13 arch::fixnumshift) imm1)
  (stb imm1 str imm))


(define-sparc-vinsn %scharcode (((code :imm))
				((str :lisp)
				 (idx :imm))
				((imm :u32)
				 ))
  (srl idx #.(simm13 arch::fixnumshift) imm)
  (add imm #.(simm13 arch::misc-data-offset) imm)
  (ldub str imm imm)
  (sll imm #.(simm13 arch::fixnumshift) code))



(define-sparc-vinsn (%debug-trap :call) (()
                                         ())
  (ta #.(simm13 sparc::trap-breakpoint))
  )

;;; vinsns dealing with FF calls & callbacks

(define-sparc-vinsn alloc-c-frame (()
                                   ((num-c-arg-words :u16const)))
  ((:pred <= num-c-arg-words 7)
   (sub sparc::%sp #.(simm13 (+ sparc::c-frame.minsiz sparc::lisp-frame.size)) sparc::%sp))
  ((:pred > num-c-arg-words 7)
   (:apply error "Compiler bug: fix alloc-c-frame vinsn")))

(define-sparc-vinsn double-float-to-register-pair (((first :u32)
                                                    (second :u32))
                                                   ((src :double-float)))
  (add sparc::%sp #.(simm13 -8) sparc::%sp)
  (stdf src sparc::%sp #.(simm13 64))
  (ld sparc::%sp #.(simm13 64) first)
  (ld sparc::%sp #.(simm13 68) second)
  (add sparc::%sp #.(simm13 8) sparc::%sp))

(define-sparc-vinsn store-c-arg (()
                                 ((src t)
                                  (paramno :u32const)))
  (st src sparc::%sp (:apply simm13 (:apply + sparc::c-frame.param0 (:apply ash paramno 2)))))
                 
(define-sparc-vinsn store-single-c-arg (()
                                 ((src :single-float)
                                  (paramno :u32const)))
  (stf src sparc::%sp (:apply simm13 (:apply + sparc::c-frame.param0 (:apply ash paramno 2)))))

; In case sparc::*sparc-opcodes* was changed since this file was compiled.
(queue-fixup
 (fixup-vinsn-templates *sparc-vinsn-templates* sparc::*sparc-opcode-numbers*))

(ccl::provide "SPARC-VINSNS")

