/*
 * Copyright (c) 2002 by Louis Zechtzer
 *
 * Permission to use, copy and distribute this software is hereby granted
 * under the terms of version 2 or any later version of the GNU General Public
 * License, as published by the Free Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED IN ITS "AS IS" CONDITION, WITH NO WARRANTY
 * WHATSOEVER. NO LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING
 * FROM THE USE OF THIS SOFTWARE WILL BE ACCEPTED.
 */
/* 
 * Authors: Louis Zechtzer (lou@clarity.net)
 */
#include "net.h"
#include "log.h"
#include "sys.h"
#include "openmosix.h"

#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <fcntl.h>
#include <errno.h>

static void message_handler(net_msg_t *, void *);
static void usage();
	
const char *program_name = "omdiscd";

extern int sys_sighandled;

int 
main(int argc, char *argv[]) 
{
	net_params_t net_handle;
	struct in_addr if_addrs[NET_MAX_IFS];
	int cnt;
	
	static struct option long_opts[] = {
		{ "interface", 1, NULL, 'i' },
		{ "help", 0, NULL, 'h' },
		{ "nodaemon", 0, NULL, 'n' },
		{ "multicast_ttl", 1, NULL, 'm' },
		{ 0, 0, 0, 0 }
	};
	int opt;
	int option_index;

	char *if_names = NULL;
	int opt_nodaemon = 0;	
	int opt_mcast_ttl = 0;

	for (;;) {
		opt = getopt_long(argc, argv, "nhi:m:", long_opts, 
			&option_index);
		if (opt == -1) {
			break;
		}
		switch (opt) {
		case 'i':
			if_names = optarg;
			break;
		case 'n':
			opt_nodaemon = 1;
			break;
		case 'm':
			opt_mcast_ttl = atoi(optarg);
			break;
		case 'h':
		case '?':
			usage();
			exit(1);	
			break;
		};
	}

	/* Initialize system aspects (e.g. daemon and logging) */
	sys_initialize(opt_nodaemon ? SYS_FOREGROUND : SYS_DAEMON); 

	/* Initialize logging facility */
	log_initialize(opt_nodaemon ? LOG_TO_STDERR : LOG_TO_SYSLOG);

	log_set_debug(DEBUG_TRACE_NONE); /* or DEBUG_TRACE_ALL,... */
	
	/* Initialize network transport */
	if (net_initialize(&net_handle, if_names, opt_mcast_ttl) == NET_FAIL) {
		log(OM_LOG_CRITICAL, "Unable to initialize network.  Exiting.");
		log_finalize();
		exit(1);
	}

	/* 
	 * Initialize openMosix specifics.  Initial entries to the kernel's
	 * map should include one node-id and interface, and aliases for 	
	 * each additional interface.
	 */
	net_get_if_addrs(&net_handle, if_addrs, &cnt);
	if (openmosix_initialize(if_addrs, cnt) == OPENMOSIX_FAIL) {
		log(OM_LOG_CRITICAL, "Unable to initialize openMosix."
			"  Exiting.");
		net_finalize(&net_handle);
		log_finalize();
		exit(1);
	}

	net_send_join_msg(&net_handle);
	for (;;) {
		net_recv_msg(&net_handle, &message_handler, 
			(void *)&net_handle);

		/* sys_sighandled global is from sys.c */
		if (sys_sighandled & SYS_GOT_SIGINT) {
			sys_sighandled &= ~SYS_GOT_SIGINT;

			log(OM_LOG_INFO, "Exiting on SIGINT.");
			net_finalize(&net_handle);
			openmosix_finalize();
			log_finalize();
			exit(0);
		}
		if (sys_sighandled & SYS_GOT_SIGHUP) {
			sys_sighandled &= ~SYS_GOT_SIGHUP;
			log(OM_LOG_DEBUG|DEBUG_TRACE_PRINT, "got SIGHUP");
		}
		if (sys_sighandled & SYS_GOT_SIGUSR1) {
			sys_sighandled &= ~SYS_GOT_SIGUSR1;
			log(OM_LOG_DEBUG|DEBUG_TRACE_PRINT, "got SIGUSR1");
			log_toggle_debug();
		}
	}

	return 0;
}


/*
 * message_handler: will be called whenever a multicast message is
 * received.  It practice it should handle adding new nodes to
 * the map.  The network functions pass (a potentially dangling reference) 
 * to a net_msg_t struct and "data" which is the net_params object.
 */
static void 
message_handler(net_msg_t *msg, void *data)
{
	switch(msg->type) {
	case NET_MSG_JOIN:
		log(OM_LOG_DEBUG|DEBUG_TRACE_RECV, "Received join from %x", 
			msg->src.s_addr);

		/*
		 * If routing is enabled, echo message out to all interfaces
		 * except the one which received the message.
		 */
		/*
		net_route_msg(&msg, (net_params_t *)data);
		*/

		/* Notify new node (and all others) of this node's existence */
		net_send_ack_msg((net_params_t *)data);

		if (openmosix_add_node(&msg->src, msg->src_ifs, msg->if_cnt) 
			== OPENMOSIX_FAIL) {

			log(OM_LOG_ALERT|OM_LOG_PERROR, "Unable to add node:"
				" %x to openMosix kernel", 
				ntohl(msg->src.s_addr));
		}


		break;

	case NET_MSG_ACK:
		log(OM_LOG_DEBUG|DEBUG_TRACE_RECV, "Received ack from %x", 
			msg->src.s_addr);
		/* If the node is already in the map, openmosix_add_node will
		   ignore it */
		if (openmosix_add_node(&msg->src, msg->src_ifs, msg->if_cnt) 
			== OPENMOSIX_FAIL) {

			log(OM_LOG_ALERT|OM_LOG_PERROR, "Unable to add node:"
				" %x to openMosix kernel", 
				ntohl(msg->src.s_addr));
		}

		/*
 		 * If routing is enabled, echo the acknowledgment to all
		 * interfaces except the one that received the message 
		 */
		/*
		net_route_msg(&msg);
		*/
		break;

	case NET_MSG_LEAVE:
		log(OM_LOG_DEBUG|DEBUG_TRACE_RECV, "Received leave"
			"notification");
		break;
	}
}

static void
usage()
{
	fprintf(stderr, "Usage: %s [ -i|--interface <interface_name> ]"
		" [ -n|--nodaemon ]\n\t[ -h|--help ]\n", program_name);
}
