/*
 * Copyright (c) 2002 by Louis Zechtzer
 *
 * Permission to use, copy and distribute this software is hereby granted
 * under the terms of version 2 or any later version of the GNU General Public
 * License, as published by the Free Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED IN ITS "AS IS" CONDITION, WITH NO WARRANTY
 * WHATSOEVER. NO LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING
 * FROM THE USE OF THIS SOFTWARE WILL BE ACCEPTED.
 */
/* 
 * Authors: Louis Zechtzer (lou@clarity.net)
 */


#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <arpa/inet.h>
#include <stdint.h>
#include <errno.h>

#include "openmosix.h"

int
main(int argc, char *argv[])
{
	int fd, rb, i, nents;
	char ipv4[16];
	char nodeid[12];
	uint32_t addr;
	struct mosixnet ents[MAX_MOSNET_ENTS];

	fd = open(OPENMOSIX_PROC_MOSPE, O_RDONLY);
	if (fd == -1) {
		fprintf(stderr, "Unable to open %s\n\n", OPENMOSIX_PROC_MOSPE);
	} else {
		rb = read(fd, nodeid, 12);
		if (rb == -1) {
			fprintf(stderr, "Unable to read %s\n\n", 
				OPENMOSIX_PROC_MOSPE);
		} else {
			int id;
			nodeid[(rb >= 12) ? 11 : rb] = '\0';
			sscanf(nodeid, "%d", &id);
			printf("My Node-Id: 0x%04x\n\n", id);
		}
	}
	close(fd);

	fd = open(OPENMOSIX_PROC_CONFIG, O_RDONLY);
	if (fd == -1) {
		if (errno == ENOSYS) {
			fprintf(stderr, "openMosix not configured.\n");
		} else {	
			fprintf(stderr, "Unable to open %s\n", 
				OPENMOSIX_PROC_CONFIG);
			return errno;
		}
	}

	rb = read(fd, &ents, sizeof(struct mosixnet) * MAX_MOSNET_ENTS);
	if (rb == -1) {
		i = errno;
		close(fd);
		fprintf(stderr, "Unable to read %s\n", OPENMOSIX_PROC_CONFIG);
		return i;
	} else if (rb == 0) {
		close(fd);
		printf("No entries openMosix map\n");
		return 0;
	}
	close(fd);

	printf("Base Node-Id Address          Count\n");
	printf("------------ ---------------- -----\n");

	nents = rb / sizeof(struct mosixnet);
	for (i = 0; i < nents; i++) {
		addr = ((struct sockaddr_in *)&ents[i].saddr)->sin_addr.s_addr;
		if (ents[i].cnt == 0) {
			printf("0x%04x       %-16s alias\n", 
				ents[i].base, 
				inet_ntop(AF_INET, &addr, ipv4, 16));
		} else {
			printf("0x%04x       %-16s %d\n", 
				ents[i].base, 
				inet_ntop(AF_INET, &addr, ipv4, 16),
				ents[i].cnt);
		}
	}
	
	return 0;
}
