// $Id: CassettePlayer.hh 5920 2006-12-11 19:59:23Z m9710797 $

#ifndef CASSETTEPLAYER_HH
#define CASSETTEPLAYER_HH

#include "EventListener.hh"
#include "CassetteDevice.hh"
#include "SoundDevice.hh"
#include "Schedulable.hh"
#include "EmuTime.hh"
#include <string>
#include <memory>

namespace openmsx {

class CassetteImage;
class MSXMixer;
class Scheduler;
class MSXCliComm;
class WavWriter;
class LoadingIndicator;
class BooleanSetting;
class TapeCommand;
class MSXCommandController;
class MSXEventDistributor;
class EventDistributor;

class CassettePlayer : public CassetteDevice, public SoundDevice,
                       private EventListener, private Schedulable
{
public:
	CassettePlayer(MSXCommandController& msxCommandController,
	               MSXMixer& mixer, Scheduler& Scheduler,
	               MSXEventDistributor& msxEventDistributor,
	               EventDistributor& eventDistributor,
	               MSXCliComm& cliComm);
	virtual ~CassettePlayer();

	// CassetteDevice
	virtual void setMotor(bool status, const EmuTime& time);
	virtual short readSample(const EmuTime& time);
	virtual void setSignal(bool output, const EmuTime& time);

	// Pluggable
	virtual const std::string& getName() const;
	virtual const std::string& getDescription() const;
	virtual void plugHelper(Connector& connector, const EmuTime& time);
	virtual void unplugHelper(const EmuTime& time);

	// SoundDevice
	virtual void setVolume(int newVolume);
	virtual void setSampleRate(int sampleRate);
	virtual void updateBuffer(
		unsigned length, int* buffer, const EmuTime& start,
		const EmuDuration& sampDur);

private:
	enum State { PLAY, RECORD, STOP };
	State getState() const;
	void setState(State newState, const EmuTime& time);
	static std::string getStateString(State state);
	void setImageName(const std::string& newImage);
	const std::string& getImageName() const;
	void checkInvariants() const;

	/** Insert a tape for use in PLAY mode.
	 */
	void playTape(const std::string& filename, const EmuTime& time);

	/** Removes tape (possibly stops recording). And go to STOP mode.
	 */
	void removeTape(const EmuTime& time);

	/** Goes to RECORD mode using the given filename as a new tape
	  * image. Finishes any old recording session.
	  */
	void recordTape(const std::string& filename, const EmuTime& time);

	/** Rewinds the tape. Also sets PLAY mode, because you can't record
	  * over an existing tape. (And it won't be useful to implement that
	  * anyway.)
	  */
	void rewind(const EmuTime& time);

	/** Enable or disable motor control.
	 */
	void setMotorControl(bool status, const EmuTime& time);

	/** True when the tape is rolling: not in STOP mode, AND [ motorcontrol
	  * is disabled OR motor is on ].
	  */
	bool isRolling() const;

	/** If motor, motorControl or state is changed, this method should
	  * be called to update the end-of-tape syncpoint and the loading
	  * indicator.
	  */
	void updateLoadingState(const EmuTime& time);

	void updateAll(const EmuTime& time);

	void updatePlayPosition(const EmuTime& time);

	/** Returns a random access sample from the inserted tape image,
	  * indexed by tape time.
	  */
	short getSample(const EmuTime& time);

	void fillBuf(size_t length, double x);
	void flushOutput();
	void autoRun();

        // EventListener
	virtual bool signalEvent(shared_ptr<const Event> event);

	// Schedulable
	virtual void executeUntil(const EmuTime& time, int userData);
	virtual const std::string& schedName() const;

	State state;
	std::auto_ptr<CassetteImage> playImage;
	bool motor, motorControl;

	/** The time in the world of the tape. Zero at the start of the tape. */
	EmuTime tapeTime;

	EmuTime recTime;

	/** Last time that tape time was synced with machine time. */
	EmuTime prevTime;

	double lastX; // last unfiltered output
	double lastY; // last filtered output
	double partialOut;
	double partialInterval;
	bool lastOutput;
	size_t sampcnt;
	static const size_t BUF_SIZE = 1024;
	unsigned char buf[BUF_SIZE];

	std::auto_ptr<WavWriter> recordImage;

	MSXCommandController& msxCommandController;

	const std::auto_ptr<TapeCommand> tapeCommand;

	// SoundDevice
	int volume;
	EmuDuration delta;
	EmuTime playTapeTime;
	std::string casImage;

	MSXCliComm& cliComm;
	EventDistributor& eventDistributor;

	std::auto_ptr<LoadingIndicator> loadingIndicator;
	std::auto_ptr<BooleanSetting> autoRunSetting;

	friend class TapeCommand;
};

} // namespace openmsx

#endif
