/* Geoid.java
 * Copyright (C) 1996 by William Giel
 *
 * E-mail: rvdi@usa.nai.net
 * WWW: http://www.nai.net/~rvdi/home.htm
 *
 ***************************************************************************
 * Abstract
 * --------
 * Performs general geodetic computations.
 ***************************************************************************
 * Permission to use, copy, modify, and distribute this software and its
 * documentation without fee for NON-COMMERCIAL purposes is hereby granted.
 * 
 * THE AUTHOR MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY
 * OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. THE AUTHOR SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 ***************************************************************************/

 
///////////////////////////////
//Container for azimuth and distance
///////////////////////////////
class AzimuthDistance
{
	public double distance,azimuth;
		
	AzimuthDistance( double d, double a )
	{
		distance=d;
		azimuth=a;
	}

}


////////////////////////////////////////////////////////////
//Geoid object defines datum and provides methods
//for basic computations. computeInverse is the only
//public method currently implemented.
//
//Reference: Bomford,G., GEODESY, 3rd ed.,repr. 1975, Oxford
////////////////////////////////////////////////////////////
public class Geoid
{
	//////////////////////////////
	//Define some useful stuff...
	//////////////////////////////
	static final double FUZZ = 0.000000000000001;	//Used for fuzzy-testing equality
	static final double dtr= Math.PI/180.0;			//degrees to radians factor
	static final double	rtd= 180.0/Math.PI;			//radians to degrees factor

	///////////////////////////////////
	//Datum definition, and some useful
	//parameters
	///////////////////////////////////
    double 	a,		//semi-major axis
            inv_f,	//inverse of flattening
            f,		//flattening
            esq,	//eccentricity squared
            e2sq;	//second eccentricity squared

	Geoid(double semiMajorAxis, double invFlattening)
	{
		//////////////////////////////////////////////////
		//Typical defining constants in U.S. literature...
		//additional constructors could be used for others
		//////////////////////////////////////////////////
		a=semiMajorAxis;
		inv_f=invFlattening;

		///////////////////////////////////
		//Calculate other useful parameters
		//(Bomford, p.564)
		///////////////////////////////////
		f=1.0/inv_f;
		esq=2.0*f-f*f;
		e2sq=esq/(1.0-esq);
	}

	///////////////////////////////////////
	//Use to test floating point equalities
	///////////////////////////////////////
	private boolean fuzzyEquals(double a, double b)
	{
		return (Math.abs(a - b) < FUZZ);
	}

	//////////////////////////////////////
	//Radius of curvature in the meridian
	//(Bomford, p.565)
	//////////////////////////////////////
	private double rho(double phi)	
	{
		//Phi in radians

		return a*(1-esq) / Math.pow(1 - esq*Math.sin(phi)*Math.sin(phi),1.5);
	}

	///////////////////////////////////////
	//Radius of curvature in prime vertical
	//(Bomford, p.565)
	///////////////////////////////////////
	private double v(double phi)
	{
		//Phi in radians

		return a / Math.sqrt(1 - esq*Math.sin(phi)*Math.sin(phi));
	}

	/////////////////////////////
	//Cunninghams Azimuth Formula
	//(Bomford, pp.135-136)
	/////////////////////////////
	private double CunninghamAzimuth(double lat1, double long1,
													double lat2, double long2)
	{

		////////////////////////////////////////////////////
		//Check for possible trig problems!
		//
		//If lat2 = 90 degrees return azimuth of 0 degrees
		//If lat1 = 90 degrees return azimuth of 180 degrees
		//and if lat1 = 0 degrees, we'll have a problem
		//so we'll resort to Robbins's azimuth formula.
		////////////////////////////////////////////////////
		if( fuzzyEquals(lat2,90.0) ) return (double)0;
		if( fuzzyEquals(lat1,90.0) ) return (double)180;
		if( fuzzyEquals(lat1,0.0) ) return RobbinsAzimuth(lat1, long1, lat2, long2);

		////////////////////////////
		//Convert coords to radians
		////////////////////////////
		lat1 *= dtr; long1 *= dtr;
		lat2 *= dtr; long2 *= dtr;

		double d_lambda = long2 - long1;

		////////////////////////////////////////////////
		//Special consideration for d_lambda=180 degrees
		////////////////////////////////////////////////
		if(fuzzyEquals(d_lambda,Math.PI)){
			if(lat1 > 0 && lat2 > 0) return (double)0;
			if(lat1 < 0 && lat2 < 0) return (double)180;
			if(lat2 < 0){
				if( (Math.PI - lat1 - lat2 ) > Math.PI) return (double)180;
				else return (double)0;
			}
			else{
				if( (Math.PI - lat2 - lat1 ) > Math.PI) return (double)180;
				else return (double)0;
			}
		}

		////////////////////////////////////////////////
		//Special consideration for d_lambda= 0 degrees
		////////////////////////////////////////////////
		if(fuzzyEquals(d_lambda,(double)0)){
			if(lat1 > lat2) return (double)180;
			if(lat1 <= lat2) return (double)0;
		}		
		
		double a12=Math.tan(lat2) / ( (1.0 + e2sq) * Math.tan(lat1) )
					+ esq * v(lat1) * Math.cos(lat1) / ( v(lat2) * Math.cos(lat2) );
		

		//////////////////////////////////////////////////////////
		//Rearranged formula to use sin(d_lambda) in numerator
		//rather than in denominator, resulting in tan(az) rather
		//than cotan(az). Use Math.atan2 to get azimuth in the
		//correct quadrant.
		//////////////////////////////////////////////////////////
		double az=Math.atan2(Math.sin(d_lambda),
					((a12 - Math.cos(d_lambda)) * Math.sin(lat1)))*rtd;

		//////////////////
		//Normalize
		//////////////////
		if(az<0)az+=360.0;
		
		//////////////////////////////////////////
		//Note:Azimuth returned in decimal degrees
		//////////////////////////////////////////
		return az;
	}
		
	/////////////////////////////
	//Robbins's Azimuth Formula
	//(Bomford, pp.136-137)
	/////////////////////////////
	private double RobbinsAzimuth(double lat1, double long1, double lat2, double long2)
	{

		////////////////////////////////////////////////////
		//Check for possible trig problems!
		//
		//If lat2 = 90 degrees return azimuth of 0 degrees
		//If lat1 = 90 degrees return azimuth of 180 degrees
		////////////////////////////////////////////////////
		if( fuzzyEquals(lat2,90.0)) return (double)0;
		if( fuzzyEquals(lat1,90.0)) return (double)180;

		////////////////////////////
		//Convert coords to radians
		////////////////////////////
		lat1 *= dtr; long1 *= dtr;
		lat2 *= dtr; long2 *= dtr;

		double d_lambda = long2 - long1;

		////////////////////////////////////////////////
		//Special consideration for d_lambda=180 degrees
		////////////////////////////////////////////////
		if(fuzzyEquals(d_lambda,Math.PI)){
			if(lat1 > 0 && lat2 > 0) return (double)0;
			if(lat1 < 0 && lat2 < 0) return (double)180;
			if(lat2 < 0){
				if( (Math.PI - lat1 - lat2 ) > Math.PI) return (double)180;
				else return (double)0;
			}
			else{
				if( (Math.PI - lat2 - lat1 ) > Math.PI) return (double)180;
				else return (double)0;
			}
		}

		////////////////////////////////////////////////
		//Special consideration for d_lambda= 0 degrees
		////////////////////////////////////////////////
		if(fuzzyEquals(d_lambda,(double)0)){
			if(lat1 > lat2) return (double)180;
			if(lat1 <= lat2) return (double)0;
		}		

		double tan_u= (1.0 - esq) * Math.tan(lat2) +
						esq * v(lat1) * Math.sin(lat1) / ( v(lat2) * Math.cos(lat2));

		//////////////////////////////////////////////////////////
		//Rearranged formula to use sin(d_lambda) in numerator
		//rather than in denominator, resulting in tan(az) rather
		//than cotan(az). Use Math.atan2 to get azimuth in the
		//correct quadrant.
		//////////////////////////////////////////////////////////
		double az = Math.atan2(Math.sin(d_lambda),
				(Math.cos(lat1) * tan_u - Math.sin(lat1)*Math.cos(d_lambda)))*rtd;

		////////////////////
		//Normalize
		////////////////////
		if(az<0)az+=360.0;

		//////////////////////////////////////////
		//Note:Azimuth returned in decimal degrees
		//////////////////////////////////////////
		return az;
	}

	///////////////////////////////////////////
	//Rudoe's inverse formula
	//(Bomford, p.136)
	///////////////////////////////////////////
	public AzimuthDistance RudoeInverse(	double lat1, double long1,
													double lat2,double long2)
	{
		
			double az12=dtr*CunninghamAzimuth(lat1,long1,lat2,long2);

			////////////////////////////
			//Convert coords to radians
			////////////////////////////
			lat1 *= dtr; long1 *= dtr;
			lat2 *= dtr; long2 *= dtr;

			double e0=e2sq*(Math.cos(lat1)*Math.cos(lat1)*Math.cos(az12)*Math.cos(az12)
									+ Math.sin(lat1)*Math.sin(lat1));

			double b0=(v(lat1)/(1.0 + e0)) * Math.sqrt((1.0 +
					e2sq*Math.cos(lat1)*Math.cos(lat1)*Math.cos(az12)*Math.cos(az12)));

			double u1= Math.atan2(Math.tan(lat1),Math.cos(az12)*Math.sqrt(1.0+e0));

			double d_lambda = long2 - long1;
			double z1= v(lat1)*(1.0-esq)*Math.sin(lat1);
			double z2= v(lat2)*(1.0-esq)*Math.sin(lat2);
			double y2= v(lat2)*Math.cos(lat2)*Math.sin(d_lambda);
			double x2= v(lat2)*Math.cos(lat2)*Math.cos(d_lambda);

			double u2= Math.atan2(v(lat1)*Math.sin(lat1)+(1.0+e0)*(z2-z1),
			 (x2*Math.cos(az12)-y2*Math.sin(lat1)*Math.sin(az12))*Math.sqrt(1.0+e0));

			double c0=1.0+(.25*e0)-((3.0/64.0)*e0*e0)+((5.0/256.0)*e0*e0*e0);

			double c2=-(1.0/8.0)*e0+((1.0/32.0)*e0*e0)-((15.0/1024.0)*e0*e0*e0);

			double c4=-(1.0/256.0)*e0*e0+((3.0/1024.0)*e0*e0*e0);

			double distance= b0*(c0*(u2-u1)+c2*(Math.sin(2.0*u2)-Math.sin(2.0*u1))
								+c4*(Math.sin(4.0*u2)-Math.sin(4.0*u1)));

			return new AzimuthDistance(distance,rtd*az12);
	}
}


