/*************************************************************************
 *
 *  $RCSfile: accessibleeventnotifier.cxx,v $
 *
 *  $Revision: 1.6 $
 *
 *  last change: $Author: vg $ $Date: 2003/05/19 12:57:08 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef COMPHELPER_ACCESSIBLE_EVENT_NOTIFIER
#include <comphelper/accessibleeventnotifier.hxx>
#endif

#ifndef _OSL_DIAGNOSE_H_
#include <osl/diagnose.h>
#endif

#ifndef _COMPHELPER_GUARDING_HXX_
#include <comphelper/guarding.hxx>
#endif

//.........................................................................
namespace comphelper
{
//.........................................................................

	using namespace ::com::sun::star::uno;
	using namespace ::com::sun::star::lang;
	using namespace ::com::sun::star::accessibility;
	using namespace ::comphelper;

	//=====================================================================
	//= AccessibleEventNotifier
	//=====================================================================
	//---------------------------------------------------------------------
	::osl::Mutex						AccessibleEventNotifier::s_aMutex;
	AccessibleEventNotifier::ClientMap	AccessibleEventNotifier::s_aClients;

	//---------------------------------------------------------------------
	AccessibleEventNotifier::TClientId AccessibleEventNotifier::generateId()
	{
		TClientId nBiggestUsedId = 0;
		TClientId nFreeId = 0;

		// look through all registered clients until we find a "gap" in the ids

		// Note that the following relies on the fact the elements in the map are traveled with
		// ascending keys (aka client ids)

		for	(	ClientMap::const_iterator aLookup = s_aClients.begin();
				aLookup != s_aClients.end();
				++aLookup
			)
		{
			TClientId nCurrent = aLookup->first;
			OSL_ENSURE( nCurrent > nBiggestUsedId, "AccessibleEventNotifier::generateId: map is expected to be sorted ascending!" );

			if ( nCurrent - nBiggestUsedId > 1 )
			{	// found a "gap"
				nFreeId = nBiggestUsedId + 1;
				break;
			}

			nBiggestUsedId = nCurrent;
		}

		if ( !nFreeId )
			nFreeId = nBiggestUsedId + 1;

		OSL_ENSURE( s_aClients.end() == s_aClients.find( nFreeId ),
			"AccessibleEventNotifier::generateId: algorithm broken!" );

		return nFreeId;
	}

	//---------------------------------------------------------------------
	AccessibleEventNotifier::TClientId AccessibleEventNotifier::registerClient( )
	{
		::osl::MutexGuard aGuard( s_aMutex );

		// generate a new client id
		TClientId nNewClientId = generateId( );

		// the event listeners for the new client
		EventListeners* pNewListeners = new EventListeners( s_aMutex );
			// note that we're using our own mutex here, so the listener containers for all
			// our clients share this same mutex.
			// this is a reminiscense to the days where the notifier was asynchronous. Today this is
			// completely nonsense, and potentially slowing down the Office me thinks ...

		// add the client
		s_aClients.insert( ClientMap::value_type( nNewClientId, pNewListeners ) );

		// outta here
		return nNewClientId;
	}

	//---------------------------------------------------------------------
	sal_Bool AccessibleEventNotifier::implLookupClient( const TClientId _nClient, ClientMap::iterator& _rPos )
	{
		// look up this client
		_rPos = s_aClients.find( _nClient );
		OSL_ENSURE( s_aClients.end() != _rPos, "AccessibleEventNotifier::implLookupClient: invalid client id (did you register your client?)!" );

		return ( s_aClients.end() != _rPos );
	}

	//---------------------------------------------------------------------
	void AccessibleEventNotifier::revokeClient( const TClientId _nClient )
	{
		::osl::MutexGuard aGuard( s_aMutex );

		ClientMap::iterator aClientPos;
		if ( !implLookupClient( _nClient, aClientPos ) )
			// already asserted in implLookupClient
			return;

		// remove it from the clients map
		delete aClientPos->second;
		s_aClients.erase( aClientPos );
	}

	//---------------------------------------------------------------------
	void AccessibleEventNotifier::revokeClientNotifyDisposing( const TClientId _nClient,
			const Reference< XInterface >& _rxEventSource ) SAL_THROW( ( ) )
	{
		::osl::MutexGuard aGuard( s_aMutex );

		ClientMap::iterator aClientPos;
		if ( !implLookupClient( _nClient, aClientPos ) )
			// already asserted in implLookupClient
			return;

		// notify the "disposing" event for this client
		EventObject aDisposalEvent;
		aDisposalEvent.Source = _rxEventSource;

		// notify the listeners
		EventListeners* pListeners = aClientPos->second;

		// we do not need the entry in the clients map anymore
		// (do this before actually notifying, because some client implementations have re-entrance
		// problems and call into revokeClient while we are notifying from hereing)
		s_aClients.erase( aClientPos );

		// now really do the notification
		pListeners->disposeAndClear( aDisposalEvent );
		delete pListeners;

	}

	//---------------------------------------------------------------------
	sal_Int32 AccessibleEventNotifier::addEventListener(
		const TClientId _nClient, const Reference< XAccessibleEventListener >& _rxListener ) SAL_THROW( ( ) )
	{
		::osl::MutexGuard aGuard( s_aMutex );

		ClientMap::iterator aClientPos;
		if ( !implLookupClient( _nClient, aClientPos ) )
			// already asserted in implLookupClient
			return 0;

		if ( _rxListener.is() )
			aClientPos->second->addInterface( _rxListener );

		return aClientPos->second->getLength();
	}

	//---------------------------------------------------------------------
	sal_Int32 AccessibleEventNotifier::removeEventListener(
		const TClientId _nClient, const Reference< XAccessibleEventListener >& _rxListener ) SAL_THROW( ( ) )
	{
		::osl::MutexGuard aGuard( s_aMutex );

		ClientMap::iterator aClientPos;
		if ( !implLookupClient( _nClient, aClientPos ) )
			// already asserted in implLookupClient
			return 0;

		if ( _rxListener.is() )
			aClientPos->second->removeInterface( _rxListener );

		return aClientPos->second->getLength();
	}

	//---------------------------------------------------------------------
	Sequence< Reference< XInterface > > AccessibleEventNotifier::getEventListeners( const TClientId _nClient ) SAL_THROW( ( ) )
	{
		Sequence< Reference< XInterface > > aListeners;

		::osl::MutexGuard aGuard( s_aMutex );

		ClientMap::iterator aClientPos;
		if ( implLookupClient( _nClient, aClientPos ) )
			aListeners = aClientPos->second->getElements();

		return aListeners;
	}

	//---------------------------------------------------------------------
	void AccessibleEventNotifier::addEvent( const TClientId _nClient, const AccessibleEventObject& _rEvent ) SAL_THROW( ( ) )
	{
		Sequence< Reference< XInterface > > aListeners;

		// --- <mutex lock> -------------------------------
		{
			::osl::MutexGuard aGuard( s_aMutex );

			ClientMap::iterator aClientPos;
			if ( !implLookupClient( _nClient, aClientPos ) )
				// already asserted in implLookupClient
				return;

			// since we're synchronous, again, we want to notify immediately
			aListeners = aClientPos->second->getElements();
		}
		// --- </mutex lock> ------------------------------

			// default handling: loop through all listeners, and notify them
		const Reference< XInterface >* pListeners = aListeners.getConstArray();
		const Reference< XInterface >* pListenersEnd = pListeners + aListeners.getLength();
		while ( pListeners != pListenersEnd )
		{
			try
			{
				static_cast< XAccessibleEventListener* >( pListeners->get() )->notifyEvent( _rEvent );
			}
			catch( const Exception& e )
			{
				e;
				// silent this
				// no assertion, because a broken access remote bridge or something like this
				// can cause this exception
			}
			++pListeners;
		}
	}

//.........................................................................
}	// namespace comphelper
//.........................................................................

